clc
clear
% function [execution_time, margin , time] = simulate()
    
    tic
   
    %Define simulation parameters
    startTime = datetime(2021,12,10,18,27,57);   % 10 December 2021, 6:27:57 PM UTC
    stopTime = startTime + hours(0.5);           
    sampleTime = 15;                           % Seconds
    sc = satelliteScenario(startTime,stopTime,sampleTime,"AutoSimulate",false);
   
    %Add the satellites
    sat = satellite(sc,"1000sats.tle");
    
  
    %Define the ground stations
    gsSource = groundStation(sc,42.3001,-71.3504, ... % Latitude and longitude in degrees
        "Name","Source Ground Station");
    gsTarget = groundStation(sc,17.4351,78.3824, ...  % Latitude and longitude in degrees
        "Name","Target Ground Station");
    
    % Calculate the scenario state corresponding to StartTime.
    advance(sc);
    
    % Retrieve the elevation angle of each satellite with respect to the ground
    % stations.
    [~,elSourceToSat] = aer(gsSource,sat);
    [~,elTargetToSat] = aer(gsTarget,sat);
    
    % Determine the elevation angles that are greater than or equal to 30
    % degrees.
    elSourceToSatGreaterThanOrEqual30 = (elSourceToSat >= 30)';
    elTargetToSatGreaterThanOrEqual30 = (elTargetToSat >= 30)';
    
    % Find the indices of the elements of elSourceToSatGreaterThanOrEqual30
    % whose value is true.
    trueID = find(elSourceToSatGreaterThanOrEqual30 == true);
    
    % These indices are essentially the indices of satellites in sat whose
    % elevation angle with respect to "Source Ground Station" is at least 30
    % degrees. Determine the range of these satellites to "Target Ground
    % Station".
    [~,~,r] = aer(sat(trueID), gsTarget);
    
    % Determine the index of the element in r bearing the minimum value.
    [~,minRangeID] = min(r);
    
    % Determine the element in trueID at the index minRangeID.
    id = trueID(minRangeID);
    
    % This is the index of the best satellite for initial access to the
    % constellation. This will be the first hop in the path. Initialize a
    % variable 'node' that stores the first two nodes of the routing - namely,
    % "Source Ground Station" and the best satellite for initial constellation
    % access.
    nodes = {gsSource sat(id)};
    
    earthRadius = 6378137;                                                   % meters
    altitude = 500000;                                                       % meters
    horizonElevationAngle = asind(earthRadius/(earthRadius + altitude)) - 90; % degrees
    
    
    minSatElevation = -15; % degrees
    
    % Flag to specify if the complete multi-hop path has been found.
    pathFound = false;
    
    % Determine nodes of the path in a loop. Exit the loop once the complete
    % multi-hop path has been found.
    while ~pathFound
        % Index of the satellite in sat corresponding to current node is
        % updated to the value calculated as index for the next node in the
        % prior loop iteration. Essentially, the satellite in the next node in
        % prior iteration becomes the satellite in the current node in this
        % iteration.
        idCurrent = id;
    
        % This is the index of the element in elTargetToSatGreaterThanOrEqual30
        % tells if the elevation angle of this satellite is at least 30 degrees
        % with respect to "Target Ground Station". If this element is true, the
        % routing is complete, and the next node is the target ground station.
        if elTargetToSatGreaterThanOrEqual30(idCurrent)
            nodes = {nodes{:} gsTarget}; %#ok<CCAT> 
            pathFound = true;
            continue
        end
    
        % If the element is false, the path is not complete yet. The next node
        % in the path must be determined from the constellation. Determine
        % which satellites have elevation angle that is greater than or equal
        % to -15 degrees with respect to the current node. To do this, first
        % determine the elevation angle of each satellite with respect to the
        % current node.
        [~,els] = aer(sat(idCurrent),sat); 
    
        % Overwrite the elevation angle of the satellite with respect to itself
        % to be -90 degrees to ensure it does not get re-selected as the next
        % node.
        els(idCurrent) = -90; 
    
        % Determine the elevation angles that are greater than or equal to -15
        % degrees.
        s = els >= minSatElevation;
    
        % Find the indices of the elements in s whose value is true.
        trueID = find(s == true);
    
        % These indices are essentially the indices of satellites in sat whose
        % elevation angle with respect to the current node is greater than or
        % equal to -15 degrees. Determine the range of these satellites to
        % "Target Ground Station".
        [~,~,r] = aer(sat(trueID), gsTarget);
    
        % Determine the index of the element in r bearing the minimum value.
        [~,minRangeID] = min(r);
    
        % Determine the element in trueID at the index minRangeID.
        id = trueID(minRangeID);
    
        % This is the index of the best satellite among those in sat to be used
        % for the next node in the path. Append this satellite to the 'nodes'
        % variable.
        nodes = {nodes{:} sat(id)}; %#ok<CCAT>
    end
    
    sc.AutoSimulate = true;
    % ac = access(nodes{:});
    % 
    % ac.LineColor = "red";
    % % 
    % accessIntervals(ac);
    
    v = satelliteScenarioViewer(sc,"ShowDetails",false);
    sat.MarkerSize = 6; % Pixels
    campos(v,60,5);     % Latitude and longitude in degrees
     
    play(sc);
   
    
    %----------------------------------------------------------------------------------
    %Add Link Between Nodes%
    %----------------------------------------------------------------------------------

    gimbalgs1 = gimbal(gsSource);
    gimbalgs2 = gimbal(gsTarget);
    
    %----------------------------------------------------------------------------------
    %Add a transmitter to ground station gs1.%
    %----------------------------------------------------------------------------------
    frequency = 30e9;                                                                          % Hz
    power = 40;                                                                                % dBW
    bitRate = 20;                                                                              % Mbps
    txGs1 = transmitter(gimbalgs1,Name="Ground Station 1 Transmitter",Frequency=frequency, ...
            Power=power,BitRate=bitRate);
    
    link_nodes{1}=txGs1;
    %----------------------------------------------------------------------------------
    %RECEIVER AT GS2%
    %----------------------------------------------------------------------------------
    requiredEbNo = 1;                                                                     % dB
    rxGs2 = receiver(gimbalgs2,Name="Ground Station 2 Receiver",RequiredEbNo=requiredEbNo);
    
    dishDiameter = 5;                                % meters
    gaussianAntenna(txGs1,DishDiameter=dishDiameter);
    gaussianAntenna(rxGs2,DishDiameter=dishDiameter);
    
    sat2=nodes(2:end-1);
    %Add gimbals to the satellites
    % p_delay = 0;
    for i=1:length(sat2)
            % disp(sat2{i})
            % nodes = {nodes{:} sat2{i}};
            % p_delay = p_delay + max(latency(nodes{i},nodes{i+1}));
            sat_Rx = receiver(gimbal(sat2{i}), "Name","Satellite Receiver",...
            "GainToNoiseTemperatureRatio",5, ... % decibels/Kelvin
            "SystemLoss",3);
            gaussianAntenna(sat_Rx,"DishDiameter",0.5,"ApertureEfficiency",0.5); 
            link_nodes = {link_nodes{:} sat_Rx} ;
                                                                          
            sat_Tx = transmitter(gimbal(sat2{i}), ...
            "Frequency",27e9, ...             % hertz
            "BitRate",20,...
            "Power",20,...
            "SystemLoss",3 );
            gaussianAntenna(sat_Tx,"DishDiameter",0.5,"ApertureEfficiency",0.5); 
            link_nodes = {link_nodes{:} sat_Tx} ;
    end
    
    link_nodes{end+1} = rxGs2;
    %----------------------------------------------------------------------------------
    %POINY AT%
    %----------------------------------------------------------------------------------
    %link 1 & 2
    pointAt(gimbalgs1,sat2{1});
    pointAt(nodes{2}.Gimbals(1),gsSource);
    
    %Point the nodes to each other
    for i=2:length(sat2)
        pointAt(nodes{i}.Gimbals(2),sat2{i});
        pointAt(nodes{i+1}.Gimbals(1),sat2{i-1});
    end
    
    %link n-1 , n
    pointAt(nodes{end-1}.Gimbals(2),gsTarget);
    pointAt(gimbalgs2,nodes{end-1});
    
    %----------------------------------------------------------------------------------
    %Create and analyze links between each pair of nodes%
    %----------------------------------------------------------------------------------
    
    for i = 1:2:length(link_nodes)-1
        % Create a link between the current node and the next node
        currentLink = link(link_nodes{i}, link_nodes{i+1});
        each_link=linkIntervals(currentLink);
        % Analyze the link
        % Check if the link is established
        status = linkStatus(currentLink);
    end
    
    lnk = link(link_nodes{:});
    linkIntervals(lnk)
    
    %--------------------------------------------------------------------------------------------
    %CALCULATE AND PLOT EB/N0%
    %--------------------------------------------------------------------------------------------
    % figure
    [e, time] = ebno(lnk);
    margin = e - rxGs2.RequiredEbNo;
    plot(time,margin,"LineWidth",2);
    xlabel("Time(hour)");
    ylabel("Link Margin (dB)");

    grid on;

    % S = whos ;
    % b=0;
    % for k = 1:length(S)
    %    b = b + S(k).bytes ;
    % end
    % display(b);

    execution_time = toc;  % Get the elapsed time
    disp(['Execution time  = ' , num2str(execution_time)])
   
 % end

