//@ts-check
/*
  Copyright: (c) 2018-2020, Guilherme Francescon Cittolin <gfcittolin@gmail.com>
  GNU General Public License v3.0+ (see LICENSE or https://www.gnu.org/licenses/gpl-3.0.txt)
*/

const util = require('util');
const debug = util.debuglog('nodes7');
//@ts-ignore
const constants = require('../constants.json');

const NodeS7Error = require('../errors.js');

/**
 * Regex to match the address format of addresses
 * Match 1: DB Number
 * Match 2: Type
 * Match 3: Address
 * Match 4: Bit address / Array length
 * Match 5: Array length (for fields that need bit address)
 */
const REGEX_NODES7_ADDR = /^(?:DB(\d+),)?([A-Z]+)(\d+)(?:\.(\d+))?(?:\.(\d+))?$/;

/**
 * Parse address strings according to NodeS7 rules
 */
class AddressParserNodeS7 {

    /**
     * 
     * @param {string} address the address to be parsed
     * @throws {NodeS7Error} ERR_PARSE_ADDR_OFFSET - Byte offset of an address is invalid
     * @throws {NodeS7Error} ERR_PARSE_AREA - Area addressed is unknown or invalid
     * @throws {NodeS7Error} ERR_PARSE_BIT_OFFSET - Bit offset is missing or is invalid
     * @throws {NodeS7Error} ERR_PARSE_DATATYPE - Datatype is unknown or invalid
     * @throws {NodeS7Error} ERR_PARSE_DB_DATATYPE - Datatype of a DB area is unknown or invalid
     * @throws {NodeS7Error} ERR_PARSE_DB_NUMBER - Number of a DB is unknown or invalid
     * @throws {NodeS7Error} ERR_PARSE_INVALID_ARR_LEN - Array length of an array specification is invalid
     * @throws {NodeS7Error} ERR_PARSE_INVALID_BIT_OFFSET - Bit offset is specified in a type that doesn't support it
     * @throws {NodeS7Error} ERR_PARSE_STRING_LEN - String length specified is missing or is invalid
     * @throws {NodeS7Error} ERR_PARSE_UNKNOWN_FORMAT - Basic format of a NODES7 address format cannot be identified
     */
    parse(address) {
        debug("S7Item parseAddress_NodeS7", address);

        let match = address.match(REGEX_NODES7_ADDR);
        if (!match) {
            throw new NodeS7Error('ERR_PARSE_UNKNOWN_FORMAT', `Could not parse item "${address}", invalid address format`, { item: address });
        }

        debug("S7Item parseAddress_NodeS7 match", match);

        // variables for the fields extracted from the address string
        let match_db = match[1];
        let match_area = match[2];
        let match_addr = match[3];
        let match_bitAddr = parseInt(match[4]);
        let match_arrLen = parseInt(match[5]);

        // variables for the parsed and validated fields
        let addrType, dataType, dataTypeLength, addressOffset,
            bitAddressOffset, arrayLength, dbNumber,
            readTransportCode, writeTransportCode, areaCode,
            byteLength, byteLengthWrite, byteLengthWithFill;

        /* Parse S7 Address */

        // test address offset
        addressOffset = parseInt(match_addr)
        if (isNaN(addressOffset)) {
            throw new NodeS7Error('ERR_PARSE_ADDR_OFFSET', `Invalid address offset on "${address}"`, { item: address });
        }

        // parse DB number
        if (match_db) {
            dbNumber = parseInt(match_db);

            if (dbNumber < 1 || isNaN(dbNumber)) {
                throw new NodeS7Error('ERR_PARSE_DB_NUMBER', `Invalid DB Number on "${address}"`, { item: address });
            }
        }

        // extract addrType and dataType
        if (dbNumber) {

            addrType = "DB";

            // validate data type
            switch (match_area) {
                case "X":
                case "BYTE":
                case "CHAR":
                case "STRING":
                case "INT":
                case "DINT":
                case "WORD":
                case "DWORD":
                case "REAL":
                case "RINT":
                case "RDINT":
                case "RWORD":
                case "RDWORD":
                case "RREAL":
                case "DT":
                case "DTZ":
                case "DTL":
                case "DTLZ":
                    dataType = match_area;
                    break;
                case "B":
                    dataType = "BYTE";
                    break;
                case "C":
                    dataType = "CHAR";
                    break;
                case "S":
                    dataType = "STRING";
                    break;
                case "I":
                    dataType = "INT";
                    break;
                case "DI":
                    dataType = "DINT";
                    break;
                case "W":
                    dataType = "WORD";
                    break;
                case "D":
                case "DW":
                    dataType = "DWORD";
                    break;
                case "R":
                    dataType = "REAL";
                    break;
                case "RI":
                    dataType = "RINT";
                    break;
                case "RDI":
                    dataType = "RDINT";
                    break;
                case "RW":
                    dataType = "RWORD";
                    break;
                case "RD":
                case "RDW":
                    dataType = "RDWORD";
                    break;
                case "RR":
                    dataType = "RREAL";
                    break;
                default:
                    throw new NodeS7Error('ERR_PARSE_DB_DATATYPE', `Unknown DB data type "${match_area}" for address "${address}"`, { item: address });
            }

        } else {

            // the first char indicates the PLC area (I, Q, M, P, ...)
            addrType = match_area.charAt(0);
            // the data type (if area is P, skip the first char (I,E,Q,A))
            dataType = match_area.substr(addrType === "P" ? 2 : 1);

            // validate address type
            switch (addrType) {
                case "E":
                    addrType = "I";
                    break;
                case "A":
                    addrType = "Q";
                case "P":
                case "I":
                case "Q":
                case "M":
                case "T":
                case "C":
                    break;
                default:
                    throw new NodeS7Error('ERR_PARSE_AREA', `Unknown address type "${addrType}" for address "${address}"`, { item: address });
            }

            // validate data type
            switch (dataType) {
                case "":
                    if (addrType === "T") {
                        dataType = "TIMER";
                    } else if (addrType === "C") {
                        dataType = "COUNTER";
                    } else {
                        dataType = "X";
                    }
                    break;
                case "B":
                    dataType = "BYTE";
                    break;
                case "C":
                    dataType = "CHAR";
                    break;
                case "I":
                    dataType = "INT";
                    break;
                case "DI":
                    dataType = "DINT";
                    break;
                case "W":
                    dataType = "WORD";
                    break;
                case "D":
                case "DW":
                    dataType = "DWORD";
                    break;
                case "R":
                    dataType = "REAL";
                    break;
                case "RI":
                    dataType = "RINT";
                    break;
                case "RDI":
                    dataType = "RDINT";
                    break;
                case "RW":
                    dataType = "RWORD";
                    break;
                case "RD":
                case "RDW":
                    dataType = "RDWORD";
                    break;
                case "RR":
                    dataType = "RREAL";
                    break;
                default:
                    throw new NodeS7Error('ERR_PARSE_DATATYPE', `Unknown data type "${dataType}" for address "${address}"`, { item: address });
            }
        }

        // handle array lengths and bit address
        if (dataType === "X") {
            if (isNaN(match_bitAddr)) {
                throw new NodeS7Error('ERR_PARSE_BIT_OFFSET', `Bit address offset required for data type "X" on "${address}"`, { item: address });
            }
            if (match_bitAddr > 7) {
                throw new NodeS7Error('ERR_PARSE_BIT_OFFSET', `Bit address offset out of range 0-7 on "${address}"`, { item: address });
            }

            bitAddressOffset = match_bitAddr;
            arrayLength = match_arrLen;

            if (isNaN(arrayLength) || arrayLength < 1) {
                arrayLength = 1;
            }

        } else if (dataType === "STRING") {
            // match_bitAddr is the string length for string types
            if (isNaN(match_bitAddr) || match_bitAddr < 1) {
                throw new NodeS7Error('ERR_PARSE_STRING_LEN', `String length required for data type "STRING" on "${address}"`, { item: address });
            }

            dataTypeLength = match_bitAddr + 2; //strings have 2 extra bytes for string length
            bitAddressOffset = 0;
            arrayLength = match_arrLen;

            if (isNaN(arrayLength) || arrayLength < 1) {
                arrayLength = 1;
            }

        } else {
            if (!isNaN(match_arrLen)) {
                // the array length should be at the bitAddr field, this is a syntax error
                throw new NodeS7Error('ERR_PARSE_INVALID_BIT_OFFSET', `Invalid use of bit address offset on "${address}"`, { item: address });
            }

            bitAddressOffset = 0;

            // the bitAddr field is actually the array length
            if (!isNaN(match_bitAddr)) {
                arrayLength = match_bitAddr;

                if (arrayLength < 1) {
                    throw new NodeS7Error('ERR_PARSE_INVALID_ARR_LEN', `Invalid array length on "${address}"`, { item: address });
                }
            } else {
                arrayLength = 1;
            }
        }

        switch (dataType) {
            case "DTL":
            case "DTLZ":
                dataTypeLength = 12;
                break;
            case "DT":
            case "DTZ":
                dataTypeLength = 8;
                break;
            case "REAL":
            case "DWORD":
            case "DINT":
            case "RREAL":
            case "RDWORD":
            case "RDINT":
                dataTypeLength = 4;
                break;
            case "INT":
            case "WORD":
            case "RINT":
            case "RWORD":
            case "TIMER":
            case "COUNTER":
                dataTypeLength = 2;
                break;
            case "X":
            case "BYTE":
            case "CHAR":
                dataTypeLength = 1;
                break;
            case "STRING":
                // For strings, arrayLength and dtypelen were assigned during parsing.
                break;
            default:
                // we have validated the dataType before, so we should never reach this
                throw new NodeS7Error('ERR_PARSE_DATATYPE', `Cannot determine the length of type "${dataType}" for address "${address}"`, { item: address });
        }

        // set area/transport codes accordingly

        //readTransportCode = 0x04; //TODO - this is WORD from the original code
        readTransportCode = constants.proto.transport.BYTE;
        writeTransportCode = constants.proto.dataTransport.BBYTE;

        switch (addrType) {
            case "DB":
            case "DI":
                areaCode = constants.proto.area.DB;
                break;
            case "I":
            case "E":
                areaCode = constants.proto.area.INPUTS;
                break;
            case "Q":
            case "A":
                areaCode = constants.proto.area.OUTPUTS;
                break;
            case "M":
                areaCode = constants.proto.area.FLAGS;
                break;
            case "P":
                areaCode = constants.proto.area.PERIPHALS;
                break;
            case "C":
                areaCode = constants.proto.area.COUNTER;
                readTransportCode = constants.proto.transport.DATE;
                break;
            case "T":
                areaCode = constants.proto.area.TIMER;
                readTransportCode = constants.proto.transport.DATE;
                break;
            default:
                throw new NodeS7Error('ERR_PARSE_AREA', `Cannot determine area codes for address "${address}"`, { item: address });
        }

        if (dataType === 'X') {
            writeTransportCode = constants.proto.dataTransport.BBIT;
        }

        // determine the length in bytes of this item

        if (dataType === 'X') {
            byteLength = Math.ceil((bitAddressOffset + arrayLength) / 8);
        } else {
            byteLength = arrayLength * dataTypeLength;
        }

        // for writes, boolen values use a whole byte for each
        byteLengthWrite = arrayLength * dataTypeLength;

        byteLengthWithFill = byteLength;
        byteLengthWithFill += byteLengthWithFill % 2;

        let result = {
            addrtype: addrType,
            datatype: dataType,
            dtypelen: dataTypeLength,
            offset: addressOffset,
            bitOffset: bitAddressOffset,
            arrayLength: arrayLength,
            dbNumber: dbNumber,
            readTransportCode: readTransportCode,
            writeTransportCode: writeTransportCode,
            areaCode: areaCode,
            byteLength: byteLength,
            byteLengthWrite: byteLengthWrite,
            byteLengthWithFill: byteLengthWithFill
        };

        debug("S7Item parseAddress_NodeS7 result", result);
        return result;
    }
}

module.exports = new AddressParserNodeS7();