function output = MLP_builder(action, MLP_struct, varargin)

% Check if the action is to initialize the structure
if strcmp(action, 'initialize')

    % Initialize the structure with default values
    MLP_struct = struct( ...
        'File_csv_name', 'yourfile.csv', ...          % Specify the dataset file name
        'ShuffleSeed', 16, ...                        % Specify the seed for shuffling
        'ShuffleRate', 1, ...                         % Specify the shuffling rate
        'InputSize', 2, ...                           % Specify the size of the input layer
        'OutputSize', 2, ...                          % Specify the size of the output layer
        'OutputActivationFun', {''}, ...              % Specify the activation function for the output layer
        'HiddenSizes', [], ...                        % Specify the sizes of hidden layers
        'HiddenActivationFun', {''}, ...              % Specify the activation function for hidden layers
        'LearningRate', 0.01, ...                     % Specify the learning rate
        'Momentum', 0.9, ...                          % Specify the decay of the momentum
        'SumSquaredWeight', 0.9, ...                  % Specify the sum squared weight
        'BatchSize', 32, ...                          % Specify the batch size
        'Epochs', 10, ...                             % Specify the number of epochs
        'OnlineLossesINFO',true, ...                  % Specify whether the cmd updates about  losses
        'FinalPlotLosses', true); ...                 % Specify whether to plot losses at the end

    % Output the initialized structure
    output = MLP_struct;

    % Check if the action is to train the structure
elseif strcmp(action, 'train')

    % Check if the structure is consistent
    consistency_check(MLP_struct)

    % Perform training based on the provided structure
    MLP_training(MLP_struct);


else
    error('Invalid action. Use ''initialize'' or ''train''.');

end
end

% ======= MAIN FUNCTION =======

function [W_conc, b_conc, decoding_matrix, scale] = MLP_training(inputStruct)
% MLP_training - Main function to manage MLP training.
%
%   [W_conc, b_conc, decoding_matrix, scale] = MLP_training(inputStruct) 
%   performs the entire training process, including data preparation, 
%   file handling, Python integration, and final evaluation through plots.
%
%   Parameters:
%       - inputStruct: Structure containing MLP training parameters.
%
%   Outputs:
%       - W_conc: Concatenated weights matrices.
%       - b_conc: Concatenated biases vectors.
%       - decoding_matrix: Matrix encoding MLP architecture.
%       - scale: Scaling factors.
%
%   Example:
%       [W, b, decoding, scaling] = MLP_training(myStruct);
%
%   See also: code_activation, losses_plot, MLP_python2matlab
%
%   Author: Marino Massimo Costantini

% Re-arranging data
MLP_py_shape= [inputStruct.InputSize, inputStruct.HiddenSizes, inputStruct.OutputSize]';
MLP_py_activation_functions=[inputStruct.HiddenActivationFun,inputStruct.OutputActivationFun];

% Preparing text file for writing in Python
fid = fopen('history_losses.txt', 'w');
fclose(fid);



% Online update about training option
OnlineLosses(inputStruct.OnlineLossesINFO)


disp('-------- Python Processing START --------');
tic;
[MLP_dict,maxes,losses] = pyrunfile("MLP_builder.py", ...
    ["final_MLP", ...
    "maxes", ...
    "losses"], ...
    mode='train', ...
    dataset_file_name=py.str(inputStruct.File_csv_name), ...
    percentage=py.float(inputStruct.ShuffleRate), ...
    seed=py.int(inputStruct.ShuffleSeed), ...
    MLP_shape=py.list(int64(MLP_py_shape)), ...
    activation_functions=MLP_py_activation_functions, ...
    learning_rate=inputStruct.LearningRate, ...
    momentum=inputStruct.Momentum, ...
    sum_squared_weight=inputStruct.SumSquaredWeight, ...
    batch_size=py.int(inputStruct.BatchSize), ...
    max_epoch_N=py.int(inputStruct.Epochs));

% Something went wrong
if islogical(MLP_dict)
    error('EXIT. MLP did not create.')
end

% Final plot option
if inputStruct.FinalPlotLosses
    
    losses_plot(losses, ...
        inputStruct.Epochs, ...
        inputStruct.LearningRate, ...
        inputStruct.BatchSize, ...
        inputStruct.Momentum, ...
        inputStruct.SumSquaredWeight, ...
        inputStruct.HiddenActivationFun, ...
        inputStruct.OutputActivationFun, ...
        MLP_py_shape)
end



scale=double(maxes);
type_activation = code_activation(length(MLP_py_shape),MLP_py_activation_functions);
decoding_matrix = [MLP_py_shape, type_activation];
[W_conc,b_conc,MLP_struct] = MLP_python2matlab(MLP_py_shape,MLP_dict);

save MLP_Matlab_data scale W_conc b_conc decoding_matrix
save MLP_Python_data scale MLP_py_shape MLP_struct MLP_py_activation_functions

end

% ==== SECONDARY NESTED FUNCTIONS ====

function OnlineLosses(settings)
if settings
    filename = 'OnlineLosses.exe';
    fullFilePath = fullfile(pwd, filename);  % Costruisci il percorso completo del file nel current folder

    if exist(fullFilePath, 'file') == 2
        disp(['The file', filename, ' is already present in the current folder.']);
    else
        disp(['The file ', filename, 'is not yet present in the current folder.']);
        copyfile(which(filename),".")
        disp('File added')
    end

    !OnlineLosses -f history_losses.txt &
end
end

function consistency_check(MLP_struct)

% consistency_check - Verify the integrity of the provided MLP struct.
%
%   consistency_check(MLP_struct) performs checks on the fields of the MLP
%   struct to ensure that they meet the required criteria. This function
%   does not return any output.
%
%   Parameters:
%       - MLP_struct: The struct describing the Multi-Layer Perceptron.
%
%   Example:
%       consistency_check(myMLP);
%
%   See also: MLP_builder
%
%   Author: Marino Massimo Costantini

disp('-------- MatLab preliminary check --------')

if exist("MLP_builder.py","file")==0
    error('File "MLP_builder.py" not found. Toolchain not available.')
end


if      ~isempty(MLP_struct.File_csv_name) && ...
        ~isempty(MLP_struct.ShuffleSeed) && ...
        ~isempty(MLP_struct.ShuffleRate) && ...
        ~isempty(MLP_struct.InputSize) && ...
        ~isempty(MLP_struct.OutputSize) && ...
        ~isempty(MLP_struct.OutputActivationFun) && ...
        ~isempty(MLP_struct.HiddenSizes) && ...
        ~isempty(MLP_struct.HiddenActivationFun) && ...
        ~isempty(MLP_struct.LearningRate) && ...
        ~isempty(MLP_struct.Momentum) && ...
        ~isempty(MLP_struct.SumSquaredWeight) && ...
        ~isempty(MLP_struct.BatchSize) && ...
        ~isempty(MLP_struct.Epochs) && ...
        ~isempty(MLP_struct.OnlineLossesINFO) && ...
        ~isempty(MLP_struct.FinalPlotLosses)

    % File name check.
    if ~ischar(MLP_struct.File_csv_name)
        error('The datatype of the field ".File_csv_name" must be a string.')
    end
    
    % Seed value check.
    if round(MLP_struct.ShuffleSeed) ~= MLP_struct.ShuffleSeed
       error(['The field ".ShuffleSeed" must be an integer. Detected value ',num2str(MLP_struct.ShuffleSeed),' instead.'])
    end

    % Shuffle Rate value check.
    if MLP_struct.ShuffleRate < 0 || MLP_struct.ShuffleRate > 1
        error('Shuffle rate must be in the range of 0 to 1')
    end

    % Input Size check.
    if MLP_struct.InputSize < 1
        error('InputSize must be at least 1');
    elseif round(MLP_struct.InputSize) ~= MLP_struct.InputSize
        error(['The field ".InputSize" must be an integer. Detected value ',num2str(MLP_struct.InputSize),' instead.'])
    end
    
    % Hidden layer sizes check.
    if ~isempty(find(MLP_struct.HiddenSizes<1, 1))
        error(['Each hidden layer size value must be at least 1. Found zero(s) in HiddenSizes: ',num2str(MLP_struct.HiddenSizes)]);
    end
    if ~isempty(find(round(MLP_struct.HiddenSizes)~=MLP_struct.HiddenSizes, 1))
        error(['The field ".HiddenSizes" must be an integer vector. Detected value ',num2str(MLP_struct.HiddenSizes),' instead.'])
    end
    
    % Output Size check.
    if MLP_struct.OutputSize < 1 || MLP_struct.OutputSize > 10
        error('OutputSize must be in the range between 1 and 10');
    elseif round(MLP_struct.OutputSize) ~= MLP_struct.OutputSize
        error(['The field ".OutputSize" must be an integer. Detected value ',num2str(MLP_struct.OutputSize),' instead.'])
    end
    
    % Learning rate check.
    if MLP_struct.LearningRate <= 0
        error('The learning rate must be strictly positive');
    end

    % Momentum decay check.
    if MLP_struct.Momentum < 0 || MLP_struct.Momentum > 1
        error('The field ".Momentum" must be in the range of 0 to 1');
    end
    
    % Sum Squared Weight check.
    if MLP_struct.SumSquaredWeight < 0 || MLP_struct.SumSquaredWeight > 1
        error('The field ".SumSquaredWeight" must be in the range of 0 to 1');
    end

    % Epoch number check.
    if MLP_struct.Epochs < 2
        error('The number of epochs must be at least 2');
    elseif round(MLP_struct.Epochs) ~= MLP_struct.Epochs
        error(['The field ".Epochs" must be an integer. Detected value ',num2str(MLP_struct.Epochs),' instead.'])
    end
    
    % Batch size check.
    if MLP_struct.BatchSize <= 0
        error('The batch size must be strictly positive');
    elseif round(MLP_struct.BatchSize) ~= MLP_struct.BatchSize
        error(['The field ".BatchSize" must be an integer. Detected value ',num2str(MLP_struct.BatchSize),' instead.'])
    end

    % Online loss updating option check
    if ~islogical(MLP_struct.OnlineLossesINFO)
        error(['The field ".OnlineLossesINFO" must be "true" or "false". Detected value ',num2str(MLP_struct.OnlineLossesINFO),' instead.'])
    elseif exist('OnlineLosses.exe','file') == 0
        error('File "OnlineLosses.exe" not found. OnlineLossesINFO not available.')
    end

    % Plot option check
    if ~islogical(MLP_struct.FinalPlotLosses)
        error(['The field ".FinalPlotLosses" must be "true" or "false". Detected value ',num2str(MLP_struct.FinalPlotLosses),' instead.'])
    end
       
    % Number layer - activation function check.
    if sum(~cellfun(@isempty,MLP_struct.HiddenActivationFun(:) )) ~= length(MLP_struct.HiddenSizes)
        error('Mismatch between number of Hidden Activation Functions and number of Hidden layers')
    end
    disp('All parameters are consistent.');
    disp('');

else
    error('MLP struct is not complete. Please fill in all fields before compiling.');
end


end

function type_activation = code_activation(N_layer,MLP_py_activation_functions)

% code_activation() encodes activation function names into numerical values
% to facilitate handling in the MATLAB function and decoding matrix. The
% input layer is automatically assigned as 0 as there are no activation
% functions at that stage.
%
% Inputs:
%   - N_layer: Number of layers in the MLP.
%   - MLP_py_activation_functions: Cell array containing activation function
%     names for each hidden layer.
%
% Output:
%   - type_activation: Numeric array representing encoded activation
%     functions for each layer.
%
% Note: Activation functions are coded as follows:
%       0 - Input layer (no activation function)
%       1 - Sigmoid
%       2 - Tanh
%       3 - ReLU
%       4 - Linear
%       5 - Softmax
%       6 - Step
%       7 - Sine
%
% Example:
%   type_activation = code_activation(3, {'tanh', 'relu'});
%
% Author: Marino Massimo Costantini

% Initialize
type_activation=zeros(N_layer,1);

% Coding
    for i=1:N_layer
        if i==1
            type_activation(i)=0;
        else
            type=MLP_py_activation_functions{i-1};
    
            switch type
                case 'sigmoid'
                    type_activation(i) = 1;
                case 'tanh'
                    type_activation(i) = 2;
                case 'relu'
                    type_activation(i) = 3;
                case 'lin'
                    type_activation(i) = 4;
                case 'softmax'
                    type_activation(i) = 5;
                case 'step'
                    type_activation(i) = 6;
                case 'sine' 
                    type_activation(i) = 7;
                otherwise
                    error('Invalid activation function type');
            end
        end
    end
end

function [W_concatenate, b_concatenate, MLP_struct] = MLP_python2matlab(MLP_shape, final_MLP)

% MLP_python2matlab - Converts a Python MLP structure to MATLAB-compatible matrices.
%
%   [W_concatenate, b_concatenate, MLP_struct] = MLP_python2matlab(MLP_shape, final_MLP) 
%   takes the MLP_shape and the dictionary final_MLP as inputs and returns the converted 
%   MATLAB vectors W_concatenate and b_concatenate, along with the updated 
%   MLP_struct.
%
%   Parameters:
%       - MLP_shape: Shape of the MLP layers.
%       - final_MLP: Python MLP structure.
%
%   Outputs:
%       - W_concatenate: Concatenated weights matrices.
%       - b_concatenate: Concatenated biases vectors.
%       - MLP_struct: Updated MATLAB structure.
%
%   Example:
%       [W, b, struct] = MLP_python2matlab(shape, py_struct);
%
%   See also: code_activation, losses_plot
%
%   Author: Marino Massimo Costantini


% Initialize the structure with biases and weights from the Python MLP
MLP_struct = struct(final_MLP);
W_concatenate = zeros(1,sum(MLP_shape(1:end-1).*MLP_shape(2:end)));
b_concatenate = zeros(1,sum(MLP_shape(2:end)));
j_in=1;k_in=1;
% Loop to populate the output structure
for i = 1:length(MLP_shape)-1
    % Retrieve field names for weights and biases
    weight_field = ['W' num2str(i)];
    bias_field = ['b' num2str(i)];

    % Convert Python arrays to MATLAB matrices
    Wn = double(MLP_struct.(weight_field));
    bn = double(MLP_struct.(bias_field));

    % Replace the Python arrays with MATLAB matrices in the structure
    MLP_struct.(weight_field) = Wn;
    MLP_struct.(bias_field) = bn;

    % Create MATLAB arrays for Simulink transfer
    % Structs cannot be directly passed to Simulink functions, so matrices are
    % encoded into concatenated MATLAB arrays. These arrays will be decoded
    % inside the MATLAB function in Simulink.
    j_fin=length(reshape(Wn, 1, []))+j_in-1;
    k_fin=length(reshape(bn, 1, []))+k_in-1;
    W_concatenate(j_in:j_fin) = reshape(Wn, 1, []);
    b_concatenate(k_in:k_fin) = reshape(bn, 1, []);
    j_in=j_fin+1;
    k_in=k_fin+1;

end
end

function losses_plot(losses, max_epoch_N, learning_rate, batch_size, momentum,RMSprop, HiddenActivationFun, OutputActivationFun, MLP_shape)
% losses_plot() - Converts Python array to MATLAB-compatible format and plots training and validation losses over epochs.

% Initialize matrix to store losses for each epoch
losses_mat = zeros(max_epoch_N, 2);

% Populate the losses matrix
for i = 1:max_epoch_N
    losses_mat(i, :) = double(losses{i});
end

% Plot data
figure(Name='Losses plot');
hold on;
plot(losses_mat(:, 1), 'r', 'LineWidth', 2,'DisplayName','Training losses');
plot(losses_mat(:, 2), 'b', 'LineWidth', 2,'DisplayName','Validation losses');
grid on;

% Convert numerical parameters to strings for subtitle
lr_str = num2str(learning_rate);
momentum_str = num2str(momentum);
RMSprop_str = num2str(RMSprop);
batch_size_str = num2str(batch_size);
MLP_shape_str  = num2str(MLP_shape');
% Title & Subtitles
title('Training & Validation Losses');
subtitle(['Learning Rate: \bf' lr_str '\rm' ...
    ', Batch Size: \bf' batch_size_str newline '\rm' ...
    ', Momentum decay: \bf' momentum_str '\rm' ...
    ', Sum Squared Weight: \bf' RMSprop_str newline '\rm' ...
    'Hidden Activation Function: \bf' strjoin(HiddenActivationFun, ', ') '\rm' newline ...
    'Output Activation Function: \bf' OutputActivationFun '\rm' newline ...
    'Overall MLP shape: \bf[' MLP_shape_str ']\rm' ]);


% Labels and legend
xlabel('Epochs'); ylabel('Loss');
legend('Location', 'Best');

% Set x-axis and y-axis limits
xlim([1 max_epoch_N]);
ylim([0 max(losses_mat(:))*1.2]);
hold off;
end





