import os
import django
import re
import requests
import datetime
import time
import pandas as pd

from selenium import webdriver
from selenium.webdriver.support.ui import Select
from selenium.webdriver.common.by import By
from selenium.webdriver.common.action_chains import ActionChains
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.common.exceptions import TimeoutException
from selenium.webdriver.chrome.options import Options
from sqlalchemy import create_engine, MetaData, Table

from core.models import URLRequest
def selenium_page_actions(url, actions):
    # Set up the browser driver
    current_directory = os.path.dirname(os.path.abspath(__file__))
    chromedriver_path = os.path.join(current_directory, "chromedriver.exe")

    # Set Chrome options to allow multiple file downloads
    chrome_options = Options()
    prefs = {
        "download.prompt_for_download": False,  # To disable the download prompt
        "download.directory_upgrade": True,
        "safebrowsing_for_trusted_sources_enabled": False,
        "safebrowsing.enabled": False,
        "profile.default_content_setting_values.automatic_downloads": 1
    }
    chrome_options.add_experimental_option("prefs", prefs)
    chrome_options.add_argument("--no-sandbox")
    chrome_options.add_argument("--disable-dev-shm-usage")
    chrome_options.add_argument("--incognito")  # incognito mode doesn't use the previous session's cookies
    chrome_options.add_argument("--disable-cache")  # Disable the cache

    driver = webdriver.Chrome()

    # Navigate to the website
    driver.get(url)

    # Wait for the page to load (can be improved with WebDriverWait)
    time.sleep(5)
    wait = WebDriverWait(driver, 10)

    # if action item includes get_element_wait, javascript is implemented
    for action in actions:
        # Determine the element locator
        if "element_id" in action or "element_css_selector" in action:
            if "element_id" in action:
                locator = (By.ID, action["element_id"])
            elif "element_css_selector" in action:
                locator = (By.CSS_SELECTOR, action["element_css_selector"])

            try:
                if "get_element_wait" in action and action["get_element_wait"] > 0:
                    wait = WebDriverWait(driver, action["get_element_wait"])
                    element = wait.until(EC.presence_of_element_located(locator))
                else:
                    element = driver.find_element(*locator)
            except TimeoutException:
                continue

            # Iterate through function, input, and wait lists
        for func, inp, wait_time in zip(action["function"], action["input"], action["wait"]):
            if func == "clear":
                element.clear()
            elif func == "send_keys":
                element.send_keys(inp)
            elif func == "click":
                element.click()
            elif func == "execute_script":
                driver.execute_script(inp, element)
            elif func == "refresh":  # The added refresh action
                driver.refresh()

            if wait_time > 0:
                time.sleep(wait_time)

    time.sleep(2)  # Wait for the download to complete

    driver.quit()

def fill_api_parameters(api_param, param_dict):
    # Regex to identify placeholders with optional date formatting
    pattern = r"\{(\w+)(?:\[(.*?)\])?\}"

    # Replacement function
    def replacer(match):
        key = match.group(1)  # Parameter name e.g. 'Series'
        format_str = match.group(2)  # Optional date format e.g. '%Y-%m-%d'

        # If the key isn't present in the param_dict, return the original placeholder
        if key not in param_dict:
            return match.group(0)

        # If the placeholder has a date format
        if format_str:
            return param_dict[key].strftime(format_str)

        return str(param_dict[key])

    return re.sub(pattern, replacer, api_param)

def load_SQL_data(db_filename, table_name, query=None, parsetime=None):
    # Construct the full path to the database file
    db_filepath = os.path.join(os.getcwd(), db_filename)
    if query is None:
        query = f'SELECT * FROM {table_name}'
    # Check if the file exists
    if os.path.exists(db_filepath):
        # File exists, so we load the existing data into a pandas DataFrame
        try:
            engine = create_engine(f'sqlite:///{db_filepath}')
            if parsetime:
                loaded_df = pd.read_sql(query, con=engine, parse_dates=parsetime)
                for column in parsetime:
                    loaded_df[column] = loaded_df[column].dt.tz_localize('UTC')
            else:
                loaded_df = pd.read_sql(query, con=engine)
            return loaded_df
        except:
            # An error occurred, most likely the table does not exist
            return None
    else:
        # The file does not exist
        return None


def save_SQL_data(df, db_filename, table_name):
    # Construct the full path to the database file
    db_filepath = os.path.join(os.getcwd(), db_filename)

    # Create an engine to interact with the SQL database
    engine = create_engine(f'sqlite:///{db_filepath}')

    # Save the DataFrame to the specified table in the database
    # If the table already exists, it will be replaced
    df.to_sql(table_name, con=engine, index=False, if_exists='replace')
    return df

def delete_SQL_table(db_filename, table_name):
    # Construct the full path to the database file
    db_filepath = os.path.join(os.getcwd(), db_filename)

    # Check if the file exists
    if os.path.exists(db_filepath):
        engine = create_engine(f'sqlite:///{db_filepath}')

        # Create MetaData object to reflect the database's current state
        metadata = MetaData()
        metadata.reflect(engine)

        # Check if the table exists in the reflected metadata
        if table_name in metadata.tables:
            # Get the table and drop it
            table_to_drop = Table(table_name, metadata)
            table_to_drop.drop(engine)
            print(f"Table '{table_name}' has been deleted from '{db_filename}'.")
        else:
            print(f"Table '{table_name}' does not exist in '{db_filename}'.")
    else:
        print(f"Database '{db_filename}' does not exist.")

def table_name_from_filepath(filepath):
    # Extract filename without extension
    base_name = os.path.splitext(os.path.basename(filepath))[0]

    # Replace spaces with underscores
    sanitized_name = base_name.replace(' ', '_')

    # Remove any characters not allowed in SQLite table names (assuming only alphanumeric and underscore are allowed)
    sanitized_name = re.sub(r'[^a-zA-Z0-9_]', '', sanitized_name)

    return sanitized_name