import pytz
import requests
import json
import numpy as np
import pandas as pd

from decimal import Decimal
from datetime import datetime, date, time, timedelta
from core.utils import fill_api_parameters, selenium_page_actions
from django.conf import settings
from django.db.models import F, Q
from core.models import variables
from form.utils import import_form_data
from form.models import FormFormat
from pandas import Timestamp
from .models import IPP, PPAData, Price, XUL115, PPA, Rates, PPAGenerator

def construct_price_matrix(start_date, end_date):
    power_resolution = 0.05
    time_resolution = 'h'
    currency = 'BZD'

    # start_date and end_date are datetime objects that must include hour, minute, and second as time naive
    start_date_utc = convert_time_naive(start_date, settings.TIME_ZONE, 'UTC')
    end_date_utc = convert_time_naive(end_date, settings.TIME_ZONE, 'UTC')
    FUNCTION_MAP = {
        "CFE_price": CFE_price,
        "PPA_price": PPA_price,
        "CHL_MOL_price": CHL_MOL_price,
        "generator_fuel": generator_fuel,
        "GWH100_Spill": GWH100_Spill,
        "belcogen_penalty": belcogen_penalty,
    }

    # Retrieve all generators that are active during the given time period
    generators = PPAGenerator.objects.filter(status=True)
    result_dict = {}  # This will store ppa.id as keys and function returned values as values
    for generator in generators:
        ppa = PPA.objects.filter(id=generator.PPA_id).first()
        if ppa.energy_price_fxn:
            function_to_call = FUNCTION_MAP.get(ppa.energy_price_fxn)
            if function_to_call:
                result = function_to_call(start_date_utc, end_date_utc, ppa, power_resolution, currency)  # Call the function
        elif ppa.fuel_cost_fxn:
            function_to_call = FUNCTION_MAP.get(ppa.fuel_cost_fxn)
            if function_to_call:
                result = function_to_call(start_date_utc, end_date_utc, ppa, generator, power_resolution, currency)
        elif ppa.energy_price is not None:
            result = PPA_price(start_date_utc, end_date_utc, ppa, power_resolution, currency)

        result_dict[generator.id] = result

    return json.dumps(result_dict)

def CHL_MOL_price(start_date_utc, end_date_utc, ppa, power_resolution=0.05, currency='BZD'):
    rate_df = get_ppa_rates(start_date_utc, end_date_utc, 'ppa_base_rate', 'BZD', ppa)
    results = serialize_time_rate(rate_df, power_resolution)
    return results

def GWH100_Spill(start_date_utc, end_date_utc, power_resolution=0.05, currency='BZD'):
    return 0

def belcogen_penalty(start_date_utc, end_date_utc, power_resolution=0.05, currency='BZD'):
    return 0

def PPA_price(start_date_utc, end_date_utc, ppa, power_resolution=0.05, currency='BZD'):
    rate_df = get_ppa_rates(start_date_utc, end_date_utc, 'ppa_base_rate', 'BZD', ppa)
    results = serialize_time_rate(rate_df, power_resolution)
    return results

def generator_fuel(start_date_utc, end_date_utc, ppa, generator, power_resolution=0.05, currency='BZD'):
    results = serialize_fuel_curve(ppa, generator, start_date_utc, end_date_utc, power_resolution, currency)
    return results

def serialize_fuel_curve(ppa, ppa_generator, start_date_utc, end_date_utc, power_resolution=0.05, currency='BZD'):
    rates_df = interval_rates_table(ppa.fuel_name, start_date_utc, end_date_utc, ppa.id)
    rates_df = currency_conversion(rates_df, currency, start_date_utc, end_date_utc)

    total_hours = int((end_date_utc - start_date_utc).total_seconds()) // 3600 + 1
    hourly_intervals = [start_date_utc + timedelta(hours=i) for i in range(total_hours)]

    p_100 = ppa_generator.P_max
    p_0 = 0
    fuel_function = generator_curve(ppa_generator)

    time_data = {}

    for interval in hourly_intervals:
        fuel_price = rates_df[rates_df['time'] == interval]['rate'].values[0]
        # Generate power curve data using the fuel_function
        curve_data = {
            "P{:03}".format(round(i*100/(p_100 - p_0))): fuel_function(i, fuel_price)
            for i in frange(power_resolution * (p_100 - p_0), p_100, power_resolution * (p_100 - p_0))
        }
        time_data[str(interval)] = curve_data

    return time_data

def get_matching_rate(rates_list, interval):
     rate_idx = 0
     while rate_idx < len(rates_list) - 1 and not rates_list[rate_idx].time <= pytz.timezone('UTC').localize(interval):
         rate_idx += 1
     current_rate = rates_list[rate_idx].rate
     current_currency = rates_list[rate_idx].currency

     return current_rate,current_currency

def query_latest_rates(type, end_date_utc, PPA_id=None):
    localized_time = pytz.timezone('UTC').localize(end_date_utc)
    if PPA_id:
        rates_queryset = Rates.objects.filter(type=type, PPA_id=PPA_id, time__lte=localized_time).order_by('-time')
    else:
        rates_queryset = Rates.objects.filter(type=type, time__lte=localized_time).order_by('-time')
    rates_list = list(rates_queryset)
    return rates_list

def interval_rates_table(type, start_date_utc, end_date_utc, PPA_id=None):
    total_hours = int((end_date_utc - start_date_utc).total_seconds()) // 3600 + 1
    hourly_intervals = [start_date_utc + timedelta(hours=i) for i in range(total_hours)]
    rates_list = query_latest_rates(type, end_date_utc, PPA_id)
    rate_data = []
    for interval in hourly_intervals:
        current_rate, current_currency = get_matching_rate(rates_list, interval)
        rate_data.append({
            'time': interval,
            'rate': current_rate,
            'currency': current_currency,
        })

    # Convert the rate_data list to a dataframe
    df = pd.DataFrame(rate_data)
    return df

def fixed_rates_table(rate, currency, start_date_utc, end_date_utc):
    total_hours = int((end_date_utc - start_date_utc).total_seconds()) // 3600 + 1
    hourly_intervals = [start_date_utc + timedelta(hours=i) for i in range(total_hours)]
    rate_data = []
    for interval in hourly_intervals:
        rate_data.append({
            'time': interval,
            'rate': rate,
            'currency': currency,
        })

    # Convert the rate_data list to a dataframe
    df = pd.DataFrame(rate_data)
    return df
def currency_conversion(rates_df, new_currency, start_date_utc, end_date_utc):
    unique_currencies = rates_df['currency'].unique()
    for currency in unique_currencies:
        if currency != new_currency:
            if currency == 'USD' and new_currency == 'BZD' or currency == 'BZD' and new_currency == 'USD':
                usd_bzd_df = interval_rates_table('USD_BZD_exchange', start_date_utc, end_date_utc)
            elif currency == 'MXN' and new_currency == 'USD' or currency == 'USD' and new_currency == 'MXN':
                usd_mxn_df = interval_rates_table('SF60653', start_date_utc, end_date_utc)
            elif currency == 'MXN' and new_currency == 'BZD' or currency == 'BZD' and new_currency == 'MXN':
                usd_mxn_df = interval_rates_table('SF60653', start_date_utc, end_date_utc)
                usd_bzd_df = interval_rates_table('USD_BZD_exchange', start_date_utc, end_date_utc)
                result_rates = usd_bzd_df['rate'] / usd_mxn_df['rate']
                mxn_bzd_df = pd.DataFrame({
                    'time': usd_mxn_df['time'],
                    'rate': result_rates,
                    'currency': 'BZD'
                })

    hourly_intervals = rates_df['time']
    rate_data = []

    for interval in hourly_intervals:
        current_rate = rates_df[rates_df['time'] == interval]['rate'].values[0]
        current_currency = rates_df[rates_df['time'] == interval]['currency'].values[0]
        if current_currency == new_currency:
            rate_value = current_rate
        else:
            if {current_currency, new_currency} == {'USD', 'BZD'}:
                rate_value = current_rate * usd_bzd_df[(usd_bzd_df['time'] == interval)]['rate'].values[0]
            elif {current_currency, new_currency} == {'BZD', 'USD'}:
                rate_value = current_rate / usd_bzd_df[(usd_bzd_df['time'] == interval)]['rate'].values[0]
            elif {current_currency, new_currency} == {'USD', 'MXN'}:
                rate_value = current_rate * usd_mxn_df[(usd_mxn_df['time'] == interval)]['rate'].values[0]
            elif {current_currency, new_currency} == {'MXN', 'USD'}:
                rate_value = current_rate / usd_mxn_df[(usd_mxn_df['time'] == interval)]['rate'].values[0]
            elif {current_currency, new_currency} == {'MXN', 'BZD'}:
                rate_value = current_rate * mxn_bzd_df[(usd_mxn_df['time'] == interval)]['rate'].values[0]
            elif {current_currency, new_currency} == {'BZD', 'MXN'}:
                rate_value = current_rate / mxn_bzd_df[(usd_mxn_df['time'] == interval)]['rate'].values[0]


        rate_data.append({
            'time': interval,
            'rate': rate_value,
            'currency': new_currency,
        })

    # Convert the rate_data list to a dataframe
    df = pd.DataFrame(rate_data)
    return df

def generator_curve(generator):
    curve = generator.fuel_cost_curve
    def function(x, p):
        return eval(curve)

    return function


def CFE_price(start_date_utc, end_date_utc, ppa, power_resolution=0.05, currency='BZD'):
    missing_records = check_XUL115_records("MDA", start_date_utc, end_date_utc)
    if missing_records:
        print('Missing records in XUL115 for time period: ' + str(start_date_utc) + ' to ' + str(end_date_utc))

    total_hours = int((end_date_utc - start_date_utc).total_seconds()) // 3600 + 1
    hourly_intervals = [start_date_utc + timedelta(hours=i) for i in range(total_hours)]
    time_df = pd.DataFrame({'time': hourly_intervals})
    time_df['time'] = time_df['time'].dt.tz_localize('UTC')

    records_MDA = XUL115.objects.filter(
        Q(time__gte=pytz.timezone('UTC').localize(start_date_utc)) &
        Q(time__lte=pytz.timezone('UTC').localize(end_date_utc)) &
        Q(Proceso='MDA')
    ).order_by('time').values('time', 'PML')

    time_values_MDA = records_MDA.values_list('time', flat=True).distinct()

    # Fetch records_EST excluding time values that are in time_values_MDA
    records_EST = XUL115.objects.filter(
        Q(time__gte=pytz.timezone('UTC').localize(start_date_utc)) &
        Q(time__lte=pytz.timezone('UTC').localize(end_date_utc)) &
        Q(Proceso='EST') &
        ~Q(time__in=time_values_MDA)  # Exclude times that are in time_values_MDA
    ).order_by('time').values('time', 'PML')

    # Convert the queryset to a pandas DataFrame
    df_MDA = pd.DataFrame.from_records(records_MDA)
    df_MDA['currency'] = 'MXN'
    df_EST = pd.DataFrame.from_records(records_EST)
    df_EST['currency'] = 'USD'
    df_union = pd.concat([df_MDA, df_EST], axis=0).reset_index(drop=True)
    df_union.rename(columns={'PML': 'rate'}, inplace=True)
    df_final = pd.merge(time_df, df_union, on='time', how='left')
    df_final['rate'].fillna(999, inplace=True)
    df_final['currency'].fillna('BZD', inplace=True)
    df_final['time'] = df_final['time'].dt.tz_localize(None)

    df_final = currency_conversion(df_final, currency, start_date_utc, end_date_utc)
    results = serialize_time_rate(df_final, power_resolution)

    return results


def check_XUL115_records(proceso, start_date_utc, end_date_utc):
    missing_intervals = get_missing_interval(start_date_utc, end_date_utc, proceso)
    missing_intervals_banxico = get_missing_interval(start_date_utc, end_date_utc, "SF60653")
    if missing_intervals_banxico:
        start_date_mex = convert_time_naive(missing_intervals_banxico[0], 'UTC', 'America/Mexico_City')
        end_date_mex = convert_time_naive(missing_intervals_banxico[-1], 'UTC', 'America/Mexico_City')
        get_banxico_api_data(start_date_mex, end_date_mex)
    # Step 1: Fetch all hourly records between start and end date for given proceso

    if not missing_intervals:
        return None

    # Step 2: Fetch missing data from CENACE API
    start_date_mex = convert_time_naive(missing_intervals[0], 'UTC', 'America/Mexico_City')
    end_date_mex = convert_time_naive(missing_intervals[-1], 'UTC', 'America/Mexico_City')
    get_cenace_api_data(start_date_mex, end_date_mex, proceso)


    missing_intervals = get_missing_interval(start_date_utc, end_date_utc, proceso)

    # If still missing any data, fetch from CSV
    if missing_intervals:
        unique_missing_dates = sorted(set(convert_time_naive(interval, 'UTC', 'America/Mexico_City').date() for interval in missing_intervals))
        grouped_dates = group_consecutive_dates(unique_missing_dates, 5)

        for date_group in grouped_dates:
            start = date_group[0]
            end = date_group[-1]
            get_cenace_csv(proceso, start, end)
    else:
        return None

    # Step 4: Check again to see if the missing data has been saved in XUL115
    missing_intervals = get_missing_interval(start_date_utc, end_date_utc, proceso)
    today_date = datetime.combine(convert_time_naive(datetime.now(), settings.TIME_ZONE, 'America/Mexico_City').date(), time.min)
    two_days_from_now = today_date + timedelta(days=2)
    missing_intervals = subtract_estimate_data(missing_intervals, two_days_from_now)


    # If still missing any data, fetch from estimate form
    if missing_intervals:
        unique_missing_dates = sorted(
            set(convert_time_naive(interval, 'UTC', 'America/Mexico_City').date() for interval in missing_intervals))
        form = FormFormat.objects.get(name='Estimated MDA 2 day Prior')
        if two_days_from_now.date() in unique_missing_dates:
            import_form_data(form.id, two_days_from_now)

        missing_intervals = get_missing_interval(start_date_utc, end_date_utc, proceso)
        missing_intervals = subtract_estimate_data(missing_intervals, two_days_from_now)
    else:
        return None

    return missing_intervals


def get_cenace_api_data(start_date_mex, end_date_mex, proceso, ip="None"):
    start_date, end_date, outside_range = check_valid_cenace_date(proceso, start_date_mex, end_date_mex, "API")
    if outside_range:
        return None

    mexico_city_tz = pytz.timezone('America/Mexico_City')
    api_endpoint = variables.objects.get(group='CFE_price', type='API', name='cenace_sim').value
    api_param = variables.objects.get(group='CFE_price', type='parameters', name='cenace_sim').value

    n_date = start_date
    while n_date < end_date:
        e_date = min(n_date + timedelta(days=6), end_date)
        param_dict = {
            "proceso": proceso,
            "start": n_date,
            "end": e_date,
        }
        url = api_endpoint + fill_api_parameters(api_param, param_dict)

        response = requests.get(url, proxies={"http": ip, "https": ip} if ip != "None" else None)

        data = response.json()  # parse the JSON data

        # Check if "Resultados" key exists in data
        if "Resultados" not in data:
            n_date += timedelta(days=7)  # increment the date
            continue  # skip to the next iteration

        for zona_data in data["Resultados"]:
            for entry in zona_data["Valores"]:
                defaults = {
                    'PML': entry['pml'],
                    'PML_ene': entry.get('pml_ene', None),
                    'PML_per': entry.get('pml_per', None),
                    'PML_cng': entry.get('pml_cng', None)
                }
                entry_date = datetime.strptime(entry['fecha'], '%Y-%m-%d').date()
                entry_time = timedelta(hours=int(entry['hora']) - 1)  # correspond to 00:00
                combined_datetime = datetime.combine(entry_date, datetime.min.time()) + entry_time
                localized_datetime = mexico_city_tz.localize(combined_datetime)
                xul_instance, created = XUL115.objects.update_or_create(
                    time=localized_datetime,
                    Proceso=proceso,
                    defaults=defaults
                )

        n_date += timedelta(days=7)

def get_banxico_api_data(start_date_mex, end_date_mex):
    mexico_city_tz = pytz.timezone('America/Mexico_City')
    api_endpoint = variables.objects.get(group='CFE_price', type='API', name='banxico_v1').value
    api_param = variables.objects.get(group='CFE_price', type='parameters', name='banxico_v1').value
    api_key = variables.objects.get(group='CFE_price', type='KEY', name='banxico_v1').value


    param_dict = {
        "APIKey": api_key,
        "start": start_date_mex,
        "end": end_date_mex,
    }
    url = api_endpoint + fill_api_parameters(api_param, param_dict)

    response = requests.get(url)
    data = json.loads(response.content)

    for entry in data["bmx"]["series"][0]["datos"]:
        dato = entry["dato"]
        fecha = entry["fecha"]
        entry_date = datetime.strptime(fecha, '%d/%m/%Y')
        localized_datetime = mexico_city_tz.localize(entry_date)

        sf60653_instance, created = Rates.objects.update_or_create(
            time=localized_datetime,
            type='SF60653',
            defaults={'rate': dato, 'currency': 'MXN', 'PPA_id': 0}
        )

def get_cenace_csv(proceso, start_date_mex, end_date_mex):
    if proceso == "MTR":
        url = variables.objects.get(group='CFE_price', type='WEB', name='reportes_MTR').value
    else:
        url = variables.objects.get(group='CFE_price', type='WEB', name='reportes_MDA').value

    start_date_mex, end_date_mex, outside_range = check_valid_cenace_date(proceso, start_date_mex, end_date_mex, "CSV")
    if outside_range:
        return None
    need_to_change_date, same_date = check_dates_for_cenace_csv(proceso, start_date_mex, end_date_mex)

    if same_date and need_to_change_date:
        start_date_mex = start_date_mex - timedelta(days=1)

    start_date_utc = convert_time_naive(start_date_mex, 'America/Mexico_City', 'UTC')
    end_date_utc = convert_time_naive(end_date_mex.replace(hour=23, minute=0, second=0, microsecond=0), 'America/Mexico_City', 'UTC')
    missing_intervals = get_missing_interval(start_date_utc, end_date_utc, proceso)

    if not missing_intervals:
        return

    num_days = (end_date_mex - start_date_mex).days + 1
    start_date = start_date_mex.strftime('%d/%m/%Y')
    end_date = end_date_mex.strftime('%d/%m/%Y')

    start_day, start_month, start_year = map(int, start_date.split('/'))
    end_day, end_month, end_year = map(int, end_date.split('/'))
    start_data_title = get_data_title_for_date(start_year, start_month, start_day)
    end_data_title = get_data_title_for_date(end_year, end_month, end_day)

    # Convert them to integers for comparison
    start_month = start_month - 1  # Because the options start from 0
    end_month = end_month - 1

    if not need_to_change_date:
        actions = [
            #{
            #    "function": ["refresh"],
            #    "input": [None],  # or None, as there's no input needed for refresh
            #    "wait": [10]  # seconds to wait after refreshing
            #},
            {
                "element_id": f"ContentPlaceHolder1_gvReportes_imgBtnCSV_0",
                "get_element_wait": 10,
                "function": ["execute_script", "execute_script"],
                "input": ["arguments[0].scrollIntoView(true);", "arguments[0].click();"],
                "wait": [2, 2]
            },
        ]
    else:
        actions = [
            {
                "element_id": "ContentPlaceHolder1_txtPeriodo",
                "function": ["click"],
                "input": [None],
                "wait": [2]
            },
            {
                "element_css_selector": ".drp-calendar.left select.monthselect",
                "function": ["click"],
                "input": [None],
                "wait": [2]
            },
            {
                "element_css_selector": f".drp-calendar.left select.monthselect option[value='{start_month}']",
                "function": ["click"],
                "input": [None],
                "wait": [2]
            },
            {
                "element_css_selector": ".drp-calendar.left select.yearselect",
                "function": ["click"],
                "input": [None],
                "wait": [2]
            },
            {
                "element_css_selector": f".drp-calendar.left select.yearselect option[value='{start_year}']",
                "function": ["click"],
                "input": [None],
                "wait": [2]
            },
            {
                "element_css_selector": ".drp-calendar.right select.monthselect",
                "function": ["click"],
                "input": [None],
                "wait": [2]
            },
            {
                "element_css_selector": f".drp-calendar.right select.monthselect option[value='{end_month}']",
                "function": ["click"],
                "input": [None],
                "wait": [2]
            },
            {
                "element_css_selector": ".drp-calendar.right select.yearselect",
                "function": ["click"],
                "input": [None],
                "wait": [2]
            },
            {
                "element_css_selector": f".drp-calendar.right select.yearselect option[value='{end_year}']",
                "function": ["click"],
                "input": [None],
                "wait": [2]
            },
            {
                "element_css_selector": f".drp-calendar.left tbody td[data-title='{start_data_title}']",
                "function": ["click"],
                "input": [None],
                "wait": [2]
            },
            {
                "element_css_selector": f".drp-calendar.right tbody td[data-title='{end_data_title}']",
                "function": ["click"],
                "input": [None],
                "wait": [2]
            },
            {
                "element_css_selector": ".drp-buttons .applyBtn",
                "get_element_wait": 10,
                "function": ["execute_script", "execute_script"],
                "input": ["arguments[0].scrollIntoView(true);", "arguments[0].click();"],
                "wait": [5, 5]
            },
        ]

        if same_date:
            csv_actions = [
                {
                    "element_id": f"ContentPlaceHolder1_gvReportes_imgBtnCSV_1",
                    "get_element_wait": 10,
                    "function": ["execute_script", "execute_script"],
                    "input": ["arguments[0].scrollIntoView(true);", "arguments[0].click();"],
                    "wait": [2, 2]
                } for i in range(num_days)
            ]
        else:
            csv_actions = [
                {
                    "element_id": f"ContentPlaceHolder1_gvReportes_imgBtnCSV_{i}",
                    "get_element_wait": 10,
                    "function": ["execute_script", "execute_script"],
                    "input": ["arguments[0].scrollIntoView(true);", "arguments[0].click();"],
                    "wait": [2, 2]
                } for i in range(num_days)
            ]
        actions.extend(csv_actions)
    selenium_page_actions(url, actions)
    form = FormFormat.objects.get(name='MDA_Expost Dia')
    if same_date:
        import_form_data(form.id, end_date_mex)
    else:
        current_date = start_date_mex
        while current_date <= end_date_mex:
            import_form_data(form.id, current_date)
            current_date += timedelta(days=1)

def get_data_title_for_date(year, month, day):
    # Get the first day of the month
    first_day = datetime(year, month, 1).weekday()

    # If the first day is Sunday, set the start row to 1, else 0
    start_row = 1 if first_day == 6 else 0  # 6 corresponds to Sunday

    # For determining the row: consider only the number of Sundays before the given day
    sundays_before_day = sum(1 for i in range(1, day + 1) if datetime(year, month, i).weekday() == 6)

    row = start_row + sundays_before_day - 1
    weekday_of_given_day = datetime(year, month, day).weekday()
    col = (weekday_of_given_day + 1) % 7

    return f"r{row}c{col}"

def frange(start, stop, step):
    """
    A range function for floats.
    """
    i = start
    while i < stop:
        yield i
        i += step

def convert_time_naive(date_value, timezone_current, timezone_new):
    if timezone_current == 'UTC':
        tz = pytz.UTC
    else:
        tz = pytz.timezone(timezone_current)

    if timezone_new == 'UTC':
        tz_new = pytz.UTC
    else:
        tz_new = pytz.timezone(timezone_new)

    date_value_current = tz.localize(date_value)
    date_value_new = date_value_current.astimezone(tz_new).replace(tzinfo=None)

    return date_value_new

def check_valid_cenace_date(proceso, start_date, end_date, mode="API"):
    original_start_date = start_date
    original_end_date = end_date
    today_date = datetime.combine(convert_time_naive(datetime.now(), settings.TIME_ZONE, 'America/Mexico_City').date(), time.min)
    outside_range = False

    # Adjust start_date and end_date based on proceso
    if proceso == "MTR":
        # Limit to 7 days in the past. If current time is before 2 pm, adjust end_date to the previous day
        if past_cenace_update_time(proceso):
            if mode == "API":
                start_date = max(start_date, today_date - timedelta(days=7))
                end_date = max(end_date, today_date - timedelta(days=7))
            else:
                start_date = min(start_date, today_date - timedelta(days=3))
                end_date = min(end_date, today_date - timedelta(days=3))
        else:
            if mode == "API":
                start_date = min(start_date, today_date - timedelta(days=8))
                end_date = min(end_date, today_date - timedelta(days=8))
            else:
                start_date = min(start_date, today_date - timedelta(days=4))
                end_date = min(end_date, today_date - timedelta(days=4))
    elif proceso == "MDA":
        # Allow up to one day in the future if after 2 pm
        if past_cenace_update_time(proceso):
            start_date = min(start_date, today_date + timedelta(days=1))
            end_date = min(end_date, today_date + timedelta(days=1))
        else:
            start_date = min(start_date, today_date)
            end_date = min(end_date, today_date)

    if end_date < original_start_date or start_date > original_end_date:
        outside_range = True

    return start_date, end_date, outside_range

def past_cenace_update_time(proceso):
    current_time = convert_time_naive(datetime.now(), settings.TIME_ZONE, 'America/Mexico_City').time()
    past_time = False
    if proceso == "MDA":
        if current_time > datetime.strptime("18:00:00", "%H:%M:%S").time():
            past_time = True
    else:
        if current_time > datetime.strptime("18:00:00", "%H:%M:%S").time():
            past_time = True
    return past_time

def check_dates_for_cenace_csv(proceso, start_date_mex, end_date_mex):
    today_date = datetime.combine(convert_time_naive(datetime.now(), settings.TIME_ZONE, 'America/Mexico_City').date(), time.min)
    same_date = False
    need_to_change_date = False
    if proceso == "MDA":
        if start_date_mex == end_date_mex:
            same_date = True
            if past_cenace_update_time(proceso):
                if start_date_mex != today_date + timedelta(days=1):
                    need_to_change_date = True
            else:
                if start_date_mex != today_date:
                    need_to_change_date = True
        else:
            need_to_change_date = True
    elif proceso == "MTR":
        if start_date_mex == end_date_mex:
            same_date = True
            if past_cenace_update_time(proceso):
                if start_date_mex != today_date - timedelta(days=7):
                    need_to_change_date = True
            else:
                if start_date_mex != today_date - timedelta(days=8):
                    need_to_change_date = True
        else:
            need_to_change_date = True
    return need_to_change_date, same_date

def group_consecutive_dates(dates, max_group_size=5):
    if not dates:
        return []

    # Convert datetime.date objects into datetime objects with time set to 00:00:00
    dates = [datetime.combine(d, datetime.min.time()) for d in dates]

    # Sort the dates
    dates = sorted(dates)

    groups = []
    current_group = [dates[0]]

    for d in dates[1:]:
        # If the current date is consecutive to the last date in the group, add it to the current group
        if d == current_group[-1] + timedelta(days=1) and len(current_group) < max_group_size:
            current_group.append(d)
        else:
            # Otherwise, add the current group to groups and start a new group
            groups.append(current_group)
            current_group = [d]

    # Add the last group
    groups.append(current_group)

    return groups

def get_missing_interval(start_date_utc, end_date_utc, proceso):
    if proceso == 'SF60653':
        start_date_mex = convert_time_naive(start_date_utc, 'UTC', 'America/Mexico_City')
        end_date_mex = convert_time_naive(end_date_utc, 'UTC', 'America/Mexico_City')
        start_date_mex = start_date_mex.replace(hour=0, minute=0, second=0, microsecond=0)
        end_date_mex = end_date_mex.replace(hour=0, minute=0, second=0, microsecond=0)
        start_date_utc = convert_time_naive(start_date_mex, 'America/Mexico_City', 'UTC')
        end_date_utc = convert_time_naive(end_date_mex, 'America/Mexico_City', 'UTC')
        intervals_utc = [pytz.utc.localize(start_date_utc + timedelta(days=i)) for i in
                               range((end_date_utc - start_date_utc).days + 1)]
        records = Rates.objects.filter(
            Q(type=proceso) &
            Q(time__gte=pytz.timezone('UTC').localize(start_date_utc)) &
            Q(time__lte=pytz.timezone('UTC').localize(end_date_utc))
        ).order_by('time').values_list('time', flat=True)
    else:
        total_hours = int((end_date_utc - start_date_utc).total_seconds()) // 3600 + 1
        hourly_intervals = [start_date_utc + timedelta(hours=i) for i in range(total_hours)]
        intervals_utc = [pytz.utc.localize(dt) for dt in hourly_intervals]
        records = XUL115.objects.filter(
            Q(time__gte=pytz.timezone('UTC').localize(start_date_utc)) &
            Q(time__lte=pytz.timezone('UTC').localize(end_date_utc)) &
            Q(Proceso=proceso)
        ).order_by('time').values_list('time', flat=True)

    # Convert the records to a set for faster lookup
    records_set = set(records)

    # Identify missing intervals
    missing_intervals = [dt.replace(tzinfo=None) for dt in intervals_utc if dt not in records_set]

    return missing_intervals

def serialize_time_rate(df_records, power_resolution=0.05):
    df_records['time'] = df_records['time'].dt.tz_localize(None) # Make the datetime index of df_records timezone-naive

    n = int(1 / power_resolution)
    new_columns = ["P{0:03}".format(int(power_resolution * 100 * i)) for i in range(1, n + 1)]

    # Repeat the price_field column n times and rename these columns
    for col in new_columns:
        df_records[col] = df_records['rate']

    # Drop the original price_field column
    df_records.drop(columns=['rate', 'currency'], inplace=True)

    # Convert the dataframe to a nested dictionary
    df_records.set_index('time', inplace=True)
    nested_dict = df_records.to_dict(orient='index')

    # Format datetime.datetime objects and convert number objects to string format
    json_data = {k.strftime('%Y-%m-%d %H:%M:%S'): {key: str(value) for key, value in v.items()} for k, v in
                 nested_dict.items()}

    return json_data

def subtract_estimate_data(missing_intervals, start_date_mex):
    end_date_mex = start_date_mex + timedelta(hours=23, minutes=59, seconds=59)
    start_date_utc = convert_time_naive(start_date_mex, 'America/Mexico_City', 'UTC')
    end_date_utc = convert_time_naive(end_date_mex, 'America/Mexico_City', 'UTC')

    records = XUL115.objects.filter(
        Q(time__gte=pytz.timezone('UTC').localize(start_date_utc)) &
        Q(time__lte=pytz.timezone('UTC').localize(end_date_utc)) &
        Q(Proceso='EST')
    ).order_by('time').values_list('time', flat=True)

    estimated_intervals = set(record.replace(tzinfo=None) for record in records)
    missing_intervals = [dt for dt in missing_intervals if dt not in estimated_intervals]
    return missing_intervals

def get_ppa_rates(start_date_utc, end_date_utc, type='ppa_base_rate', currency='BZD', ppa=None):
    if type == 'fixed_rate':
        rates_df = fixed_rates_table(ppa.energy_price, ppa.energy_price_cur, start_date_utc, end_date_utc)
    if ppa:
        rates_df = interval_rates_table(type, start_date_utc, end_date_utc, ppa.id)
    else:
        rates_df = interval_rates_table(type, start_date_utc, end_date_utc)
    rates_df = currency_conversion(rates_df, currency, start_date_utc, end_date_utc)
    return rates_df





