import os
import django

# Set the DJANGO_SETTINGS_MODULE environment variable to your project's settings
os.environ.setdefault("DJANGO_SETTINGS_MODULE", "EMS.settings")
django.setup()

import xml.etree.ElementTree as ET
import os
import re
import math
import networkx as nx
import json
from LF.models import Feeder
from PPA.models import PPAGenerator
from django.apps import apps
from django.http import HttpResponse, JsonResponse
from django.views.decorators.csrf import csrf_exempt
from django.db.models import Q, Count, F
from django.db import IntegrityError
from collections import defaultdict
from Network.models import Connection, UploadedFile, Generator, Transformer, Conductor, \
    ComponentAttributes, Bus, Node, Edge, Island, LoadEtap, Capacitor, \
    Label, Switch, NodeMin, EdgeMin

def get_generator_constraints(fringe_removal_level=6):
    for level in range(1, fringe_removal_level + 1):
        get_minimized_network(level)
    gen_matrix = build_generator_constraints()
    bus_matrix, branch_matrix = get_minimized_network_matrices()
    return bus_matrix, branch_matrix, gen_matrix


def get_minimized_network_matrices():
    # Retrieve all nodes, edges, generators, capacitors, and loads.
    nodes = Node.objects.all()
    edges = Edge.objects.all()
    generators = Generator.objects.all()
    capacitors = Capacitor.objects.all()
    loads = Load.objects.all()

    # Construct a dictionary with etap_iid as the key for easier access.
    generator_dict = {gen.etap_iid: gen for gen in generators}
    capacitor_dict = {cap.etap_iid: cap for cap in capacitors}
    load_dict = {load.feeder_id: load for load in loads}

    # Initialize the node matrix.
    node_matrix = []

    for idx, node in enumerate(nodes, start=1):  # Label nodes from 1 to n.
        gen_id = generator_dict.get(node.etap_iid, None)
        cap_id = capacitor_dict.get(node.etap_iid, None)
        load_id = load_dict.get(node.etap_iid, None)

        node_matrix.append([
            idx,  # Node label.
            node.etap_iid,
            node.reference_voltage,
            gen_id,  # Generator ID. It could be None.
            cap_id,  # Capacitor ID. It could be None.
            load_id  # Load ID. It could be None.
        ])

    # Initialize the edge matrix.
    edge_matrix = []

    for edge in edges:
        from_node_label = nodes.filter(etap_iid=edge.from_bus_iid).first()
        to_node_label = nodes.filter(etap_iid=edge.to_bus_iid).first()

        # Compute conductance and susceptance if needed (you might have these directly in the Edge model).
        conductance = 1 / edge.R_pos  # Assuming R_pos is resistance. Adjust if needed.
        susceptance = 1 / edge.L_pos  # Assuming L_pos is inductance. Adjust if needed.

        edge_matrix.append([
            from_node_label,
            to_node_label,
            conductance,
            susceptance,
            edge.current_limit,
            edge.power_limit  # Adjust if you have a different field name.
        ])

    bus_dict = {
        'bus_matrix': node_matrix
    }

    branch_dict = {
        'branch_matrix': edge_matrix
    }

    # Convert the dictionaries to JSON strings.
    bus_json = json.dumps(bus_dict)
    branch_json = json.dumps(branch_dict)

    return bus_json, branch_json

def build_generator_constraints():
    # Assuming you have imported the model at the top of your file
    generators = PPAGenerator.objects.all()
    etap_generators = Generator.objects.all()
    constraints = {}
    for generator in generators:
        etap_generator = Generator.objects.get(etap_iid=generator.etap_iid)
        if etap_generator:
            P_max = generator.P_max
            P_min = generator.P_min
            Q_max = etap_generator.Q_max
            Q_min = etap_generator.Q_min
        else:
            P_max = generator.P_max
            P_min = generator.P_min
            Q_max = 0
            Q_min = 0
        constraints[generator.id] = {
            "P_max": P_max,
            "P_min": P_min,
            "Q_max": Q_max,
            "Q_min": Q_min,
            "must_run": generator.must_run,
            "must_run_condition": generator.must_run_condition,
            "start_up_cost": generator.start_up_cost,
            "shut_down_cost": generator.shut_down_cost,
            "ramp_up": generator.ramp_up,
            "ramp_down": generator.ramp_down,
            "min_up_time": generator.min_up_time,
            "min_down_time": generator.min_down_time,
        }

    result = {
        "generator_constraints": constraints
    }

    # Convert the result dictionary to a JSON string.
    json_result = json.dumps(result, cls=DjangoJSONEncoder)

    return json_result

def get_minimized_network(fringe_level=1):
    nodes, edges_from, edges_to, fringe_nodes, injection_nodes, multi_tx_nodes, tx3w_nodes = get_special_nodes(fringe_level)
    starting_points = {
        59056: True,  # Main Network XULHA
        99474: True,  # Caye Caulker Bus37
    }
    visited_nodes = set()  # Set to keep track of visited nodes
    visited_edges = set()
    nodes_to_include_set = set()  # Set to store nodes to be included in NodeMin
    edges_to_include_set = set()  # Set to store edges to be included in EdgeMin
    # include the starting_points nodes
    for starting_bus_iid in starting_points.keys():
        nodes_to_include_set.add(nodes[starting_bus_iid])
        nodes_to_include_set.add(nodes[starting_bus_iid])
    stack = list(starting_points.items())
    while stack:
        current_node_iid, included = stack.pop()
        if current_node_iid in visited_nodes:
            continue
        visited_nodes.add(current_node_iid)
        current_node = nodes.get(current_node_iid)
        print(current_node.bus_etap_iid, current_node.etap_id)
        current_edges = edges_from.get(current_node_iid, []) + edges_to.get(current_node_iid, [])
        for edge in current_edges:
            if edge in visited_edges:
                continue
            visited_edges.add(edge)
            if edge.from_bus_iid == current_node_iid:
                next_node_iid = edge.to_bus_iid
            else:
                next_node_iid = edge.from_bus_iid
            next_node = nodes.get(next_node_iid)
            if next_node_iid in fringe_nodes or edge.status == 'Open': #skip this node and edge
                if edge in edges_from[edge.from_bus_iid]:
                    edges_from[edge.from_bus_iid].remove(edge)
                if edge in edges_to[edge.to_bus_iid]:
                    edges_to[edge.to_bus_iid].remove(edge)
                continue

            if (next_node_iid in tx3w_nodes or next_node_iid in multi_tx_nodes):
                edges_to_include_set.add(edge)
                nodes_to_include_set.add(next_node)
                stack.append((next_node_iid, True))
            elif EndTraceAtEdge(edge, next_node, edges_from, edges_to, stack, fringe_nodes, visited_edges, nodes, injection_nodes):
                # requery tertiary_node because it may have been updated
                new_node_iid = edge.from_bus_iid if edge.to_node == current_node else edge.to_bus_iid
                new_node = nodes.get(new_node_iid)
                edges_to_include_set.add(edge)
                nodes_to_include_set.add(new_node)
            elif next_node_iid in injection_nodes:
                edges_to_include_set.add(edge)
                nodes_to_include_set.add(next_node)
                stack.append((next_node_iid, True))
            else:
                stack = trace_to_end(next_node, current_node, edge, edges_from, edges_to, fringe_nodes,
                             injection_nodes, multi_tx_nodes, tx3w_nodes, visited_edges,
                             nodes, edges_to_include_set, nodes_to_include_set, stack)

    add_min_network_to_models(nodes_to_include_set, edges_to_include_set)

def trace_to_end(next_node, current_node, edge, edges_from, edges_to, fringe_nodes, injection_nodes, multi_tx_nodes,
                 tx3w_nodes, visited_edges, nodes, edges_to_include_set, nodes_to_include_set, stack):
    next_node_iid = next_node.bus_etap_iid
    next_edges = [e for e in (edges_from.get(next_node_iid, []) + edges_to.get(next_node_iid, [])) if e != edge]
    for next_edge in next_edges:
        node_removed, tertiary_node_iid, tertiary_node = ReplaceNodeWithPrevious(current_node, next_node, next_edge, edges_from, edges_to, fringe_nodes, visited_edges, nodes)
        if node_removed:
            continue
        if (tertiary_node_iid in tx3w_nodes or tertiary_node_iid in multi_tx_nodes):
            visited_edges.add(next_edge)
            edges_to_include_set.add(next_edge)
            nodes_to_include_set.add(tertiary_node)
            stack.append((tertiary_node_iid, True))
        elif EndTraceAtEdge(next_edge, tertiary_node, edges_from, edges_to, stack, fringe_nodes, visited_edges, nodes, injection_nodes):
            # requery tertiary_node because it may have been updated
            new_node_iid = next_edge.from_bus_iid if next_edge.to_node == current_node else next_edge.to_bus_iid
            new_node = nodes.get(new_node_iid)
            visited_edges.add(next_edge)
            edges_to_include_set.add(next_edge)
            nodes_to_include_set.add(new_node)
        elif tertiary_node_iid in injection_nodes:
            visited_edges.add(next_edge)
            edges_to_include_set.add(next_edge)
            nodes_to_include_set.add(tertiary_node)
            stack.append((tertiary_node_iid, True))
        elif tertiary_node_iid == current_node.bus_etap_iid:
            visited_edges.add(next_edge)
            edges_to_include_set.add(next_edge)
            nodes_to_include_set.add(tertiary_node)
        else:
            stack = trace_to_end(tertiary_node, current_node, next_edge, edges_from, edges_to, fringe_nodes, injection_nodes,
                        multi_tx_nodes, tx3w_nodes, visited_edges, nodes, edges_to_include_set, nodes_to_include_set, stack)
    return  stack

def ReplaceNodeWithPrevious(current_node, next_node, next_edge, edges_from, edges_to, fringe_nodes, visited_edges, nodes):
    node_removed = False
    # Remove the old next_edge from its previous location
    if next_edge in edges_from[next_edge.from_bus_iid]:
        edges_from[next_edge.from_bus_iid].remove(next_edge)
    if next_edge in edges_to[next_edge.to_bus_iid]:
        edges_to[next_edge.to_bus_iid].remove(next_edge)

    tertiary_node_iid = next_edge.from_bus_iid if next_edge.to_node == next_node else next_edge.to_bus_iid
    if tertiary_node_iid in fringe_nodes or next_edge.status == 'Open':  # skip this node and edge
        visited_edges.add(next_edge)
        if next_edge in edges_from[next_edge.from_bus_iid]:
            edges_from[next_edge.from_bus_iid].remove(next_edge)
        if next_edge in edges_to[next_edge.to_bus_iid]:
            edges_to[next_edge.to_bus_iid].remove(next_edge)
        node_removed = True
    else:
        if next_edge.to_node == next_node:
            next_edge.to_node = current_node
            next_edge.to_bus_iid = current_node.bus_etap_iid
        else:
            next_edge.from_node = current_node
            next_edge.from_bus_iid = current_node.bus_etap_iid
        # adjust the high side node as well
        if next_edge.high_side == next_node.bus_etap_iid:
            next_edge.high_side = current_node.bus_etap_iid
        # sort the bus_iids so that the from_bus_iid is always less than the to_bus_iid
        if next_edge.from_bus_iid > next_edge.to_bus_iid:
            next_edge.from_bus_iid, next_edge.to_bus_iid = next_edge.to_bus_iid, next_edge.from_bus_iid
            next_edge.from_node, next_edge.to_node = next_edge.to_node, next_edge.from_node

    # Add the updated next_edge to its new location
    tertiary_node = nodes.get(tertiary_node_iid)
    edges_from[next_edge.from_bus_iid].append(next_edge)
    edges_to[next_edge.to_bus_iid].append(next_edge)
    return node_removed, tertiary_node_iid, tertiary_node
def EndTraceAtEdge(edge, next_node, edges_from, edges_to, stack, fringe_nodes, visited_edges, nodes, injection_nodes):
    if edge.transformers > 0 or edge.conductors > 0:
        next_edges = [e for e in (edges_from.get(next_node.bus_etap_iid, []) + edges_to.get(next_node.bus_etap_iid, [])) if e != edge]
        if next_node.bus_etap_iid in injection_nodes:
            return False
        else:
            if len(next_edges) == 1:
                next_edge = next_edges[0] if next_edges else None
                if next_edge.transformers > 0 or next_edge.conductors > 0:
                    stack.append((next_node.bus_etap_iid, True))
                    return True
                else:
                    tertiary_node_iid = next_edge.from_bus_iid if next_edge.to_node == next_node else next_edge.to_bus_iid
                    tertiary_node = nodes.get(tertiary_node_iid)
                    if tertiary_node_iid in fringe_nodes or next_edge.status == 'Open':  # skip this node and edge
                        visited_edges.add(next_edge)
                        if next_edge in edges_from[next_edge.from_bus_iid]:
                            edges_from[next_edge.from_bus_iid].remove(next_edge)
                        if next_edge in edges_to[next_edge.to_bus_iid]:
                            edges_to[next_edge.to_bus_iid].remove(next_edge)
                    else:
                        _, _, _ = ReplaceNodeWithPrevious(tertiary_node, next_node, edge, edges_from, edges_to, fringe_nodes, visited_edges, nodes)
                        stack.append((tertiary_node_iid, True))
                    return True
            else:
                stack.append((next_node.bus_etap_iid, True))
                return True
    else:
        return False


def get_special_nodes(fringe_level):
    if fringe_level == 1:
        NodeModel = Node
        EdgeModel = Edge
    else:
        NodeModel = NodeMin
        EdgeModel = EdgeMin

    node_queryset = NodeModel.objects.all()
    edge_queryset = EdgeModel.objects.all()
    nodes_with_edge_counts = NodeModel.objects.annotate(
        total_edge_count=Count('outgoing_edges') + Count('incoming_edges'))
    filtered_edges = EdgeModel.objects.filter(Q(transformers__gt=0))
    nodes_multi_tx = NodeModel.objects.annotate(
        outgoing_edge_count=Count('outgoing_edges', filter=Q(outgoing_edges__in=filtered_edges)),
        incoming_edge_count=Count('incoming_edges', filter=Q(incoming_edges__in=filtered_edges)),
    ).annotate(
        total_edge_count=F('outgoing_edge_count') + F('incoming_edge_count')
    ).filter(total_edge_count__gt=1).values_list('bus_etap_iid', flat=True)

    nodes = {node.bus_etap_iid: node for node in node_queryset}
    tx3w_nodes = [node.bus_etap_iid for node in nodes_with_edge_counts if node.type == 'XFORM3W']
    edges_from = defaultdict(list)
    edges_to = defaultdict(list)
    for edge in edge_queryset:
        edges_from[edge.from_bus_iid].append(edge)
        edges_to[edge.to_bus_iid].append(edge)

    node_edges = defaultdict(list)
    for edge in edge_queryset:
        node_edges[edge.from_bus_iid].append(edge)
        node_edges[edge.to_bus_iid].append(edge)

    fringe_nodes = []
    for node in nodes_with_edge_counts:
        if node.total_edge_count == 1 and node.loads_feeder == 0 and node.capacitors == 0 and node.generators == 0:
            fringe_nodes.append(node.bus_etap_iid)
        elif node.total_edge_count == 2:
            edge1, edge2 = node_edges[node.bus_etap_iid]
            if (edge1.from_bus_iid == edge2.from_bus_iid and edge1.to_bus_iid == edge2.to_bus_iid) or \
               (edge1.from_bus_iid == edge2.to_bus_iid and edge1.to_bus_iid == edge2.from_bus_iid):
                fringe_nodes.append(node.bus_etap_iid)

    injection_nodes = [node.bus_etap_iid for node in nodes_with_edge_counts if
                       node.loads_feeder > 0 or node.capacitors > 0 or node.generators > 0]
    multi_tx_nodes = set(nodes_multi_tx)
    return nodes, edges_from, edges_to, fringe_nodes, injection_nodes, multi_tx_nodes, tx3w_nodes

def add_min_network_to_models(nodes_to_include_set, edges_to_include_set):
    # Construct minimized_nodes
    minimized_nodes = [NodeMin(
        island_id=node.island_id,
        bus_id=node.bus_id,
        bus_etap_iid=node.bus_etap_iid,
        loads_lumped=node.loads_lumped,
        loads_single=node.loads_single,
        loads_feeder=node.loads_feeder,
        capacitors=node.capacitors,
        generators=node.generators,
        reference_voltage=node.reference_voltage,
        voltage_phase_A=node.voltage_phase_A,
        voltage_phase_B=node.voltage_phase_B,
        voltage_phase_C=node.voltage_phase_C,
        user_LocX=node.user_LocX,
        user_LocY=node.user_LocY,
        type=node.type,
        loads_string=node.loads_string,
        generators_string=node.generators_string,
        capacitors_string=node.capacitors_string,
        transformers_string=node.transformers_string,
        switches_string=node.switches_string,
        voltage_string=node.voltage_string,
        etap_id=node.etap_id,
        layer=node.layer,
        status=node.status,
    ) for node in nodes_to_include_set]
    # Bulk create minimized nodes
    NodeMin.objects.all().delete()
    NodeMin.objects.bulk_create(minimized_nodes)

    nodemin_queryset = NodeMin.objects.all()
    nodes = {node.bus_etap_iid: node for node in nodemin_queryset}
    minimized_edges = [EdgeMin(
        island_id=edge.island_id,
        from_node=nodes[edge.from_bus_iid],
        to_node=nodes[edge.to_bus_iid],
        from_bus_iid=edge.from_bus_iid,
        to_bus_iid=edge.to_bus_iid,
        transformers=edge.transformers,
        switches=edge.switches,
        threeWinding=edge.threeWinding,
        transformer_ratio=edge.transformer_ratio,
        conductors=edge.conductors,
        R_pos=edge.R_pos,
        R_zero=edge.R_zero,
        C_pos=edge.C_pos,
        C_zero=edge.C_zero,
        L_pos=edge.L_pos,
        L_zero=edge.L_zero,
        current_limit=edge.current_limit,
        voltage_ratio=edge.voltage_ratio,
        voltage_shift=edge.voltage_shift,
        status=edge.status,
        high_side=edge.high_side,
        transformers_string=edge.transformers_string,
        conductors_string=edge.conductors_string,
        layer=edge.layer,
    ) for edge in edges_to_include_set]
    # Bulk create minimized edges
    EdgeMin.objects.all().delete()
    EdgeMin.objects.bulk_create(minimized_edges)
    # remove edges where the from_node and to_node are the same
    EdgeMin.objects.filter(from_node=F('to_node')).delete()
def parse_xml_and_save():
    uploaded_file = UploadedFile.objects.get(id=1)
    file_path = uploaded_file.file.path  # Get the full path to the uploaded file

    # Check if the file exists
    if os.path.exists(file_path):
        with open(file_path, 'r', encoding='utf-8') as xml_file:
            xml_data = xml_file.read()  # Read the file content
            root = ET.fromstring(xml_data)  # Parse the XML data into an ElementTree object
    else:
        return HttpResponse("File does not exist.")

    extract_connections(root)

    extract_components(root)

    return HttpResponse("Data parsed and saved.")


def extract_connections(root):
    connection_fields = [field.name for field in Connection._meta.get_fields()]
    attributes = {}
    connection_elements = root.find('CONNECTIONS').findall('CONNECT')
    connections_to_create = []

    # Parse the connections
    for connection_entry in connection_elements:
        attributes = {attr: value for attr, value in connection_entry.items() if attr in connection_fields}
        connections_to_create.append(Connection(**attributes))

    # Delete all existing data
    Connection.objects.all().delete()
    Connection.objects.bulk_create(connections_to_create)

    # Extract etap_iid values from Generator model
    etap_iids = set(Generator.objects.values_list('etap_iid', flat=True))

    # Add connections from PPAGenerator records not in Etap Generators
    for generator in PPAGenerator.objects.exclude(etap_iid__in=etap_iids):
        gen_iid = generator.etap_iid
        bus_iid = generator.bus_iid

        connection = Connection.objects.filter(Q(FromIID=gen_iid) | Q(ToIID=gen_iid)).first()

        if connection:
            if connection.FromIID == gen_iid:
                if connection.FromElement == 'UNIVERSALRELAY':
                    connection.FromElement = 'SYNGEN'
            else:
                if connection.ToElement == 'UNIVERSALRELAY':
                    connection.ToElement = 'SYNGEN'
            connection.save()
        else:
            connection_type = Connection.objects.filter(Q(FromIID=bus_iid) | Q(ToIID=bus_iid)).first()
            if connection_type.FromIID == bus_iid:
                type = connection_type.FromElement
                etap_id = connection_type.FromID
                pin = 1 + Connection.objects.filter(FromIID=bus_iid).order_by('-FromPin').values_list('FromPin', flat=True).first()
            else:
                type = connection_type.ToElement
                etap_id = connection_type.ToID
                pin = 1 + Connection.objects.filter(ToIID=bus_iid).order_by('-ToPin').values_list('ToPin',flat=True).first()
            attributes = {
                'FromElement': 'SYNGEN',
                'FromID': generator.etap_id,
                'FromIID': gen_iid,
                'FromPin': 0,
                'ToElement': type,
                'ToIID': bus_iid,
                'ToID': etap_id,
                'ToPin': pin,
            }
            Connection.objects.create(**attributes)

def extract_components(root):
    table_list = ComponentAttributes.objects.values('table').distinct()
    #table_list = ComponentAttributes.objects.filter(table='Conductor').values('table').distinct()
    for table_entry in table_list:
        table = table_entry['table']
        component_list = ComponentAttributes.objects.filter(table=table).values('component').distinct()
        # Dynamically get the model class based on the 'table' value
        model_class = apps.get_model(app_label='Network', model_name=table)
        model_class.objects.all().delete()
        required_fields = [field.name for field in model_class._meta.get_fields()
                           if (not field.null or field.blank is False) and field.name != 'id']
        instances_to_create = []  # List to accumulate new model instances
        for component_entry in component_list:
            component_elements,attribute_list, component_name = get_components_and_required_attributes(root, component_entry, table)

            for component in component_elements:
                attributes = {}
                attributes['type'] = component_name
                for attribute_entry in attribute_list:
                    if attribute_entry.operation:
                        expression = attribute_entry.operation
                        attribute_value=evaluate_operation(expression, component)
                    else:
                        attribute_name = attribute_entry.attribute
                        attribute_value = component.get(attribute_name)
                    attributes[attribute_entry.field] = attribute_value

                # add values for required fields
                for field_name in required_fields:
                    field = model_class._meta.get_field(field_name)
                    if field.get_internal_type() == 'AutoField' or field_name in attributes:
                        # AutoField will be auto-incremented by the database
                        continue
                    default_value = field.get_default()
                    attributes[field_name] = default_value

                # Create a new instance of the model class and add it to the list
                new_instance = model_class(**attributes)
                instances_to_create.append(new_instance)

        # Bulk create all the accumulated instances
        model_class.objects.bulk_create(instances_to_create)


def get_components_and_required_attributes(root,component_entry, table):
    component_name = component_entry['component']
    pattern = r"(\w+)(\[[^\]]+\])"
    match = re.match(pattern, component_name)
    if match:
        component_name= match.group(1).strip()
        component_exp =  match.group(2).strip()
        component_elements = root.findall(component_exp)
    else:
        component_elements = root.find('COMPONENTS').findall(component_name)
    attribute_list = ComponentAttributes.objects.filter(table=table, component=component_name)
    return component_elements, attribute_list, component_name

def evaluate_operation(expression,component):
    # Replace all the variable names in [variable]{dictionary} with their dictionary substitutions
    pattern = r"\[([a-zA-Z]+)\]\{([a-zA-Z0-9:, ]+)\}"
    matches = re.findall(pattern, expression)
    for match in matches:
        variable_name = match[0].strip()
        attribute_value = component.get(variable_name)
        dictionary_str = match[1].strip()
        dictionary = {}
        for item in dictionary_str.split(","):
            key, value = item.strip().split(":")
            dictionary[key.strip()] = value.strip()

        if attribute_value in dictionary:
            variable_value = dictionary[attribute_value]
            expression = expression.replace('{'+dictionary_str+'}', '')
            expression = expression.replace('['+variable_name+']', variable_value)

    #see if there are any custom functions
    expression=run_custom_functions(expression,int(component.get('IID')))

    # Replace all the attribute names in [] with their values
    pattern = re.compile(r'\[([^\]]+)\]')
    matches = pattern.findall(expression)
    for match in matches:
        attribute_value = component.get(match.strip())
        expression = expression.replace('['+match.strip() +']', attribute_value)

    try:
        result = eval(expression)
        return result
    except:
        return expression

def run_custom_functions(expression,component_IID):
    # Dictionary of functions
    function_mapping = {
        "get_nearest_bus": find_nearest_bus,
        "get_pin_nearest_bus": find_nearest_pin_bus,
    }
    # Search for function placeholders in the expression
    function_placeholders = re.findall(r'{(\w+)(?:\((\w+)\))?\}', expression)
    if component_IID == 3600 or component_IID ==3808:
        myluck=function_placeholders
    for func_name, input_value in function_placeholders:
        if func_name in function_mapping:
            if input_value:
                func_result = str(function_mapping[func_name](component_IID, input_value))
            else:
                func_result = str(function_mapping[func_name](component_IID))

            expression = expression.replace(f'{{{func_name}{f"({input_value})" if input_value else ""}}}', func_result)

    return expression


def find_nearest_bus(component_iid, visited=None, depth=0):
    if visited is None:
        visited = set()

    bus_components = ['BUS', 'XFORM3W', 'CAPACITOR', 'SYNGEN', 'SINGLESWITCH', 'DOUBLESWITCH', 'FUSE', 'HVCB', 'LVCB', 'RECLOSER']

    # Check if the component is directly connected to a bus
    direct_connection = Connection.objects.filter(
        (Q(FromIID=component_iid, ToElement__in=bus_components) & ~Q(ToIID__in=visited)) |
        (Q(ToIID=component_iid, FromElement__in=bus_components) & ~Q(FromIID__in=visited))
    ).first()

    if direct_connection:
        return direct_connection.FromIID if direct_connection.ToIID == component_iid else direct_connection.ToIID

    # Mark this component as visited to avoid loops
    visited.add(component_iid)

    # Get all connections involving this component
    connections = Connection.objects.filter(
        Q(FromIID=component_iid) | Q(ToIID=component_iid)
    )

    nearest_bus_iid = 0
    shortest_depth = float('inf')

    for connection in connections:
        if connection.FromIID == component_iid:
            connected_iid = connection.ToIID
        else:
            connected_iid = connection.FromIID

        connected_element = connection.ToElement if connection.FromIID == component_iid else connection.FromElement

        if connected_element not in bus_components and connected_iid not in visited:
            # Recurse into the connected component
            bus_iid = find_nearest_bus(connected_iid, visited, depth + 1)
            if bus_iid !=0 and depth + 1 < shortest_depth:
                shortest_depth = depth + 1
                nearest_bus_iid = bus_iid

    return nearest_bus_iid
def find_nearest_pin_bus(component_iid, pin, visited=None, depth=0):
    if visited is None:
        visited = set()

    bus_components = ['BUS', 'XFORM3W', 'CAPACITOR', 'SYNGEN', 'SINGLESWITCH', 'DOUBLESWITCH', 'FUSE', 'HVCB', 'LVCB', 'RECLOSER']

    if depth == 0:
        # Check if the component pin is directly connected to a bus or XFORM3W
        direct_connection = Connection.objects.filter(
            Q(FromIID=component_iid, FromPin=pin, ToElement__in=bus_components) |
            Q(ToIID=component_iid, ToPin=pin, FromElement__in=bus_components)
        ).first()
    else:
        # Check if the component is directly connected to a bus or XFORM3W in either direction
        direct_connection = Connection.objects.filter(
            (Q(FromIID=component_iid, ToElement__in=bus_components) & ~Q(FromIID__in=visited)) |
            (Q(ToIID=component_iid, FromElement__in=bus_components) & ~Q(ToIID__in=visited))
        ).first()

    if direct_connection:
        return direct_connection.FromIID if direct_connection.ToIID == component_iid else direct_connection.ToIID

    # Mark this component as visited to avoid loops
    visited.add(component_iid)

    # Get all connections involving this component
    if depth == 0:
        connections = Connection.objects.filter(
            Q(FromIID=component_iid, FromPin=pin) | Q(ToIID=component_iid, ToPin=pin)
        )
    else:
        connections = Connection.objects.filter(
            Q(FromIID=component_iid) | Q(ToIID=component_iid)
        )

    nearest_bus_iid = 0
    shortest_depth = float('inf')

    for connection in connections:
        if connection.FromIID == component_iid:
            connected_iid = connection.ToIID
        else:
            connected_iid = connection.FromIID

        connected_element = connection.ToElement if connection.FromIID == component_iid else connection.FromElement

        if connected_element not in bus_components and connected_iid not in visited:
            # Recurse into the connected component
            bus_iid = find_nearest_bus(connected_iid, visited, depth + 1)
            if bus_iid !=0 and depth + 1 < shortest_depth:
                shortest_depth = depth + 1
                nearest_bus_iid = bus_iid

    return nearest_bus_iid
def populate_network_from_components():
    populate_nodes_from_buses()
    populate_edges_from_nodes()
    # add voltages to nodes starting at Xul-Ha
    construct_node_voltages()
    # Add strings to nodes for display
    construct_node_strings()
    # Add strings to edges for display
    construct_edge_strings()
    get_islands()
def populate_nodes_from_buses():
    buses = Bus.objects.all()

    # Fetch loads, capacitors, and generators related to all buses
    loads_counts_lumped = LoadEtap.objects.filter(type='LUMPEDLOAD').values('bus_iid').annotate(load_count=Count('id'))
    loads_counts_single = LoadEtap.objects.filter(type='STLOAD').values('bus_iid').annotate(load_count=Count('id'))
    loads_counts_feeder = Feeder.objects.all().values('etap_iid').annotate(load_count=Count('id'))

    # Create a dictionary to store counts by bus ID
    loads_counts_lumped_dict = {item['bus_iid']: item['load_count'] for item in loads_counts_lumped}
    loads_counts_single_dict = {item['bus_iid']: item['load_count'] for item in loads_counts_single}
    loads_counts_feeder_dict = {item['etap_iid']: item['load_count'] for item in loads_counts_feeder}

    # et existing bus_etap_iids and identify Nodes to delete
    existing_bus_etap_iids = set(Node.objects.values_list('bus_etap_iid', flat=True))
    current_bus_etap_iids = {bus.etap_iid for bus in buses}
    transformer_etap_iids = set(Transformer.objects.filter(type='XFORM3W').values_list('etap_iid', flat=True))
    switch_etap_iids = set(Switch.objects.all().values_list('etap_iid', flat=True))
    generator_etap_iids = set(Generator.objects.all().values_list('etap_iid', flat=True))
    ppa_etap_iids = set(PPAGenerator.objects.all().values_list('etap_iid', flat=True))
    capacitor_etap_iids = set(Capacitor.objects.all().values_list('etap_iid', flat=True))

    # Find and delete nodes that are not in buses.etap_iid
    to_delete_bus_etap_iids = existing_bus_etap_iids - current_bus_etap_iids - transformer_etap_iids - switch_etap_iids
    to_delete_bus_etap_iids = to_delete_bus_etap_iids - generator_etap_iids - capacitor_etap_iids - ppa_etap_iids
    Node.objects.filter(bus_etap_iid__in=to_delete_bus_etap_iids).delete()

    common_dict = {
        'loads_string': '',
        'generators_string': '',
        'capacitors_string': '',
        'switches_string': '',
        'transformers_string': '',
        'voltage_string': '',
        'layer': '',
    }

    # Loop through all buses and update or create Node entries
    count = 0
    buses_count = len(buses)
    for bus in buses:
        count += 1
        print('Processing bus node {} of {}'.format(count, buses_count))
        Node.objects.update_or_create(
            bus_etap_iid=bus.etap_iid,
            defaults={
                **common_dict,
                'bus_id': bus.id,
                'type': bus.type,
                'loads_lumped': loads_counts_lumped_dict.get(bus.etap_iid, 0),
                'loads_single': loads_counts_single_dict.get(bus.etap_iid, 0),
                'loads_feeder': loads_counts_feeder_dict.get(bus.etap_iid, 0),
                'capacitors': 0,
                'generators': 0,
                'etap_id': bus.etap_id,
            }
        )
    # Loop through all three winding transformers & update or create Node entries
    print('Processing transformer nodes')
    for transformer in Transformer.objects.filter(type='XFORM3W'):
        Node.objects.update_or_create(
            bus_etap_iid=transformer.etap_iid,
            defaults={
                **common_dict,
                'bus_id': transformer.id,
                'type': transformer.type,
                'loads_lumped': 0,
                'loads_single': 0,
                'loads_feeder': 0,
                'capacitors': 0,
                'generators': 0,
                'etap_id': transformer.etap_id,
            }
        )
    print('Processing switch nodes')
    for switch in Switch.objects.all():
        Node.objects.update_or_create(
            bus_etap_iid=switch.etap_iid,
            defaults={
                **common_dict,
                'bus_id': switch.id,
                'type': switch.type,
                'loads_lumped': 0,
                'loads_single': 0,
                'loads_feeder': 0,
                'capacitors': 0,
                'generators': 0,
                'etap_id': switch.etap_id,
                'status': switch.status,
            }
        )
    print('Processing capacitor nodes')
    for capacitor in Capacitor.objects.all():
        Node.objects.update_or_create(
            bus_etap_iid=capacitor.etap_iid,
            defaults={
                **common_dict,
                'bus_id': capacitor.id,
                'type': capacitor.type,
                'loads_lumped': 0,
                'loads_single': 0,
                'loads_feeder': 0,
                'capacitors': 1,
                'generators': 0,
                'etap_id': capacitor.etap_id,
            }
        )
    print('Processing generator nodes')
    for generator in Generator.objects.all():
        Node.objects.update_or_create(
            bus_etap_iid=generator.etap_iid,
            defaults={
                **common_dict,
                'bus_id': generator.id,
                'type': generator.type,
                'loads_lumped': 0,
                'loads_single': 0,
                'loads_feeder': 0,
                'capacitors': 0,
                'generators': 1,
                'etap_id': generator.etap_id,
            }
        )
    print('Processing ppa_generator nodes')
    etap_iids = set(Generator.objects.values_list('etap_iid', flat=True))
    for generator in PPAGenerator.objects.exclude(etap_iid__in=etap_iids):
        node_defaults = {
            **common_dict,
            'bus_id': generator.id,
            'type': 'SYNGEN',
            'loads_lumped': 0,
            'loads_single': 0,
            'loads_feeder': 0,
            'capacitors': 0,
            'generators': 1,
            'etap_id': generator.etap_id,
        }

        # Check if the Node object exists
        node, created = Node.objects.update_or_create(
            bus_etap_iid=generator.etap_iid,
            defaults=node_defaults
        )

        # If the object is newly created, set the 'user_LocX' and 'user_LocY' values
        if created:
            node.user_LocX = 0
            node.user_LocY = 0
            node.save()

def get_closest_voltage(voltage, voltage_min):
    sorted_levels = sorted(voltage_min.keys(), reverse=True)
    for i, level in enumerate(sorted_levels):
        if voltage >= voltage_min[level]:
            return level
    # If the voltage doesn't fall into any specified range, return the original voltage or handle as needed
    return voltage
def construct_node_voltages():
    node_queryset = Node.objects.all()
    edge_queryset = Edge.objects.all()
    buses = {bus.etap_iid: bus for bus in Bus.objects.filter(etap_iid__in=node_queryset.values('bus_etap_iid'))}
    nodes = {node.bus_etap_iid: node for node in node_queryset}
    edges_from = defaultdict(list)
    edges_to = defaultdict(list)
    for edge in edge_queryset:
        edges_from[edge.from_bus_iid].append(edge)
        edges_to[edge.to_bus_iid].append(edge)
    visited = set()
    voltage_min = {
        480: 200,
        115: 90,
        69: 60,
        34.5: 31,
        22: 20,
        14.5: 14,
        13.8: 13.4,
        13.2: 12.6,
        11: 10,
        6.9: 6.75,
        6.6: 5,
        4.16: 3,
        0.48: 0.3,
        0.24: 0,

    }
    starting_points = {
        14176: 115,  # Main Network XULHA
        23584: 22,  # Citrus Company of Belize (POC Hope Creek)
        99474: 6.6,  # Caye Caulker Bus37
        100484: 115,  # Bus44
        105396: 13.8,  # CMB=13.8_2
        103591: 0.48,  # Bus56
        66544: 115,  # M-SUB XFMR HS
    }
    # Initialize the stack with all starting points
    stack = list(starting_points.items())

    while stack:
        current_bus, current_voltage = stack.pop()
        if current_bus in visited:
            continue
        visited.add(current_bus)
        current_node = nodes.get(current_bus)
        if current_node:
            current_node.reference_voltage = current_voltage
            current_node.save()

            # Retrieve edges directly from the dictionaries
            current_edges = edges_from.get(current_bus, []) + edges_to.get(current_bus, [])
            for edge in current_edges:
                if edge.from_bus_iid == current_bus:
                    next_bus = edge.to_bus_iid
                else:
                    next_bus = edge.from_bus_iid
                if edge.transformer_ratio:
                    if edge.high_side == current_bus:
                        next_voltage = current_voltage / edge.transformer_ratio
                    else:
                        next_voltage = current_voltage * edge.transformer_ratio
                    next_voltage = get_closest_voltage(next_voltage, voltage_min)
                else:
                    next_voltage = current_voltage

                stack.append((next_bus, next_voltage))

def construct_node_strings():
    node_queryset = Node.objects.all()
    generators, ppa_generators, capacitors, loads, switches, loads_feeder = get_node_meta(node_queryset)
    transformers, conductors = get_edge_meta(node_queryset)
    buses = {bus.etap_iid: bus for bus in Bus.objects.filter(etap_iid__in=node_queryset.values('bus_etap_iid'))}
    nodes = {node.bus_etap_iid: node for node in node_queryset}
    node_list = [(node_element.bus_etap_iid) for node_element in node_queryset]
    for node in node_list:
        bus_instance = buses.get(node)
        node_instance = nodes.get(node)
        generator_string, capacitor_string, load_string, switch_string = build_node_meta(generators, ppa_generators, capacitors, loads, loads_feeder, switches, node, node_instance)
        layer = get_node_layer(node_instance)
        if bus_instance is not None:
            node_instance.loads_string = load_string
            node_instance.generators_string = generator_string
            node_instance.capacitors_string = capacitor_string
            node_instance.layer = layer
            node_instance.voltage_string = f"Voltage:{node_instance.reference_voltage}kV"
            node_instance.save()
        elif node_instance.type == 'XFORM3W':
            transformer_string, _, voltage_string = build_edge_meta(transformers, None, node, -1, 1, 0)
            node_instance.layer = "TX3Winding"
            node_instance.transformers_string = transformer_string
            node_instance.voltage_string = voltage_string
            node_instance.save()
        elif node_instance.type in ['SINGLESWITCH', 'DOUBLESWITCH', 'FUSE', 'HVCB', 'LVCB', 'RECLOSER']:
            if node_instance.type in ['FUSE', 'LVCB']:
                node_instance.layer = "lvSwitch"
            elif node_instance.type == 'DOUBLESWITCH':
                node_instance.layer = "doubleSwitch"
            else:
                node_instance.layer = "switch"
            node_instance.switches_string = switch_string
            node_instance.voltage_string = f"kV:{node_instance.reference_voltage}"
            node_instance.save()
        elif node_instance.type in ['CAPACITOR', 'SYNGEN']:
            node_instance.layer = layer
            node_instance.capacitors_string = capacitor_string
            node_instance.generators_string = generator_string
            node_instance.voltage_string = f"Voltage:{node_instance.reference_voltage}kV"
            node_instance.save()

def determine_bus_connection(from_node_id, to_node_id, max_depth, connections_i, elements_i, visited=None, depth=0):
    if visited is None:
        visited = set()
    bus_components = ['BUS', 'XFORM3W', 'CAPACITOR', 'SYNGEN', 'SINGLESWITCH', 'DOUBLESWITCH', 'FUSE', 'HVCB', 'LVCB', 'RECLOSER', 'UNIVERSALRELAY']
    # Check if the component is directly connected to the other component using the connections_i
    connections_for_from_node = connections_i.get(from_node_id, set())
    direct_connection = to_node_id in connections_for_from_node and from_node_id not in visited

    if direct_connection:
        return 1

    if depth == max_depth:
        return 0

    # Mark this component as visited to avoid loops
    visited.add(from_node_id)

    # Extract connections involving the current component from the provided indexed connections
    connections = connections_i.get(from_node_id, set())

    for connected_iid in connections:
        if connected_iid != from_node_id and connected_iid not in visited:
            connected_elements = elements_i.get(connected_iid, set())
            connected_element = next(iter(connected_elements), None)
            if connected_element not in bus_components:
                # Recurse into the connected component
                if determine_bus_connection(connected_iid, to_node_id, max_depth, connections_i, elements_i, visited, depth + 1):
                    return 1

    return 0

def get_connection_set(nodes):
    node_ids_set = set(node.bus_etap_iid for node in nodes)

    conductors = set(Conductor.objects.filter(from_bus_iid__in=node_ids_set).values_list('from_bus_iid', 'to_bus_iid'))

    switches_1 = set(Switch.objects.filter(
        Q(from_bus_iid__in=node_ids_set) |
        Q(etap_iid__in=node_ids_set)
    ).values_list('etap_iid', 'from_bus_iid'))
    switches_2 = set(Switch.objects.filter(
        Q(to_bus_iid__in=node_ids_set) |
        Q(etap_iid__in=node_ids_set)
    ).values_list('etap_iid', 'to_bus_iid'))
    switches_3 = set(Switch.objects.filter(
        Q(to_bus_iid_2__in=node_ids_set) |
        Q(etap_iid__in=node_ids_set)
    ).values_list('etap_iid', 'to_bus_iid_2'))
    switches = switches_1 | switches_2 | switches_3

    transformers = set(Transformer.objects.filter(
        Q(type='XFORM2W', primary_bus_iid__in=node_ids_set) |
        Q(type='XFORM2W', secondary_bus_iid__in=node_ids_set)
    ).values_list('primary_bus_iid', 'secondary_bus_iid'))
    transformers_1 = set(Transformer.objects.filter(
        Q(type='XFORM3W', etap_iid__in=node_ids_set) |
        Q(type='XFORM3W', primary_bus_iid__in=node_ids_set)
    ).values_list('etap_iid', 'primary_bus_iid'))
    transformers_2 = set(Transformer.objects.filter(
        Q(type='XFORM3W', etap_iid__in=node_ids_set) |
        Q(type='XFORM3W', secondary_bus_iid__in=node_ids_set)
    ).values_list('etap_iid', 'secondary_bus_iid'))
    transformers_3 = set(Transformer.objects.filter(
        Q(type='XFORM3W', etap_iid__in=node_ids_set) |
        Q(type='XFORM3W', tertiary_bus_iid__in=node_ids_set)
    ).values_list('etap_iid', 'tertiary_bus_iid'))
    transformers = transformers | transformers_1 | transformers_2 | transformers_3

    capacitors = set(Capacitor.objects.filter(etap_iid__in=node_ids_set).values_list('etap_iid', 'bus_iid'))

    generators = set(Generator.objects.filter(etap_iid__in=node_ids_set).values_list('etap_iid', 'bus_iid'))

    ppa_generators = set(PPAGenerator.objects.filter(etap_iid__in=node_ids_set).values_list('etap_iid', 'bus_iid'))

    connections = set(Connection.objects.all().values_list('FromIID', 'ToIID'))
    connections_i = {}
    for from_id, to_id in connections:
        connections_i.setdefault(from_id, set()).add(to_id)
        connections_i.setdefault(to_id, set()).add(from_id)

    elements = set(Connection.objects.all().values_list('FromElement', 'FromIID', 'ToElement', 'ToIID'))
    elements_i = {}

    for from_element, from_iid, to_element, to_iid in elements:
        elements_i.setdefault(from_iid, set()).add(from_element)
        elements_i.setdefault(to_iid, set()).add(to_element)

    return conductors, switches, transformers, capacitors, generators, ppa_generators, connections, connections_i, elements_i

def count_connection_set(objects, from_node_id, to_node_id):
    return ((from_node_id, to_node_id) in objects) + ((to_node_id, from_node_id) in objects)

def populate_edges_from_nodes():
    nodes = Node.objects.all()

    # Fetch conductors and transformers related to all nodes
    conductors_m = Conductor.objects.filter(
        Q(from_bus_iid__in=[node.bus_etap_iid for node in nodes]) |
        Q(to_bus_iid__in=[node.bus_etap_iid for node in nodes])
    )
    switches_m = Switch.objects.filter(
        Q(from_bus_iid__in=[node.bus_etap_iid for node in nodes]) |
        Q(to_bus_iid__in=[node.bus_etap_iid for node in nodes]) |
        Q(to_bus_iid_2__in=[node.bus_etap_iid for node in nodes])
    )
    transformers_m = Transformer.objects.filter(
        Q(primary_bus_iid__in=[node.bus_etap_iid for node in nodes]) |
        Q(secondary_bus_iid__in=[node.bus_etap_iid for node in nodes]) |
        Q(tertiary_bus_iid__in=[node.bus_etap_iid for node in nodes])
    )

    conductors, switches, transformers, capacitors, generators, ppa_generators, connections, connections_i, elements_i = get_connection_set(nodes)

    # Create a dictionary to store the status for each edge
    edge_statuses = construct_edge_status(switches_m, nodes)

    # Create the Edge entries in bulk
    edge_list = []
    seen_edges = set()
    count = 0
    nodes_count = len(nodes)
    for node1 in nodes:
        count += 1
        print('Processing node {} of {}'.format(count, nodes_count))
        for node2 in nodes:
            if node1 != node2:
                from_node_id, to_node_id = sorted((node1.bus_etap_iid, node2.bus_etap_iid))

                # Skip this pair if we've seen it before
                if (from_node_id, to_node_id) in seen_edges:
                    continue
                conductors_count = count_connection_set(conductors, from_node_id, to_node_id)
                switches_count = count_connection_set(switches, from_node_id, to_node_id)
                transformers_count = count_connection_set(transformers, from_node_id, to_node_id)
                direct_count = 0
                if conductors_count == 0 and transformers_count == 0 and switches_count == 0:
                    # if node is a generator, capacitor, or bus check for direct connection to bus
                    if node1.type == 'CAPACITOR' or node2.type == 'CAPACITOR':
                        direct_count = count_connection_set(capacitors, from_node_id, to_node_id)
                    elif node1.type == 'SYNGEN' or node2.type == 'SYNGEN':
                        direct_count = count_connection_set(generators, from_node_id, to_node_id)
                        if direct_count == 0:
                            direct_count += count_connection_set(ppa_generators, from_node_id, to_node_id)
                    elif node1.type == 'BUS' or node2.type == 'BUS':
                        direct_count = count_connection_set(connections, from_node_id, to_node_id)
                        if direct_count == 0 and node1.type == 'BUS' and node2.type == 'BUS':
                            max_depth = 5
                            direct_count = determine_bus_connection(from_node_id, to_node_id, max_depth, connections_i, elements_i)

                status = edge_statuses.get((from_node_id, to_node_id))

                R_zero_sum, C_pos_sum, C_zero_sum, L_pos_sum, L_zero_sum, min_current_limit = get_conductor_parameters(
                    conductors_m, conductors_count, from_node_id, to_node_id)

                threeWinding, transformer_ratio, high_node_iid = get_transformer_parameters(transformers_m, transformers_count, from_node_id, to_node_id)

                if from_node_id == node1.bus_etap_iid:
                    from_node = node1
                    to_node = node2
                else:
                    from_node = node2
                    to_node = node1
                if conductors_count > 0 or transformers_count > 0 or switches_count > 0 or direct_count > 0:
                    edge_list.append(
                        Edge(
                            island=node1.island,
                            from_node=from_node,
                            to_node=to_node,
                            from_bus_iid=from_node_id,
                            to_bus_iid=to_node_id,
                            conductors=conductors_count,
                            switches=switches_count,
                            transformers=transformers_count,
                            threeWinding=threeWinding,
                            transformer_ratio=transformer_ratio,
                            high_side=high_node_iid,
                            status=status,
                            R_zero=R_zero_sum,
                            C_pos=C_pos_sum,
                            C_zero=C_zero_sum,
                            L_pos=L_pos_sum,
                            L_zero=L_zero_sum,
                            current_limit=min_current_limit,
                            transformers_string='',
                            conductors_string='',
                        )
                    )
                seen_edges.add((from_node_id, to_node_id))

    # Delete existing edges
    Edge.objects.all().delete()

    try:
        Edge.objects.bulk_create(edge_list)
    except IntegrityError as e:
        # Handle any IntegrityError if necessary
        pass

def construct_edge_status(switches,nodes):
    edge_status = {}

    for switch in switches:
        # pin 0 is always closed for double switches only
        from_node_id, to_node_id = sorted((switch.etap_iid, switch.from_bus_iid))
        if switch.to_bus_iid_2:
            edge_status[(from_node_id, to_node_id)] = "Closed"
        else:
            current_status = edge_status.get((from_node_id, to_node_id))
            switch_status = "Closed" if switch.status is None else switch.status
            if current_status is None or current_status == "Closed":
                edge_status[(from_node_id, to_node_id)] = switch_status

        # set pin 1 status
        from_node_id, to_node_id = sorted((switch.etap_iid, switch.to_bus_iid))
        current_status = edge_status.get((from_node_id, to_node_id))
        switch_status = "Closed" if switch.status is None else switch.status
        switch_status = "Open" if switch_status == "Position B" else switch_status
        switch_status = "Closed" if switch_status == "Position A" else switch_status
        if current_status is None or current_status == "Closed":
            edge_status[(from_node_id, to_node_id)] = switch_status

        # set pin 2 status for double switches
        if switch.to_bus_iid_2:
            from_node_id, to_node_id = sorted((switch.etap_iid, switch.to_bus_iid_2))
            current_status = edge_status.get((from_node_id, to_node_id))
            switch_status = "Closed" if switch.status is None else switch.status
            switch_status = "Open" if switch_status == "Position A" else switch_status
            switch_status = "Closed" if switch_status == "Position B" else switch_status
            if current_status is None or current_status == "Closed":
                edge_status[(from_node_id, to_node_id)] = switch_status

    # Calculate the status for each edge based on switches
    edge_statuses = {}
    for node1 in nodes:
        for node2 in nodes:
            if node1 != node2:
                from_node_id, to_node_id = sorted((node1.bus_etap_iid, node2.bus_etap_iid))
                status = edge_status.get((from_node_id, to_node_id))
                if status is None:
                    status = "Closed"
                edge_statuses[(from_node_id, to_node_id)] = status

    return edge_statuses
def construct_edge_strings():
    edge_queryset = Edge.objects.all()
    node_queryset = Node.objects.all()

    edges = {f'{edge.from_bus_iid}-{edge.to_bus_iid}': edge for edge in edge_queryset}
    edge_list = [(edge_element.from_bus_iid, edge_element.to_bus_iid) for edge_element in edge_queryset]
    transformers, conductors = get_edge_meta(node_queryset)

    for edge in edge_list:
        source, target = sorted((edge[0], edge[1]))
        edge_instance = edges.get(f"{source}-{target}")
        transformer_string, conductor_string, _ = build_edge_meta(transformers, conductors,
                                                               source, target, edge_instance.transformers,
                                                               edge_instance.conductors)
        edge_instance.transformers_string = transformer_string
        edge_instance.conductors_string = conductor_string
        if edge_instance.conductors > 0:
            edge_instance.layer = "conductor"
        else:
            edge_instance.layer = "edge"
        edge_instance.save()

def get_transformer_parameters(transformers, transformers_count, from_node_id, to_node_id):
    threeWinding = None
    transformer_ratio = None
    high_node_iid = None
    if transformers_count > 0:
        filtered_transformers = transformers.filter(
            Q(type='XFORM2W', primary_bus_iid=from_node_id, secondary_bus_iid=to_node_id) |
            Q(type='XFORM2W', primary_bus_iid=to_node_id, secondary_bus_iid=from_node_id) |
            Q(type='XFORM3W', etap_iid__in=[from_node_id, to_node_id], primary_bus_iid__in=[from_node_id, to_node_id]) |
            Q(type='XFORM3W', etap_iid__in=[from_node_id, to_node_id], secondary_bus_iid__in=[from_node_id, to_node_id]) |
            Q(type='XFORM3W', etap_iid__in=[from_node_id, to_node_id], tertiary_bus_iid__in=[from_node_id, to_node_id])
        )
        for transformer in filtered_transformers:
            if transformer.tertiary_S:
                threeWinding = transformer.tertiary_bus_iid

            # Define a dictionary to map bus_iids to their corresponding voltage attributes
            bus_to_voltage = {
                transformer.primary_bus_iid: transformer.primary_V,
                transformer.secondary_bus_iid: transformer.secondary_V,
                transformer.tertiary_bus_iid: transformer.tertiary_V
            }

            # Get the voltages corresponding to from_node_id and to_node_id
            from_voltage = bus_to_voltage.get(from_node_id)
            to_voltage = bus_to_voltage.get(to_node_id)

            if transformer.type == 'XFORM2W':
                if from_voltage > to_voltage:
                    high_node_iid = from_node_id
                    high_voltage = from_voltage
                    low_voltage = to_voltage
                else:
                    high_node_iid = to_node_id
                    high_voltage = to_voltage
                    low_voltage = from_voltage
                if high_voltage is not None and low_voltage is not None:
                    transformer_ratio = high_voltage / low_voltage
            else:
                terminal_iid = to_node_id if from_node_id == transformer.etap_iid else from_node_id
                if transformer.primary_bus_iid == terminal_iid:
                    high_node_iid = transformer.primary_bus_iid
                    transformer_ratio = 1
                else:
                    high_node_iid = transformer.etap_iid
                    high_voltage = transformer.primary_V
                    if transformer.secondary_bus_iid == terminal_iid:
                        low_voltage = transformer.secondary_V
                    else:
                        low_voltage = transformer.tertiary_V
                    if high_voltage is not None and low_voltage is not None:
                        transformer_ratio = high_voltage / low_voltage

            # Store the transformer ratio in the transformer object (or however you wish to store this value)
            transformer.transformer_ratio = transformer_ratio
    return threeWinding, transformer_ratio, high_node_iid

def get_conductor_parameters(conductors, conductors_count, from_node_id, to_node_id):
    R_zero_sum = 0
    C_pos_sum = 0
    C_zero_sum = 0
    L_pos_sum = 0
    L_zero_sum = 0
    min_current_limit = None
    if conductors_count > 0:
        filtered_conductors = conductors.filter(
            Q(type__in=["CABLE", "XLINE"]) &
            (
                    (Q(from_bus_iid=from_node_id) & Q(to_bus_iid=to_node_id)) |
                    (Q(from_bus_iid=to_node_id) & Q(to_bus_iid=from_node_id))
            )
        )
        for conductor in filtered_conductors:
            # Sum the values of the fields (if they are not None)
            if conductor.R_zero is not None:
                R_zero_sum += conductor.R_zero
            if conductor.C_pos is not None:
                C_pos_sum += conductor.C_pos
            if conductor.C_zero is not None:
                C_zero_sum += conductor.C_zero
            if conductor.L_pos is not None:
                L_pos_sum += conductor.L_pos
            if conductor.L_zero is not None:
                L_zero_sum += conductor.L_zero
            if conductor.current_limit is not None and conductor.current_limit > 0:
                if min_current_limit is None or conductor.current_limit < min_current_limit:
                    min_current_limit = conductor.current_limit
    return R_zero_sum, C_pos_sum, C_zero_sum, L_pos_sum, L_zero_sum, min_current_limit
def get_islands():
    island_name_dict = {
        'Main': 2432,
        'CCK': 99443,
    }

    Node.objects.update(island=None)
    Edge.objects.update(island=None)

    Island.objects.all().delete()
    # Assuming you have a model named "Connection" with fields "from_bus_iid" and "to_bus_iid"
    edge_queryset = Edge.objects.all()

    # Create a list of connections (edges) as tuples: [(from_bus, to_bus), ...]
    edges = [(edge_element.from_bus_iid, edge_element.to_bus_iid) for edge_element in edge_queryset]

    # Create a graph and add edges
    G = nx.Graph()
    G.add_edges_from(edges)

    # Find connected components (islands)
    islands = list(nx.connected_components(G))

    # Iterate through islands to count buses
    for idx, island in enumerate(islands, start=1):
        island_buses = list(island)
        num_buses_in_island = len(island_buses)
        num_edges_in_island = sum(G.has_edge(bus1, bus2) for bus1 in island_buses for bus2 in island_buses)
        print(f"Island {idx}: Number of Buses = {num_buses_in_island}")
        island_instance = Island.objects.create(
            name=f"Island {idx}",
            nodes=num_buses_in_island,
            edges=num_edges_in_island,
        )
        island_instance.save()

        # Update nodes in the current island
        Node.objects.filter(bus_etap_iid__in=island).update(island=island_instance)

        # Update edges in the current island
        Edge.objects.filter(from_bus_iid__in=island, to_bus_iid__in=island).update(island=island_instance)

        # Check if any node in the current island matches a bus_etap_iid in the dictionary
        for desired_name, bus_iid in island_name_dict.items():
            if bus_iid in island_buses:
                island_instance.name = desired_name
                island_instance.save()
                break

def graph_normalization():
    # Define the canvas size and normalization factor
    canvas_width = 2500
    canvas_height = 2500
    x_normalization_factor = canvas_width / 5000
    y_normalization_factor = canvas_height / 5000
    return x_normalization_factor, y_normalization_factor
def return_network_graph(min_version = False):
    if min_version:
        edge_queryset = EdgeMin.objects.all()
        node_queryset = NodeMin.objects.all()
    else:
        edge_queryset = Edge.objects.all()
        node_queryset = Node.objects.all()

    # Create a graph and add edges
    nodes = [(node_element.bus_etap_iid) for node_element in node_queryset]
    edges = [(edge_element.from_bus_iid, edge_element.to_bus_iid) for edge_element in edge_queryset]
    G = nx.Graph()
    G.add_nodes_from(nodes)
    G.add_edges_from(edges)



    # Create a dictionary to store Bus and Node instances by their etap_iid
    buses = {bus.etap_iid: bus for bus in Bus.objects.filter(etap_iid__in=node_queryset.values('bus_etap_iid'))}
    transformers = {tx.etap_iid: tx for tx in Transformer.objects.filter(etap_iid__in=node_queryset.values('bus_etap_iid'))}
    switches = {switch.etap_iid: switch for switch in Switch.objects.filter(etap_iid__in=node_queryset.values('bus_etap_iid'))}
    nodes = {node.bus_etap_iid: node for node in node_queryset}
    edges = {f'{edge.from_bus_iid}-{edge.to_bus_iid}': edge for edge in edge_queryset}



    # To only display the largest island
    #largest_island = max(nx.connected_components(G), key=len)
    #G_subgraph = G.subgraph(largest_island)
    G_subgraph = G

    if G_subgraph.number_of_nodes() == 0 or G_subgraph.number_of_edges() == 0:
        json_data = '{}'
    else:
        json_data = {
                "nodes": [],
                "edges": []
        }

        # Define the canvas size and normalization factor
        x_normalization_factor, y_normalization_factor = graph_normalization()
        json_data = add_LC_labels(json_data, x_normalization_factor, y_normalization_factor)
        # Assuming G_subgraph is your networkx subgraph
        for node in G_subgraph.nodes():
            bus_instance = buses.get(node)
            tx_instance = transformers.get(node)
            switch_instance = switches.get(node)
            node_instance = nodes.get(node)

            if node_instance.user_LocX is not None and node_instance.user_LocY is not None:
                x_normalized = node_instance.user_LocX * x_normalization_factor
                y_normalized = node_instance.user_LocY * y_normalization_factor
            elif node_instance.type == 'XFORM3W':
                x_normalized = tx_instance.etap_LocX * x_normalization_factor
                y_normalized = tx_instance.etap_LocY * y_normalization_factor
            elif node_instance.type in ['SINGLESWITCH', 'DOUBLESWITCH', 'FUSE', 'HVCB', 'LVCB', 'RECLOSER']:
                x_normalized = switch_instance.etap_LocX * x_normalization_factor
                y_normalized = switch_instance.etap_LocY * y_normalization_factor
            elif node_instance.type == 'BUS':
                x_normalized = bus_instance.etap_LocX * x_normalization_factor
                y_normalized = bus_instance.etap_LocY * y_normalization_factor

            json_data["nodes"].append({
                "data": {
                    "id": node,
                    "label": node_instance.etap_id,  # Use Etap_iid as label
                    "X": x_normalized,  # Normalized X coordinate
                    "Y": y_normalized,  # Normalized Y coordinate
                    "layer": node_instance.layer,  # color differentiation
                    "type": "bus",  # type of node. Can be label
                    "generators": node_instance.generators_string if node_instance.generators_string else '',
                    "loads": node_instance.loads_string if node_instance.loads_string else '',
                    "capacitors": node_instance.capacitors_string if node_instance.capacitors_string else '',
                    "transformers": node_instance.transformers_string if node_instance.transformers_string else '',
                    "switches": node_instance.switches_string if node_instance.switches_string else '',
                    "voltage": node_instance.voltage_string if node_instance.voltage_string else '',
                }
            })

        for edge in G_subgraph.edges():
            source, target = sorted((edge[0], edge[1]))
            edge_instance = edges.get(f"{source}-{target}")
            if edge_instance:
                status = edge_instance.status if edge_instance else "Closed"
                transformers_string = edge_instance.transformers_string if edge_instance.transformers_string else ''
                conductors_string = edge_instance.conductors_string if edge_instance.conductors_string else ''
                layer = edge_instance.layer
                if edge_instance.transformers > 0:
                    if edge_instance.threeWinding or edge_instance.threeWinding == 0:
                        TX = "None"
                    else:
                        TX = "Transformer"
                else:
                    TX = "None"
                if edge_instance.high_side:
                    high_node = source if source == edge_instance.high_side else target
                    low_node = target if source == edge_instance.high_side else source
                else:
                    high_node = source
                    low_node = target
            else:
                status = "Closed"
                TX = "None"
                high_node = source
                low_node = target
                transformers_string = ''
                conductors_string = ''
                layer = "edge"

            json_data["edges"].append({
                "data": {
                    "id": f"{source}_{target}",
                    "source": high_node,
                    "target": low_node,
                    "status": status,  # Set the "layer" attribute to the status value
                    "TX": TX,
                    "transformers": transformers_string,
                    "conductors": conductors_string,
                    "layer": layer,
                }
            })

    json_string = json.dumps(json_data)
    return json_string
def add_LC_labels(json_data, x_normalization_factor, y_normalization_factor):
    labels = Label.objects.all()
    for label in labels:
        json_data["nodes"].append({
            "data": {
                "id": label.id,  # Unique ID for the label
                "label": label.label,  # Label text
                "X": label.user_LocX*x_normalization_factor,  # X coordinate
                "Y": label.user_LocY*y_normalization_factor,  # Y coordinate
                "type": "label",  # Node type indicating that this is a label
                "layer": "text",
                "generators": None,
                "loads": None,
                "capacitors": None,
                "voltage": None,
            }
        })
    return json_data

def get_node_layer(node_instance):
    if (node_instance.generators > 0 or node_instance.capacitors > 0) and node_instance.loads_feeder > 0:
        layer = "both"
    elif node_instance.loads_feeder > 0:
        layer = "load_feeder"
    elif node_instance.generators > 0:
        layer = "generator"
    elif node_instance.capacitors > 0:
        layer = "capacitor"
    elif node_instance.loads_single > 0 or node_instance.loads_lumped > 0:
        layer = "load_etap"
    else:
        layer = "none"
    return layer

def get_node_meta(node_queryset):
    # Initialize defaultdicts with list as the default factory function
    generators = defaultdict(list)
    ppa_generators = defaultdict(list)
    capacitors = defaultdict(list)
    loads = defaultdict(list)
    loads_feeder = defaultdict(list)
    switches = defaultdict(list)

    # Group Generator objects by bus_iid
    for generator in Generator.objects.filter(etap_iid__in=node_queryset.values('bus_etap_iid')):
        generators[generator.etap_iid].append(generator)

    # Group PPA_Generator objects by bus_iid
    for generator in PPAGenerator.objects.filter(etap_iid__in=node_queryset.values('bus_etap_iid')):
        ppa_generators[generator.etap_iid].append(generator)

    # Group Capacitor objects by bus_iid
    for capacitor in Capacitor.objects.filter(etap_iid__in=node_queryset.values('bus_etap_iid')):
        capacitors[capacitor.etap_iid].append(capacitor)

    # Group LoadEtap objects by bus_iid
    for load in LoadEtap.objects.filter(bus_iid__in=node_queryset.values('bus_etap_iid')):
        loads[load.bus_iid].append(load)

    for feeder in Feeder.objects.filter(etap_iid__in=node_queryset.values('bus_etap_iid')):
        loads_feeder[feeder.etap_iid].append(feeder)

    for switch in Switch.objects.filter(etap_iid__in=node_queryset.values('bus_etap_iid')):
        switches[switch.etap_iid].append(switch)

    return generators, ppa_generators, capacitors, loads, switches, loads_feeder

def get_edge_meta(node_queryset):
    # Initialize defaultdicts with list as the default factory function
    transformers = defaultdict(list)
    conductors = defaultdict(list)

    # Group Transformer objects by edge
    node_id_list = list(node_queryset.values_list('bus_etap_iid', flat=True))

    transformer_ids = Transformer.objects.filter(
        Q(primary_bus_iid__in=node_id_list) |
        Q(secondary_bus_iid__in=node_id_list) |
        Q(tertiary_bus_iid__in=node_id_list)
    ).values_list('id', flat=True)

    for transformer in Transformer.objects.filter(id__in=transformer_ids):
        if transformer.primary_bus_iid:
            transformers[transformer.primary_bus_iid].append(transformer)
        if transformer.secondary_bus_iid:
            transformers[transformer.secondary_bus_iid].append(transformer)
        if transformer.tertiary_bus_iid:
            transformers[transformer.tertiary_bus_iid].append(transformer)

    # Group conductor objects by edge
    conductor_ids = Conductor.objects.filter(
        Q(from_bus_iid__in=node_id_list) |
        Q(to_bus_iid__in=node_id_list)
    ).values_list('id', flat=True)
    for conductor in Conductor.objects.filter(id__in=conductor_ids):
        conductors[conductor.from_bus_iid].append(conductor)
        conductors[conductor.to_bus_iid].append(conductor)

    return transformers, conductors

def build_node_meta(generators, ppa_generators, capacitors, loads, loads_feeder, switches, node, node_instance):
    if node_instance.generators > 0:
        generator_strings = []

        for generator_instance in generators.get(node, []):
            generator_str = f"{generator_instance.etap_id} -> P<sub>max</sub>:{generator_instance.P_max:.2f} kV:{generator_instance.op_voltage:.1f}"
            generator_strings.append(generator_str)

        for generator_instance in ppa_generators.get(node, []):
            generator_str = f"{generator_instance.etap_id} -> P<sub>max</sub>:{generator_instance.P_max:.2f} P<sub>min</sub>:{generator_instance.P_min:.1f}"
            generator_strings.append(generator_str)

        generator_string = "\n".join(generator_strings).strip('\n') if generator_strings else ''
        generator_string = '<span style="font-size: larger;">Generators:</span>\n' + generator_string
    else:
        generator_string = ''

    if node_instance.capacitors > 0:
        capacitor_strings = []

        for capacitor_instance in capacitors.get(node, []):
            capacitor_str = f'{capacitor_instance.etap_id} -> Q<sub>total</sub>(MVAR):{capacitor_instance.Q:.3f} kV:{capacitor_instance.op_voltage:.1f}'
            capacitor_strings.append(capacitor_str)

        capacitor_string = "\n".join(capacitor_strings).strip('\n') if capacitor_strings else ''
        capacitor_string = '<span style="font-size: larger;">Capacitors:</span>\n' + capacitor_string
    else:
        capacitor_string = ''

    if node_instance.loads_single > 0 or node_instance.loads_lumped > 0 or node_instance.loads_feeder > 0:
        load_strings = []
        if node_instance.loads_single > 0 or node_instance.loads_lumped > 0:
            for load_instance in loads.get(node, []):
                load_str = f'{load_instance.etap_id} -> P(MW):{load_instance.P:.3f} Q(MVAR):{load_instance.Q:.3f}\n         kV:{load_instance.op_voltage:.1f} Type:{load_instance.type}'
                load_strings.append(load_str)
        if node_instance.loads_feeder > 0:
            for load_instance in loads_feeder.get(node, []):
                load_str = f'{load_instance.FeederName} -> P(MW):{load_instance.load_typical:.3f} Type:Forecast'
                load_strings.append(load_str)
        load_string = "\n".join(load_strings).strip('\n') if load_strings else ''
        load_string = '<span style="font-size: larger;">Loads:</span>\n' + load_string
    else:
        load_string = ''

    if node_instance.type in ['SINGLESWITCH', 'DOUBLESWITCH', 'FUSE', 'HVCB', 'LVCB', 'RECLOSER']:
        switch_strings = []
        for switch_instance in switches.get(node, []):
            switch_str = f'{switch_instance.etap_id} -> Type:{switch_instance.type} Status:{switch_instance.status}'
            switch_strings.append(switch_str)
        switch_string = "\n".join(switch_strings).strip('\n') if switch_strings else ''
        switch_string = '<span style="font-size: larger;">Switches:</span>\n' + switch_string
    else:
        switch_string = ''

    return generator_string, capacitor_string, load_string, switch_string

def build_edge_meta(transformers, conductors, source, target, transformer_no, conductor_no):
    if transformer_no > 0:
        transformer_strings = []
        voltage_strings = []
        unique_transformer_instances = set()
        if target == -1:
            transformer_instances_source = [Transformer.objects.filter(etap_iid=source).first()]
            transformer_instances_target = []
        else:
            transformer_instances_source = transformers.get(source, [])
            transformer_instances_target = transformers.get(target, [])
        for tx_instance in transformer_instances_source + transformer_instances_target:
            if tx_instance.id not in unique_transformer_instances:
                unique_transformer_instances.add(tx_instance.id)
                if (
                        (tx_instance.primary_bus_iid == source and tx_instance.secondary_bus_iid == target)
                        or (tx_instance.primary_bus_iid == target and tx_instance.secondary_bus_iid == source)
                        or (target == -1 and tx_instance.etap_iid == source)
                ):
                    transformer_str = (
                        f"{tx_instance.etap_id} -> S<sub>p</sub>:{tx_instance.primary_S:.2f} "
                        f"{'Δ' if tx_instance.primary_type.lower() == 'delta' else tx_instance.primary_type}"
                        f" V<sub>p</sub>:{tx_instance.primary_V:.1f}\n"
                        f"        S<sub>s</sub>:{tx_instance.secondary_S:.2f} "
                        f"{'Δ' if tx_instance.secondary_type.lower() == 'delta' else tx_instance.secondary_type}"
                        f" V<sub>s</sub>:{tx_instance.secondary_V:.1f}"
                    )
                    voltage_str = f"Voltage:{tx_instance.primary_V}/{tx_instance.secondary_V}"
                    if tx_instance.tertiary_S:
                        transformer_str += (
                            f"\n        S<sub>t</sub>:{tx_instance.tertiary_S:.2f} "
                            f"{'Δ' if tx_instance.tertiary_type.lower() == 'delta' else tx_instance.secondary_type}"
                            f" V<sub>t</sub>:{tx_instance.tertiary_V:.1f}"
                        )
                        voltage_str += f"/{tx_instance.tertiary_V}kV"
                    else:
                        voltage_str += f"kV"
                    transformer_strings.append(transformer_str)
                    voltage_strings.append(voltage_str)
                elif tx_instance.etap_iid in [source, target] and tx_instance.primary_bus_iid in [source, target]:
                    transformer_str = (
                        f"{tx_instance.etap_id}(3W) -> S<sub>p</sub>:{tx_instance.primary_S:.2f} "
                        f"{'Δ' if tx_instance.primary_type.lower() == 'delta' else tx_instance.primary_type}"
                        f" V<sub>p</sub>:{tx_instance.primary_V:.1f}"
                    )
                    transformer_strings.append(transformer_str)
                elif tx_instance.etap_iid in [source, target] and tx_instance.secondary_bus_iid in [source, target]:
                    transformer_str = (
                        f"{tx_instance.etap_id}(3W) -> S<sub>s</sub>:{tx_instance.secondary_S:.2f} "
                        f"{'Δ' if tx_instance.secondary_type.lower() == 'delta' else tx_instance.secondary_type}"
                        f" V<sub>s</sub>:{tx_instance.secondary_V:.1f}"
                    )
                    transformer_strings.append(transformer_str)
                elif tx_instance.etap_iid in [source, target] and tx_instance.tertiary_bus_iid in [source, target]:
                    transformer_str = (
                        f"{tx_instance.etap_id}(3W) -> S<sub>t</sub>:{tx_instance.tertiary_S:.2f} "
                        f"{'Δ' if tx_instance.tertiary_type.lower() == 'delta' else tx_instance.tertiary_type}"
                        f" V<sub>t</sub>:{tx_instance.tertiary_V:.1f}"
                    )
                    transformer_strings.append(transformer_str)
        transformer_string = "\n".join(transformer_strings).strip('\n') if transformer_strings else ''
        transformer_string = '<span style="font-size: larger;">Transformers:</span>\n' + transformer_string
        voltage_string = "\n".join(voltage_strings).strip('\n') if voltage_strings else ''
    else:
        transformer_string = ''
        voltage_string = ''

    if conductor_no > 0:
        conductor_strings = []
        switch_strings = []
        unique_conductor_instances = set()
        conductor_instances_source = conductors.get(source, [])
        conductor_instances_target = conductors.get(target, [])
        for cd_instance in conductor_instances_source + conductor_instances_target:
            if (
                    cd_instance.from_bus_iid == source and cd_instance.to_bus_iid == target
                    or (cd_instance.from_bus_iid == target and cd_instance.to_bus_iid == source)
            ):
                if cd_instance.type in ["CABLE", "XLINE"] and cd_instance.id not in unique_conductor_instances:
                    unique_conductor_instances.add(cd_instance.id)
                    conductor_str = f'{cd_instance.etap_id} -> A<sub>limit</sub>:{cd_instance.current_limit:.1f} '
                    conductor_str = conductor_str + f'Length:{cd_instance.length:.2f}{cd_instance.length_unit}'
                    conductor_strings.append(conductor_str)
        if conductor_strings:
            conductor_string = "\n".join(conductor_strings).strip('\n')
            conductor_string = '<span style="font-size: larger;">Conductors:</span>\n' + conductor_string
        else:
            conductor_string = ''
    else:
        conductor_string = ''
        switch_string = ''

    return transformer_string, conductor_string, voltage_string

@csrf_exempt
def save_positions_to_node(request):
    data = json.loads(request.body.decode('utf-8'))

    # Define the canvas size and normalization factor
    x_normalization_factor, y_normalization_factor = graph_normalization()

    # Get all relevant node ids from the positions data
    node_ids = [position_data.get('id') for position_data in data.get('positions', []) if position_data.get('type') == 'bus']
    node_queryset = Node.objects.filter(bus_etap_iid__in=node_ids)
    nodes = {node.bus_etap_iid: node for node in node_queryset}

    # get all labels
    node_ids = [position_data.get('id') for position_data in data.get('positions', []) if position_data.get('type') == 'label']
    label_queryset = Label.objects.filter(id__in=node_ids)
    labels = {label.label: label for label in label_queryset}



    # Create a list to hold Node instances that need to be updated
    nodes_to_update = []
    labels_to_update = []

    # Loop through the positions data and update the Node objects
    for position_data in data.get('positions', []):
        if position_data.get('type') == 'bus':
            node = nodes.get(int(position_data.get('id')))
            if node:
                node.user_LocX = int(position_data.get('position').get('x') / x_normalization_factor)
                node.user_LocY = int(position_data.get('position').get('y') / y_normalization_factor)
                nodes_to_update.append(node)
        else:
            label = labels.get(position_data.get('label'))
            if label:
                label.user_LocX = int(position_data.get('position').get('x') / x_normalization_factor)
                label.user_LocY = int(position_data.get('position').get('y') / y_normalization_factor)
                labels_to_update.append(label)


    # Use Django's bulk_update to update all nodes at once, reducing database hits
    if nodes_to_update:
        Node.objects.bulk_update(nodes_to_update, ['user_LocX', 'user_LocY'])
    if labels_to_update:
        Label.objects.bulk_update(labels_to_update, ['user_LocX', 'user_LocY'])

    # Return a JSON response indicating success
    return JsonResponse({'status': 'success'})


get_generator_constraints(6)
#busiid = find_nearest_bus(99493)
#parse_xml_and_save()
#populate_network_from_components()
#populate_nodes_from_buses()

#populate_edges_from_nodes()
#construct_node_voltages()
#construct_node_strings()
#construct_edge_strings()
#get_islands()

