%% l computing function
% Function that computes the  obligor's quality vector
% IMPORTANT: the vector must have each value under 1 and greater than 0.8

% INPUTS
% - BD: Basedati loan data tape
% - SW: Tassiswap dataset
% - method: method to be used to compute l. The choices are "Uniform",
%           "Random", "ClassicalSwaps" and "Swapsplusdefaults".
% - n: number of obligors in the pool
% - arg: function to be used in the "Classicalswaps" method. The choices
%        are "Minmax", "Sigmoid", "ReLu".

% OUTPUT
% - l: obligor's quality function.

function [l, spread] = compute_l(BD, SW, method, n, arg)

% Uniform assignment
if method == "Uniform"

    l = 0.8*ones(1,n);
    spread = l;

% Pseudo-Random assignment
elseif method == "Random"

    rng(300829);
    l = rand(1,n);
    l(l < 0.8) = 0.8;
    l(l > 0.99) = 0.99;
    spread = l;

% Assignment using functions and swap rates
elseif method == "ClassicalSwaps"
    
    % Recalling the classical_swaps function
    % IMPORTANT: requires high computational time for the calculation
    spread = classical_swaps(BD, SW);

    if arg == "Minmax"
        l = 0.8 + 0.19*(spread'-min(spread))/(max(spread)-min(spread));
    elseif arg == "Sigmoid"
        l = 0.8 + 0.19*(1./(1+exp(-spread')));
    elseif arg == "ReLu"
        l = 0.8 + 0.19*max((spread')-1, 0)/max(spread);
    end

% Assignment using Swap rates and defaults of single obligors
elseif method == "Swapsplusdefaults"

    % Recalling the classical_swaps function
    % IMPORTANT: requires high computational time for the calculation
    spread = classical_swaps(BD, SW);

    l = zeros(1,n);
    % High value because they are already defaulted
    l(BD.Defaulted == "Y") = 0.99; 

    % Rescaling w.r.t. the number of defaults during the issue's life
    MD = BD.MonthsDelinq';
    condN = BD.Defaulted == "N";
    l(condN) = (1/(5*pi) * (atan(MD(condN)))) + ...
               0.1*(1./(1+exp(-spread(condN)'))) + 0.79; 

end
