%% Spreads computing function
% Function that calculates the spread of each loan

% INPUTS
% - BD: Basedati loan data tape
% - SW: Tassiswap dataset

% OUTPUT
% - spread: vector of spreads of each obligor in the pool

function [spread] = classical_swaps(BD, SW)

% 'BD' important data
c = BD.AmountLoan;
T = BD.OrigMaturityTerm;
Coup = BD.GrossCoupon;
Issues = BD.IssueDate;
MD = BD.MonthsDelinq;
Def = BD.Defaulted;

% 'SW' important data
BuD = SW.BUSINESS_DATE;
Val = SW.VALUE;

% Computing the years to maturity of each obligor
ytm = zeros(4139,1);
fl = floor(T/12);
ce = ceil(T/12);
cond1 = ((T/12 - floor(T/12)) < 0.5) + (T/12 >= 15) > 0;
cond2 = ((T/12 - floor(T/12)) >= 0.5) + (T/12 < 15) == 2;
ytm(cond1) = fl(cond1); % Because ceil gives back 16 years loans too
ytm(cond2) = ce(cond2);

% Computing the relative indexes
ytmindex = zeros(length(c), 1);
for i = 1:length(T)
    if ytm(i) == 1
        ytmindex(i) = 10;
    elseif ytm(i) < 12
        ytmindex(i) = ytm(i)-2;
    else
        ytmindex(i) = ytm(i)-1;
    end
end

% Searching for the correlated swap rates
corr_swap = zeros(length(T), 1);
for i = 1:length(T)
    [y,m,d] = split(between(Issues(i),...
                  BuD((ytmindex(i)*1307 + 1):((ytmindex(i)+1)*1307))),...
                  {'years', 'months', 'days'});
    timedistance = 365*abs(y) + 30*abs(m) + abs(d);
    [~, diffind] = min(timedistance);
    corr_swap(i) = Val(diffind + ytmindex(i)*1307);
end

% IMPORTANT: some issues cannot be correctly classified because their issue
% date is not present in the Swap table. Then the attribution is performed
% as follows: 
% - if the issue is defaulted the spread assigned is the maximum of all the
% other spreads;
% - if the issue has MonthsDelinq ~= 0 the spread assigned is 5/6 of the
% maximum spread.

% Last date available in the Swap table
mindate = datetime(2018,10,1);
old_issues = Issues < mindate;
old_defaulted = old_issues & (Def == "Y");
MDnotzero = (MD ~= 0) & (Def == "N") & old_issues;

% Multiplying by 100 to obtain percetages
corr_rate = corr_swap*100;

% Computing the spread
spread = Coup - corr_rate;
max_spread = max(spread);
spread(old_defaulted) = max_spread;
spread(MDnotzero) = 5/6 * max_spread;
