%% General function for estimating portfolio losses' characteristics
% Function that estimates the probability of large losses of a given bond
% portfolio as well as the relative CVaR.

% INPUTS
% n : number of obligors in the pool
% nrep: number of repetition for the Simulative method
% alpha: iterdependence parameter (heavy tailedness parameter). It admits
%        values in [1,Inf).
% b : threshold for the loss
% c: vector of the expositions of the loss
% l: quality vector of the obligors
% f_n: quality function of the pool
% copula: copula function to be used in the Exact calculation. It admits
%         the strings "Gumbel" and "Clayton"
% seed: seed of the RNG for reproducibilty of the results
% method: method to be used. It admits the strings "Simulative" or "Exact"

% OUTPUTS
% prob: probability of a default larger than n*b
% CVaR: expected shortfall
% vstar: threshold value of the systematic factor V. Only computed when
%        method == "Exact"


function [prob, CVaR, vstar] = archest(n, nrep, alpha, b, c, l, f_n, copula, seed, method)

% Setting the seed of the RNG for reproducibility
rng(seed)

% Copula function to be used
if copula == "Gumbel"

    % Gumbel's copula generator
    phi = @(x, a) (-log(x)).^a;
    cs = (cos(pi/(2*alpha)))^(alpha);
    if alpha < 2
        delta = 0.5;
    else
        delta = 0.25;
    end
    % Mixing variable: one-sided Stable standard random variable
    varb = makedist('Stable', 'alpha', 1/alpha, 'beta', 1, 'gam', cs, 'delta', delta);

elseif copula == "Clayton"

    %Clayton's copula generator
    phi = @(x, a) x.^(-a) - 1;
    % Mixing variable: Gamma randoma variable
    varb = makedist('Gamma', 'a', 1/alpha, 'b',  1);

end

% Simulative method
if method == "Simulative"
    
    % Initializations
    Vtot = zeros(nrep,1);
    Loss = zeros(nrep,1);
    phival = phi(ones(1,n)-f_n(n)*l, alpha);
    
    % Simulation process
    for step = 1:nrep

        V = random(varb, 1, 1);
        R = exprnd(1, 1, n);
        O = R./phival;

        % Ordering the obtained values
        [ordO, indexes] = sort(O, 'descend');
        tot = 0;

        % Finding the ranked threshold value
        for i=1:n
            if tot >= n*b
               break
            else
                tot = tot + c(indexes == i);
            end
        end
        reqO = ordO(n-i+1);

        % Evaluating the exceeding of the threshold
        Vtot(step) = V > reqO;

        % Evaluating the total loss
        Loss(step) = (V > O)*c';
    end

    % Computing the probability of default
    prob = sum(Vtot)/nrep;
    fprintf('The estimated probability of a loss larger than %d is %0.8f. \n', n*b, prob);

    % Computing teh excpected shortfall
    expectedvalue = sum(max(Loss - n*b*ones(nrep,1), 0))/nrep;
    CVaR = n*b + expectedvalue/prob;
    fprintf('The estimated CVaR is %0.4f. \n', CVaR);

    % For computational purposes, it is necessary to explicit vstar
    % in another way
    vstar = "ONLY for Exact method";


% Exact asymptotic method
elseif method == "Exact"
    
    % Computing the frequency vector
    w = zeros(1,n);
    vals = [c;l];
    couples = zeros(2,n);
    count = 1;

    % Finding the couples then the frequency of each couple
    for i = 1:n
        if vals(1,i) == 0
            continue
        else
            ind1 = find(vals(1,:) == vals(1,i));
            ind2 = find(vals(2,:) == vals(2,i));
            int = intersect(ind1,ind2);
            couples(:,count) = vals(:,int(1,1));
            w(count) = size(int,2)/n;
            count = count + 1;
            vals(1,int) = 0;
        end
    end
    couples = couples(:,1:count-1);
    w = w(1:count-1);

    % Expliciting some useful functions:

    % r(v) function:
    rv = @(v) sum(couples(1,:).* w.* (1-exp(-v.*(couples(2,:).^alpha))));

    % Scaled r(v) function used to calculate the value of vstar numerically
    rvminusb = @(v) rv(v) - b;
    vstar = fzero(rvminusb, 1);

    % Derivative of r(v) multiplied by v^(-1/alpha) (used to calculate
    % numerically the integral.
    rvprime = @(v) (sum((couples(1,:).*w.*(couples(2,:).^alpha))*(exp(-v'*(couples(2,:).^alpha)))',1)).*v.^(-1/alpha);

    % Computing the asymptotic probability of losses larger than a
    % threshold
    if alpha == 1
        prob = f_n(n)*vstar^(-1/alpha);
    else
        prob = f_n(n)*vstar^(-1/alpha)/gamma(1-1/alpha);
    end
    fprintf('The asymptotic probability of a loss larger than %d is %0.8f. \n', n*b, prob);

    % Computing the expected shortfall
    CVaR = n*b + n*integral(rvprime, vstar, +Inf)/(vstar^(-1/alpha));
    fprintf('The asymptotic CVaR is %0.4f. \n', CVaR);

end
