# Author: Daniel Guarecuco Aguiar <deaguiar@stud.ntnu.no>
#   <daniel.aguiar@nordicsemi.no>
# Created Date: 14/02/2023
# This script synthesizes RTL design, exports timing report
# Usage: Intented to be called from main.sh
#   dc_shell -f syn.tcl
# # ---------------------------------------------------------------------------

# Get library name from env varible defined in bash script
set TECH_LIB $env(TECH_LIB)
# Get relevant name to be used for saving all files
set OUT_DIR $env(OUT_DIR)
exec mkdir -p $OUT_DIR
# Get period
set PERIOD $env(PERIOD)
# Get exlude_cells flag
set exclude_cells $env(exclude_cells)
set TECH_CHAN $env(TECH_CHAN)

suppress_message LINT-52
suppress_message LINT-31
suppress_message LINT-29

# ############### Specify Libraries # ##########################
set LIB_PATH "/library_path"

set ADDITIONAL_SEARCH_PATH    [list {*}[subst {
  [glob $LIB_PATH/path_to_library]
  [glob ./]
  [glob $OUT_DIR]
}]]  ;

set search_path ""
append search_path " $ADDITIONAL_SEARCH_PATH"

set link_library "*"
lappend link_library $TECH_LIB

set target_library  $TECH_LIB

# ############### Read Design # ###############################
# Define workpath
define_design_lib -path ./work WORK

# Read verilog source files
source -echo -verbose ./design.tcl

set DESIGN_NAME "ibex_top"

# Initialize name mapping database
set hdlin_enable_upf_compatible_naming true
saif_map -start

analyze -library WORK -format sverilog -vcs ${RTL_SOURCE_LIST}
set top_entity ${DESIGN_NAME}

# Elaborate design
elaborate ${DESIGN_NAME}

# Convert FSDB to SAIF
sh fsdb2saif $OUT_DIR/ibex_simple_system.fsdb -o $OUT_DIR/ibex_top.saif  -flatten_genhier
# Map RTL simulation names to RTL design
saif_map -create_map -source_instance ibex_simple_system/u_top/u_ibex_top -input $OUT_DIR/ibex_top.saif

# ############### Define Design Environment # ###################
#set_operating_condition
#set_wire_load_model
#set_load

# ############### Set Design Constraints # ######################
create_clock clk_i -name "CLK" -period $PERIOD

#set_input_delay
#set_output_delay
#set_max_transition
#set_driving_cell
#set_max_area

if {$exclude_cells != 0} {
  source -echo -verbose $OUT_DIR/../${TECH_CHAN}_exclude_cells_${exclude_cells}.tcl
}

# ############### Optimize Design # #############################
# Compile design
compile -exact_map

# ############### Save Design # #################################
write_file -hierarchy -format verilog -output $OUT_DIR/ibex_top_map.v ibex_top
write_file -format svsim -output $OUT_DIR/ibex_top_wrapper.sv
write_file -hierarchy -format ddc -output $OUT_DIR/ibex_top_map.ddc
write_sdc $OUT_DIR/ibex_top_sdc.sdc
saif_map -write_map $OUT_DIR/ibex_top_name_mapping.tcl -type primepower

# Full. Include name of input pin in the cell
report_timing -input_pins > $OUT_DIR/report_timing_full.txt
# Input-to-Register
report_timing -from [all_inputs] -to [all_registers -data_pins] > $OUT_DIR/report_timing_in_reg.txt
# Register-to-Register
report_timing -from [all_registers -clock_pins] -to [all_registers -data_pins] > $OUT_DIR/report_timing_reg_reg.txt
# Register-to-Output
report_timing -from [all_registers -clock_pins] -to [all_outputs] > $OUT_DIR/report_timing_reg_out.txt
# Input-to-Output
report_timing -from [all_inputs] -to [all_outputs] > $OUT_DIR/report_timing_in_out.txt

quit
