#!/bin/bash

# Author: Daniel Guarecuco Aguiar <deaguiar@stud.ntnu.no>
#   <daniel.aguiar@nordicsemi.no>
# This is the main script
# 1.Build Ibex Software
# 2.Run RTL simulation
# 3.Run Synthesis
# 4.Run Power estimation, export netlist
# 5.Clean netlist and create testbenches
# 6.Import netlist and run analog Simulation
# # ---------------------------------------------------------------------------

# Library name and Period in ps
#   SLVT20 3000
#   SLVT24 4000
#   SLVT28 4000
#   SLVT32 5000
#   SLVT36 6000   
#   LVT24 15000
#   LVT28 18000
#   LVT32 24000
#   LVT36 28000
#   HVT20 1760
#   HVT24 2040
#   HVT28 2300
#   UHVT28 6000
#   UHVT32 7150
#   UHVT36 8900


arg_lib=$1 #Library name
arg_period=$2 #Period
exclude_cells=$3 # % to exclude (0,20,30,40,50)
arg1=$4 #Options: syn import analogSim
#**************************************************************************************
#                   EDIT THIS ACCORDINGLY
#**************************************************************************************
array_lib=(
    "${arg_lib} ${arg_period}"
)

project_path="/my_path/mep_thesis"
make_sw_path="${project_path}/design/ibex/examples/sw/simple_system/hello_test"
build_path="${project_path}/build/lowrisc_ibex_ibex_simple_system_0/sim-modelsim"

#Matches SLVT and LVT 
regex1='(.*)_([a-zA-Z]+VT[0-9]+)_(.+)_0d([0-9]+)(V_m?[0-9]+C)(.+)\s+(\w+)'
#Matches HVT and UHVT
regex2='(.*)_([a-zA-Z]+[0-9]+[a-zA-Z]+)(_TT_0P)([0-9]+V).+(_[0-9]+C)(.+)\s+(\w+)'

#**************************************************************************************
# Setting environment
module load synopsys/verdi/2021.09-sp2-2

export PATH=$PATH\:$HOME/.local/bin
export PATH=$PATH\:$HOME/lowrisc-toolchain/bin
export ARCH=rv32imc
export MODEL_TECH=/path_tool/questasim/2022.3/questasim/bin


export exclude_cells=$exclude_cells
#125, 25, m40
temp=25
#**************************************************************************************
# Starts here

# Loop through libraries
for i in "${array_lib[@]}"
do
    #Evalute lib file name against regex
    if [[ "$i" =~ $regex1 ]]; then 
        echo "Matches lib"
        PERIOD=$(( ${BASH_REMATCH[7]}/1000 ))
        echo "Period is ${PERIOD} ns"
    elif [[ "$i" =~ $regex2 ]]; then  
        echo "Matches lib"
        PERIOD=$(( ${BASH_REMATCH[7]} ))
        echo "Period is ${PERIOD} ps"
    else 
        echo "No Match - Library name invalid"
        exit
    fi 
    #Get lib name
    LIB_NAME="${BASH_REMATCH[2]}_0d${BASH_REMATCH[4]}${BASH_REMATCH[5]}_${exclude_cells}"
    #Set output directory (e.g /my_path/mep_thesis/syn_out/LVT24_25C)
    OUT_DIR="${project_path}/analysis/syn_out/$LIB_NAME"

    #Set env variable used by tcl scripts
    export OUT_DIR=$OUT_DIR
    #Example LVT24
    TECH_CHAN="${BASH_REMATCH[2]}"
    export TECH_CHAN=$TECH_CHAN
    TESTBENCH="${TECH_CHAN}_Testbench"
    stringarray=($i)
    #Example lib_name.nldm.db
    export TECH_LIB=$stringarray
    export PERIOD=$PERIOD
    SIM_PERIOD=$(( ${BASH_REMATCH[7]} ))

    echo "**************************************************************************************"
    echo "RUNNING FLOW USING $i"
    echo "**************************************************************************************"
    
    current_dir=$PWD
    cd $project_path

    if [ "$arg1" = "syn" -o "$arg1" = "all"  ];
    then
        ## Make software  
        echo "Making software" 
        make -C $make_sw_path
        ## Generate VMEM from binary
        echo "Generating VMEM"
        hexdump -v -e '1/4 "%08x " "\n"' $make_sw_path/hello_test.bin > $make_sw_path/hello_test.vmem
        ## Build Simulation
        echo "Building Simulation"
        fusesoc --cores-root=. run --target=sim --tool=modelsim --setup --build lowrisc:ibex:ibex_simple_system --RV32E=0 --RV32M=ibex_pkg::RV32MFast --SRAMInitFile="${make_sw_path}/hello_test.vmem" --Period=$SIM_PERIOD
        ## Run simulation
        echo "Running Simulation"
        cd $build_path
        make run
        echo "Simulation Log:"
        cat ibex_simple_system.log
        cd $current_dir;
        ## Create output directory
        mkdir -p $OUT_DIR
        ## Copy FSDB
        cp "${build_path}/ibex_simple_system.fsdb" ${OUT_DIR}

        ## Run sythesis
        echo "STARTING DC_SHELL..."
        dc_shell -f syn.tcl -o $OUT_DIR/compile.log

        ## Export histogram
        echo "EXPORTING CELL HISTOGRAM..."
        grep "LVT_\|HVT_" $OUT_DIR/ibex_top_map.v | awk '{print $1}' | sort | uniq -c | sort -nr | awk '{print $2 "," $1}' > $OUT_DIR/cell_histogram_ibex_$LIB_NAME.csv

        ## Run power stimation
        echo "STARTING PT_SHELL..."
        pt_shell -f power.tcl
    fi

    if [ "$arg1" = "import" ];
    then
        cd $project_path/analysis
        ## Build SPICE testbench
        echo "Building spice testbench"
        python3 generate_testbench.py $OUT_DIR

        ## Build OceanScript for simulation
        python3 generate_ocn.py $OUT_DIR $TECH_CHAN $temp

        ## Importing to Virtuoso
        cd $project_path/virtuoso
        echo "Importing netlist to Virtuoso"
        echo "************************************Creating Testbench"
        spiceIn -netlistFile $OUT_DIR/netlist_testbench.spi -outputLib $TESTBENCH -language SPICE -refLibList "cmos22fdsoi analogLib" -devMapFile /my_path/mep_thesis/virtuoso/mapping
    fi

    if [ "$arg1" = "analogSim" ];
    then
        temp=25
        echo "Simulating for 25C"
        cd virtuoso
        echo "Simulating for input connected to gnd"
        virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_1.ocn
        echo "Simulating for input connected to vdd"
        virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_2.ocn
        echo "Simulating for input connected to Z"
        virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_3.ocn
        # Run next simulation if previous one didnt oscillate
        if  grep "Frequency" ./results/${TECH_CHAN}_${temp}C_*_top_results_3.csv | grep -q "eval err" ;
        then
            echo "Simulating for input connected to !Z"
            virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_4.ocn
        fi

        temp=m40
        echo "Simulating for -40C"
        echo "Simulating for input connected to gnd"
        virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_1.ocn
        echo "Simulating for input connected to vdd"
        virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_2.ocn
        echo "Simulating for input connected to Z"
        virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_3.ocn
        # Run next simulation if previous one didnt oscillate
        if  grep "Frequency" ./results/${TECH_CHAN}_${temp}C_*_top_results_3.csv | grep -q "eval err" ;
        then
            echo "Simulating for input connected to !Z"
            virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_4.ocn
        fi

        temp=125
        echo "Simulating for 125C"
        echo "Simulating for input connected to gnd"
        virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_1.ocn
        echo "Simulating for input connected to vdd"
        virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_2.ocn
        echo "Simulating for input connected to Z"
        virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_3.ocn
        # Run next simulation if previous one didnt oscillate
        if  grep "Frequency" ./results/${TECH_CHAN}_${temp}C_*_top_results_3.csv | grep -q "eval err" ;
        then
            echo "Simulating for input connected to !Z"
            virtuoso -nograph -restore $OUT_DIR/top_sim_${temp}C_Excluding_${exclude_cells}%_4.ocn
        fi
    fi
    
done


echo "All done"
echo

exit