# Author: Daniel Guarecuco Aguiar <deaguiar@stud.ntnu.no>
#   <daniel.aguiar@nordicsemi.no>
# Created Date: 14/02/2023
# This script cleans the spice netlist generated by PrimeTime.
# Generates four testbenches, where input is connected to gnd, vdd, Z and !Z.
# Generates four spiceIn.params 
#
# Usage: Intented to be called from main.sh
#   python3 generate_testbenches.py /my_path/master_thesis/analysis/syn_out/LVT24_25C LVT24_Testbench
# # ---------------------------------------------------------------------------

import fileinput
import re
import shutil
import os
import sys

def copy_file(dest_path,spice_file_source,spice_file):
    if not os.path.exists(dest_path):
        os.mkdir(dest_path)
    shutil.copy(spice_file_source, spice_file)

# Append current line to previous one if it starts with +
def remove_break(file_path):
    previousLine = ""
    # Read spice file line by line
    for line in fileinput.input(file_path, inplace=True):
        if line.startswith("+"):
            previousLine = previousLine.replace("\n","")  + line.replace("+","")
        else:
            print('{}'.format(previousLine), end='')
            previousLine = line

# Return a list of cell names in the critical path, obtained from the timing report
def get_all_cell_names(file_path):
    with open(file_path, 'r') as file:
        # read all content of the file
        content = file.read()
        names = re.findall('\s+(.+)\/\w+ \(.+\)',content)
        # remove duplicates (input + output)
        res = []
        res_prev = ""
        for i in names:
            if (i != res_prev):
                res_prev = i
                res.append(i)
        return res

def remove_cell(file_path, cell_name):
    # Read spice file line by line
    for line in fileinput.input(file_path, inplace=True):
        # Check if cell_name is in the line and remove it
        if (line.find(cell_name) != -1):
            newline = "* LINE REMOVED: " + line
            print('{}'.format(newline), end='')
        else:
            # Do nothing, keep the same line
            print('{}'.format(line), end='')

def get_cell_pins(file_path, cell_name):
    with open(file_path, 'r') as file:
        # read all content of the file
        pins = []
        content = file.read()
        pins = re.findall('\s+(' + re.escape(cell_name) + '\/\w+) \(.+\)',content)
    return pins  

def append_to_file(file_path, string):
    # Open a file with access mode 'a'
    with open(file_path, "a") as file_object:
        # Append string at the end of file
        file_object.write(string)

def replace_slash(file_path):
    # Read spice file line by line
    for line in fileinput.input(file_path, inplace=True):
        # Do nothing if it stats with a .
        if line.startswith("."):
             print('{}'.format(line), end='')
        else:
            line = line.replace('/', '_')
            print('{}'.format(line), end='')

def replace_brackets(file_path):
    # Read spice file line by line
    for line in fileinput.input(file_path, inplace=True):
        # Do nothing if it stats with a .
        if line.startswith("."):
             print('{}'.format(line), end='')
        else:
            line = line.replace('[', '_')
            line = line.replace(']', '_')
            print('{}'.format(line), end='')

def replace_parenthesis(file_path):
    # Read spice file line by line
    for line in fileinput.input(file_path, inplace=True):
        # Do nothing if it stats with a .
        if line.startswith("."):
             print('{}'.format(line), end='')
        else:
            line = line.replace('(', '_')
            line = line.replace(')', '_')
            print('{}'.format(line), end='')

def remove_vsource(file_path, string):
    # Read spice file line by line
    for line in fileinput.input(file_path, inplace=True):
        # Do nothing if it stats with a .
        if line.startswith("v"):
            if (line.find(string) != -1):
                line = "* LINE REMOVED: " + line
        print('{}'.format(line), end='')

def replace_input(file_path, string):
    # Read spice file line by line
    for line in fileinput.input(file_path, inplace=True):
        if line.startswith("x"):
            # If _VDD
            if (line.find(string) != -1):
                if (string == "_VDD"):
                    list_line = re.search('(^x.+)(\s\S+_VDD)(.+)',line)
                    line = list_line.group(1) + " vdd" + list_line.group(3) + "\n"
                elif (string == "_VNW_N"):
                    list_line = re.search('(^x.+)(\s\S+_VNW_N)(.+)',line)
                    line = list_line.group(1) + " vnw_n" + list_line.group(3) + "\n"
                elif (string == "_VPW_P"):
                    list_line = re.search('(^x.+)(\s\S+_VPW_P)(.+)',line)
                    line = list_line.group(1) + " vpw_p" + list_line.group(3) + "\n"
                elif (string == "_VNW_P"):
                    list_line = re.search('(^x.+)(\s\S+_VNW_P)(.+)',line)
                    line = list_line.group(1) + " vnw_n" + list_line.group(3) + "\n"
                elif (string == "_VPW_N"):
                    list_line = re.search('(^x.+)(\s\S+_VPW_N)(.+)',line)
                    line = list_line.group(1) + " vpw_p" + list_line.group(3) + "\n"
        print('{}'.format(line), end='')

def unify_vdd(file_path):
    # Read spice file line by line
    for line in fileinput.input(file_path, inplace=True):
        if line.startswith("vvdd"):
            # Extract value of vdd
            vdd_value = re.search('^vvdd\s+vdd\s+\d\s+(\d+\.\d+)',line).group(1)
            line = "vvdd vdd 0 vdd\n"
        elif line.startswith("v"):
            if (re.match('^v\S+\s+\S+\s+0\s+' + vdd_value, line)):
                newline = re.search('(^v\S+\s+\S+\s)+0\s+' + vdd_value,line).group(1)
                line = newline + "vdd 0\n"
        print('{}'.format(line), end='')

def lower_case_vss(file_path):
    # Read spice file line by line
    for line in fileinput.input(file_path, inplace=True):
        line = line.replace('VSS', 'vss')
        print('{}'.format(line), end='')

# Remove stimulus, vvdd, vvss, .temp, .global
def remove_stim(file_path):
    # Read spice file line by line
    for line in fileinput.input(file_path, inplace=True):
        # Remove the following line
        if line.startswith(".include"):
            if (re.match('^^\.include\s+\S+stim\.sp\S', line)):
                line = "* LINE REMOVED: " + line
        print('{}'.format(line), end='')

def remove_corners(file_path):
    # Read spice file line by line
    ret_text = ""
    for line in fileinput.input(file_path, inplace=True):
        # Remove the following lines
        if line.startswith("vvdd vdd"):
            ret_text += line
            line = "* LINE REMOVED: " + line
        elif line.startswith(".global vss"):
            ret_text += line
            line = "* LINE REMOVED: " + line
        elif line.startswith("vvss vss"):
            ret_text += line
            line = "* LINE REMOVED: " + line
        elif line.startswith(".temp"):
            ret_text += line
            line = "* LINE REMOVED: " + line
        print('{}'.format(line), end='')
    return ret_text

def creat_subckt(file_path):
    # Read spice file line by line
    for line in fileinput.input(file_path, inplace=True):
        # Append subcircuit definition after .temp
        if line.startswith(".temp"):
            subckt="\n.SUBCKT CRITICAL_PATH VDD VSS VNW_N VPW_P A Z\n"
            line = line + subckt
        print('{}'.format(line), end='')

def connect_ext_pins(spice_file, outer_in, outer_out):
    conn_str = "\n* CONNECTION TO EXTERNAL PINS\n\
vext_vdd VDD vdd 0\n\
vext_vss VSS vss 0\n\
vext_vnw_n VNW_N vnw_n 0\n\
vext_vpw_p VPW_P vpw_p 0\n\
vext_in A "+ outer_in + " 0\n\
vext_out Z "+ outer_out + " 0\n\
.ENDS"   
    # Append line to spice file
    append_to_file(spice_file, conn_str)


def write_to_file(file_path, content):
    f = open(file_path, "w")
    f.write(content)
    f.close()

def write_testbench(file_path, content):
    tb_path = file_path + '/netlist_testbench.spi'
    feedback_str = content + "\n* Input A is connected to a sp4tswitch. Inputs are VSS, VDD, Z or !Z\n\
* Switch is controlled via sw_pos variable\n\n\
einverter nZ vss vdd Z 1\n\
simulator lang=spectre\n\
S0 (A vss vdd Z nZ) sp4tswitch paramTyp=string tranPosition_str=sw_pos model=switch\n\
simulator lang=spice\n\n\
.end"
    write_to_file(tb_path, feedback_str)
    print('File ' +tb_path+ ' created.')


####################################################################################
#dir_path = "/my_path/mep_thesis/analysis/syn_out/LVT24_25C"
# Path taken from argument, set at main.sh
dir_path = sys.argv[1]
spice_file_source =  dir_path + "/netlist_critical_path.sp"
report_file = dir_path + "/report_timing_full.txt"

dest_path= dir_path
spice_file_name = "netlist_critical_path.spi"
spice_file = dest_path + "/" + spice_file_name

####################################################################################

# Copy netlist to spice directory
copy_file(dest_path,spice_file_source,spice_file)

# Gather data from timing report
# Get all cell names in critical path
list_cells = get_all_cell_names(report_file)
# Get name of input pin. Used for feedback
outer_in = get_cell_pins(report_file, list_cells[1])[0]
# Get name of output pin. Used for feedback
outer_out = get_cell_pins(report_file, list_cells[-2])[1]


# Processing netlist to be imported by Virtuoso
# Remove line breaks when line is too long. Easier to process
remove_break(spice_file)
# Remove first FF. Interested only in combinational circuits between two FF
remove_cell(spice_file, list_cells[0])
# Remove last FF
remove_cell(spice_file, list_cells[-1])
# Remove all individual vsource used for VDD, VNW_N, VPW_P 
remove_vsource(spice_file, "_VDD")
remove_vsource(spice_file, "_VNW_N")
remove_vsource(spice_file, "_VPW_P")
remove_vsource(spice_file, "_VNW_P")
remove_vsource(spice_file, "_VPW_N")

replace_input(spice_file, "_VDD")
replace_input(spice_file, "_VNW_N")
replace_input(spice_file, "_VPW_P")
replace_input(spice_file, "_VNW_P")
replace_input(spice_file, "_VPW_N")
# Instead of having multiple vsource with same value as vdd, short-circuit them to vdd
unify_vdd(spice_file)
# Replace all VSS with vss
lower_case_vss(spice_file)

# Create subckt definition after .temp
creat_subckt(spice_file)

# Remove stimulus. Not needed
remove_stim(spice_file)
# Remove vvdd, vvss, .temp, .global. To be declared outside in testbench
corners=remove_corners(spice_file)

# Remove final .end. So we can append connecting signals 
remove_cell(spice_file, ".end")
# Connect to external pins and add .ENDS
connect_ext_pins(spice_file, outer_in, outer_out)
# Replace slash / by _. To avoid issues with Virtuoso
replace_slash(spice_file)
# Replace [] by _. To avoid issues with Virtuoso
replace_brackets(spice_file)
# Replace () by _. To avoid issues with Virtuoso
replace_parenthesis(spice_file)


# Create testbench
content = ".include " + spice_file_name + "\n\n" + corners + "vvnw_n vnw_n 0 vnw_n\n"\
+ "vvpw_p vpw_p 0 vpw_p\n"\
+ "\nxtop_01 vdd vss vnw_n vpw_p A Z CRITICAL_PATH\n"
write_testbench(dest_path, content)

print('Done')