function [dist,prev] = dijkstra_forward_star(nodes, edges, start)
% nodes - list of nodes in the graph
% edges - list of edges in the graph, where each edge is represented as a
%         structure with fields 'start', 'end', and 'weight'
% start - starting node
% dist - vector of shortest distances from the start node
% prev - vector of previous nodes in the shortest path

n = numel(nodes); % number of nodes
dist = inf(1,n); % initialize distances to inf
prev = zeros(1,n); % initialize previous node to 0
heap = start; % initialize heap with the starting node

while ~isempty(heap)
    node = heap(1); % extract node with smallest distance
    heap(1) = heap(end);
    heap = heap(1:end-1);
    heap = siftdown(heap, dist, 1);
    
    for i = 1:numel(edges)
        edge = edges(i);
        if edge.start == node
            neighbor = edge.end;
            alt = dist(node) + edge.weight;
            if alt < dist(neighbor)
                dist(neighbor) = alt;
                prev(neighbor) = node;
                heap = siftup(heap, dist, neighbor);
            end
        end
    end
end

end

function heap = siftdown(heap, dist, i)
% siftdown helper function to maintain the heap property

minIndex = i;
left = 2*i;
right = 2*i + 1;

if left <= numel(heap) && dist(heap(left)) < dist(heap(minIndex))
    minIndex = left;
end

if right <= numel(heap) && dist(heap(right)) < dist(heap(minIndex))
    minIndex = right;
end

if minIndex ~= i
    temp = heap(i);
    heap(i) = heap(minIndex);
    heap(minIndex) = temp;
    heap = siftdown(heap, dist, minIndex);
end

end

function heap = siftup(heap, dist, i)
% siftup helper function to maintain the heap property

while i > 1
    parent = floor(i/2);
    if dist(heap(parent)) > dist(heap(i))
        temp = heap(i);
        heap(i) = heap(parent);
        heap(parent) = temp;
        i = parent;
    else
        break;
    end
end

end
