import os.path
import sys
import numpy as np
import seaborn as sns
import pandas as pd
import argparse
from datetime import datetime
from scipy.sparse import save_npz, csr_matrix
from sklearn.model_selection import train_test_split, cross_validate
from sklearn.ensemble import RandomForestClassifier, HistGradientBoostingClassifier
from sklearn import metrics, svm, preprocessing
from sklearn.neighbors import KNeighborsClassifier
from sklearn.metrics import confusion_matrix, accuracy_score, precision_score, recall_score, f1_score, roc_curve, \
    classification_report, roc_auc_score
from matplotlib import pyplot as plt
from Utils.Extract_feature import extract_ngrams, extract_bytes_sequence, extract_opcode_sequence, count_asm_opcode, \
    extract_syscall, save_syscall, check_syscall, load_syscall, extract_syscall_sequence, extract_entropy, \
    extract_apis, save_apis, load_apis, check_apis, extract_binary_sizes, extract_entropy_section, \
    extract_bytes_microsoft
from pathlib import Path
from Utils.Integrate_features import integrated_features

data_dir = Path("data")
smp_dir = data_dir.joinpath("samples")
ftr_dir = Path("pre-extract")
if not ftr_dir.exists():
    ftr_dir.mkdir()
if not Path("results").exists():
    Path("results").mkdir()


def cross_validation(model, _X, _y, _cv=5):
    """
    Performs cross validation with 5 divisions of the train dataset
    :param model: Classifier selected
    :param _X: DataFrame with train data
    :param _y: DataFrame with labels
    :param _cv: Number of Fold
    :return: Results of cross validation
    """
    scoring = "f1_weighted"
    res = cross_validate(estimator=model, X=_X, y=_y, cv=_cv, scoring=scoring, return_train_score=True)
    return res["test_score"]


def plot_result(x_label, y_label, plot_title, val_data):
    """
    Plot results of cross validation
    :param x_label: Title x-axis
    :param y_label: Title y-axis
    :param plot_title: Graph title
    :param val_data: Data to be plotted
    """
    plt.figure(figsize=(12, 6))
    labels = ["1st Fold", "2nd Fold", "3rd Fold", "4th Fold", "5th Fold"]
    x_axis = np.arange(len(labels))
    plt.gca()
    plt.ylim(0, 1)
    plt.bar(x_axis + 0.2, val_data, 0.4, color='red', label='Validation')
    plt.title(plot_title, fontsize=30)
    plt.xticks(x_axis, labels)
    plt.xlabel(x_label, fontsize=14)
    plt.ylabel(y_label, fontsize=14)
    plt.legend()
    plt.grid(True)
    plt.plot()


def show_confusion_matrix(pred, true, title, dataset, feature):
    """
    Trace the confusion matrix of a model
    :param pred: Data predicted by the model
    :param true: Actual data
    :param title: Graph title
    :param dataset: Chosen dataset
    :param feature: Chosen features
    """
    conf_matrix = confusion_matrix(true, pred)
    font_scale = 3 if dataset == "malign" else 4
    plt.figure(figsize=(12, 11))
    sns.set(font_scale=font_scale)
    ax = sns.heatmap(conf_matrix, annot=True, fmt="d", cbar=False, cmap="Blues")
    ax.set_title(title, fontsize=34)
    ax.set_xlabel("Predicted", fontsize=34)
    ax.set_ylabel("Actual", fontsize=34)
    plt.savefig("results/"+title+"_"+dataset+"_"+feature+".svg")


def fit_and_pred(clf, X_train, X_test, y_train, y_test, results, dataset, features):
    """
    Train the model through cross validation and then test it by reporting the results
    :param clf: Selected classifier
    :param X_train: DataFrame with train data
    :param X_test: DataFrame with test data
    :param y_train: DataFrame with labels of train data
    :param y_test: DataFrame with labels of test data
    :param results: Results to be updated
    :param dataset: Chosen dataset
    :param features: Chosen features
    :return: Updated results
    """
    clf_result = cross_validation(clf, X_train, y_train)
    model_name = type(clf).__name__
    plot_result(model_name, "F1", "F1 scores in 5 Folds", clf_result)

    clf.fit(X_train, y_train)
    clf_y_pred = clf.predict(X_test)
    if model_name == "RandomForestClassifier":
        title = "Random Forest"
    elif model_name == "KNeighborsClassifier":
        title = "K-NN"
    elif model_name == "SVC":
        title = "SVM"
    else:
        title = "Gradient Boosting"

    show_confusion_matrix(clf_y_pred, y_test["Class"], "Confusion Matrix of " + title, dataset=dataset, feature=features)
    accuracy = accuracy_score(y_test, clf_y_pred)
    print(f"Accuracy = {accuracy:.4f}")
    if dataset == "benign" or dataset == "ST-WinMal-benign":
        precision = precision_score(y_test, clf_y_pred)
        recall = recall_score(y_test, clf_y_pred)
        f1score = f1_score(y_test, clf_y_pred)
        print(f"Precision = {precision:.4f}")
        print(f"Recall = {recall:.4f}")
        print(f"F1 Score = {f1score:.4f}")

        clf_y_pred = clf.predict_proba(X_test)[:, 1]
        fpr, tpr, _ = roc_curve(y_test, clf_y_pred)
        auc = round(metrics.roc_auc_score(y_test, clf_y_pred), 4)
        results[model_name].extend([fpr, tpr, auc])
    else:
        print(classification_report(y_test, clf_y_pred, digits=4, zero_division=0))
        clf_y_pred = clf.predict_proba(X_test)
        micro_roc_auc_ovr = roc_auc_score(y_test, clf_y_pred, multi_class="ovr", average="micro")
        macro_roc_auc_ovr = roc_auc_score(y_test, clf_y_pred, multi_class="ovr", average="macro")
        weighted_roc_auc_ovr = roc_auc_score(y_test, clf_y_pred, multi_class="ovr", average="weighted")
        print(f"Micro-averaged One-vs-Rest ROC AUC score: {micro_roc_auc_ovr:.4f}")
        print(f"Macro-averaged One-vs-Rest ROC AUC score: {macro_roc_auc_ovr:.4f}")
        print(f"Weighted-averaged One-vs-Rest ROC AUC score: {weighted_roc_auc_ovr:.4f}")

    print()
    print("---------------------------")
    return results


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="ML for malware characterization and identification")
    parser.add_argument("--dataset",
                        type=str,
                        help="benign for microsoft dataset with benign file or malign for microsoft dataset without "
                             "benign file, ST-WinMal-benign for second dataset with benign file or ST-WinMal-malign "
                             "for second dataset with malign file",
                        choices=["benign", "malign", "ST-WinMal-benign", "ST-WinMal-malign"],
                        required=True)
    parser.add_argument("--features",
                        nargs="+",
                        help="insert feature(s) to use",
                        choices=["binary_size",
                                 "3_gram_bytes",
                                 "4_gram_opcode",
                                 "4_gram_API",
                                 "entropy",
                                 "opcode_counter",
                                 "API_check"],
                        required=True)
    args = parser.parse_args()
    ben_set = {"binary_size", "3_gram_bytes", "entropy"}

    mal_set = {"4_gram_opcode",
               "4_gram_API",
               "opcode_counter",
               "API_check"}

    not_andrea_microsoft_set = {"4_gram_opcode",
                                "4_gram_API",
                                "opcode_counter"}
    andrea_microsoft_set = {"API_check",
                            "3_gram_bytes",
                            "entropy"}

    feat_set = set(args.features)
    if args.dataset == "benign" and (feat_set & mal_set):
        print("Features with benign dataset: ")
        for i in ben_set:
            print(" - " + i)
        sys.exit()
    if (args.dataset == "andrea_microsoft_benign"
        or args.dataset == "ST-WinMal-malign") \
            and (feat_set & not_andrea_microsoft_set):
        print("Features with andrea microsoft dataset: ")
        for i in andrea_microsoft_set:
            print(" - " + i)
        sys.exit()

    if args.dataset == "benign":
        lbls = pd.read_csv(data_dir.joinpath("trainLabels_benign.csv"))
    elif args.dataset == "malign":
        lbls = pd.read_csv(data_dir.joinpath("trainLabels_malign.csv"))
    elif args.dataset == "ST-WinMal-benign":
        lbls = pd.read_csv(data_dir.joinpath("ST-WinMal_benign.csv"))
    elif args.dataset == "ST-WinMal-malign":
        lbls = pd.read_csv(data_dir.joinpath("ST-WinMal_malign.csv"))
    else:
        sys.exit()

    lbls["Id"] = lbls["Id"].astype(str)
    lbls["Class"] = lbls["Class"].astype("category")
    lbls.set_index("Id", inplace=True)

    if args.dataset == "ST-WinMal-benign" or args.dataset == "ST-WinMal-malign":
        suffix = args.dataset.split("-")[-1]
        if "API_check" in args.features and not os.path.exists("pre-extract/API_check_ST-WinMal_"+suffix+".npz"):
            start_time = datetime.now()
            if not os.path.exists("Utils/apis_"+suffix+".txt"):
                for file in lbls.index:
                    extract_apis(file)
                save_apis(suffix)

            apis_vct = []
            load_apis(suffix)
            for file in lbls.index:
                apis_vct.append(check_apis(file))
            feature_to_save = np.array(apis_vct)
            col_sum = feature_to_save.sum(axis=0)
            top_frqn_ftr_idx = np.argsort(col_sum)[::-1][:5000]
            print(col_sum[top_frqn_ftr_idx])
            save_npz(ftr_dir.joinpath("API_check_ST-WinMal_"+suffix+".npz"),
                     csr_matrix(feature_to_save)[:, top_frqn_ftr_idx])
            total_time = datetime.now() - start_time
            print("Time extraction API check: %s" % total_time)
        if "entropy" in args.features and not os.path.exists("pre-extract/entropy_ST-WinMal_"+suffix+".npz"):
            start_time = datetime.now()

            entropy_matrix = []
            for file in lbls.index:
                entropy_row = extract_entropy_section(file)
                entropy_matrix.append(entropy_row)

            feature_to_save = np.array(entropy_matrix)
            save_npz(ftr_dir.joinpath("entropy_ST-WinMal_"+suffix+".npz"), csr_matrix(feature_to_save))
            total_time = datetime.now() - start_time
            print("Time extraction entropy for Andrea dataset: %s" % total_time)
        if "3_gram_bytes" in args.features and not os.path.exists("pre-extract/3_gram_bytes_ST-WinMal_"+suffix+".npz"):
            start_time = datetime.now()
            bytes_ngram_vct, bytes_ngrams = extract_ngrams(lbls.index, extract_bytes_microsoft, 3)
            print(bytes_ngrams.shape)

            col_sum = bytes_ngrams.sum(axis=0).A1
            top_frqn_ftr_idx = np.argsort(col_sum)[::-1][:5000]
            print(col_sum[top_frqn_ftr_idx])
            save_npz(ftr_dir.joinpath("3_gram_bytes_ST-WinMal_"+suffix+".npz"), bytes_ngrams[:, top_frqn_ftr_idx])
            total_time = datetime.now() - start_time
            print("Time extraction ngrams from Andrea dataset: %s" % total_time)
    else:
        if "binary_size" in args.features and not os.path.exists("pre-extract/binary_size_" + args.dataset + ".npz"):
            start_time = datetime.now()
            size_matrix = []
            for file in lbls.index:
                size_matrix.append([extract_binary_sizes(file)])
            feature_to_save = np.array(size_matrix)
            save_npz(ftr_dir.joinpath("binary_size_" + args.dataset + ".npz"), csr_matrix(feature_to_save))
            total_time = datetime.now() - start_time
            print("Time extraction API check: %s" % total_time)

        if "3_gram_bytes" in args.features and not os.path.exists("pre-extract/3_gram_bytes_" + args.dataset + ".npz"):
            start_time = datetime.now()
            bytes_ngram_vct, bytes_ngrams = extract_ngrams(lbls.index, extract_bytes_sequence, 3)
            print(bytes_ngrams.shape)

            col_sum = bytes_ngrams.sum(axis=0).A1
            top_frqn_ftr_idx = np.argsort(col_sum)[::-1][:5000]
            print(col_sum[top_frqn_ftr_idx])
            save_npz(ftr_dir.joinpath("3_gram_bytes_" + args.dataset + ".npz"), bytes_ngrams[:, top_frqn_ftr_idx])
            total_time = datetime.now() - start_time
            print("Time extraction API check: %s" % total_time)

        if "4_gram_opcode" in args.features and not os.path.exists(
                "pre-extract/4_gram_opcode_malign.npz") and args.dataset == "malign":
            start_time = datetime.now()
            opcodes_ngram_vct, opcodes_ngrams = extract_ngrams(lbls.index, extract_opcode_sequence, 4)
            print(opcodes_ngrams.shape)

            col_sum = opcodes_ngrams.sum(axis=0).A1
            top_frqn_ftr_idx = np.argsort(col_sum)[::-1][:5000]
            print(col_sum[top_frqn_ftr_idx])

            save_npz(ftr_dir.joinpath("4_gram_opcode_malign.npz"), opcodes_ngrams[:, top_frqn_ftr_idx])
            total_time = datetime.now() - start_time
            print("Time extraction API check: %s" % total_time)

        if "opcode_counter" in args.features and not os.path.exists(
                "pre-extract/opcode_counter_malign.npz") and args.dataset == "malign":
            start_time = datetime.now()
            opcode_counter = []

            for file in lbls.index:
                opc_counter = count_asm_opcode(file)
                opcode_counter.append(opc_counter)

            feature_to_save = np.array(opcode_counter)
            save_npz(ftr_dir.joinpath("opcode_counter_malign.npz"), csr_matrix(feature_to_save))
            total_time = datetime.now() - start_time
            print("Time extraction API check: %s" % total_time)

        if "API_check" in args.features and not os.path.exists(
                "pre-extract/API_check_malign.npz") and args.dataset == "malign":
            start_time = datetime.now()
            if not os.path.exists("Utils/syscalls.txt"):
                for file in lbls.index:
                    extract_syscall(file)
                save_syscall()

            syscall_vct = []
            load_syscall()
            for file in lbls.index:
                syscall_vct.append(check_syscall(file))

            feature_to_save = np.array(syscall_vct)

            col_sum = feature_to_save.sum(axis=0)
            top_frqn_ftr_idx = np.argsort(col_sum)[::-1][:5000]
            print(col_sum[top_frqn_ftr_idx])

            save_npz(ftr_dir.joinpath("API_check_malign.npz"), csr_matrix(feature_to_save)[:, top_frqn_ftr_idx])
            total_time = datetime.now() - start_time
            print("Time extraction API check: %s" % total_time)
        if "4_gram_API" in args.features and not os.path.exists(
                "pre-extract/4_gram_API_malign.npz") and args.dataset == "malign":
            start_time = datetime.now()
            api_ngrm_vct, api_ngrms = extract_ngrams(lbls.index, extract_syscall_sequence, 4)
            print(api_ngrms.shape)

            col_sum = api_ngrms.sum(axis=0).A1
            top_frqn_ftr_idx = np.argsort(col_sum)[::-1][:5000]
            print(col_sum[top_frqn_ftr_idx])

            save_npz(ftr_dir.joinpath("4_gram_API_malign.npz"), api_ngrms[:, top_frqn_ftr_idx])
            total_time = datetime.now() - start_time
            print("Time extraction API check: %s" % total_time)

        if "entropy" in args.features and not os.path.exists("pre-extract/entropy_" + args.dataset + ".npz"):
            start_time = datetime.now()
            entropy_matrix = []
            for file in lbls.index:
                entropy_matrix.append([extract_entropy(file)])
            feature_to_save = np.array(entropy_matrix)
            save_npz(ftr_dir.joinpath("entropy_" + args.dataset + ".npz"), csr_matrix(feature_to_save))
            total_time = datetime.now() - start_time
            print("Time extraction API check: %s" % total_time)

    X = integrated_features(args.features, args.dataset)
    print(X.head())
    print(X.shape)

    print()

    X_train, X_test, y_train, y_test = train_test_split(X, lbls, test_size=0.2, stratify=lbls)

    scaler = preprocessing.StandardScaler()

    scaler.fit(X_train)
    X_train_scaled = scaler.transform(X_train)
    X_test_scaled = scaler.transform(X_test)

    y_train = y_train.values.ravel()

    results = {"RandomForestClassifier": [],
               "KNeighborsClassifier": [],
               "SVC": [],
               "HistGradientBoostingClassifier": []}

    if len(args.features) == 1:
        feat = args.features[0]
    else:
        feat = "_".join(args.features)

    rf = RandomForestClassifier()
    print("Random Forest Results")
    results = fit_and_pred(rf, X_train, X_test, y_train, y_test, results, args.dataset, feat)

    knn = KNeighborsClassifier(n_neighbors=5)
    print("K-Nearest Neighbors Results")
    results = fit_and_pred(knn, X_train, X_test, y_train, y_test, results, args.dataset, feat)

    if args.dataset == "benign" or args.dataset == "malign":
        svm = svm.SVC(probability=True, class_weight='balanced')
    else:
        svm = svm.SVC(probability=True)
    print("Support Vector Machine Results")
    results = fit_and_pred(svm, X_train_scaled, X_test_scaled, y_train, y_test, results, args.dataset, feat)

    if args.dataset == "malign":
        gb = HistGradientBoostingClassifier(class_weight='balanced')
    else:
        gb = HistGradientBoostingClassifier()
    print("Gradient Boosting Results")
    results = fit_and_pred(gb, X_train, X_test, y_train, y_test, results, args.dataset, feat)

    if args.dataset == "benign" or args.dataset == "ST-WinMal-benign":
        plt.figure(0).clf()
        plt.rcdefaults()
        plt.figure(figsize=(14, 10))
        plt.plot(results["RandomForestClassifier"][0],
                 results["RandomForestClassifier"][1],
                 label="Random Forest, AUC="+str(results["RandomForestClassifier"][2]))
        plt.plot(results["KNeighborsClassifier"][0],
                 results["KNeighborsClassifier"][1],
                 label="KNN, AUC=" + str(results["KNeighborsClassifier"][2]))
        plt.plot(results["SVC"][0],
                 results["SVC"][1],
                 label="SVM, AUC=" + str(results["SVC"][2]))
        plt.plot(results["HistGradientBoostingClassifier"][0],
                 results["HistGradientBoostingClassifier"][1],
                 label="Gradient Boosting, AUC=" + str(results["HistGradientBoostingClassifier"][2]))
        plt.legend(fontsize=30)
        plt.grid()
        plt.xlabel("False Positive Rate (Positive label: 1)", fontsize=28)
        plt.ylabel("True Positive Rate (Positive label: 1)", fontsize=28)
        plt.xticks(fontsize=26)
        plt.yticks(fontsize=26)
        plt.savefig("results/ROC_Curve_"+args.dataset+"_"+feat+".svg")
