import os.path
import re
from typing import Callable
import numpy as np
from sklearn.feature_extraction.text import CountVectorizer
from Utils.ReadFile import ReadFile
from math import log

SeqReader = Callable[[str], str]

opcodes = ['add', 'align', 'and', 'bt', 'call', 'cdq', 'cld', 'cli', 'cmc', 'cmp', 'cwd', 'daa', 'db', 'dd', 'dec',
           'dw', 'faddp', 'fchs', 'fdiv', 'fdivp', 'fdivr', 'fild', 'fistp', 'fld', 'fstcw', 'fstcwimul', 'fstp',
           'fword', 'fxch', 'imul', 'inc', 'ins', 'ja', 'jb', 'jbe', 'je', 'jg', 'jge', 'jl', 'jle', 'jmp', 'jnb',
           'jno', 'jnz', 'jo', 'jz', 'lea', 'loope', 'mov', 'movsw', 'movzx', 'mul', 'neg', 'not', 'or', 'out', 'outs',
           'pop', 'popf', 'push', 'pusha', 'pushf', 'rcl', 'rcr', 'rdtsc', 'rep', 'ret', 'retn', 'rol', 'ror', 'sal',
           'sar', 'sbb', 'scas', 'setb', 'setle', 'setnle', 'setnz', 'setz', 'shl', 'shld', 'shr', 'sidt', 'stc', 'std',
           'sti', 'stos', 'stosw', 'sub', 'test', 'wait', 'xchg', 'xor']

sections = ['.bss', '.cormeta', '.data', 'debug$F', '.debug$P', '.debug$S', '.debug$T', '.drective', '.edata', '.idata',
           '.idlsym', '.pdata', '.rdata', '.reloc', '.rsrc', '.sbss', '.sdata', '.srdata', '.sxdata', '.text', '.tls',
           '.tls$', '.vsdata', '.xdata']

api_rgx = re.compile(r"extrn\s+(?:__imp_)?([^\s:]+)", flags=re.IGNORECASE)
call_rgx = re.compile(r"\scall\s+(?:ds:)(?:__imp_)?([^\s]+)", flags=re.IGNORECASE)

all_apis = []


def extract_binary_sizes(id):
    """
    Extracts the size of a binary file passed the id
    :param id: binary file id
    :return: binary file size
    """
    return os.path.getsize("data/samples/" + id + ".bytes")


def extract_ngrams(ids, seq_reader, n):
    """
    Extracts the n-grams of a specific object, which may be bytes, opcodes or APIs
    :param ids: id of files from which to extract n-grams
    :param seq_reader: function to be used to extract the sequence of the specified object (bytes, opcode or API)
    :param n: gram size
    :return: The CountVectorizer object with a matrix in which the occurrences of n-grams and found n-grams are counted
    """
    class Reader:
        def __init__(self, id: str):
            self.id = id

        def read(self):
            return seq_reader(self.id)

    seqs = [Reader(id) for id in ids]
    ngram_vct = CountVectorizer(ngram_range=(n, n), stop_words=None, token_pattern=r"(?u)\b[\w@?]{2,}\b", lowercase=False, input="file")
    ngrams = ngram_vct.fit_transform(seqs)
    return ngram_vct, ngrams


def extract_bytes_sequence(id):
    """
    Extracts the sequence of bytes from a specified file
    :param id: file id
    :return: Sequence of file bytes
    """
    sequence_bytes = []
    for byte in ReadFile.bytes_reader(id):
        sequence_bytes.append(byte.lower())
    return " ".join(sequence_bytes)


def extract_opcode_sequence(id):
    """
    Extracts the sequence of opcode from a specified file
    :param id: file id
    :return: Sequence of file opcodes
    """
    sequence_opcodes = []
    for line in ReadFile.asm_lines(id):
        row = line.split()
        for opcode in opcodes:
            if opcode in row:
                sequence_opcodes.append(opcode)
                break
    return " ".join(sequence_opcodes)


def count_asm_opcode(id):
    """
    Counts the occurrences of each assembly function within a specified asm file
    :param id: file id
    :return: array with the occurrences of each opcode
    """
    opcode_counter = np.zeros(len(opcodes))
    for line in ReadFile.asm_lines(id):
        parts = line.split()
        for opcode in opcodes:
            if opcode in parts:
                opcode_counter[opcodes.index(opcode)] += 1
                break
    return opcode_counter


def extract_syscall(id):
    """
    Inserts in the list of all syscalls those found in a specified file
    :param id: file id
    """
    asm = ReadFile.asm(id)
    apis = set(api_rgx.findall(asm))
    calls = call_rgx.findall(asm)
    syscalls = [i for i in calls if i in apis]
    list_syscalls = list(dict.fromkeys(syscalls))
    for syscall in list_syscalls:
        if syscall not in all_apis:
            all_apis.append(syscall)


def save_syscall():
    """
    Saves the syscalls entered up to that point in the global list within the syscall.txt file
    """
    with open('Utils/syscalls.txt', 'w') as f:
        for api in all_apis:
            f.write(f"{api}\n")


def load_syscall():
    """
    Load the syscalls into the syscall.txt file in the global list
    """
    with open('Utils/syscalls.txt', 'r') as f:
        data = f.read()
        global all_apis
        all_apis = data.split("\n")


def check_syscall(id):
    """
    Checks for the presence of syscalls within a specified asm file
    :param id: file id
    :return: Array with a 0 in the columns of syscalls not found and 1 in those found
    """
    syscall_presence = np.zeros(len(all_apis))
    asm = ReadFile.asm(id)
    apis = set(api_rgx.findall(asm))
    calls = call_rgx.findall(asm)
    syscalls = [i for i in calls if i in apis]
    list_syscalls = list(dict.fromkeys(syscalls))
    for sys in list_syscalls:
        if sys in all_apis:
            syscall_presence[all_apis.index(sys)] = 1
    return syscall_presence


def extract_syscall_sequence(id):
    """
    Extracts the syscall sequence within an asm file
    :param id: file id
    :return: syscall sequence of the file
    """
    asm = ReadFile.asm(id)
    apis = set(api_rgx.findall(asm))
    calls = call_rgx.findall(asm)
    syscalls = [i for i in calls if i in apis]
    return " ".join(syscalls)


def extract_entropy(id):
    """
    Calculates the entropy of a binary file
    :param id: file id
    :return: file entropy
    """
    data = ReadFile.bytes_reader(id)
    possible = dict(((x, 0) for x in range(0, 256)))
    for byte in data:
        possible[int(byte, 16)] += 1

    data_len = len(data)
    entropy = 0.0

    for i in possible:
        if possible[i] == 0:
            continue

        p = float(possible[i] / data_len)
        entropy -= p * log(p, 2)

    return entropy


def extract_apis(id):
    """
    Extracts APIs from a sample of the ST-WinMal dataset
    :param id: file id
    """
    apis = ReadFile.apis(id)
    for api in apis:
        if api not in all_apis:
            all_apis.append(api)


def save_apis(suffix):
    """
    Save the API in the global list within the apis.txt file
    """
    with open('Utils/apis_'+suffix+'.txt', 'w') as f:
        for api in all_apis:
            f.write(f"{api}\n")


def load_apis(suffix):
    """
    Load the API within the apis.txt file into the global list
    """
    with open('Utils/apis_'+suffix+'.txt', 'r') as f:
        data = f.read()
        global all_apis
        all_apis = data.split("\n")


def check_apis(id):
    """
    Checks for the presence of apis within a sample of the ST-WinMal dataset
    :param id: file id
    :return: Array with a 0 in the columns of API not found and 1 in those found
    """
    apis_presence = np.zeros(len(all_apis))
    apis = ReadFile.apis(id)
    for api in apis:
        if api in all_apis:
            apis_presence[all_apis.index(api)] = 1
    return apis_presence


def extract_entropy_section(id):
    """
    Extracts entropy from the .entropy files of the ST-WinMal dataset
    :param id: file id
    :return: Array with entropy for each section
    """
    sections_entropies = np.full(len(sections), 9, dtype=float)
    list_sections = ReadFile.entropy(id)
    for elem in list_sections:
        if elem[0] in sections:
            sections_entropies[sections.index(elem[0])] = float(elem[1])
    return sections_entropies


def extract_bytes_microsoft(id):
    """
    Extracts the byte sequence from a sample of the ST-WinMal dataset
    :param id: file id
    :return: Sequence of bytes
    """
    sequence_bytes = []
    for byte in ReadFile.bytes_microsoft(id):
        sequence_bytes.append(byte.lower())
    return " ".join(sequence_bytes)
