classdef parkingStateValidator < nav.StateValidator

% 'parkingStateValidator' configures a custom state validation for RRT*
% planner for perpendicular parking problem.

    properties
        Obstacles
        EgoCar
        ValidationDistance
    end

    methods
        function obj = parkingStateValidator(ss)
            
            % Validator Obstacles List:
            obj@nav.StateValidator(ss);

            % Obstacles:
            obj.Obstacles = createObstacles();

            % Ego Vehicle:
            obj.EgoCar = collisionBox(4.7, 1.8, 0);

            % Safety Distance
            obj.ValidationDistance = 0.1;

        end

        % 'isStateValid' checks if state is valid:
        function isValid = isStateValid(obj, state)

            isValid = true(size(state,1), 1);
            for k = 1:size(state, 1)
                
                relPose = [1.4, 0, 0];
                st = robotics.core.internal.SEHelpers.accumulatePoseSE2(state(k,:), relPose);

                pos = [st(1), st(2), 0];
                quat = eul2quat([st(3), 0, 0]);

                for i = 1:length(obj.Obstacles)
                    [~, dist] = ...
                        robotics.core.internal.intersect(obj.Obstacles{i}.GeometryInternal, ...
                        obj.Obstacles{i}.Position, ...
                        obj.Obstacles{i}.Quaternion,...
                        obj.EgoCar.GeometryInternal,...
                        pos, quat, 1);
                    
                    if dist <= 0.15
                        isValid(k) = false;
                        return;
                    end
                end
            end
        end

        % 'isMotionValid' checks if path between states is valid:
        function [isValid, lastValid] = isMotionValid(obj, state1, state2)
            
            % Verify that state1 is valid
            if ~obj.isStateValid(state1)
                isValid = false;
                lastValid = nan(1,obj.StateSpace.NumStateVariables);
                return
            end

            % Interpolate between state1 and state2 with ValidationDistance
            dist = obj.StateSpace.distance(state1, state2);
            interval = obj.ValidationDistance/dist;
            interpStates = obj.StateSpace.interpolate(state1, state2, [0:interval:1 1]);

            % Check all interpolated states for validity
            interpValid = obj.isStateValid(interpStates);

            if nargin == 1
                if any(~interpValid)
                    isValid = false;
                else
                    isValid = true;
                end
            else
                % Find the first invalid index. Note that the maximum
                % non-empty value of 'firstInvalidIdx' can be 2, since we
                % always check state1 and state 2, and state1 is already
                % verified above.
                firstInvalidIdx = find(~interpValid, 1);

                if isempty(firstInvalidIdx)
                    % The whole motion is valid:
                    isValid = true;
                    lastValid = state2;
                else
                    isValid = false;
                    lastValid = interpStates(firstInvalidIdx-1, :);
                end
            end
        end


        %COPY Create deep copy of object
        function copyObj = copy(obj)
            % not used
        end
    end
end

% =========================================================================
% LOCAL FUNCTION
% =========================================================================

function obstacles = createObstacles()

% Obstacles: 2 occupied parking lots, road curbside and yellow line
obsLength = 6.2;
egoLength = 4.7;
egoWidth = 1.8;

% Parked Car 1:
obs1 = collisionBox(egoWidth, egoLength, 0);
T1 = trvec2tform([-9.3, -1.75, 0]);
obs1.Pose = T1;

% Parked Car 2:
obs2 = collisionBox(egoWidth, egoLength, 0);
T2 = trvec2tform([-6.2, -1.75, 0]);
obs2.Pose = T2;

% Right Bottom Line:
obs3 = collisionBox(4*obsLength, 0.5, 0);
T3 = trvec2tform([11.35, 1.35, 0]);
obs3.Pose = T3;

% Parking Vertical Rightmost Line:
obs4 = collisionBox(0.5, obsLength+1, 0);
T4 = trvec2tform([-1.3, -2, 0]);
obs4.Pose = T4;

% Parking Bottom Line:
obs5 = collisionBox(10.3, 0.5, 0);
T5 = trvec2tform([-6.2, -5.35, 0]);
obs5.Pose = T5;

% Parking Vertical Leftmost Line:
obs6 = collisionBox(0.5, obsLength+1, 0);
T6 = trvec2tform([-11.1, -2, 0]);
obs6.Pose = T6;

% Left Bottom Line:
obs7 = collisionBox(2*obsLength, 0.5, 0);
T7 = trvec2tform([-17.55, 1.35, 0]);
obs7.Pose = T7;

% Other Line:
obs8 = collisionBox(47.5, 0.5, 0);
% T8 = trvec2tform([0, 5.35, 0]);
T8 = trvec2tform([0, 6, 0]);
obs8.Pose = T8;

obstacles = {obs1, obs2, obs3, obs4, obs5, obs6, obs7, obs8};

end
