%% PERPENDICULAR PARKING USING RRT PLANNER AND MPC TRACKING CONTROLLER

clear all, close all, clc

%% Ego Vehicle Definition and Parking Environment

vdims = vehicleDimensions;          % Ego Vehicle Dimensions [m].
egoLength    = vdims.Length;        % Ego Vehicle Length [m].
egoWidth     = vdims.Width;         % Ego Vehicle Width [m].
egoWheelbase = vdims.Wheelbase;     % Ego Vehicle Wheelbase [m].

obsLength = 6.2;                    % Length of the Obstacle [m].
distToCenter = 0.5*egoWheelbase;    % [m].

% Ego Inital and Target Locations [x-pos (m), y-pos (m), yaw-angle (rad)]:
egoInitialPose = [1, 3, 0];
egoTargetPose = [-3.1, -3.15, pi/2];

Tv = 0.1;
helperSLVisualizeParking(egoInitialPose,0);

pause(1)

%% Path Planning from RRT*

xlim = [-5 5];   
ylim = [-6 6];     
yawlim = [-pi pi]; 
bounds = [xlim; ylim; yawlim];
stateSpace = stateSpaceReedsShepp(bounds);
% stateSpace.MinTurningRadius = 3;
% stateSpace.MinTurningRadius = 2.4;
stateSpace.MinTurningRadius = 2.3;

stateValidator = parkingStateValidator(stateSpace);

planner = plannerRRTStar(stateSpace, stateValidator);
planner.MaxConnectionDistance = 4;
% planner.MaxConnectionDistance = 2;
planner.ContinueAfterGoalReached = true;
% planner.MaxIterations = 2000;
planner.MaxIterations = 2e3;

rng(9, 'twister')
[pathObj, solnInfo] = plan(planner, egoInitialPose, egoTargetPose);

f = findobj('Name', 'Automated Perpendicular Parking');
ax = gca(f);
hold(ax, 'on');
plot(ax, solnInfo.TreeData(:,1), solnInfo.TreeData(:,2), 'y.-', 'LineWidth', 2) % Tree expansion

p = 100;
% p = 140;
pathObj.interpolate(p+1);
xRef = pathObj.States;

plot(ax, xRef(:,1), xRef(:,2), 'b-', 'LineWidth',2)

% return
pause(5)

%% Design the Nonlinear MPC Tracking Controller

mpcverbosity('off');

nx = 3;     % Number of States.
ny = 3;     % Number of Outputs.
nu = 2;     % Number of Inputs.

nlobjTracking = nlmpc(nx, ny, nu);

Ts = 0.1;
pTracking = 10;
nlobjTracking.Ts = Ts;
nlobjTracking.PredictionHorizon = pTracking;
nlobjTracking.ControlHorizon = pTracking;

nlobjTracking.MV(1).Min = -2;
nlobjTracking.MV(1).Max = 2;
nlobjTracking.MV(2).Min = -pi/6;
nlobjTracking.MV(2).Max = pi/6;
% nlobjTracking.MV(2).Min = -pi/4;
% nlobjTracking.MV(2).Max = pi/4;

nlobjTracking.Weights.OutputVariables = [3 3 3]; % x,y 2.5 ok
nlobjTracking.Weights.ManipulatedVariablesRate = [0.1 0.005];   % 2nd MV values: 0.1, 0.05, 0.01

nlobjTracking.Model.StateFcn = "parkingVehicleStateFcnRRT";
nlobjTracking.Jacobian.StateFcn = "parkingVehicleStateJacobianFcnRRT";

nlobjTracking.Optimization.CustomEqConFcn = "parkingTerminalConFcn";

validateFcns(nlobjTracking, randn(3,1), randn(2,1));

%% Closed-Loop Simulation in MATLAB

x = egoInitialPose';

u = zeros(2,1);

[coredata, onlinedata] = getCodeGenerationData(nlobjTracking, x, u);

mexfcn = buildMEX(nlobjTracking, 'parkingRRTMex', coredata, onlinedata);
[coredata, onlinedata] = getCodeGenerationData(nlobjTracking, x, u);

xTrackHistory = x;
uTrackHistory = u; 
mv = u;
Duration = 14;
Tsteps = Duration/Ts;
Xref = [xRef(2:p+1,:); repmat(xRef(end,:),Tsteps-p,1)];

for ct = 1:Tsteps
    
    % States
    xk = x;
    
    % Compute optimal control moves with MEX function
    onlinedata.ref = Xref(ct:min(ct+pTracking-1,Tsteps),:);
    [mv,onlinedata,info] = mexfcn(xk,mv,onlinedata);
    
    % Implement first optimal control move and update plant states.
    ODEFUN = @(t,xk) parkingVehicleStateFcnRRT(xk,mv);
    [TOUT,YOUT] = ode45(ODEFUN,[0 Ts], xk);
    x = YOUT(end,:)';
    
    % Save plant states for display.
    xTrackHistory = [xTrackHistory x]; %#ok<*AGROW>
    uTrackHistory = [uTrackHistory mv];
    
end

plotAndAnimateParkingRRT(p, xRef, xTrackHistory, uTrackHistory, info);

% %% Closed-Loop Simulation in Simulink
% 
% mdl = 'mpcVDAutoParkingRRT';
% open_system(mdl)
% 
% f = findobj('Name','Automated Parallel Parking');
% close(f)
% 
% sim(mdl)

%% Conclusion

% Enable message display
mpcverbosity('on');

% % Close Simulink model
% bdclose(mdl)

% % Close animation plots
% f = findobj('Name','Automated Parallel Parking');
% close(f)