%% PLAN A PERPENDICULAR PARKING PATH USING MULTISTAGE NONLINEAR MPC

clear all
close all
clc

%% Vehicle Definition and Parking Environment:

vdims = vehicleDimensions;          % Ego Vehicle Dimensions [m].
egoLength    = vdims.Length;        % Ego Vehicle Length [m].
egoWidth     = vdims.Width;         % Ego Vehicle Width [m].
egoWheelbase = vdims.Wheelbase;     % Ego Vehicle Wheelbase [m].

obsLength = 6.2;                    % Length of the Obstacle [m].
distToCenter = 0.5*egoWheelbase;    % Distance to Centre [m].

% Ego Inital and Target Locations [x-pos (m), y-pos (m), yaw-angle (rad)]:
egoInitialPose = [1, 3, 0];
egoTargetPose = [-3.1, -3.15, pi/2];

% Visualization of the Parking Environment:
helperSLVisualizeParking(egoInitialPose, 0);
pause(1)

%% Configuration of the Vehicle Path Planner System Block

% Number of Obstacles:
numObs = 8;

% Obstacles' Matrix - each row represents an obstacle with 5 elements:
% obsMat(a, b) = [x-pos, y-pos, heading angle, length, width]
obsMat = [-9.3      , -1.75, 0,      egoLength,     egoWidth ; ...
          -obsLength, -1.75, 0,      egoLength,     egoWidth ; ...
           11.35    ,  1.35, 0,  4 * obsLength, 0.5          ; ...
          -1.3      , -2   , 0,  0.5          , 1 + obsLength; ...
          -obsLength, -5.35, 0, 10.3          , 0.5          ; ...
          -11.1     , -2   , 0,  0.5          , 1 + obsLength; ...
          -17.55    ,  1.35, 0,  2 * obsLength, 0.5          ; ...
            0       ,  6   , 0, 47.5          , 0.5         ];

Ts = 1;     % Sampling Time [s].
hp = 10;    % Prediciton Horizon [s].

v_range = [-3.5, 3.5];          % Velocity Range: +- 3.5 m/s.
steer_range = [-pi/5, pi/5];    % Steering Angle Range: +- 36 deg.
% v_range = [-3.5, 3.5];          
% steer_range = [-pi/6, pi/6];    

safetyDistance = 0.1;           % Safety Distance [m].

%% Simulation of the Vehicle Path Planner in Simulink Model

Tsim = 1/hp;     % Simulation Time [s].

% open_system("VehiclePathPlannerSystem.slx") % Open the Simulink Model.
out = sim("VehiclePathPlannerSystem.slx");  % Simulate the Simulink Model.

% Simulation Outputs:
out_x_seq = out.x_seq;          % Optimal State Sequence.
out_ExitFlag = out.exitflag;    % Optimization Status.
out_mv_seq = out.mv_seq;        % Optimal Control Sequence.
out_mv_opti = out.mv_opti;      % Optimal MV Control Action.
out_cost = out.cost;            % Optimal Cost.

% Animate the Simulation:
timeLength = size(out_x_seq, 1);
for ct = 1:timeLength
    helperSLVisualizeParking(out_x_seq(ct,:), 0);
    pause(0.1);
end

% Plot of the Vehicle States:
figure()
subplot(311), plot(out_x_seq(:, 1), 'linewidth', 1.5), grid on
xlabel('Time, $t$ [$s$]', 'FontSize', 15, 'Interpreter', 'latex')
ylabel('$x$ position [$m$]', 'FontSize', 15, 'Interpreter', 'latex')
subplot(312), plot(out_x_seq(:, 2), 'linewidth', 1.5), grid on
xlabel('Time, $t$ [$s$]', 'FontSize', 15, 'Interpreter', 'latex')
ylabel('$y$ position [$m$]', 'FontSize', 15, 'Interpreter', 'latex')
subplot(313), plot(rad2deg(out_x_seq(:, 3)), 'linewidth', 1.5), grid on
xlabel('Time, $t$ [$s$]', 'FontSize', 15, 'Interpreter', 'latex')
ylabel('Yaw angle, $\psi$ [$deg$]', 'FontSize', 15, 'Interpreter', 'latex')
sgtitle('\bf Behaviour of the Ego Vehicle States', 'FontSize', 20, 'Interpreter', 'latex')

% Plot of the Vehicle Control Inputs:
figure()
subplot(211), stairs(out_mv_seq(:, 1), 'linewidth', 1.5), grid on
xlabel('Time, $t$ [$s$]', 'FontSize', 15, 'Interpreter', 'latex')
ylabel('Speed, $v$ [$m/s$]', 'FontSize', 15, 'Interpreter', 'latex')
subplot(212), stairs(rad2deg(out_mv_seq(:, 2)), 'linewidth', 1.5), grid on
xlabel('Time, $t$ [$s$]', 'FontSize', 15, 'Interpreter', 'latex')
ylabel('Steering Angle, $\delta$ [$deg$]', 'FontSize', 15, 'Interpreter', 'latex')
sgtitle('\bf Behaviour of the Ego Vehicle Control Inputs', 'FontSize', 20, 'Interpreter', 'latex')

% Analyze Parking Results:
fprintf('1) Optimization exit flag = %d (Successful when positive)\n', ...
    out_ExitFlag)
if out_ExitFlag > 0
    fprintf('1.1) Optimal solution for speed [m/s] and steering angle [deg]: %2.4f %2.4f\n', ...
        out_mv_opti(1), out_mv_opti(2)')
end
fprintf('2) Optimal Cost = %i\n', out_cost)
fprintf('3) Elapsed time [s] for nlmpcmove = %.4f\n', out.SimulationMetadata.TimingInfo.TotalElapsedWallTime)

% [4] final position of ego:
e1 = out_x_seq(end, 1) - egoTargetPose(1);
e2 = out_x_seq(end, 2) - egoTargetPose(2);
e3 = rad2deg(out_x_seq(end, 3) - egoTargetPose(3));
fprintf('4) Final states error in x [m], y [m], and psi [deg]: %2.4f, %2.4f, %2.4f\n', ...
    e1, e2, e3)

% [5] final controller values:
vFinal = out_mv_seq(end, 1);
deltaFinal = rad2deg(out_mv_seq(end, 2));
fprintf('5) Final control inputs speed [m/s] and steering angle [deg]: %2.4f %2.4f\n', ...
    vFinal, deltaFinal)

fprintf('\n')

% %% Conclusion
% 
% % Enable Message Display:
% mpcverbosity('off');
% 
% % Close Animation Plots:
% f = findobj('Name', 'Automated Parallel Parking');
