function helperSLVisualizeParking(pose, steer)

% This function creates and initialize visualizers for ego and obstacles
% for parking.

persistent vehicleBodyHandle axesHandle vehicleDims

pose(3) = rad2deg(pose(3));
steer = rad2deg(steer);

if isempty(vehicleDims)
    vehicleDims = vehicleDimensions;
end

if isempty(axesHandle) || ~isvalid(axesHandle)
    % Initialize figure
    % fh1 = figure('Visible', 'off', 'Position', [287 501 1412 480]);
    fh1 = figure('Visible', 'off');
    fh1.Name        = 'Automated Perpendicular Parking';
    fh1.NumberTitle = 'off';
    fh1.WindowState = 'Maximized';
    axesHandle      = axes(fh1);
    legend off, axis equal, grid on
    title(axesHandle, '\bf Perpendicular Parking using Multistage NMPC', ...
        'FontSize', 20, 'Interpreter', 'latex');
    xlabel('$x$ axis', 'FontSize', 15, 'Interpreter', 'latex')
    ylabel('$y$ axis', 'FontSize', 15, 'Interpreter', 'latex')
    hold(axesHandle, 'on');
    
    axesHandle.XLim = [-25 25];
    axesHandle.YLim = [-7 7];
    
    width = 3.1;
    length = 6.2;
    
    % Occupied parking lots [left corner (a, b), width c, height d]:
    rectangle('Position', [-10.85 -5.1 width length+0.5], 'EdgeColor', 'r')
    rectangle('Position', [- 7.75 -5.1 width length+0.5], 'EdgeColor', 'r')

    % Target parking lot for ego [left corner (a, b), width c, height d]:
    rectangle('Position', [-4.65 -5.1 width length+0.5], 'EdgeColor', 'g', 'LineWidth', 2)
    % Origin:
    plot(-3.1, -1.75, '*', 'color', 'g')

   % Post processing of figure:
    obstacles = createObstacles();
    for ct = 1:numel(obstacles)
        show(obstacles{ct})
    end
    % Remove line style for yellow line and curbside:
    ax = gca(fh1);
    yellowLine = ax.Children(1);
    yellowLine.LineStyle = 'none';
    curbside = ax.Children(2);
    curbside.LineStyle = 'none';
end

% Vehicle's plot:
if isempty(vehicleBodyHandle) || any(~isvalid(vehicleBodyHandle))
    vehicleBodyHandle = helperPlotVehicle(pose, vehicleDims, steer, ...
        'Parent', axesHandle);
else
    vehicleShapes = helperVehiclePolyshape(pose, vehicleDims, steer);
    for n = 1:numel(vehicleBodyHandle)
        vehicleBodyHandle(n).Shape = vehicleShapes(n);
    end
end

plot(axesHandle, pose(1), pose(2), '.', 'Color', 'r')

fh1.Visible = 'on';
drawnow('limitrate');

end

% =========================================================================
% LOCAL FUNCTION
% =========================================================================

function obstacles = createObstacles()

% Obstacles: 2 occupied parking lots, road curbside and yellow line
obsLength = 6.2;
egoLength = 4.7;
egoWidth = 1.8;

% Parked Car 1:
obs1 = collisionBox(egoWidth, egoLength, 0);
T1 = trvec2tform([-9.3, -1.75, 0]);
obs1.Pose = T1;

% Parked Car 2:
obs2 = collisionBox(egoWidth, egoLength, 0);
T2 = trvec2tform([-6.2, -1.75, 0]);
obs2.Pose = T2;

% Right Bottom Line:
obs3 = collisionBox(4*obsLength, 0.5, 0);
T3 = trvec2tform([11.35, 1.35, 0]);
obs3.Pose = T3;

% Parking Vertical Rightmost Line:
obs4 = collisionBox(0.5, obsLength+1, 0);
T4 = trvec2tform([-1.3, -2, 0]);
obs4.Pose = T4;

% Parking Bottom Line:
obs5 = collisionBox(10.3, 0.5, 0);
T5 = trvec2tform([-6.2, -5.35, 0]);
obs5.Pose = T5;

% Parking Vertical Leftmost Line:
obs6 = collisionBox(0.5, obsLength+1, 0);
T6 = trvec2tform([-11.1, -2, 0]);
obs6.Pose = T6;

% Left Bottom Line:
obs7 = collisionBox(2*obsLength, 0.5, 0);
T7 = trvec2tform([-17.55, 1.35, 0]);
obs7.Pose = T7;

% Other Line:
obs8 = collisionBox(47.5, 0.5, 0);
% T8 = trvec2tform([0, 5.35, 0]);
T8 = trvec2tform([0, 6, 0]);
obs8.Pose = T8;

obstacles = {obs1, obs2, obs3, obs4, obs5, obs6, obs7, obs8};

end
