%% PERPENDICULAR PARKING WITH NLMPC

clear all, close all, clc

%% Vehicle Definition

fprintf('\nVehicle Definition initiated...\n')

vdims = vehicleDimensions;
egoWheelbase = vdims.Wheelbase;
distToCenter = 0.5 * egoWheelbase;

egoInitialPose = [1, 3, 0];

egoTargetPose = [-3.1, -1.75-distToCenter, pi/2];

% The x-position of egoTargetPose is given by the position of the center of
% the parking spot plus half of the vehicle width, in this case: -1.75-0.9

% The y-position of egoTargetPose is given by the position of the center of
% the parking spot plus half of the vehicle wheelbase, in this case:
% -3.1-distToCenter

% We have the yaw angle different from 0 because this time the car is
% rotated of 90° with respect to the parallel parking case.

Tv = 0.25;
helperSLVisualizeParking(egoInitialPose, 0)

fprintf('\nPlease, enter any key to proceed!\n')
% pause
pause(1)

% fprintf('\nVehicle Definition ended successfully!\n')
fprintf('Vehicle Definition ended successfully!\n')

%% NMPC Design

fprintf('\nNLMPC Design initiated...\n')

Ts = 0.25;                  % Sample Time.
hp = 35;                    % Prediction Horizon.
hc = 35;                    % Control Horizon.

Qp = diag([50 50 1]);       % State Process Weight Matrix.
Rp = 0.01 * eye(2);         % Input Process Weight Matrix.
Qt = diag([1 5 100]);         % State Terminal Weight Matrix.
Rt = 0.1 * eye(2);         % Input Terminal Weight Matrix.
% Qp = diag([10 10 1]);       % State Process Weight Matrix.
% Rp = 0.01 * eye(2);         % Input Process Weight Matrix.
% Qt = diag([1 5 5]);         % State Terminal Weight Matrix.
% Rt = 0.04 * eye(2);         % Input Terminal Weight Matrix.

safetyDistance = 0.1;       % [m].      

maxIter = 40;               % NLMPC Solver's Number of Iterations.

mpcverbosity('off');

nx = 3;     % States: x-, y-position and psi, vehicle's yaw angle.
ny = 3;     % Outputs: x-, y- position and psi, vehicle's yaw angle.
nu = 2;     % Inputs/Manipulated Variables: vehicle speed and steering angle.
nlobj = nlmpc(nx, ny, nu);

nlobj.Ts = Ts;
nlobj.PredictionHorizon = hp;
nlobj.ControlHorizon = hc;

nlobj.States(1).Units = "m";
nlobj.States(2).Units = "m";
nlobj.States(3).Units = "rad";

nlobj.OutputVariables(1).Units = "m";
nlobj.OutputVariables(2).Units = "m";
nlobj.OutputVariables(3).Units = "rad";

nlobj.ManipulatedVariables(1).Units = "m/s";
nlobj.ManipulatedVariables(2).Units = "rad";

nlobj.MV(1).Min = -2;       % Lower Bound on Ego Speed [m/s]
nlobj.MV(1).Max = 2;        % Upper Bound on Ego Speed [m/s]
nlobj.MV(2).Min = -pi/4;    % Lower Bound on Ego Steering Angle [rad]
nlobj.MV(2).Max = pi/4;     % Upper Bound on Ego Steering Angle [rad]

nlobj.Model.StateFcn = "parkingVehicleStateFcn";
nlobj.Jacobian.StateFcn = "parkingVehicleStateJacobianFcn";

nlobj.Optimization.CustomCostFcn = "parkingCostFcn";
nlobj.Optimization.ReplaceStandardCost = true;
nlobj.Jacobian.CustomCostFcn = "parkingCostJacobian";

nlobj.Optimization.CustomIneqConFcn = "parkingIneqConFcn";
nlobj.Jacobian.CustomIneqConFcn = "parkingIneqConFcnJacobian";

nlobj.Optimization.SolverOptions.FunctionTolerance = 0.01;
nlobj.Optimization.SolverOptions.StepTolerance = 0.01;
nlobj.Optimization.SolverOptions.ConstraintTolerance = 0.01;
nlobj.Optimization.SolverOptions.OptimalityTolerance = 0.01;
nlobj.Optimization.SolverOptions.MaxIter = maxIter;

% Options for the 'nlmpcmove' function:
opt = nlmpcmoveopt;

% opt.X0 = [linspace(egoInitialPose(1), egoTargetPose(1), hp)', ...
%           linspace(egoInitialPose(2), egoTargetPose(2), hp)', ...
%           linspace(egoInitialPose(3), egoTargetPose(3), hp)'];    
opt.X0 = [];    

opt.MV0 = zeros(hp, nu);

paras = {egoTargetPose, Qp, Rp, Qt, Rt, distToCenter, safetyDistance}';
nlobj.Model.NumberOfParameters = numel(paras);
opt.Parameters = paras;

fprintf('NLMPC Design ended successfully!\n')

fprintf('\nParking Results Analysis:\n')

useMex = 0; 
runParkingAndPlot

% useMex = 1;
% runParkingAndPlot
% 
% %% Simulink
% 
% Duration = hp * Ts;
% mdl = 'mpcVDAutoParking';
% open_system(mdl)
% 
% createParameterBus(nlobj,[mdl '/Nonlinear MPC Controller'],'parasBusObject',paras);
% 
% f = findobj('Name','Automated Parallel Parking');
% % close(f)
% 
% sim(mdl)
% 
% open_system([mdl '/Ego Vehicle Model/Ego Vehicle Pose'])
% open_system([mdl '/Controls'])
% 
% % Enable message display
% mpcverbosity('on');
% 
% % % Close Simulink model
% % bdclose(mdl)
% 
% % Close animation plots
% f = findobj('Name','Automated Parallel Parking');
% % close(f)
