"""
Photovoltaic simulator for Mosaik.

Research group of Politecnico di Torino Energy Center Lab.

- author: Lorenzo Bottaccioli
- copyright: Copyright 2020. Energy Center Lab - Politecnico di Torino"
- credits: Lorenzo Bottaccioli, Daniele Salvatore Schiera
- maintainer: Lorenzo Bottaccioli
- email: lorenzo.bottaccioli@polito.it
- status: Development

"""

import sys
from pathlib import Path

import mosaik_api
import pandas as pd
import os

# current_Path = os.getcwd().replace('\\', '/')
# api_Path = current_Path+'/models/pv_sim'
# os.chdir(api_Path)
# from pv_sim import pv_sim

PROJECT_ROOT = str(Path(__file__).resolve().parents[2]).replace('\\', '/')
sys.path.insert(0,PROJECT_ROOT)
from models.pv_sim import pv_sim


# META = {
#     'models': {
#         'PVsimulator': {
#             'public': True,
#             'params': ['P_system', 'slope', 'aspect', 'latitude', 'longitude', 'elevation',
#                        'Tc_noct', 'T_ex_noct', 'a_p', 'ta', 'P_mpdule',
#                        'G_noct', 'G_stc', 'Area', 'Tc_stc', 'start_date'],
#             'attrs': ['power_dc', 'ghi', 'T_ext'],
#         },
#     },
# }

META = {
    'models': {}}


class PVSim(mosaik_api.Simulator):

    def __init__(self):
        super().__init__(META)
        self.simulator = pv_sim.PVsimulator()
        self.eid_prefix = 'PV_'
        self.entities = {}  # Maps EIDs to model indices in self.simulator

    def init(self, sid, eid_prefix=None, start_date=None, step_size=None, sim_meta=None):
        self.start_date = pd.to_datetime(start_date)
        if eid_prefix is not None:
            self.eid_prefix = eid_prefix
        self.step_size = step_size

        if self.meta['models'] == {}:
            self.meta['models'] = sim_meta['models']
        return self.meta

    def create(self, num, model, P_system=5000, slope=35, aspect=0, latitude=45.7, longitude=7.6, elevation=230,
               Tc_noct=45, T_ex_noct=20, a_p=0.0038, ta=0.9, P_module=283,
               G_noct=800, G_stc=1000, Area=1.725, Tc_stc=25.0):
        next_eid = len(self.entities)
        entities = []

        for i in range(next_eid, next_eid + num):
            eid = '%s_%d' % (model, i)
            self.simulator.add_PV(P_system=P_system, slope=slope, aspect=aspect, latitude=latitude, longitude=longitude,
                                  elevation=elevation,
                                  Tc_noct=Tc_noct, T_ex_noct=T_ex_noct, a_p=a_p, ta=ta, P_module=P_module,
                                  G_noct=G_noct, G_stc=G_stc, Area=Area, Tc_stc=Tc_stc)
            self.entities[eid] = i
            entities.append({'eid': eid, 'type': model})

        return entities

    def step(self, time, inputs):

        sim_time = self.start_date + pd.Timedelta(time, unit='s')
        # Perform simulation steps
        attrs = {}
        for dest_eid, attrs in inputs.items():
            for attr, values in attrs.items():
                if attr == 'ghi':
                    ghi = sum(values.values())
                elif attr == 'T_ext':
                    T_ext = sum(values.values())

            self.simulator.step(ghi, T_ext, sim_time)

        return time + self.step_size  # Step size is 1 minute

    def get_data(self, outputs):
        models = self.simulator.models
        data = {}
        for eid, attrs in outputs.items():
            model_idx = self.entities[eid]
            model_type = eid.split('_')[0]
            data[eid] = {}
            for attr in attrs:
                if attr not in self.meta['models'][model_type]['attrs']:
                    raise ValueError('Unknown output attribute: %s' % attr)

                # Get model.val or model.delta:
                data[eid][attr] = getattr(models[model_idx], attr)

        return data




if __name__ == '__main__':
    mosaik_api.start_simulation(PV_Sim())

