"""
Photovoltaic Simulator for for Helics.
Research group of Politecnico di Torino Energy Center Lab.
- author: Juan Gilberto Rueda Vásquez
- email: juan.rueda@polito.it
- status: Development
"""
import sys
from pathlib import Path

import mosaik_api
import pandas as pd
import os

from models.pv_sim import pv_sim

class PV_Sim():
    
    def __init__(self, eid_prefix=None, start_date=None, step_size=None, sim_meta=None):
        self.simulator = pv_sim.PVsimulator()
        self.eid_prefix = 'PV_'
        self.entities = {}  # Maps EIDs to model indices in self.simulator
        self.meta = sim_meta

        self.start_date = pd.to_datetime(start_date)
        if eid_prefix is not None:
            self.eid_prefix = eid_prefix
        self.step_size = step_size

        if self.meta['models'] == {}:
            self.meta['models'] = sim_meta['models']
        
        # print(self.meta)
    
    def create(self, num, model, P_system=5000, slope=35, aspect=0, latitude=45.7, longitude=7.6, elevation=230,
               Tc_noct=45, T_ex_noct=20, a_p=0.0038, ta=0.9, P_module=283,
               G_noct=800, G_stc=1000, Area=1.725, Tc_stc=25.0):
        
        next_eid = len(self.entities)
        entities = []

        for i in range(next_eid, next_eid + num):
            eid = '%s_%d' % (model, i)
            self.simulator.add_PV(P_system=P_system, slope=slope, aspect=aspect, latitude=latitude, longitude=longitude,
                                  elevation=elevation,
                                  Tc_noct=Tc_noct, T_ex_noct=T_ex_noct, a_p=a_p, ta=ta, P_module=P_module,
                                  G_noct=G_noct, G_stc=G_stc, Area=Area, Tc_stc=Tc_stc)
            self.entities[eid] = i
            entities.append({'eid': eid, 'type': model})
        
        self.name_eid = eid # Variable con el nombre del modelo generado
        
        return self.name_eid

    def run(self, time, data):

        sim_time = self.start_date + pd.Timedelta(time, unit='s')
        # Perform simulation steps
        var = []
        attrs = {}
        ghi = 0
        T_ext = 0
        for dest_eid, attrs in data.items():
            for attr, values in attrs.items():
                var.append(attr)
                if attr == 'ghi':
                    ghi = sum(values.values())
                elif attr == 'T_ext':
                    T_ext = sum(values.values())

            self.simulator.step(ghi, T_ext, sim_time)
        
        outputs = {self.name_eid:var} # Variable creada para get data
        models = self.simulator.models
        data = {}
        for eid, attrs in outputs.items():
            model_idx = self.entities[eid]
            model_type = eid.split('_')[0]
            data[eid] = {}
            for attr in attrs:
                if attr not in self.meta['models'][model_type]['attrs']:
                    raise ValueError('Unknown output attribute: %s' % attr)

                # Get model.val or model.delta:
                data[eid][attr] = getattr(models[model_idx], attr)

        return data, time + self.step_size  # Step size is 1 minute
    

# META = {'models': {'PVsimulator': {'public': True,'params': ['P_system', 'slope', 'aspect', 'latitude', 'longitude', 'elevation'],'attrs': ['power_dc', 'ghi', 'T_ext'],},},}

# input = {}
# data = {'ghi':{0:20},'T_ext':{0:10}} # Ejemplo de como deben entrar los datos
# # data={}
# prueba = PV_Sim(start_date='2015-01-01 00:00:00', step_size=600, sim_meta=META)
# name = prueba.create(num=1, model='PVsimulator')
# input [name] = data

# salida, time = prueba.run(0,input)
# # print(time)
# # print(salida)