"""
Meter Simulator for Mosaik.

Research group of Politecnico di Torino Energy Center Lab.

- author: Daniele Salvatore Schiera
- copyright: Copyright 2020. Energy Center Lab - Politecnico di Torino"
- credits: Daniele Salvatore Schiera
- maintainer: Daniele Salvatore Schiera
- email: daniele.schiera@polito.it
- status: Development

"""
import collections
import pprint
import pickle
import numpy as np
from pathlib import Path
import json

import mosaik_api


# TODO da rivedere!

# META = {
#     'models': {
#         'SmartMeter': {
#             'public': True,
#             'params': ['instance_name'],
#             'attrs': ['Load', 'Prod', 'Pnet', 'Pprod', 'Pload', 'Pexport', 'Pinport'],
#         },
#     },
# }


META = {
    'models': {}}


class SMModel:

    def __init__(self, name):
        self.name = name

        self.Pnet = 0.0
        self.Pload = 0.0
        self.Pprod = 0.0
        self.Pexport = 0.0
        self.Pinport = 0.0
        self.Pnetbatt = 0.0


class Meter(mosaik_api.Simulator):

    def __init__(self):
        super().__init__(META)
        self.eid_prefix = 'Meter_'
        self.eid = None
        self.entities = {}

        self.step_size = None

    def init(self, sid, step_size, collect_data = False, output_name=None, sim_meta= None):
        self.step_size = step_size
        self.collect_data = collect_data # TODO: il collettore è da rivedere ma non è fondamentale (sostituisce il
        # collector in parte
        if collect_data:
            self.data = collections.defaultdict(collections.defaultdict(list))
            if output_name == None:
                self.output_name = 'test'
            else:
                self.output_name = output_name

        if self.meta['models'] == {}:
            self.meta['models'] = sim_meta['models']
        return self.meta

    def create(self, num, model, instance_name=None):
        next_eid = len(self.entities)
        entities = []
        if instance_name == None:
            instance_name = list(range(next_eid, next_eid + num))

        for i in range(len(instance_name)):
            eid = f'{model}_{instance_name[i]}'
            self.entities[eid] = SMModel(instance_name[i])
            entities.append({'eid': eid, 'type': model})
        return entities

    def step(self, time, inputs):
        NetBatt = 0.0
        Pload = 0.0
        Pprod = 0.0
        for dest_eid, attrs in inputs.items():
            meter = self.entities[dest_eid]
            for attr, values in attrs.items():
                # System reference from the consumers (load is positive)
                if attr == 'Load':
                    Pload = sum(values.values())
                elif attr == 'Prod':
                    Pprod = sum(values.values())
                elif attr == 'NetBatt':
                    NetBatt = sum(values.values())

            meter.Pload = Pload
            meter.Pprod = Pprod
            meter.Pnetbatt = NetBatt
            meter.Pnet = Pload - Pprod - NetBatt
            # TODO si puo fare meglio con numpy invece di usare if e usare data type diverso dal float per output
            if meter.Pnet < 0.0:
                meter.Pexport = meter.Pnet
                meter.Pinport = 0.0
            else:
                meter.Pexport = 0.0
                meter.Pinport = meter.Pnet

            if self.collect_data:
                self.data[time][f'{meter.name}:Pload'].append(Pload)
                self.data[time][f'{meter.name}:Pprod'].append(Pprod)
                self.data[time][f'{meter.name}:Pnet'].append(meter.Pnet)
                self.data[time][f'{meter.name}:Pinport'].append(meter.Pinport)
                self.data[time][f'{meter.name}:Pexport'].append(meter.Pexport)


        return time + self.step_size

    def get_data(self, outputs):
        data = {}
        for eid, attrs in outputs.items():
            meter = self.entities[eid]
            model_type = eid.split('_')[0]
            data[eid] = {}
            for attr in attrs:
                if attr not in self.meta['models'][model_type]['attrs']:
                    raise ValueError('Unknown output attribute: %s' % attr)
                data[eid][attr] = getattr(meter,attr)

        return data

    def finalize(self):
        if self.collect_data:

            # print('Collected data from meters:')
            # print(self.data)
            # TODO: collection da rivedere ma non è utile
            Path("./OutputSimResults").mkdir(parents=True, exist_ok=True)
            with open(f'./OutputSimResults/output_sim_meters_{self.output_name}.json', 'w') as outfile:
                json.dump(self.data, outfile)


if __name__ == '__main__':
    mosaik_api.start_simulation(Meter())
