#include "ld_slam/loopClosureDetection.hpp"


namespace ldslam
{
/* ************************************  CONSTRUCTOR  ************************************************ */
LoopClosure::LoopClosure(const rclcpp::NodeOptions & options)
: Node("loopClosure", options)
/* *************************************************************************************************** */
{
    RCLCPP_INFO(get_logger(), "Initialization init");

    setParams();
    initializePubSub();
    allocateMemory();

    //this->timer_loop_ = this->create_wall_timer(std::chrono::seconds(3), std::bind(&LoopClosure::performLoopClosure, this));
    //this->timer_rendezvous_ = this->create_wall_timer(std::chrono::seconds(10), std::bind(&LoopClosure::publishScanContext, this));
    //this->timer_debug_ = this->create_wall_timer(std::chrono::seconds(30), std::bind(&LoopClosure::debugPerform, this));
    this->timer_inter_loop_ = this->create_wall_timer(std::chrono::seconds(30), std::bind(&LoopClosure::performInterLoopClosure, this));
    this->timer_perform_loop_ = this->create_wall_timer(std::chrono::seconds(2), std::bind(&LoopClosure::performInterLoopClosure_callback, this));

    RCLCPP_INFO(get_logger(), "Initialization end");
}

void LoopClosure::initializePubSub()
{
    std::string ld_slam = "/ld_slam";
    std::string input_cloud = "/" + this->robot_name_ + ld_slam +  "/input_cloud";
    std::string odometry = "/" + this->robot_name_ + ld_slam + "/odometry";
    std::string cloud_surface = "/" + this->robot_name_ + ld_slam + "/feature/cloud_surface";
    std::string cloud_corner = "/" + this->robot_name_ + ld_slam + "/feature/cloud_corner";
    std::string pos = "/" + this->robot_name_ + ld_slam + "/pos";
    std::string scan_context_pub = "/" + this->robot_name_ + ld_slam + "/sc_context_pub";
    std::string scan_context_sub = "/" + this->robot_name_ + ld_slam + "/sc_context_sub";

    // queste devono essere dettate opportunamente rispetto ai robot che vi sono in circolazione (in particolare rispetto al nome)
    std::string loop_id = "/" + this->robot_name_ + ld_slam + "/loop_closure/id";
    std::string loop_cloud = "/" + this->robot_name_ + ld_slam + "/loop_closure/cloud";

    input_cloud = "/ld_slam/point_cloud_raw";
    odometry = "/ld_slam/odom";
    cloud_surface = "/ld_slam/feature/cloud_surface";
    cloud_corner = "/ld_slam/feature/cloud_corner";
    pos = "/ld_slam/pos";
    scan_context_pub = "/prova_matrix";
    scan_context_sub = "/prova_matrix";
    loop_id = "/loop_closure/id";
    loop_cloud = "/loop_closure/cloud";

    cloud_raw_sub_ = this->create_subscription<sensor_msgs::msg::PointCloud2>(
        input_cloud, 100, std::bind(&LoopClosure::cloud_raw_callback, this, std::placeholders::_1));

    cloud_surface_sub_ = this->create_subscription<sensor_msgs::msg::PointCloud2>(
        cloud_surface, 100, std::bind(&LoopClosure::cloud_surface_callback, this, std::placeholders::_1));

    cloud_corner_sub_ = this->create_subscription<sensor_msgs::msg::PointCloud2>(
        cloud_corner, 100, std::bind(&LoopClosure::cloud_corner_callback, this, std::placeholders::_1));

    odom_sub_ = this->create_subscription<geometry_msgs::msg::PoseStamped>(
        pos, 10, std::bind(&LoopClosure::odometry_callback, this, std::placeholders::_1));    

    sc_context_sub_ = this->create_subscription<std_msgs::msg::Float64MultiArray>(
        scan_context_sub, 10, std::bind(&LoopClosure::sc_context_callback, this, std::placeholders::_1));  

    loop_id_sub_ = this->create_subscription<std_msgs::msg::Int32MultiArray>(
        loop_id, 1, std::bind(&LoopClosure::inter_loop_id_callback, this, std::placeholders::_1));

    loop_cloud_sub_ = this->create_subscription<sensor_msgs::msg::PointCloud2>(
        loop_cloud, 200, std::bind(&LoopClosure::store_targetCloud_callback, this, std::placeholders::_1));

    sc_context_pub_ = this->create_publisher<std_msgs::msg::Float64MultiArray>(
        scan_context_pub, 100);

    loop_id_pub_ = this->create_publisher<std_msgs::msg::Int32MultiArray>(
        loop_id, 100);

    loop_cloud_pub_ = this->create_publisher<sensor_msgs::msg::PointCloud2>(
        loop_cloud, 100);

    // debug
    debug_pointCloud_sub_ = this->create_subscription<sensor_msgs::msg::PointCloud2>(
     "/points", 100, std::bind(&LoopClosure::debug_pointCloud_callback, this, std::placeholders::_1));

}




void LoopClosure::debug_pointCloud_callback(const sensor_msgs::msg::PointCloud2::SharedPtr msg)
{
    pcl::PointCloud<PointType>::Ptr raw_cloud_tmp_ptr (new pcl::PointCloud<PointType>);

    pcl::fromROSMsg(*msg, *raw_cloud_tmp_ptr);
    this->scManager.makeAndSaveOtherScancontextAndKeys(*raw_cloud_tmp_ptr);
    this->other_raw_clouds_.push_back(*raw_cloud_tmp_ptr);
}

void LoopClosure::debugPerform() 
{
    auto result = this->scManager.detectInterLoopClosureID();

    std::cout << "Result [" << 0 << "]: " << result.first[0].idx1 << " | " << result.first[0].idx2 << std::endl;

    for (int i = 0; i < int(result.first.size()); i ++ ) {
        std::cout << "Result [" << i << "]: " << result.first[i].idx1 << " | " << result.first[i].idx2 << std::endl;
    }
}

void LoopClosure::allocateMemory() 
{
    this->SClatestSurfKeyFrameCloud.reset(new pcl::PointCloud<PointType>());
    this->SCnearHistorySurfKeyFrameCloud.reset(new pcl::PointCloud<PointType>());
    this->SCnearHistorySurfKeyFrameCloudDS.reset(new pcl::PointCloud<PointType>());
    this->SCinterTargetCloudDS.reset(new pcl::PointCloud<PointType>());
}

void LoopClosure::setParams()
{
    this->declare_parameter("loopClosureEnableFlag", true);
    this->get_parameter("loopClosureEnableFlag", loopClosureEnableFlag);
    this->declare_parameter("loopClosureFrequency", 1.0);
    this->get_parameter("loopClosureFrequency", loopClosureFrequency);

    loopClosureEnableFlag = true;
    loopClosureFrequency = 0.5;
    potentialLoopFlag = false;
    historyKeyframeSearchNum = 20;
    historyKeyframeFitnessScore = 1.5;
    interClosureEnableFlag = true;

    potentialInterLoopFlag = false;
    interLoop = true;
    activePerformLoop = false;
    
    latestPointCloud = 0;
}

void LoopClosure::cloud_raw_callback(const sensor_msgs::msg::PointCloud2::SharedPtr msg)
{
    std::lock_guard<std::mutex> lock(this->mtx_clouds_);

    //this->count3 ++ ;
    //std::cout << "Count raw: " << count3 << std::endl;

    pcl::PointCloud<PointType>::Ptr raw_cloud_tmp_ptr (new pcl::PointCloud<PointType>);

    pcl::fromROSMsg(*msg, *raw_cloud_tmp_ptr);

    this->raw_clouds_.push_back(*raw_cloud_tmp_ptr);

    // add scan context, index latestPointCLoud is a progressive number
    this->scManager.makeAndSaveScancontextAndKeys(*raw_cloud_tmp_ptr);
}

void LoopClosure::sc_context_callback(const std_msgs::msg::Float64MultiArray::SharedPtr msg) 
{
    const float height = msg->layout.dim[0].size;
    const float width = msg->layout.dim[1].size;

    std::vector<double> data = msg->data;
    Eigen::Map<Eigen::Matrix<double, Eigen::Dynamic, Eigen::Dynamic, Eigen::RowMajor> > mat(data.data(), height, width);

    std::shared_ptr<Eigen::MatrixXd> map_ptr (new Eigen::MatrixXd);
    *map_ptr = mat;
    
    this->scManager.saveOtherScancontextAndKeys(map_ptr);
    this->interClosureEnableFlag = true;
}


void LoopClosure::publishScanContext() 
{
    // publish all scan context on topic
    for (int z = 0; z < int(this->raw_clouds_.size()); z++) {
        // get last scan Context saved
        std::shared_ptr<Eigen::MatrixXd> sc_tmp_ptr (new Eigen::MatrixXd);
        this->scManager.getScanContext(sc_tmp_ptr, z);

        // getMaxDimensions API outputs height and width used in Float64MultiArray message preparation.
        auto dimensions_ = this->scManager.getMaxDimensions();
        int height = dimensions_.first;
        int width = dimensions_.second;

        /* Float64MultiArray preparation. 
            Two fields: 
            - layout: used as hyperparameters collector 
            - data: where matrix is written, data is a single vector, matrix can be stored using indexing properly */
        auto matrix_msg = std::make_shared<std_msgs::msg::Float64MultiArray>();
        matrix_msg->layout.dim.push_back(std_msgs::msg::MultiArrayDimension());
        matrix_msg->layout.dim.push_back(std_msgs::msg::MultiArrayDimension());
        matrix_msg->layout.dim[0].label = "height";
        matrix_msg->layout.dim[1].label = "width";
        matrix_msg->layout.dim[0].size = height;
        matrix_msg->layout.dim[1].size = width;
        matrix_msg->layout.dim[0].stride = width*height;
        matrix_msg->layout.dim[1].stride = width;
        matrix_msg->layout.data_offset = 0;

        for (int i = 0; i < width*height; i++) 
            matrix_msg->data.push_back(0);
        
        // int dstride0 = matrix_msg->layout.dim[0].stride;
        int dstride1 = matrix_msg->layout.dim[1].stride;
        int offset = matrix_msg->layout.data_offset;

        // storing matrix in data vector using indexing properly
        for (int i = 0; i < height; i++) {
            for (int j = 0; j < width; j++) {
                matrix_msg->data[offset + i*dstride1 + j] = (*sc_tmp_ptr)(i,j);
            }
        }

        // publish matrix
        this->sc_context_pub_->publish(*matrix_msg);
    }
} // LoopClosure::publishScanContext() 


void LoopClosure::cloud_surface_callback(const sensor_msgs::msg::PointCloud2::SharedPtr msg)
{
    std::lock_guard<std::mutex> lock(this->mtx_clouds_);

    //this->count ++ ;
    //std::cout << "Count surface: " << count << std::endl;

    pcl::PointCloud<PointType>::Ptr surface_cloud_tmp_ptr (new pcl::PointCloud<PointType>);

    pcl::fromROSMsg(*msg, *surface_cloud_tmp_ptr);

    this->surface_clouds_.push_back(*surface_cloud_tmp_ptr);
}

void LoopClosure::cloud_corner_callback(const sensor_msgs::msg::PointCloud2::SharedPtr msg)
{
    std::lock_guard<std::mutex> lock(this->mtx_clouds_);

    //this->count1 ++ ;
    //std::cout << "Count corner: " << count1 << std::endl;

    pcl::PointCloud<PointType>::Ptr corner_cloud_tmp_ptr (new pcl::PointCloud<PointType>);

    pcl::fromROSMsg(*msg, *corner_cloud_tmp_ptr);

    this->corner_clouds_.push_back(*corner_cloud_tmp_ptr);
}

void LoopClosure::odometry_callback(const geometry_msgs::msg::PoseStamped::SharedPtr msg)
{
    //this->count2 ++ ;
    //std::cout << "Count odometry: " << count2 << std::endl;
    
    PointTypePose pose;
    Eigen::Quaterniond quat_eig;

    std::lock_guard<std::mutex> lock(this->mtx_clouds_);

    pose.x = msg->pose.position.x;
    pose.y = msg->pose.position.y;
    pose.z = msg->pose.position.z;
    geometry_msgs::msg::Quaternion quat = msg->pose.orientation;
    tf2::fromMsg(quat, quat_eig);

    // quat_eig = quaterion
    // need to convert quaternion into eulerAngles
    Eigen::Vector3d euler = quat_eig.toRotationMatrix().eulerAngles(0, 1, 2);
    pose.roll = euler[0];
    pose.pitch = euler[1];
    pose.yaw = euler[2];
    // insert odometry on vector having odometry results
    this->poses_.push_back(pose);
}

void LoopClosure::performInterLoopClosure() 
{
    std::lock_guard<std::mutex> lock(this->mtx_rendezvous_);

    
    std::cout << "[DEBUG]: Perform InterLoopClosure - init" << std::endl;

    if (this->interClosureEnableFlag == false) {
        return;
    }

    this->interClosureEnableFlag = false;
    // try to find close key frame if there are any
    if (potentialInterLoopFlag == false) {
        loop = 0;
        // resInterLoop == std::vector<Indexes> Indexes: struct{idxALPHA, idxBETA}
        auto result = this->scManager.detectInterLoopClosureID();
        std::vector<Indexes> _resInterLoop = result.first;
        /*
        for (int k = 0; k < int(_resInterLoop.size()); k++) {
            std::cout << "[DEBUG]: No filtered_resInterLoop (" << k <<"): " << _resInterLoop[k].idx1 << " | " << _resInterLoop[k].idx2 << std::endl;

        } */

        std::copy_if (_resInterLoop.begin(), _resInterLoop.end(), std::back_inserter(resInterLoop), [](Indexes id){return id.idx1 != -1;} );

        /*
        for (int k = 0; k < int(resInterLoop.size()); k++) {
            std::cout << "[DEBUG]: resInterLoop (" << k <<"): " << resInterLoop[k].idx1 << " | " << resInterLoop[k].idx2 << std::endl;

        } */

        if (resInterLoop.size() != 0) {
            std::cout << "[DEBUG]: Perform InterLoopClosure - Not Empty" << std::endl;

            potentialInterLoopFlag = true; // find some key frames that is old enough or close enough for loop closure
            timeSaveFirstCurrentScanForInterLoopClosure = timeLaserInterOdometry; 
        }
        if (potentialInterLoopFlag == false) {
            return;
        }
    }
    
    std::cout << "[DEBUG]: Perform InterLoopClosure - after detectInterLoop" << std::endl;
    potentialInterLoopFlag = false;

    std_msgs::msg::Int32MultiArray msg_id;
    int size = int(resInterLoop.size());

    msg_id.layout.dim.push_back(std_msgs::msg::MultiArrayDimension());
    msg_id.layout.dim[0].label = "dimension";
    msg_id.layout.dim[0].size = size;
    msg_id.layout.dim[0].stride = size;
    msg_id.layout.data_offset = 0;

    for (int i = 0; i < size; i++) 
        msg_id.data.push_back(0);

    for (int i = 0; i < size; i++) {
        Indexes idxs = resInterLoop[i];
        int alpha = idxs.idx1;
        int beta = idxs.idx2;

        // point cloud alpha is stored in robot (source)
        // point cloud beta returned as response from client request.
    
        // Source point cloud preparation
        pcl::PointCloud<PointType>::Ptr point_cloud_tmp ( new pcl::PointCloud<PointType>() );
        pcl::PointCloud<PointType>::Ptr point_cloud_source ( new pcl::PointCloud<PointType>() );

        *point_cloud_tmp += *transformPointCloud(boost::make_shared<pcl::PointCloud<PointType>>(this->raw_clouds_[alpha]), &this->poses_[alpha]);         
        SCinterSourceCloud.push_back(*point_cloud_tmp);
        std::cout << "[DEBUG]: Perform InterLoopClosure - alpha setted" << std::endl;

        // Target point cloud preparation

        msg_id.data.push_back(beta);
    }

    // qui pubblico un messaggio, nella parte della ricezione passo al resto.  
    loop_id_pub_->publish(msg_id);

} // LoopClosure::performInterLoopClosure

void LoopClosure::store_targetCloud_callback(const sensor_msgs::msg::PointCloud2::SharedPtr msg)
{
    std::lock_guard<std::mutex> lock(this->mtx_rendezvous_);
    pcl::PointCloud<PointType>::Ptr point_cloud_target_tmp ( new pcl::PointCloud<PointType>() );
    // target setting
    pcl::fromROSMsg(*msg, *point_cloud_target_tmp);
    this->SCinterTargetCloud.push_back(*point_cloud_target_tmp);

    if ( this->SCinterTargetCloud.size() == this->resInterLoop.size() ) {
        activePerformLoop = true;
    }
} // LoopClosure::store_targetCloud_callback

void LoopClosure::performInterLoopClosure_callback()
{
    bool isValidSCloopFactor;

    if (activePerformLoop == false) 
        return;

    std::lock_guard<std::mutex> lock(this->mtx_rendezvous_);

    for (int i = 0; i < int(this->resInterLoop.size()); i++) {
        Indexes idxs = resInterLoop[i];
        int alpha = idxs.idx1;
        int beta = idxs.idx2;

        // alpha re-setting
        pcl::PointCloud<PointType>::Ptr point_cloud_source_tmp ( new pcl::PointCloud<PointType>() );
        *point_cloud_source_tmp = this->SCinterSourceCloud[i];

        pcl::PointCloud<PointType>::Ptr _point_cloud_target_tmp ( new pcl::PointCloud<PointType>() );
        pcl::PointCloud<PointType>::Ptr point_cloud_target_tmp ( new pcl::PointCloud<PointType>() );
        // target setting
        *_point_cloud_target_tmp = this->SCinterTargetCloud[i];
        *point_cloud_target_tmp += *transformPointCloud(_point_cloud_target_tmp, &this->poses_[alpha]);         
        //*point_cloud_target = *point_cloud_tmp;
        std::cout << "[DEBUG]: Perform InterLoopClosure - beta setted" << std::endl;

        pcl::VoxelGrid<pcl::PointXYZI> voxel_grid;
        double vg_size_for_input_ = 1.5;
        voxel_grid.setLeafSize(vg_size_for_input_, vg_size_for_input_, vg_size_for_input_);
        voxel_grid.setInputCloud(point_cloud_target_tmp);
        voxel_grid.filter(*SCinterTargetCloudDS);

        double ndt_resolution = 2.0;
        pcl::Registration<pcl::PointXYZI, pcl::PointXYZI>::Ptr registration_;
        pcl::PointCloud<PointType>::Ptr unused_result(new pcl::PointCloud<PointType>());
        
        pclomp::NormalDistributionsTransform<pcl::PointXYZI, pcl::PointXYZI>::Ptr ndt(new pclomp::NormalDistributionsTransform<pcl::PointXYZI, pcl::PointXYZI>());
        ndt->setResolution(ndt_resolution);
        ndt->setTransformationEpsilon(0.0001);
        // ndt_omp
        ndt->setNeighborhoodSearchMethod(pclomp::DIRECT7);
        //if (ndt_num_threads > 0) {ndt->setNumThreads(ndt_num_threads);}

        registration_ = ndt;
        
        registration_->setInputSource(point_cloud_source_tmp);
        registration_->setInputTarget(SCinterTargetCloudDS);
        registration_->align(*unused_result);
        
        std::cout << "[SC] InterLoop: ICP fit score: " << registration_->getFitnessScore() << std::endl;
        if ( registration_->hasConverged() == false) {
            std::cout << "[SC] InterLoop: Reject this loop (not converged) " << std::endl;
            isValidSCloopFactor = false;
        } 
        else {
            if( registration_->getFitnessScore() > 1.5 ) {
                std::cout << "[SC] InterLoop: Reject this loop (bad icp fit score, > " << 1.5 
                                                                                << ")" << std::endl;
                isValidSCloopFactor = false;
            }
            else {
            std::cout << "[SC] InterLoop: The detected loop factor is added between Current [ " << alpha << 
                                            " ] and SC nearest [ " << beta << " ]" << std::endl;
            isValidSCloopFactor = true;
            }
        } 
        
        if( isValidSCloopFactor == true ) {
            // VALID LOOP CLOSURE 
            RCLCPP_INFO(this->get_logger(), "InterLoop: To pass on optimization.");
        }

    }
 
    this->interClosureEnableFlag = true;
    this->activePerformLoop = false;
} // LoopClosure::performInterLoopClosure_callback

// sostituire qui al posto di other_raw_clouds raw_clouds
void LoopClosure::inter_loop_id_callback(const std_msgs::msg::Int32MultiArray::SharedPtr msg)
{
    int size = msg->layout.dim[0].size;
    for (int i = 0; i < size; i++) {
        int id = msg->data[i];

        if (int(this->other_raw_clouds_.size()) >= id) {
            sensor_msgs::msg::PointCloud2 tempCloud;
            pcl::toROSMsg(this->other_raw_clouds_[id], tempCloud);
            this->loop_cloud_pub_->publish(tempCloud);
        }
    }
} // LoopClosure::inter_loop_id_callback


void LoopClosure::performLoopClosure() 
{
    std::lock_guard<std::mutex> lock(this->mtx_clouds_);

    if (this->raw_clouds_.size() < 20 || this->poses_.size() < 20) {
        /*
        RCLCPP_INFO(this->get_logger(), "[DEBUG]: Empty.");
        RCLCPP_INFO(this->get_logger(), "[DEBUG]: Raw: %d | Empty: %d", this->raw_clouds_.size(), this->raw_clouds_.empty());
        RCLCPP_INFO(this->get_logger(), "[DEBUG]: Corner: %d | Surface: %d | Poses: %d", this->corner_clouds_.size(), this->surface_clouds_.size(), this->poses_.size());
        */
        return;
    }
    /*
    RCLCPP_INFO(this->get_logger(), "[DEBUG]: Full.");
    RCLCPP_INFO(this->get_logger(), "[DEBUG]: Raw: %d | Empty: %d", this->raw_clouds_.size(), this->raw_clouds_.empty());
    RCLCPP_INFO(this->get_logger(), "[DEBUG]: Corner: %d | Surface: %d | Poses: %d", this->corner_clouds_.size(), this->surface_clouds_.size(), this->poses_.size());
    */
    // try to find close key frame if there are any
    if (potentialLoopFlag == false) {
        if (detectLoopClosure() == true) {
            potentialLoopFlag = true; // find some key frames that is old enough or close enough for loop closure
            timeSaveFirstCurrentScanForLoopClosure = timeLaserOdometry; 
        }
        if (potentialLoopFlag == false) {
            return;
        }
    }

    potentialLoopFlag = false;
    bool isValidSCloopFactor = false;

    if( SCclosestHistoryFrameID != -1 ) {
        /*
        pcl::IterativeClosestPoint<PointType, PointType> icp;
        icp.setMaxCorrespondenceDistance(100);
        icp.setMaximumIterations(100);
        icp.setTransformationEpsilon(1e-9);
        icp.setEuclideanFitnessEpsilon(1e-9);
        icp.setRANSACIterations(0);

        icp.setInputSource(SClatestSurfKeyFrameCloud);
        icp.setInputTarget(SCnearHistorySurfKeyFrameCloudDS);
        pcl::PointCloud<PointType>::Ptr unused_result(new pcl::PointCloud<PointType>());
        icp.align(*unused_result); 
        // icp.align(*unused_result, icpInitialMat); // PCL icp non-eye initial is bad ... don't use (LeGO LOAM author also said pcl transform is weird.)
        */

        double ndt_resolution = 2.0;
        pcl::Registration<pcl::PointXYZI, pcl::PointXYZI>::Ptr registration_;
        pcl::PointCloud<PointType>::Ptr unused_result(new pcl::PointCloud<PointType>());
        
        pclomp::NormalDistributionsTransform<pcl::PointXYZI, pcl::PointXYZI>::Ptr ndt(new pclomp::NormalDistributionsTransform<pcl::PointXYZI, pcl::PointXYZI>());
        ndt->setResolution(ndt_resolution);
        ndt->setTransformationEpsilon(0.0001);
        // ndt_omp
        ndt->setNeighborhoodSearchMethod(pclomp::DIRECT7);
        //if (ndt_num_threads > 0) {ndt->setNumThreads(ndt_num_threads);}

        registration_ = ndt;
        
        /*
        pclomp::GeneralizedIterativeClosestPoint<pcl::PointXYZI, pcl::PointXYZI>::Ptr gicp(new pclomp::GeneralizedIterativeClosestPoint<pcl::PointXYZI, pcl::PointXYZI>());
        gicp->setMaxCorrespondenceDistance(0.005);
        gicp->setTransformationEpsilon(1e-8);
        gicp->setMaximumIterations(100)

        registration_ = gicp; */

        //registration_ = icp;
        
        registration_->setInputSource(SClatestSurfKeyFrameCloud);
        registration_->setInputTarget(SCnearHistorySurfKeyFrameCloudDS);
        registration_->align(*unused_result);
        
        std::cout << "[SC] ICP fit score: " << registration_->getFitnessScore() << std::endl;
        if ( registration_->hasConverged() == false) {
            std::cout << "[SC] Reject this loop (not converged) " << std::endl;
            isValidSCloopFactor = false;
        } 
        else {
            if( registration_->getFitnessScore() > historyKeyframeFitnessScore ) {
                std::cout << "[SC] Reject this loop (bad icp fit score, > " << historyKeyframeFitnessScore 
                                                                                << ")" << std::endl;
                isValidSCloopFactor = false;
            }
            else {
            std::cout << "[SC] The detected loop factor is added between Current [ " << latestFrameIDLoopCloure << 
                                            " ] and SC nearest [ " << SCclosestHistoryFrameID << " ]" << std::endl;
            isValidSCloopFactor = true;
            }
        } 
        /*
        std::cout << "[SC] ICP fit score: " << icp.getFitnessScore() << std::endl;
        if ( icp.hasConverged() == false) {
            std::cout << "[SC] Reject this loop (not converged) " << std::endl;
            isValidSCloopFactor = false;
        } 
        else {
            if( icp.getFitnessScore() > historyKeyframeFitnessScore ) {
                std::cout << "[SC] Reject this loop (bad icp fit score, > " << historyKeyframeFitnessScore 
                                                                                << ")" << std::endl;
                isValidSCloopFactor = false;
            }
            else {
            std::cout << "[SC] The detected loop factor is added between Current [ " << latestFrameIDLoopCloure << 
                                            " ] and SC nearest [ " << SCclosestHistoryFrameID << " ]" << std::endl;
            isValidSCloopFactor = true;
            }
        }
        */
        
        if( isValidSCloopFactor == true ) {
            // VALID LOOP CLOSURE 
            RCLCPP_INFO(this->get_logger(), "To pass on optimization.");
           /* std::ofstream MyFile("loop_closure_results.txt");

            // Write to the file
            MyFile << "Loop detected. Raw cloud size: " << this->raw_clouds_.size();

            // Close the file
            MyFile.close(); */

        }
    }
}


// VERSIONE CON TRASFORMAZIONE RELATIVA ALLA POSA con RAW CLOUDS
bool LoopClosure::detectLoopClosure() 
{
    SClatestSurfKeyFrameCloud->clear();
    SCnearHistorySurfKeyFrameCloud->clear();
    SCnearHistorySurfKeyFrameCloudDS->clear();
    // std::lock_guard<std::mutex> lock(mtx);        
    latestFrameIDLoopCloure = this->raw_clouds_.size() - 1;
    SCclosestHistoryFrameID = -1; // init with -1

    auto detectResult = scManager.detectLoopClosureID(); // first: nn index, second: yaw diff 
    SCclosestHistoryFrameID = detectResult.first;
    yawDiffRad = detectResult.second; // not use for v1 (because pcl icp with initial somthing wrong...)
    // if all close, reject
    if (SCclosestHistoryFrameID == -1) { 
        RCLCPP_INFO(this->get_logger(), "SC %d | odometry_size: %d | clouds_size: %d", SCclosestHistoryFrameID, this->poses_.size(), this->raw_clouds_.size());
        return false;
    }

    RCLCPP_INFO(this->get_logger(), "SC %d | odometry_size: %d | clouds_size: %d", SCclosestHistoryFrameID, this->poses_.size(), this->raw_clouds_.size());

    pcl::PointCloud<PointType>::Ptr raw_latest(new pcl::PointCloud<PointType>());

    *raw_latest = this->raw_clouds_[latestFrameIDLoopCloure];

    *SClatestSurfKeyFrameCloud+=*transformPointCloud(raw_latest, &this->poses_[SCclosestHistoryFrameID]);         
    //*SClatestSurfKeyFrameCloud = this->raw_clouds_[latestFrameIDLoopCloure];
    /*
    pcl::PointCloud<PointType>::Ptr SChahaCloud(new pcl::PointCloud<PointType>());
    int cloudSize = SClatestSurfKeyFrameCloud->points.size();
    for (int i = 0; i < cloudSize; ++i){
        if ((int)SClatestSurfKeyFrameCloud->points[i].intensity >= 0){
            SChahaCloud->push_back(SClatestSurfKeyFrameCloud->points[i]);
        }
    }
    
    SClatestSurfKeyFrameCloud->clear();
    *SClatestSurfKeyFrameCloud = *SChahaCloud;
    */

    pcl::PointCloud<PointType>::Ptr raw_history(new pcl::PointCloud<PointType>());
    *raw_history = this->raw_clouds_[SCclosestHistoryFrameID];
    *SCnearHistorySurfKeyFrameCloud+=*transformPointCloud(raw_history, &this->poses_[SCclosestHistoryFrameID]);         
   // *SCnearHistorySurfKeyFrameCloud = this->raw_clouds_[SCclosestHistoryFrameID];

    pcl::VoxelGrid<pcl::PointXYZI> voxel_grid;
    double vg_size_for_input_ = 1.5;
    voxel_grid.setLeafSize(vg_size_for_input_, vg_size_for_input_, vg_size_for_input_);
    voxel_grid.setInputCloud(SCnearHistorySurfKeyFrameCloud);
    voxel_grid.filter(*SCnearHistorySurfKeyFrameCloudDS);
    
    //downSizeFilterHistoryKeyFrames.setInputCloud(SCnearHistorySurfKeyFrameCloud);
   // downSizeFilterHistoryKeyFrames.filter(*SCnearHistorySurfKeyFrameCloudDS);

    if (SClatestSurfKeyFrameCloud->empty() ||  SCnearHistorySurfKeyFrameCloudDS->empty()) {
        std::cout<<"SCnearHistorySurfKeyFrameCloud: " << SCnearHistorySurfKeyFrameCloud->empty() << " | SClatestSurfKeyFrameCloud: " << SClatestSurfKeyFrameCloud->empty() << std::endl;
    }

    return true;
}


/*
// VERSIONE SENZA UTILIZZO DELLA POSA CALCOLATA TRAMITE ODOMETRIA
bool LoopClosure::detectLoopClosure() 
{
    SClatestSurfKeyFrameCloud->clear();
    SCnearHistorySurfKeyFrameCloud->clear();
    SCnearHistorySurfKeyFrameCloudDS->clear();
    // std::lock_guard<std::mutex> lock(mtx);        
    latestFrameIDLoopCloure = this->corner_clouds_.size() - 1;
    SCclosestHistoryFrameID = -1; // init with -1

    auto detectResult = scManager.detectLoopClosureID(); // first: nn index, second: yaw diff 
    SCclosestHistoryFrameID = detectResult.first;
    yawDiffRad = detectResult.second; // not use for v1 (because pcl icp with initial somthing wrong...)
    // if all close, reject
    if (SCclosestHistoryFrameID == -1){ 
        RCLCPP_INFO(this->get_logger(), "SC %d | odometry_size: %d | clouds_size: %d", SCclosestHistoryFrameID, this->poses_.size(), this->surface_clouds_.size());
        return false;
    }

    RCLCPP_INFO(this->get_logger(), "SC %d | odometry_size: %d", SCclosestHistoryFrameID, this->poses_.size());

    pcl::PointCloud<PointType>::Ptr corner_latest(new pcl::PointCloud<PointType>());
    pcl::PointCloud<PointType>::Ptr surface_latest(new pcl::PointCloud<PointType>());
    *corner_latest = this->corner_clouds_[latestFrameIDLoopCloure];
    *surface_latest = this->surface_clouds_[latestFrameIDLoopCloure];

    *SClatestSurfKeyFrameCloud+=*corner_latest;         
    *SClatestSurfKeyFrameCloud+=*surface_latest; 

    pcl::PointCloud<PointType>::Ptr SChahaCloud(new pcl::PointCloud<PointType>());
    int cloudSize = SClatestSurfKeyFrameCloud->points.size();
    for (int i = 0; i < cloudSize; ++i){
        if ((int)SClatestSurfKeyFrameCloud->points[i].intensity >= 0){
            SChahaCloud->push_back(SClatestSurfKeyFrameCloud->points[i]);
        }
    }
    SClatestSurfKeyFrameCloud->clear();
    *SClatestSurfKeyFrameCloud = *SChahaCloud;

    // save history near key frames: map ptcloud (icp to query ptcloud)
    for (int j = -historyKeyframeSearchNum; j <= historyKeyframeSearchNum; ++j){
        if (SCclosestHistoryFrameID + j < 0 || SCclosestHistoryFrameID + j > latestFrameIDLoopCloure)
            continue;
        pcl::PointCloud<PointType>::Ptr corner_history(new pcl::PointCloud<PointType>());
        pcl::PointCloud<PointType>::Ptr surface_history(new pcl::PointCloud<PointType>());
        *corner_history = this->corner_clouds_[SCclosestHistoryFrameID+j];
        *surface_history = this->surface_clouds_[SCclosestHistoryFrameID+j]; 
        *SCnearHistorySurfKeyFrameCloud += *corner_history;
        *SCnearHistorySurfKeyFrameCloud += *surface_history;
    }
    downSizeFilterHistoryKeyFrames.setInputCloud(SCnearHistorySurfKeyFrameCloud);
    downSizeFilterHistoryKeyFrames.filter(*SCnearHistorySurfKeyFrameCloudDS);

    return true;
}
*/


pcl::PointCloud<PointType>::Ptr LoopClosure::transformPointCloud(pcl::PointCloud<PointType>::Ptr cloudIn, PointTypePose* transformIn)
{
    pcl::PointCloud<PointType>::Ptr cloudOut(new pcl::PointCloud<PointType>());

    PointType *pointFrom;
    PointType pointTo;

    int cloudSize = cloudIn->points.size();
    cloudOut->resize(cloudSize);
    
    for (int i = 0; i < cloudSize; ++i){

        pointFrom = &cloudIn->points[i];
        float x1 = cos(transformIn->yaw) * pointFrom->x - sin(transformIn->yaw) * pointFrom->y;
        float y1 = sin(transformIn->yaw) * pointFrom->x + cos(transformIn->yaw)* pointFrom->y;
        float z1 = pointFrom->z;

        float x2 = x1;
        float y2 = cos(transformIn->roll) * y1 - sin(transformIn->roll) * z1;
        float z2 = sin(transformIn->roll) * y1 + cos(transformIn->roll)* z1;

        pointTo.x = cos(transformIn->pitch) * x2 + sin(transformIn->pitch) * z2 + transformIn->x;
        pointTo.y = y2 + transformIn->y;
        pointTo.z = -sin(transformIn->pitch) * x2 + cos(transformIn->pitch) * z2 + transformIn->z;
        pointTo.intensity = pointFrom->intensity;

        cloudOut->points[i] = pointTo;
    }
    return cloudOut;
}

}


int main(int argc, char** argv)
{
    rclcpp::init(argc, argv);

    rclcpp::NodeOptions options;
    options.use_intra_process_comms(true);
    rclcpp::executors::SingleThreadedExecutor exec;

    auto IP = std::make_shared<ldslam::LoopClosure>(options);
    exec.add_node(IP);

    RCLCPP_INFO(rclcpp::get_logger("rclcpp"), "\033[1;32m----> Loop Clousure detection started.\033[0m");

    exec.spin();

    rclcpp::shutdown();
    return 0;
}
