function [Latm,atm_att] = atmloss(f,teta,pos,D,eff,atm_data)
% Atmospheric Attenuation Loss calculation (based on "Satellite 
% Communications Systems Engineering" by Louis J. Ippolito Jr.)

% Ionospheric effects are neglected for frequencies below 1 GHz

f = f*1E-3;          % f: Frequency [MHz] -> [GHz]

% teta: Elevation [deg]

LAT = pos(1);       % Latitude [deg]
LONG = pos(2);      % Longitude [deg]
hS = pos(3);        % Altitude of the ground receiver site from sea level [km]

% D: Antenna diameter [m]
% eff: Antenna efficiency [adim]

% Polarization tilt angle [deg]
tau = atm_data(3,1);

% Link unavailability [%]
P = 100 - atm_data(2,1);

% Atmospheric model uncertainty [%]
uncer = atm_data(1,:);


%% ITU-R PARAMETERS FROM GLOBAL MAPS

% Surface Temperature [K]

if LONG > 180
    LONG = LONG-360;
end

LAT_frac = (90+LAT)/0.75+1;
LONG_frac = (180+LONG)/0.75+1;

LAT_int = fix(LAT_frac);
LONG_int = fix(LONG_frac);

T_map = importweather('T_Annual.txt');
T = extract(T_map,LAT_int,LONG_int,LAT_frac,LONG_frac);


% Wet term of the radio refractivity [hPa/K^2]

Nwet_map = importweather('NWET_Annual_50.txt');
Nwet = extract(Nwet_map,LAT_int,LONG_int,LAT_frac,LONG_frac);


% Rainfall rate exceeded for 0.01% of an average year

LAT_frac = (90+LAT)/0.125+1;
LONG_frac = (180+LONG)/0.125+1;

LAT_int = fix(LAT_frac);
LONG_int = fix(LONG_frac);

R001_map = importweather('R001.txt');
R001 = extract(R001_map,LAT_int,LONG_int,LAT_frac,LONG_frac);


% Surface Water Vapour Density [g/m^3]

if LONG < 0
    LONG = LONG+360;
end

LAT_frac = (90-LAT)/1.125+1;
LONG_frac = LONG/1.125+1;

LAT_int = fix(LAT_frac);
LONG_int = fix(LONG_frac);

ro_map = importweather('RHO_1_v4.txt');
% Maps related to a a Probability of exceedance = 1%

ro = extract(ro_map,LAT_int,LONG_int,LAT_frac,LONG_frac);


% Columnar Liquid Water Content [kg/m^2]

L_map = importweather('Lred_1_v4.txt');
% Maps related to a a Probability of exceedance = 1%

L = extract(L_map,LAT_int,LONG_int,LAT_frac,LONG_frac);


% Average annual 0°C isotherm height [km]

LAT_frac = (90-LAT)/1.5 + 1;
LONG_frac = LONG/1.5 + 1;

LAT_int = fix(LAT_frac);
LONG_int = fix(LONG_frac);

h_is_map = importweather('h0.txt');
h_is = extract(h_is_map,LAT_int,LONG_int,LAT_frac,LONG_frac);



%% GAS ATTENUATION (ITU-R model, approximation method)

p = 1013;       % Surface Pressure [hPa]

% Additional parameters [adim.]
rp = p/1013;
rt = 288/T;


% Specific attenuation for dry air [dB/km], for frequency f <= 54 GHz

xi1 = fi(rp,rt,0.0717,-1.8132,0.0156,-1.6515);
xi2 = fi(rp,rt,0.5146,-4.6368,-0.1921,-5.7416);
xi3 = fi(rp,rt,0.3414,-6.5851,0.2130,-8.5854);

gamma0 = (7.2.*rt.^2.8./(f.^2+0.34.*rp.^2.*rt.^1.6) + 0.62.*xi3./ ...
    ((54-f).^(1.16.*xi1)+0.83.*xi2)) .* f.^2.*rp.^2*1E-3;


% Specific attenuation for water vapour [g/m^3]

eta1 = 0.995*rp.*rt.^0.68 + 0.006*ro;
eta2 = 0.735*rp.*rt.^0.5 + 0.0353*rt.^4.*ro;

gammaw = (3.98*eta1.*exp(2.23*(1-rt))./((f-22.235).^2+9.42*eta1.^2).*g(f,22) ...
    + 11.96*eta1.*exp(0.7*(1-rt))./((f-183.31).^2+11.41*eta1.^2) + ...
    0.081*eta1.*exp(6.44*(1-rt))./((f-321.226).^2+6.29*eta1.^2) + ...
    3.66*eta1.*exp(1.6*(1-rt))./((f-325.153).^2+9.22*eta1.^2) + ...
    25.37*eta1.*exp(1.09*(1-rt))./((f-380).^2) + ...
    17.4*eta1.*exp(1.46*(1-rt))./((f-488).^2) + ...
    844.6*eta1.*exp(0.17*(1-rt))./((f-557).^2).*g(f,557) + ...
    290*eta1.*exp(0.41*(1-rt))./((f-752).^2).*g(f,752) + ...
    8.3328*1E4*eta2.*exp(0.99*(1-rt))./((f-1780).^2).*g(f,1780)) ...
    .* f.^2.*rt.^2.5.*ro*1E-4;


% Equivalent height for dry air

t1 = 4.64./(1+0.066*rp.^(-2.3)) .* exp(-((f-59.7)./(2.87+12.4*exp(-7.9*rp))).^2);
t2 = 0.14*exp(2.12*rp) ./ ((f-118.75).^2 + 0.031*exp(2.2*rp));
t3 = f .* 0.0114./(1+0.14*rp.^(-2.6)) .* (-0.0247 + 0.0001*f + 1.61*1E-6*f.^2) ...
    ./ (1 - 0.0169*f + 4.1*1E-5*f.^2 + 3.2*1E-7*f.^3);

h0 = 6.1./(1+0.17*rp.^(-1.1)) .* (1+t1+t2+t3);


% Equivalent height for water vapour

sigmaw = 1.013./(1+exp(-8.6.*(rp-0.57)));

hw = 1.66*(1+ 1.39*sigmaw./((f-22.235).^2+2.56*sigmaw) + 3.37*sigmaw./ ...
    ((f-183.31).^2+4.69*sigmaw) + 1.58*sigmaw./((f-325.1).^2+2.89*sigmaw));


% Total zenith attenuation (elevation angle = 90°) [dB]
Az = gamma0.*h0 + gammaw.*hw;

% Total path attenuation (for 5° <= teta <= 90°) [dB]
Ag = Az/sind(teta);



%% CLOUD ATTENUATION (ITU-R Model)
% Significant above 20 GHz

% Inverse temperature constant [adim.] -> Fi = 300/T
% For cloud attenuation we assume T_cloud = 273.15 K, therefore...
Fi = 1.098;

% Principal and secondary Relaxation frequencies
fp = 20.09 - 142*(Fi-1) + 294*(Fi-1).^2;
fs = 590 - 1500*(Fi-1);


% Real and imaginary components of the Complex Dielectric Permittivity of water

eps0 = 77.6 + 103.3*(Fi-1);
eps1 = 5.48;
eps2 = 3.51;

epsII = f.*(eps0-eps1) ./ (fp.*(1+(f./fp).^2)) + ...
    f.*(eps1-eps2) ./ (fs.*(1+(f./fs).^2));
epsI = (eps0-eps1) ./ (1+(f./fp).^2) + (eps1-eps2) ./ (1+(f./fs).^2);


% Specific attenuation coefficient
eta = (2+epsI)./epsII;
Kl = 0.819*f./(epsII.*(1+eta.^2));       % (dB/km)/(g/m^3)

% Total Cloud Attenuation 
Ac = L*Kl/sind(teta);



%% RAIN ATTENUATION (ITU-R Model)
% Dominant above 10 GHz

% Mean annual rain height above mean sea level [km]
hR = h_is + 0.36;

% Slant path length (for teta >= 5°) [km]
LS = (hR-hS)./sind(teta);

% Horizontal projection [km]
LG = LS.*cosd(teta);

% Specific attenuation
gammaR = sp_att_rain(f,teta,tau,R001);

% Horizontal reduction factor
r001 = 1./ (1 + 0.78*sqrt(LG.*gammaR./f) - 0.38*(1-exp(-2*LG)));


% Vertical adjustment factor

zeta = atand((hR-hS)./(LG.*r001));

if zeta > teta
    LR = LG.*r001./cosd(teta);       % [km]
elseif zeta <= teta
    LR = (hR-hS)./sind(teta);        % [km]
end

if abs(LAT) < 36
    chi = 36 - abs(LAT);        % [deg]
elseif abs(LAT) >= 36
    chi = 0;
end

nu001 = 1./ (1 + sqrt(sind(teta)).*(31*(1-exp(-(teta./(1+chi)))) .* sqrt(LR.*gammaR)./(f.^2) -0.45));


% Effective path length [km]
LE = LR .* nu001;

% Attenuation exceeded for 0.01% of an average year [dB]
A001 = gammaR .* LE;

% Attenuation exceeded for other percentages (link unavailability)

if P>=1 || abs(LAT)>=36
    beta = 0;
elseif P<1 && abs(LAT)<36 && teta>=25
    beta = -0.005*(abs(LAT)-36);
else
    beta = -0.005*(abs(LAT)-36) + 1.8 - 4.25*sind(teta);
end

Ar = A001 .* (P/0.01) .^ (-(0.655 + 0.033.*log(P) - 0.045.*log(A001) ...
    - beta.*(1-P).*sind(teta)));



%% TROPOSPHERIC SCINTILLATION ATTENUATION (ITU-R Model)
% Important above 10 GHz
% Model valid for frequencies up to at least 20 GHz

% Standard deviation of the signal amplitude [dB]
sigma_ref = 3.6*1E-3 + Nwet*1E-4;

hL = 1000;       % Height of the turbulent layer [m]

% Effective path length [m]
L = 2*hL ./ (sqrt(sind(teta).^2 + 2.35*1E-4) + sind(teta));

% Effective antenna diameter [m]
Deff = sqrt(eff).*D;

% Antenna averaging factor [adim]
x = 1.22*Deff.^2.*f./L;
if (3.86* (x.^2+1).^(11/12) .* sin((11/6).*atan(1./x))) < 0
    gx = 0;
else
    gx = sqrt(3.86* (x.^2+1).^(11/12) .* sin((11/6).*atan(1./x)) - 7.08*x.^(5/6));
end

% Standard deviation of the signal for the period and propagation path [dB]
sigma = sigma_ref .* f.^(7/12) .* gx./(sind(teta).^1.2);

% Time percentage factor (for the percent outage P)
aP = -0.061* log10(P).^3 + 0.072*log10(P).^2 - 1.71*log10(P) +3;

% Scintillation fade depth [dB]
As = aP.*sigma;



%% TOTAL ATMOSPHERIC ATTENUATION

% Atmospheric loss (gas + rain + cloud + scintillation) [dB]
At = Ag + sqrt((Ac+Ar).^2 + As.^2);

% Add the Uncertainty to the NOM value to obtain ADV and FAV value
Latm = At + At.*uncer./100;
atm_att = [NaN,Ag,NaN
       NaN,Ac,NaN
       NaN,Ar,NaN
       NaN,As,NaN];

end