close all
clear all
clc

%% DATA ACQUISITION & PREPROCESSING

filename = 'C:\Users\hp\Desktop\Link_Budget_Calculator\Link_Budget.xlsx';
sheet_input = 'Input';
sheet_outup = 'Output Up';
sheet_outdw = 'Output Dw';


IN = importfile(filename,sheet_input);
IN1 = IN(1:4,[1,3:5]);              % Orbit Configuration
IN2 = IN(6:18,:);                   % Modulation Configuration
IN3 = IN([21:31,33:41],3:5);        % Ground Station Parameters
F3 = IN([21:31,33:41],1);           % Flags
IN4 = IN([44:52,54:67],3:5);        % Satellite Parameters
F4 = IN([44:52,54:67],1);
IN5 = IN(69:87,3:5);                % Earth-to-Space Path Parameters
F5 = IN(69:87,1);
IN6 = IN(89:112,3:5);               % Communication Parameters
F6 = IN(89:112,1);
IN7 = IN(114:132,3:5);              % Modulation Parameters
F7 = IN(114:132,1);

clear IN

space = NaN(1,3);
head = {'NOM','ADV','FAV','Adv tol','Fav tol','Mean','s2','AT2'};

% Boltzmann Costant = kb = 1.38*1E-23 W/(Hz*K) = -228.6 dBW/(Hz*K)
kb = -228.6;

c = 300;             % Speed of light [Mm/s]



%% ORBIT & MODULATION CONFIGURATION

% Orbit Configuration
ORB = IN1(1,1);               % Orbit: 1=LEO/MEO - 2=GEO
posG = IN1(2:4,2);            % GS position

if ORB<=2
    hS = IN1(4,3);            % LEO - MEO Nominal height [km]
elseif ORB==3
    posS = IN1(2:4,4);        % GEO SC position (Lat, Long, h)
elseif ORB==4
    hs = IN1(4,4);            % GEO height for LEOP [km]
end

clear IN1


% Modulation Configuration
MOD = rmmissing(IN2([1:9,11:13],1));
% MOD1/2 -> Up/Dw Modulation type    MOD3 -> RG type
% MOD4 -> RG Measurement Accuracy    MOD5/6 -> PCM Code type
% MOD7 -> TM Coding    MOD8/9 -> TC/TM BER Requirement
% NB: Usually No Code for TC (except for Hamming intrinsic codification)

PRT = rmmissing(IN2(10,1));          % Protocol (standard CCSDS or DVB-S2)

MODU = IN2(5,2:4);
% Uplink carrier can be modulated by:
% 1) TC ONLY -> TC Up Carrier Recovery + TC Recovery
% 2) TC + RG SEPARATELY -> TC Up Carrier Recovery + TC Recovery 
%       + RG Up Carrier Recovery + RG (Major) Tone Recovery
% 3) TC + RG SIMULTANEOUS -> TC Up Carrier Recovery + TC Recovery
%       + RG (Major) Tone Recovery

MODD = IN2(6,2:4);
% Downlink carrier can be modulated by:
% 1) TM ONLY -> Dw Carrier Recovery + TM Recovery
% 2) TM + RG SIMULTANEOUS -> Dw Carrier Recovery + TM Recovery
%       + RG (Major) Tone Recovery
% 3) TC ECHO in downlink

DWO = IN2(6,5);             % Downlink only

% BBM = IN2(11,2);          % TM Baseband modulation for CCSDS
COD = IN2(11,3:4);          % TM Codification for CCSDS (VT+RS)
DVB = IN2(11,5);            % TM Codification for DVB-S2 (FEC)

clear IN2


% Communication basic paraemeters
f_up = IN6(1,:)*1E3;        % Uplink Carrier Frequency [GHz] -> [MHz]
f_dw = IN6(2,:)*1E3;        % Downlink Carrier Frequency [GHz] -> [MHz]
R_TC = IN6(3,:);            % TC Bit Rate [bps]
R_TM = IN6(4,:);            % TM Bit Rate [bps]



%% GROUND STATION PARAMETERS

if F3(1)==1
    D = IN3(1,:);             % GS Antenna Diameter [M]
elseif F3(1)==0
    D = zeros(1,3);
end

% GS Antenna Axial Ratio [dB]
if F3(3)==1
    XG = IN3(3,:);
    xpdG = XPD(XG);         % XPD computation from AR
elseif F3(3)==0
    xpdG = IN3(4,:);         % GS Crosspolar Discrimination [dB]
    XG = AR(xpdG);          % AR computation from XPD
end


if DWO==0

etaT = IN3(2,:);             % GS Antenna Tx Efficiency [adim.]

% GS Transmission Loss [dB]
if F3(7)==1
    LT_G = IN3(7,:);
elseif F3(7)==0
    Lt_G = IN3(8,:);         % GS Tx Chain Loss [dB]
    LA_G = IN3(9,:);         % GS Tx Antenna Loss [dB]
    LT_G = Lt_G + LA_G;
end

% GS Antenna Tx Gain [dBi]
if F3(10)==1
    Gt_G = IN3(10,:);
elseif F3(10)==0
    Gt_G = 10*log10(etaT.*(pi*D.*f_up/300).^2);
end

% GS EIRP [dBW]
if F3(5)==1
    eirpG = IN3(5,:);
    Pt_G = NaN(1,3);
elseif F3(5)==0
    Pt_G = IN3(6,:);             % GS Tx output Power [W]
    eirpG = 10*log10(Pt_G) - LT_G + Gt_G;
end

% Uplink Wavelength [m]
lambda_up = c./f_up;

% Uplink Half Power Beamwidth [deg]
if D(2)>0
    hpbw_up = 72.8*lambda_up./D;      % [deg]
elseif D(2)==0
    hpbw_up = IN3(11,:);
end

end



etaR = IN3(12,:);             % GS Antenna Rx Efficiency [adim.]

% Downlink Wavelength [m]
lambda_dw = c./f_dw;

% Downlink Half Power Beamwidth [deg]
if D(2)>0
    hpbw_dw = 72.8*lambda_dw./D;      % [deg]
elseif D(2)==0
    hpbw_dw = IN3(13,:);
end

% GS Rx Antenna G/T [dB/K]
if F3(14)==1
    GT_G = IN3(14,:);
    Gr_G = space;
    Ts_G = space;
    TA_G = NaN(1,3);
    TR_G = NaN(1,3);
elseif F3(14)==0

    % GS Antenna Rx Gain [dBi]
    if F3(15)==1
        Gr_G = IN3(15,:);
    elseif F3(15)==0
        Gr_G = 10*log10(etaR.*(pi*D.*f_dw/300).^2);
    end

    % GS System Noise Temperature [dBK]
    if F3(16)==1
        Ts_G = IN3(16,:);
        TA_G = NaN(1,3);
        TR_G = NaN(1,3);
    elseif F3(14)==0
        TA_G = IN3(17,:);        % GS Antenna Noise Temperature [K]

        % GS Rx Noise Temperature [K]
        if F3(18)==1
            TR_G = IN3(18,:);
        elseif F3(18)==0
            TR_G_data = IN3(19:20,:);
            TR_G = RXnoiseT_GS(TR_G_data);
        end

        Ts_G = 10*log10(TA_G + TR_G);
    end

    GT_G = Gr_G - Ts_G;
end

clear IN3
clear F3

    

%% SATELLITE PARAMETERS
             
% SC Antenna Axial Ratio (Tx) [dB]
if F4(1)==1
    XS_Tx = IN4(1,:);
    xpdS_Tx = XPD(XS_Tx);       % XPD computation from AR
elseif F4(1)==0
    xpdS_Tx = IN4(2,:);         % SC Crosspolar Discrimination (Tx) [dB]
    XS_Tx = AR(xpdS_Tx);        % AR computation from XPD
end


% SC EIRP [dBW]
if F4(3)==1
    eirpS = IN4(3,:);
    Pt_S = space;
    LT_S = space;
    Gt_S = space;
elseif F4(3)==0
    Pt_S = IN4(4,:);        % SC Tx output Power [W]

    % SC Transmission Loss [dB]
    if F4(5)==1
        LT_S = IN4(5,:);
    elseif F4(5)==0
        LinsT_S = IN4(6,:);         % SC Tx insertion Loss [dB]

        % SC Tx reflection Loss [dB]
        if F4(7)==1
            LrefT_S = IN4(7,:);         % SC Tx reflection Loss [dB]
        elseif F4(7)==0
            Wtx = IN4(8,:);        % Total Tx VSWR [adim.]
            LrefT_S = 10*log10((1+Wtx).^2./(4*Wtx));
        end

        LT_S = LinsT_S + LrefT_S;
    end

    Gt_S = IN4(9,:);        % SC Antenna Tx Gain [dBi]
    eirpS = 10*log10(Pt_S) - LT_S + Gt_S;
end



if DWO==0
    
% SC Antenna Axial Ratio (Rx) [dB]
if F4(10)==1
    XS_Rx = IN4(10,:);
    xpdS_Rx = XPD(XS_Rx);       % XPD computation from AR
elseif F4(10)==0
    xpdS_Rx = IN4(11,:);         % SC Crosspolar Discrimination (Rx) [dB]
    XS_Rx = AR(xpdS_Rx);        % AR computation from XPD
end

% SC Rx Antenna G/T [dB/K]
if F4(12)==1
    GT_S = IN4(12,:);
    Gr_S = space;
    Ts_S = space;
    TA_S = NaN(1,3);
    TR_S = NaN(1,3);
elseif F4(12)==0
    Gr_S = IN4(13,:);        % SC Antenna Rx Gain [dBi]

    % SC System Noise Temperature [dBK]
    if F4(14)==1
        Ts_S = IN4(14,:);
        TA_S = NaN(1,3);
        TR_S = NaN(1,3);
    elseif F4(14)==0
        TA_S = IN4(15,:);        % SC Antenna Noise Temperature [K]

        % SC Rx Noise Temperature [K]
        if F4(16)==1
            TR_S = IN4(16,:);
        elseif F4(16)==0
            TR_S_data = IN4(17:21,:);
            TR_S_flag = F4(17:21);
            TR_S = RXnoiseT_SC(TR_S_data,TR_S_flag);
        end

        Ts_S = 10*log10(TA_S + TR_S);
    end

    % SC Rx Reflection Loss
    if F4(22)==1
        LR_S = IN4(22,:);
    elseif F4(22)==0

        if F4(23)==1
            Wrx = IN4(23,:);        % Total Rx VSWR [adim.]
            LR_S = 10*log10((1+Wrx).^2./(4*Wrx));
        elseif F4(23)==0
            LR_S = zeros(1,3);
        end
    end

    GT_S = Gr_S - Ts_S - LR_S;
end

end

clear IN4
clear F4

    

%% EARTH-TO-SPACE PATH PARAMETERS

% GS Antenna Pointing Accuracy Loss (up/dw) [dB]
if F5(2)==1
    if DWO==0
    Lu_up = IN5(1,:);
    end
    Lu_dw = IN5(2,:);
    teta_u = space;
else
    if F5(3)==1
        teta_u = IN5(3,:);      % GS Antenna Pointing Accuracy (Error) [deg]
    elseif F5(3)==0
        teta_u = zeros(1,3);
    end
    
    if DWO==0
    Lu_up = pointloss(f_up,D,teta_u,hpbw_up);
    end    
    Lu_dw = pointloss(f_dw,D,teta_u,hpbw_dw);
end


% GS Antenna Elevation Angle [deg] (in input for LEO orbits)
if ORB==1
    eps = IN5(8,:);
end

% Path Loss [dB/m^2] (up=dw)
if F5(9)==1
    LP = IN5(9,:);
elseif F5(9)==0

    % Slant Range [km] (up=dw)
    if F5(10)==1
        S = IN5(10,:);
    elseif F5(10)==0
        if ORB<=2 || ORB==4         % LEO - MEO - LEOP
            S = slantrange_LEO(hS,eps);
        elseif ORB==3               % GEO
            [S,eps] = slantrange_GEO(posG,posS);
        end
    end

    LP = 10*log10(4*pi*1E6*S.^2);
end


% Pointing Offset Loss (up/dw) [dB]
if F5(5)==1
    if DWO==0
    Lo_up = IN5(4,:);
    end
    Lo_dw = IN5(5,:);
    teta_o = space;
    do = space;
else
    
    if F5(6)==1
        teta_o = IN5(6,:);      % Pointing Offset [deg]
        do = space;
    elseif F5(6)==0
        if F5(7)==1
            do = IN5(7,:)*1E-3;      % Offset distance [m] -> [km]
            teta_o = asind(do./S);
        elseif F5(7)==0
            do = space;
            teta_o = zeros(1,3);
        end
    end

    if DWO==0
    Lo_up = 12*(teta_o./hpbw_up).^2;
    end    
    Lo_dw = 12*(teta_o./hpbw_dw).^2;
end


% Polarization Loss [dB] (up/dw)
if F5(11)==1
    LH = IN5(11,:);
elseif F5(11)==0
    LH = polarizloss(XG,XS_Tx);
end

% Ionospheric loss (up)
if F5(12)==1
    Lion_up = IN5(12,:);
elseif F5(12)==0
    Lion_up = zeros(1,3);
end

% Ionospheric loss (dw)
if F5(13)==1
    Lion_dw = IN5(13,:);
elseif F5(13)==0
    Lion_dw = zeros(1,3);
end

% Atmospheric Loss (up) [dB]
if DWO==0
if F5(14)==1
    Latm_up = IN5(14,:);
elseif F5(14)==0
    atm_data = IN5(16:18,:);
    [Latm_up,att_up] = atmloss(f_up(2),eps(2),posG,D(2),etaT(2),atm_data);
end
end

% Atmospheric Loss (dw) [dB]
if F5(15)==1
    Latm_dw = IN5(15,:);
elseif F5(15)==0
    atm_data = IN5(16:18,:);
    [Latm_dw,att_dw] = atmloss(f_dw(2),eps(2),posG,D(2),etaR(2),atm_data);
end

% GS Antenna Radome Loss
if F5(19)==1
    Lrad = IN5(19,:);            % GS Radome Loss [dB]
elseif F5(19)==0
    Lrad = zeros(1,3);
end

clear IN5
clear F5



%% UPLINK BASIC

if DWO==0

% Power Flux Density at SC in Free Space [dBW/m^2]
Phi_freeS = eirpG - LP;

% Uplink Free Space Loss [dB]
Lup = LP - 10*log10(22500./(pi*f_up.^2));

% Uplink Total Propagation Loss [dB]
LupT = Lup + LH + Lion_up + Latm_up + Lrad;

% Power Flux Density at SC (including atmospheric loss) [dBW/m^2]
Phi_S = Phi_freeS - Lu_up - Lo_up - LH - Lion_up - Latm_up - Lrad;

% Uplink S/N0 [dBHz] (Signal-to-Noise Ratio at SC)
SN0_S = eirpG - Lu_up - Lo_up - LupT + GT_S - kb;


% Uplink Basic Output
OUT_UP1 = [f_up*1E-3;lambda_up;Pt_G;D;etaT;XG;xpdG;LT_G;Gt_G;hpbw_up;eirpG;
    space;teta_u;Lu_up;do;teta_o;Lo_up;eps;S;LP;Phi_freeS;Lup;LH;Lion_up;att_up;
    Latm_up;Lrad;LupT;Phi_S;space;XS_Rx;xpdS_Rx;Gr_S;TA_S;TR_S;Ts_S;GT_S;SN0_S];

end



%% UP MODULATION LOSS

if DWO==0

if F7(2)==1
    XupC_C = IN7(1,:);
    XTC = IN7(2,:);
    XupC_R = IN7(3,:);
    XMjTup = IN7(4,:);
    XMnTup = IN7(5,:);
    mi_up = NaN(3,3);

elseif F7(2)==0

    if MOD(1)==1           % PM Modulation with subcarriers
        mi_up = IN7(6:8,:);
        [XupC_C,XTC,XupC_R,XMjTup,XMnTup,parzCup,parzTC,parzRup] = ...
            PMsc_up(mi_up,MODU,MOD(3));

    elseif MOD(1)==2       % PM direct Modulation
        mi_up = IN7(6:8,:);
        [XupC_C,XTC,XupC_R,XMjTup,XMnTup,parzCup,parzTC,parzRup] = ...
            PMdir_up(mi_up,MODU,MOD(3));

    elseif MOD(1)>=3 && MOD(1)<=9       % PSK Modulation
        alfaTC = IN7(9,:);
        [XupC_C,XTC,XupC_R,XMjTup,XMnTup] = PSK(alfaTC,MOD(5));
        parzTC = [XTC;space];
        mi_up = NaN(3,3);

    elseif MOD(1)==10       % FSK Modulation
        Delta_fup = IN7(10,:);
        [XupC_C,XTC,XupC_R,XMjTup,XMnTup] = FSK(Delta_fup,R_TC,MOD(5));
        parzTC = [XTC;space];
        mi_up = NaN(3,3);

    elseif MOD(1)==11       % FM Modulation with subcarriers
        FM_freq = IN7(10:11,:);
        RG_freq = IN6(21:22,:);
        [XupC_C,XTC,XupC_R,XMjTup,XMnTup] = FM(FM_freq,RG_freq,MODU,MOD(3));
        parzTC = [XTC;space];
        parzRup = [XMjTup;space];
        mi_up = NaN(3,3);
    end
end
OUT_UP1 = [OUT_UP1;space;mi_up];

end



%% (TC) UP CARRIER RECOVERY
% (TC) Carrier Recovery budget is not present (doesn't have sense) for PSK
% direct modulation and FM with s/c modulation

if DWO==0

if MOD(1)==1 || MOD(1)==2

    PLL_S = IN6(5,:);       % SC PLL (phase-locked loop) Bandwidth (2BL) [Hz]
    % -> to be converted to [dBHz]
    PLL_S_dB = 10*log10(PLL_S);

    % SC PLL Acquisition treshold [dB] (C/N req. for Carrier acq.)
    S_PLL = IN6(6,:);

    ThL_UpC = IN6(7,:);     % Up Carrier Acquisition technical Loss [dB]

    % XupC_C = TC Up Carrier Modulation Loss [dB]

    % Uplink C/N0 [dBHz]
    CN0_S = SN0_S - XupC_C - ThL_UpC;

    % Uplink C/N (SNR - signal to noise ratio - in the loop) [dBHz]
    CN_S = CN0_S - PLL_S_dB;

    % Required C/N0 at SC for (TC) Carrier Acquisition
    CN0_req_UpC = XupC_C + ThL_UpC + PLL_S_dB + S_PLL;

    % Margin for (TC) Up Carrier Acquisition (> 3 dB)
    MupC_C = CN_S - S_PLL;


    % (TC) Up Carrier Recovery Output
    OUT_UP2 = [parzCup;XupC_C;ThL_UpC;CN0_S;PLL_S;PLL_S_dB;CN_S;S_PLL;
        CN0_req_UpC;MupC_C;space;space];

else
    OUT_UP2 = NaN(13,3);
end

end



%% TC RECOVERY
% TC Recovery budget is always present when there is data transmission in uplink

if DWO==0

% S_TC = SC TC Acquisition treshold [dB] (Eb/N0 req. for TC dem.)
if F6(8)==1
    S_TC = IN6(8,:);
elseif F6(8)==0
    S_TC = coding(MOD(7),[1,1],MOD(1));        % 1 = No Code
end

ThL_TC = IN6(9,:);          % TC Demodulation technical Loss [dB]

% XTC = TC Modulation Loss [dB]

% TC S/N0 [dBHz]
SN0_TC = SN0_S - XTC - ThL_TC;

% R_TC = TC Bit Rate [bps] -> to be converted to [dBHz]
R_TC_dB = 10*log10(R_TC);

% TC Eb/N0 (SNR - signal to noise ratio - in the loop) [dBHz]
EbN0_TC = SN0_TC - R_TC_dB;

% Required C/N0 at SC for Carrier Acquisition
CN0_req_TC = XTC + ThL_TC + R_TC_dB + S_TC;

% Margin for Carrier Acquisition (> 3 dB)
MTC = EbN0_TC - S_TC;


% TC Recovery Output
OUT_UP3 = [parzTC;XTC;ThL_TC;SN0_TC;R_TC;R_TC_dB;EbN0_TC;S_TC;
    CN0_req_TC;MTC;space;space];

end



%% RNG CARRIER RECOVERY
% RNG Carrier Recovery budget is present only for PM Modulation with
% TC + RG separately

if DWO==0

if MODU(2) == 1 && (MOD(1)==1 || MOD(1)==2)

    % XupC_R = RG Up Carrier Modulation Loss [dB]
    % ThL_UpC = Up Carrier Acquisition technical Loss [dB]

    % Uplink C/N0 [dBHz]
    CN0_R = SN0_S - XupC_R - ThL_UpC;

    % PLL_S = SC PLL (phase-locked loop) Bandwidth [Hz]

    % Uplink C/N (SNR - signal to noise ratio - in the loop) [dBHz]
    CN_R = CN0_R - PLL_S_dB;

    % S_PLL = SC PLL Acquisition treshold [dB] (C/N req. for Carrier acq.)

    % Required C/N0 at SC for Carrier Acquisition
    CN0_req_UpCR = XupC_R + ThL_UpC + PLL_S_dB + S_PLL;

    % Margin for Carrier Acquisition (> 3 dB)
    MupC_R = CN_R - S_PLL;


    % RNG Carrier Recovery Output
    OUT_UP4 = [XupC_R;ThL_UpC;CN0_R;PLL_S;PLL_S_dB;CN_R;S_PLL;
        CN0_req_UpCR;MupC_R;space;space];

else
    OUT_UP4 = NaN(11,3);
end

end



%% RNG (MAJOR) TONE RECOVERY (UP)
% RNG (Major) Tone Recovery (up) budget is present in case of RG 
% measurements (with or without TC)

if DWO==0

if (MODU(2)==1 || MODU(3)==1) && MOD(1)~=3

    % RG Up effective loop BW (2BL) [KHz] -> [Hz] (Ranging noise bandwidth)
    BWup_RG = IN6(10,:)*1E3;
    % -> to be converted to [dBHz]
    BWup_RG_dB = 10*log10(BWup_RG);

    TupL_RG = IN6(11,:);       % RG Up Demodulation technical Loss [dB]

    % XMjTup = RG Major Tone Modulation Loss (up) [dB]

    % RG (Mj) Tone S/N0 (up) [dBHz]
    SN0_MT = SN0_S - XMjTup - TupL_RG;
    
    % RG Major Tone SNR (up) in RG video-bandwidth [dB]
    % (S/N for Major Tone)
    SN_MT = SN0_MT - BWup_RG_dB;

    % S/N for TC
    if MODD(3)==1       % in case of TC echo
        SN_TC = SN0_S - BWup_RG_dB - XTC - ThL_TC;
    else
        SN_TC = zeros(1,3);
    end

    % S/N for Minor Tone [dB]
    if MOD(3)==2
        % XMnTup = RG Minor Tone Modulation Loss (up) [dB]
        SN_mt = SN0_S - XMnTup - TupL_RG - BWup_RG_dB;
    else
        SN_mt = zeros(1,3);
    end

    % S/N TOT (RGMjT + RGMnT + TC) [linear]
    SN_tot = 1 + (MODU(2)==1 || MODU(3)==1) .* (10.^(SN_MT/10) + ...
        (MOD(3)==2).*10.^(SN_mt/10) + (MODD(3)==1).*10.^(SN_TC/10));
    

    % RNG Tone Recovery (up) Output
    OUT_UP5 = [parzRup;XMjTup;TupL_RG;SN0_MT;BWup_RG*1E-3;BWup_RG_dB;
        SN_MT;SN_mt;SN_tot];

else
    OUT_UP5 = NaN(10,3);
end

end



%% DOWNLINK BASIC

% Power Flux Density at GS in Free Space [dBW/m^2]
Phi_freeG = eirpS - LP;

% Downlink Free Space Loss [dB]
Ldw = LP - 10*log10(22500./(pi*f_dw.^2));

% Downlink Total Propagation Loss [dB]
LdwT = Ldw + LH + Lion_dw + Latm_dw + Lrad;

% Power Flux Density at GS (including atmospheric loss) [dBW/m^2]
Phi_G = Phi_freeG - Lu_dw - Lo_dw - LH - Lion_dw - Latm_dw - Lrad;

% Downlink S/N0 [dBHz] (Signal-to-Noise Ratio at GS)
SN0_G = eirpS - Lu_dw - Lo_dw - LdwT + GT_G - kb;


% Downlink Basic Output
OUT_DW1 = [f_dw*1E-3;lambda_dw;Pt_S;XS_Tx;xpdS_Tx;LT_S;Gt_S;eirpS;space;teta_u;
    Lu_dw;do;teta_o;Lo_dw;eps;S;LP;Phi_freeG;Ldw;LH;Lion_dw;att_dw;Latm_dw;
    Lrad;LdwT;Phi_G;space;XG;xpdG;Gr_G;hpbw_dw;TA_G;TR_G;Ts_G;GT_G;SN0_G];



%% DW MODULATION LOSS

if F7(13)==1
    XdwC_C = IN7(12,:);
    XTM = IN7(13,:);
    XMjTdw = IN7(14,:);
    XMnTdw = IN7(15,:);
    mi_dw1 = NaN(2,3);
    mi_dw2 = NaN(4,3);

elseif F7(13)==0

    if MOD(2)==1           % PM Modulation with subcarriers
        mi_dw1 = IN7(16:17,:);
        [XdwC_C,XTM,XMjTdw,XMnTdw,parzCdw,parzTM,parzRdw,mi_dw2] = ...
            PMsc_dw(mi_dw1,SN_MT,SN_mt,SN_TC,SN_tot,MODD,MOD(3));

    elseif MOD(2)==2       % PM direct Modulation
        mi_dw1 = IN7(16:17,:);
        [XdwC_C,XTM,XMjTdw,XMnTdw,parzCdw,parzTM,parzRdw,mi_dw2] = ...
            PMdir_dw(mi_dw1,SN_MT,SN_mt,SN_TC,SN_tot,MODD,MOD(3));

    elseif MOD(2)>=3 && MOD(2)<=9       % PSK Modulation
        alfaTM = IN7(18,:);
        [XdwC_C,XTM,~,XMjTdw,XMnTdw] = PSK(alfaTM,MOD(6));
        parzTM = [XTM;space;space;space];
        mi_dw1 = NaN(2,3);
        mi_dw2 = NaN(4,3);

    elseif MOD(2)==10       % FSK Modulation
        Delta_fdw = IN7(19,:);
        [XdwC_C,XTM,~,XMjTdw,XMnTdw] = FSK(Delta_fdw,R_TM,MOD(6));
        parzTM = [XTM;space;space;space];
        mi_dw1 = NaN(2,3);
        mi_dw2 = NaN(4,3);
    end
end
OUT_DW1 = [OUT_DW1;space;mi_dw1;mi_dw2];



%% DW CARRIER RECOVERY
% Carrier Recovery budget is not present (doesn't have sense) for PSK
% direct modulation and FSK direct modulation

if MOD(2)==1 || MOD(2)==2

    PLL_G = IN6(12,:);       % GS PLL (phase-locked loop) Bandwidth (2BL) [Hz]
    % -> to be converted to [dBHz]
    PLL_G_dB = 10*log10(PLL_G);

    % GS PLL Acquisition treshold [dB] (C/N req. for Carrier acq.)
    G_PLL = IN6(13,:);

    ThL_DwC = IN6(14,:);     % Dw Carrier Acquisition technical Loss [dB]

    % XdwC_C = TM Dw Carrier Modulation Loss [dB]

    % Downlink C/N0 [dBHz]
    CN0_G = SN0_G - XdwC_C - ThL_DwC;

    % Downlink C/N (SNR - signal to noise ratio - in the loop) [dBHz]
    CN_G = CN0_G - PLL_G_dB;

    % Required C/N0 at GS for Carrier Acquisition
    CN0_req_DwC = XdwC_C + ThL_DwC + PLL_G_dB + G_PLL;

    % Margin for Dw Carrier Acquisition (> 3 dB)
    MdwC_C = CN_G - G_PLL;


    % Dw Carrier Recovery Output
    OUT_DW2 = [parzCdw;XdwC_C;ThL_DwC;CN0_G;PLL_G;PLL_G_dB;CN_G;G_PLL; ...
        CN0_req_DwC;MdwC_C;space;space];

else
    OUT_DW2 = NaN(15,3);
end



%% TM RECOVERY
% TM Recovery budget is always present when there is data transmission in downlink

% G_TM = GS TM Acquisition treshold [dB] (Eb/N0 req. for TC dem.)
if F6(15)==1
    G_TM = IN6(15,:);
elseif F6(15)==0
    if PRT==1 || PRT==3           % protocol CCSDS / AX.25
        G_TM = coding(MOD(8),COD,MOD(2));
    elseif PRT==2       % protocol DVB-S2
        G_TM = dvbs2(DVB);
    end
end

ThL_TM = IN6(16,:);          % TC Demodulation technical Loss [dB]

% XTM = TM Modulation Loss [dB]

% TM S/N0 [dBHz]
SN0_TM = SN0_G - XTM - ThL_TM;

% R_TM = TM Bit Rate [bps] -> to be converted to [dBHz]
R_TM_dB = 10*log10(R_TM);

% TM Eb/N0 (SNR - signal to noise ratio - in the loop) [dBHz]
EbN0_TM = SN0_TM - R_TM_dB;

% Required C/N0 at GS for TM Acquisition
CN0_req_TM = XTM + ThL_TM + R_TM_dB + G_TM;

% Margin for Carrier Acquisition (> 3 dB)
MTM = EbN0_TM - G_TM;


% TM Recovery Output
OUT_DW3 = [parzTM;XTM;ThL_TM;SN0_TM;R_TM;R_TM_dB;EbN0_TM;G_TM; ...
    CN0_req_TM;MTM;space;space];



%% RNG (MAJOR) TONE RECOVERY (DW)
% RNG (Major) Tone Recovery (dw) budget is present in case of RG 
% measurements (with or without TC)

if MODD(2)==1 && (MOD(1)==1 || MOD(1)==2)

    % RG Dw effective loop BW (2BL) [Hz] (Ranging noise bandwidth)
    BWdw_RG = IN6(17,:);
    % -> to be converted to [dBHz]
    BWdw_RG_dB = 10*log10(BWdw_RG);

    SNR_MT = IN6(18,:);         % RG Major Tone required S/N [dB]
    TdwL_RG = IN6(20,:);        % RG Up Demodulation technical Loss [dB]

    % XMjTdw = RG Major Tone Modulation Loss (dw) [dB]

    % RG (Mj) Tone S/N0 (dw) [dBHz]
    SN0_MTdw = SN0_G - XMjTdw - TdwL_RG;
    
    % RG (Major) Tone SNR (dw) in RG video-bandwidth (S/N for Major Tone) [dB]
    SN_MTdw = SN0_MTdw - BWdw_RG_dB;

    % Required C/N0 at GS for RG (Mj) Tone Acquisition
    CN0_req_MT = XMjTdw + TdwL_RG + BWdw_RG_dB + SNR_MT;

    % Margin for RG (Mj) Tone Dw Acquisition (> 3 dB)
    MMT_R = SN_MTdw - SNR_MT;
    

    % RNG (Major) Tone Recovery (dw) Output
    OUT_DW4 = [parzRdw;XMjTdw;TdwL_RG;SN0_MTdw;BWdw_RG;BWdw_RG_dB;SN_MTdw; ...
        SNR_MT;CN0_req_MT;MMT_R;space;space];

else
    OUT_DW4 = NaN(15,3);
end



%% RNG (MINOR) TONE RECOVERY (DW)
% RNG (Minor) Tone Recovery (dw) budget is present in case of Tone RG

if  MOD(3)==2 && MODD(2)==1 && (MOD(1)==1 || MOD(1)==2)

    SNR_mt = IN6(19,:);         % RG Minor Tone required S/N [dB]

    % XMnTdw = RG Minor Tone Modulation Loss (dw) [dB]

    % RG (Mj) Tone S/N0 (dw) [dBHz]
    SN0_mtdw = SN0_G - XMnTdw - TdwL_RG;
    
    % RG (Major) Tone SNR (dw) in RG video-bandwidth (S/N for Major Tone) [dB]
    SN_mtdw = SN0_mtdw - BWdw_RG_dB;

    % Required C/N0 at GS for RG (Mj) Tone Acquisition
    CN0_req_mt = XMnTdw + TdwL_RG + BWdw_RG_dB + SNR_mt;

    % Margin for RG (Mj) Tone Dw Acquisition (> 3 dB)
    Mmt_R = SN_mtdw - SNR_mt;
    

    % RNG (Major) Tone Recovery (dw) Output
    OUT_DW5 = [XMnTdw;TdwL_RG;SN0_mtdw;BWdw_RG;BWdw_RG_dB;SN_mtdw; ...
        SNR_mt;CN0_req_mt;Mmt_R;space;space];

else
    OUT_DW5 = NaN(11,3);
end



%% RNG MEASUREMENT ACCURACY
% Analyse the system's ability to meet a requirement of ranging accuracy
% (applicable to both code and tone ranging)

if MOD(4)==1 && MODD(2)==1 && (MOD(1)==1 || MOD(1)==2)

    teta_m = IN6(23,:);
    sigma_R = IN6(24,:);
    f_MT = IN6(21,:);
    
    % RG Measurement Accuracy S/N treshold [dB]
    SN_acc = -10*log10(2) + 20*log10(75000./(pi*f_MT.*sigma_R));
    
    % BWdw_RG = RG Dw effective loop BW (2BL) [Hz] (Ranging noise bandwidth)
    % -> to be converted to [dBHz]
    BWdw_RG_dB = 10*log10(BWdw_RG);
    
    % RG Code modulation loss [dB]
    XRG_acc = -20*log10(cos(teta_m));
    
    % Required C/N0 at GS RG Measurement Accuracy
    CN0_req_acc = SN_acc + BWdw_RG_dB + XRG_acc + XMjTdw + TdwL_RG;
    
    % Margin for RG Measurement Accuracy (> 3 dB)
    M_accR = SN0_G - CN0_req_acc;
    
    
    % RNG Meas. Acc. Output
    OUT_DW6 = [f_MT;sigma_R;SN_acc;BWdw_RG;BWdw_RG_dB;XMjTdw; ...
        TdwL_RG;teta_m;XRG_acc;CN0_req_acc;M_accR;space;space];

else
    OUT_DW6 = NaN(13,3);
end

clear IN6
clear F6
clear IN7
clear F7



%% POSTPROCESSING - UP BASIC

if DWO==0

% STAT: 0 = no stat, 1 = mean, 2 = tol,
% 3 = uniform, 4 = triangular, 5 = gaussian
STU = importstatup(filename,sheet_outup);
STU1 = STU(1:45);
STU2 = STU(47:57);
STU3 = STU(61:71);
STU4 = STU(75:83);
STU5 = STU(87:96);
clear STU

AT = NaN(length(OUT_UP1),1);        % adv tol = adv - nom
FT = NaN(length(OUT_UP1),1);        % fav tol = fav - nom
mean = NaN(length(OUT_UP1),1);
s2 = NaN(length(OUT_UP1),1);
AT2 = NaN(length(OUT_UP1),1);

for i=1:length(OUT_UP1)
    if STU1(i)>=2
        AT(i) = OUT_UP1(i,2) - OUT_UP1(i,1);
        FT(i) = OUT_UP1(i,3) - OUT_UP1(i,1);
    end

    if STU1(i)==1
        mean(i) = (OUT_UP1(i,2) + OUT_UP1(i,3)) ./2;
    elseif STU1(i)==4
        mean(i) = (OUT_UP1(i,1) + OUT_UP1(i,2) + OUT_UP1(i,3)) ./3;
        s2(i) = mean(i).^2./2 - (OUT_UP1(i,1).*OUT_UP1(i,2) + OUT_UP1(i,2) ...
            .*OUT_UP1(i,3) + OUT_UP1(i,1).*OUT_UP1(i,3)) ./6;
    elseif STU1(i)==3
        mean(i) = (OUT_UP1(i,2) + OUT_UP1(i,3)) ./2;
        s2(i) = (OUT_UP1(i,2) - OUT_UP1(i,3)).^2 ./12;
    elseif STU1(i)==5
        mean(i) = (OUT_UP1(i,2) + OUT_UP1(i,3)) ./2;
        s2(i) = (OUT_UP1(i,2) - OUT_UP1(i,3)).^2 ./36;
    end

    if STU1(i)>=3
        AT2(i) = AT(i).^2;
    end
end

s2tot = sum(rmmissing(s2));
AT2tot = sum(rmmissing(AT2));
s2(length(OUT_UP1)-4) = s2tot;
AT2(length(OUT_UP1)-4) = AT2tot;

OUT_UP = num2cell([OUT_UP1,AT,FT,mean,s2,AT2]);
clear AT
clear FT
clear mean
clear s2
clear AT2
clear STU1
clear OUT_UP1

end



%% POSTPROCESSING - (TC) UP CARRIER RECOVERY

if DWO==0

AT = NaN(length(OUT_UP2),1);
FT = NaN(length(OUT_UP2),1);
mean = NaN(length(OUT_UP2),1);
s2 = NaN(length(OUT_UP2),1);
AT2 = NaN(length(OUT_UP2),1);

if MOD(1)==1 || MOD(1)==2
    for i=1:(length(OUT_UP2)-2)
        if STU2(i)>=2
            AT(i) = OUT_UP2(i,2) - OUT_UP2(i,1);
            FT(i) = OUT_UP2(i,3) - OUT_UP2(i,1);
        end
    
        if STU2(i)==1
            mean(i) = (OUT_UP2(i,2) + OUT_UP2(i,3)) ./2;
        elseif STU2(i)==4
            mean(i) = (OUT_UP2(i,1) + OUT_UP2(i,2) + OUT_UP2(i,3)) ./3;
            s2(i) = mean(i).^2./2 - (OUT_UP2(i,1).*OUT_UP2(i,2) + OUT_UP2(i,2) ...
                .*OUT_UP2(i,3) + OUT_UP2(i,1).*OUT_UP2(i,3)) ./6;
        elseif STU2(i)==3
            mean(i) = (OUT_UP2(i,2) + OUT_UP2(i,3)) ./2;
            s2(i) = (OUT_UP2(i,2) - OUT_UP2(i,3)).^2 ./12;
        elseif STU2(i)==5
            mean(i) = (OUT_UP2(i,2) + OUT_UP2(i,3)) ./2;
            s2(i) = (OUT_UP2(i,2) - OUT_UP2(i,3)).^2 ./36;
        end
    
        if STU2(i)>=3
            AT2(i) = AT(i).^2;
        end
    end

    s2(length(OUT_UP2)-3) = sum(rmmissing(s2));
    AT2(length(OUT_UP2)-3) = sum(rmmissing(AT2));
    
    s3 = s2tot + s2(length(OUT_UP2)-3);
    RSS = AT2tot + AT2(length(OUT_UP2)-3);
    
    AT2(length(OUT_UP2)-1) = s3;
    AT2(length(OUT_UP2)) = RSS;
    
    OUT_UP2(length(OUT_UP2)-1,1) = mean(length(OUT_UP2)-2) - 3*sqrt(s3);
    OUT_UP2(length(OUT_UP2),1) = OUT_UP2(length(OUT_UP2)-2,1) - sqrt(RSS);
end

OUT_UP2 = num2cell([OUT_UP2,AT,FT,mean,s2,AT2]);
clear AT
clear FT
clear mean
clear s2
clear AT2
clear s3
clear RSS
clear STU2

OUT_UP = [OUT_UP;head;OUT_UP2];
clear OUT_UP2

end



%% POSTPROCESSING - TC RECOVERY

if DWO==0

AT = NaN(length(OUT_UP3),1);
FT = NaN(length(OUT_UP3),1);
mean = NaN(length(OUT_UP3),1);
s2 = NaN(length(OUT_UP3),1);
AT2 = NaN(length(OUT_UP3),1);

for i=1:(length(OUT_UP3)-2)
    if STU3(i)>=2
        AT(i) = OUT_UP3(i,2) - OUT_UP3(i,1);
        FT(i) = OUT_UP3(i,3) - OUT_UP3(i,1);
    end

    if STU3(i)==1
        mean(i) = (OUT_UP3(i,2) + OUT_UP3(i,3)) ./2;
    elseif STU3(i)==4
        mean(i) = (OUT_UP3(i,1) + OUT_UP3(i,2) + OUT_UP3(i,3)) ./3;
        s2(i) = mean(i).^2./2 - (OUT_UP3(i,1).*OUT_UP3(i,2) + OUT_UP3(i,2) ...
            .*OUT_UP3(i,3) + OUT_UP3(i,1).*OUT_UP3(i,3)) ./6;
    elseif STU3(i)==3
        mean(i) = (OUT_UP3(i,2) + OUT_UP3(i,3)) ./2;
        s2(i) = (OUT_UP3(i,2) - OUT_UP3(i,3)).^2 ./12;
    elseif STU3(i)==5
        mean(i) = (OUT_UP3(i,2) + OUT_UP3(i,3)) ./2;
        s2(i) = (OUT_UP3(i,2) - OUT_UP3(i,3)).^2 ./36;
    end

    if STU3(i)>=3
        AT2(i) = AT(i).^2;
    end
end

s2(length(OUT_UP3)-3) = sum(rmmissing(s2));
AT2(length(OUT_UP3)-3) = sum(rmmissing(AT2));

s3 = s2tot + s2(length(OUT_UP3)-3);
RSS = AT2tot + AT2(length(OUT_UP3)-3);

AT2(length(OUT_UP3)-1) = s3;
AT2(length(OUT_UP3)) = RSS;

OUT_UP3(length(OUT_UP3)-1,1) = mean(length(OUT_UP3)-2) - 3*sqrt(s3);
OUT_UP3(length(OUT_UP3),1) = OUT_UP3(length(OUT_UP3)-2,1) - sqrt(RSS);

OUT_UP3 = num2cell([OUT_UP3,AT,FT,mean,s2,AT2]);
clear AT
clear FT
clear mean
clear s2
clear AT2
clear s3
clear RSS
clear STU3

OUT_UP = [OUT_UP;head;OUT_UP3];
clear OUT_UP3

end



%% POSTPROCESSING - RNG CARRIER RECOVERY

if DWO==0

AT = NaN(length(OUT_UP4),1);
FT = NaN(length(OUT_UP4),1);
mean = NaN(length(OUT_UP4),1);
s2 = NaN(length(OUT_UP4),1);
AT2 = NaN(length(OUT_UP4),1);

if MODU(2) == 1 && (MOD(1)==1 || MOD(1)==2)
    for i=1:(length(OUT_UP4)-2)
        if STU4(i)>=2
            AT(i) = OUT_UP4(i,2) - OUT_UP4(i,1);
            FT(i) = OUT_UP4(i,3) - OUT_UP4(i,1);
        end
    
        if STU4(i)==1
            mean(i) = (OUT_UP4(i,2) + OUT_UP4(i,3)) ./2;
        elseif STU4(i)==4
            mean(i) = (OUT_UP4(i,1) + OUT_UP4(i,2) + OUT_UP4(i,3)) ./3;
            s2(i) = mean(i).^2./2 - (OUT_UP4(i,1).*OUT_UP4(i,2) + OUT_UP4(i,2) ...
                .*OUT_UP4(i,3) + OUT_UP4(i,1).*OUT_UP4(i,3)) ./6;
        elseif STU4(i)==3
            mean(i) = (OUT_UP4(i,2) + OUT_UP4(i,3)) ./2;
            s2(i) = (OUT_UP4(i,2) - OUT_UP4(i,3)).^2 ./12;
        elseif STU4(i)==5
            mean(i) = (OUT_UP4(i,2) + OUT_UP4(i,3)) ./2;
            s2(i) = (OUT_UP4(i,2) - OUT_UP4(i,3)).^2 ./36;
        end
    
        if STU4(i)>=3
            AT2(i) = AT(i).^2;
        end
    end

    s2(length(OUT_UP4)-3) = sum(rmmissing(s2));
    AT2(length(OUT_UP4)-3) = sum(rmmissing(AT2));
    
    s3 = s2tot + s2(length(OUT_UP4)-3);
    RSS = AT2tot + AT2(length(OUT_UP4)-3);
    
    AT2(length(OUT_UP4)-1) = s3;
    AT2(length(OUT_UP4)) = RSS;
    
    OUT_UP4(length(OUT_UP4)-1,1) = mean(length(OUT_UP4)-2) - 3*sqrt(s3);
    OUT_UP4(length(OUT_UP4),1) = OUT_UP4(length(OUT_UP4)-2,1) - sqrt(RSS);
end

OUT_UP4 = num2cell([OUT_UP4,AT,FT,mean,s2,AT2]);
clear AT
clear FT
clear mean
clear s2
clear AT2
clear s3
clear RSS
clear STU4

OUT_UP = [OUT_UP;head;OUT_UP4];
clear OUT_UP4

end



%% POSTPROCESSING - RNG (MAJOR) TONE RECOVERY (UP)

if DWO==0

AT = NaN(length(OUT_UP5),1);
FT = NaN(length(OUT_UP5),1);
mean = NaN(length(OUT_UP5),1);
s2 = NaN(length(OUT_UP5),1);
AT2 = NaN(length(OUT_UP5),1);

if (MODU(2)==1 || MODU(3)==1) && MOD(1)~=3
    for i=1:(length(OUT_UP5)-2)
        if STU5(i)>=2
            AT(i) = OUT_UP5(i,2) - OUT_UP5(i,1);
            FT(i) = OUT_UP5(i,3) - OUT_UP5(i,1);
        end
    
        if STU5(i)==1
            mean(i) = (OUT_UP5(i,2) + OUT_UP5(i,3)) ./2;
        elseif STU5(i)==4
            mean(i) = (OUT_UP5(i,1) + OUT_UP5(i,2) + OUT_UP5(i,3)) ./3;
            s2(i) = mean(i).^2./2 - (OUT_UP5(i,1).*OUT_UP5(i,2) + OUT_UP5(i,2) ...
                .*OUT_UP5(i,3) + OUT_UP5(i,1).*OUT_UP5(i,3)) ./6;
        elseif STU5(i)==3
            mean(i) = (OUT_UP5(i,2) + OUT_UP5(i,3)) ./2;
            s2(i) = (OUT_UP5(i,2) - OUT_UP5(i,3)).^2 ./12;
        elseif STU5(i)==5
            mean(i) = (OUT_UP5(i,2) + OUT_UP5(i,3)) ./2;
            s2(i) = (OUT_UP5(i,2) - OUT_UP5(i,3)).^2 ./36;
        end
    
        if STU5(i)>=3
            AT2(i) = AT(i).^2;
        end
    end
end

OUT_UP5 = num2cell([OUT_UP5,AT,FT,mean,s2,AT2]);
clear AT
clear FT
clear mean
clear s2
clear AT2
clear STU5

OUT_UP = [OUT_UP;head;OUT_UP5];
clear OUT_UP5

end



%% POSTPROCESSING - DW BASIC

STD = importstatdw(filename,sheet_outdw);
STD1 = STD(1:46);
STD2 = STD(48:60);
STD3 = STD(64:76);
STD4 = STD(80:92);
STD5 = STD(96:104);
STD6 = STD(108:118);

AT = NaN(length(OUT_DW1),1);        % adv tol = adv - nom
FT = NaN(length(OUT_DW1),1);        % fav tol = fav - nom
mean = NaN(length(OUT_DW1),1);
s2 = NaN(length(OUT_DW1),1);
AT2 = NaN(length(OUT_DW1),1);

% STAT: 0 = no stat, 1 = mean, 2 = tol,
% 3 = triangular, 4 = uniform, 5 = gaussian

for i=1:length(OUT_DW1)
    if STD1(i)>=2
        AT(i) = OUT_DW1(i,2) - OUT_DW1(i,1);
        FT(i) = OUT_DW1(i,3) - OUT_DW1(i,1);
    end

    if STD1(i)==1
        mean(i) = (OUT_DW1(i,2) + OUT_DW1(i,3)) ./2;
    elseif STD1(i)==4
        mean(i) = (OUT_DW1(i,1) + OUT_DW1(i,2) + OUT_DW1(i,3)) ./3;
        s2(i) = mean(i).^2./2 - (OUT_DW1(i,1).*OUT_DW1(i,2) + OUT_DW1(i,2) ...
            .*OUT_DW1(i,3) + OUT_DW1(i,1).*OUT_DW1(i,3)) ./6;
    elseif STD1(i)==3
        mean(i) = (OUT_DW1(i,2) + OUT_DW1(i,3)) ./2;
        s2(i) = (OUT_DW1(i,2) - OUT_DW1(i,3)).^2 ./12;
    elseif STD1(i)==5
        mean(i) = (OUT_DW1(i,2) + OUT_DW1(i,3)) ./2;
        s2(i) = (OUT_DW1(i,2) - OUT_DW1(i,3)).^2 ./36;
    end

    if STD1(i)>=3
        AT2(i) = AT(i).^2;
    end
end

s2tot = sum(rmmissing(s2));
AT2tot = sum(rmmissing(AT2));
s2(length(OUT_DW1)-7) = s2tot;
AT2(length(OUT_DW1)-7) = AT2tot;

OUT_DW = num2cell([OUT_DW1,AT,FT,mean,s2,AT2]);
clear AT
clear FT
clear mean
clear s2
clear AT2
clear STD1
clear OUT_DW1



%% POSTPROCESSING - DW CARRIER RECOVERY

AT = NaN(length(OUT_DW2),1);
FT = NaN(length(OUT_DW2),1);
mean = NaN(length(OUT_DW2),1);
s2 = NaN(length(OUT_DW2),1);
AT2 = NaN(length(OUT_DW2),1);

if MOD(2)==1 || MOD(2)==2
    for i=1:(length(OUT_DW2)-2)
        if STD2(i)>=2
            AT(i) = OUT_DW2(i,2) - OUT_DW2(i,1);
            FT(i) = OUT_DW2(i,3) - OUT_DW2(i,1);
        end
    
        if STD2(i)==1
            mean(i) = (OUT_DW2(i,2) + OUT_DW2(i,3)) ./2;
        elseif STD2(i)==4
            mean(i) = (OUT_DW2(i,1) + OUT_DW2(i,2) + OUT_DW2(i,3)) ./3;
            s2(i) = mean(i).^2./2 - (OUT_DW2(i,1).*OUT_DW2(i,2) + OUT_DW2(i,2) ...
                .*OUT_DW2(i,3) + OUT_DW2(i,1).*OUT_DW2(i,3)) ./6;
        elseif STD2(i)==3
            mean(i) = (OUT_DW2(i,2) + OUT_DW2(i,3)) ./2;
            s2(i) = (OUT_DW2(i,2) - OUT_DW2(i,3)).^2 ./12;
        elseif STD2(i)==5
            mean(i) = (OUT_DW2(i,2) + OUT_DW2(i,3)) ./2;
            s2(i) = (OUT_DW2(i,2) - OUT_DW2(i,3)).^2 ./36;
        end
    
        if STD2(i)>=3
            AT2(i) = AT(i).^2;
        end
    end

    s2(length(OUT_DW2)-3) = sum(rmmissing(s2));
    AT2(length(OUT_DW2)-3) = sum(rmmissing(AT2));
    
    s3 = s2tot + s2(length(OUT_DW2)-3);
    RSS = AT2tot + AT2(length(OUT_DW2)-3);
    
    AT2(length(OUT_DW2)-1) = s3;
    AT2(length(OUT_DW2)) = RSS;
    
    OUT_DW2(length(OUT_DW2)-1,1) = mean(length(OUT_DW2)-2) - 3*sqrt(s3);
    OUT_DW2(length(OUT_DW2),1) = OUT_DW2(length(OUT_DW2)-2,1) - sqrt(RSS);
end

OUT_DW2 = num2cell([OUT_DW2,AT,FT,mean,s2,AT2]);
clear AT
clear FT
clear mean
clear s2
clear AT2
clear s3
clear RSS
clear STD2

OUT_DW = [OUT_DW;head;OUT_DW2];
clear OUT_DW2



%% POSTPROCESSING - TM RECOVERY

AT = NaN(length(OUT_DW3),1);
FT = NaN(length(OUT_DW3),1);
mean = NaN(length(OUT_DW3),1);
s2 = NaN(length(OUT_DW3),1);
AT2 = NaN(length(OUT_DW3),1);

for i=1:(length(OUT_DW3)-2)
    if STD3(i)>=2
        AT(i) = OUT_DW3(i,2) - OUT_DW3(i,1);
        FT(i) = OUT_DW3(i,3) - OUT_DW3(i,1);
    end

    if STD3(i)==1
        mean(i) = (OUT_DW3(i,2) + OUT_DW3(i,3)) ./2;
    elseif STD3(i)==4
        mean(i) = (OUT_DW3(i,1) + OUT_DW3(i,2) + OUT_DW3(i,3)) ./3;
        s2(i) = mean(i).^2./2 - (OUT_DW3(i,1).*OUT_DW3(i,2) + OUT_DW3(i,2) ...
            .*OUT_DW3(i,3) + OUT_DW3(i,1).*OUT_DW3(i,3)) ./6;
    elseif STD3(i)==3
        mean(i) = (OUT_DW3(i,2) + OUT_DW3(i,3)) ./2;
        s2(i) = (OUT_DW3(i,2) - OUT_DW3(i,3)).^2 ./12;
    elseif STD3(i)==5
        mean(i) = (OUT_DW3(i,2) + OUT_DW3(i,3)) ./2;
        s2(i) = (OUT_DW3(i,2) - OUT_DW3(i,3)).^2 ./36;
    end

    if STD3(i)>=3
        AT2(i) = AT(i).^2;
    end
end

s2(length(OUT_DW3)-3) = sum(rmmissing(s2));
AT2(length(OUT_DW3)-3) = sum(rmmissing(AT2));

s3 = s2tot + s2(length(OUT_DW3)-3);
RSS = AT2tot + AT2(length(OUT_DW3)-3);

AT2(length(OUT_DW3)-1) = s3;
AT2(length(OUT_DW3)) = RSS;

OUT_DW3(length(OUT_DW3)-1,1) = mean(length(OUT_DW3)-2) - 3*sqrt(s3);
OUT_DW3(length(OUT_DW3),1) = OUT_DW3(length(OUT_DW3)-2,1) - sqrt(RSS);

OUT_DW3 = num2cell([OUT_DW3,AT,FT,mean,s2,AT2]);
clear AT
clear FT
clear mean
clear s2
clear AT2
clear s3
clear RSS
clear STD3

OUT_DW = [OUT_DW;head;OUT_DW3];
clear OUT_DW3



%% POSTPROCESSING - RNG (MAJOR) TONE RECOVERY (DW)

AT = NaN(length(OUT_DW4),1);
FT = NaN(length(OUT_DW4),1);
mean = NaN(length(OUT_DW4),1);
s2 = NaN(length(OUT_DW4),1);
AT2 = NaN(length(OUT_DW4),1);

if MODD(2)==1 && (MOD(1)==1 || MOD(1)==2)
    for i=1:(length(OUT_DW4)-2)
        if STD4(i)>=2
            AT(i) = OUT_DW4(i,2) - OUT_DW4(i,1);
            FT(i) = OUT_DW4(i,3) - OUT_DW4(i,1);
        end
    
        if STD4(i)==1
            mean(i) = (OUT_DW4(i,2) + OUT_DW4(i,3)) ./2;
        elseif STD4(i)==4
            mean(i) = (OUT_DW4(i,1) + OUT_DW4(i,2) + OUT_DW4(i,3)) ./3;
            s2(i) = mean(i).^2./2 - (OUT_DW4(i,1).*OUT_DW4(i,2) + OUT_DW4(i,2) ...
                .*OUT_DW4(i,3) + OUT_DW4(i,1).*OUT_DW4(i,3)) ./6;
        elseif STD4(i)==3
            mean(i) = (OUT_DW4(i,2) + OUT_DW4(i,3)) ./2;
            s2(i) = (OUT_DW4(i,2) - OUT_DW4(i,3)).^2 ./12;
        elseif STD4(i)==5
            mean(i) = (OUT_DW4(i,2) + OUT_DW4(i,3)) ./2;
            s2(i) = (OUT_DW4(i,2) - OUT_DW4(i,3)).^2 ./36;
        end
    
        if STD4(i)>=3
            AT2(i) = AT(i).^2;
        end
    end

    s2(length(OUT_DW4)-3) = sum(rmmissing(s2));
    AT2(length(OUT_DW4)-3) = sum(rmmissing(AT2));
    
    s3 = s2tot + s2(length(OUT_DW4)-3);
    RSS = AT2tot + AT2(length(OUT_DW4)-3);
    
    AT2(length(OUT_DW4)-1) = s3;
    AT2(length(OUT_DW4)) = RSS;
    
    OUT_DW4(length(OUT_DW4)-1,1) = mean(length(OUT_DW4)-2) - 3*sqrt(s3);
    OUT_DW4(length(OUT_DW4),1) = OUT_DW4(length(OUT_DW4)-2,1) - sqrt(RSS);
end

OUT_DW4 = num2cell([OUT_DW4,AT,FT,mean,s2,AT2]);
clear AT
clear FT
clear mean
clear s2
clear AT2
clear s3
clear RSS
clear STD4

OUT_DW = [OUT_DW;head;OUT_DW4];
clear OUT_DW4



%% POSTPROCESSING - RNG MINOR TONE RECOVERY (DW)

AT = NaN(length(OUT_DW5),1);
FT = NaN(length(OUT_DW5),1);
mean = NaN(length(OUT_DW5),1);
s2 = NaN(length(OUT_DW5),1);
AT2 = NaN(length(OUT_DW5),1);

if MOD(3)==2 && MODD(2)==1 && (MOD(1)==1 || MOD(1)==2)
    for i=1:(length(OUT_DW5)-2)
        if STD5(i)>=2
            AT(i) = OUT_DW5(i,2) - OUT_DW5(i,1);
            FT(i) = OUT_DW5(i,3) - OUT_DW5(i,1);
        end
    
        if STD5(i)==1
            mean(i) = (OUT_DW5(i,2) + OUT_DW5(i,3)) ./2;
        elseif STD5(i)==4
            mean(i) = (OUT_DW5(i,1) + OUT_DW5(i,2) + OUT_DW5(i,3)) ./3;
            s2(i) = mean(i).^2./2 - (OUT_DW5(i,1).*OUT_DW5(i,2) + OUT_DW5(i,2) ...
                .*OUT_DW5(i,3) + OUT_DW5(i,1).*OUT_DW5(i,3)) ./6;
        elseif STD5(i)==3
            mean(i) = (OUT_DW5(i,2) + OUT_DW5(i,3)) ./2;
            s2(i) = (OUT_DW5(i,2) - OUT_DW5(i,3)).^2 ./12;
        elseif STD5(i)==5
            mean(i) = (OUT_DW5(i,2) + OUT_DW5(i,3)) ./2;
            s2(i) = (OUT_DW5(i,2) - OUT_DW5(i,3)).^2 ./36;
        end
    
        if STD5(i)>=3
            AT2(i) = AT(i).^2;
        end
    end

    s2(length(OUT_DW5)-3) = sum(rmmissing(s2));
    AT2(length(OUT_DW5)-3) = sum(rmmissing(AT2));
    
    s3 = s2tot + s2(length(OUT_DW5)-3);
    RSS = AT2tot + AT2(length(OUT_DW5)-3);
    
    AT2(length(OUT_DW5)-1) = s3;
    AT2(length(OUT_DW5)) = RSS;
    
    OUT_DW5(length(OUT_DW5)-1,1) = mean(length(OUT_DW5)-2) - 3*sqrt(s3);
    OUT_DW5(length(OUT_DW5),1) = OUT_DW5(length(OUT_DW5)-2,1) - sqrt(RSS);
end

OUT_DW5 = num2cell([OUT_DW5,AT,FT,mean,s2,AT2]);
clear AT
clear FT
clear mean
clear s2
clear AT2
clear s3
clear RSS
clear STD5

OUT_DW = [OUT_DW;head;OUT_DW5];
clear OUT_DW5



%% POSTPROCESSING - RNG MEASUREMENT ACCURACY

AT = NaN(length(OUT_DW6),1);
FT = NaN(length(OUT_DW6),1);
mean = NaN(length(OUT_DW6),1);
s2 = NaN(length(OUT_DW6),1);
AT2 = NaN(length(OUT_DW6),1);

if MOD(4)==1 && MODD(2)==1 && (MOD(1)==1 || MOD(1)==2)
    for i=1:(length(OUT_DW6)-2)
        if STD6(i)>=2
            AT(i) = OUT_DW6(i,2) - OUT_DW6(i,1);
            FT(i) = OUT_DW6(i,3) - OUT_DW6(i,1);
        end
    
        if STD6(i)==1
            mean(i) = (OUT_DW6(i,2) + OUT_DW6(i,3)) ./2;
        elseif STD6(i)==4
            mean(i) = (OUT_DW6(i,1) + OUT_DW6(i,2) + OUT_DW6(i,3)) ./3;
            s2(i) = mean(i).^2./2 - (OUT_DW6(i,1).*OUT_DW6(i,2) + OUT_DW6(i,2) ...
                .*OUT_DW6(i,3) + OUT_DW6(i,1).*OUT_DW6(i,3)) ./6;
        elseif STD6(i)==3
            mean(i) = (OUT_DW6(i,2) + OUT_DW6(i,3)) ./2;
            s2(i) = (OUT_DW6(i,2) - OUT_DW6(i,3)).^2 ./12;
        elseif STD6(i)==5
            mean(i) = (OUT_DW6(i,2) + OUT_DW6(i,3)) ./2;
            s2(i) = (OUT_DW6(i,2) - OUT_DW6(i,3)).^2 ./36;
        end
    
        if STD6(i)>=3
            AT2(i) = AT(i).^2;
        end
    end

    s2(length(OUT_DW6)-3) = sum(rmmissing(s2));
    AT2(length(OUT_DW6)-3) = sum(rmmissing(AT2));
    
    s3 = s2tot + s2(length(OUT_DW6)-3);
    RSS = AT2tot + AT2(length(OUT_DW6)-3);
    
    AT2(length(OUT_DW6)-1) = s3;
    AT2(length(OUT_DW6)) = RSS;
    
    OUT_DW6(length(OUT_DW6)-1,1) = mean(length(OUT_DW6)-2) - 3*sqrt(s3);
    OUT_DW6(length(OUT_DW6),1) = OUT_DW6(length(OUT_DW6)-2,1) - sqrt(RSS);
end

OUT_DW6 = num2cell([OUT_DW6,AT,FT,mean,s2,AT2]);
clear AT
clear FT
clear mean
clear s2
clear AT2
clear s3
clear RSS
clear STD6

OUT_DW = [OUT_DW;head;OUT_DW6];
clear OUT_DW6



%% OUTPUT TRANSCRIPTION

if DWO==0
writecell(OUT_UP,filename,'Sheet',sheet_outup, ...
    'Range','E4','AutoFitWidth',0)
end
writecell(OUT_DW,filename,'Sheet',sheet_outdw, ...
    'Range','E4','AutoFitWidth',0)
winopen C:\Users\hp\Desktop\Link_Budget_Calculator\Link_Budget.xlsx