import numpy as np
import pandas as pd
from glob import glob
import itertools
import time
from datetime import datetime

from Input import create_distribution, print_heat_map_and_orders, compatible_cells, load
from Customer import simulate_customers, selection_customers

from OrTools_solver import ortools_solver
from TabuSearch_solver import TabuSearch


def main():

    # CREA DISTRIBUZIONE PERSONALIZZATA
    flag_create_distribution = False
    if flag_create_distribution:
        # starting time for simulation
        start = time.time()

        seed = 202
        df_distribution, base_info, distribution = create_distribution(num_city=3, print_scatter=True,
                                                                       print_heat_map=True, seed=seed)
        np.random.seed(seed)
        customers_simulated = simulate_customers(df=df_distribution, num_customer=200,
                                                 length=base_info['length'], height=base_info['height'],
                                                 current_day=0, num_previous_customers=0, num_days=5)

        print_heat_map_and_orders(base_info, distribution, customers_simulated)

        # ending time for simulation
        end = time.time()
        time_str = time.strftime("%H:%M:%S", time.gmtime(end - start))
        print('Time for simulation: ' + time_str + '\n')

    # FINETUNING TABUSEARCH
    flag_finetuning_tabusearch = False
    if flag_finetuning_tabusearch:
        # starting time for simulation
        start = time.time()
        path = "./InputCVRP/Vrp-Set-A/"
        instance_name = "A-n63-k10.vrp"
        path_output = './OutputFile/'
        file_name_output = 'finetuning_tabusearch.csv'
        df = pd.read_csv(path_output + file_name_output)
        seed = 100
        customers_imported, base_info = load(path, instance_name)
        for n_iters in range(1000, 10001, 1000):
            for tabu_size in range(20, 161, 20):
                start_instance_ts = datetime.now()
                ts = TabuSearch(customers_imported, base_info)
                objective_value_ts = ts.search(n_iters=n_iters, tabu_size=tabu_size, print_solution=False, seed=seed)
                end_instance_ts = datetime.now()
                perc_worst_ts = (objective_value_ts - base_info['best_known_solution']) / base_info[
                    'best_known_solution'] * 100
                df = df.append({'path': path,
                                'instance': instance_name,
                                'n_iters': n_iters,
                                'tabu_size': tabu_size,
                                'datetime': datetime.now(),
                                'objective_real': base_info['best_known_solution'],
                                'objective_value': objective_value_ts,
                                'perc': perc_worst_ts,
                                'duration': end_instance_ts - start_instance_ts}, ignore_index=True)
                print(n_iters, tabu_size, objective_value_ts)

        # ending time for simulation
        end = time.time()
        time_str = time.strftime("%H:%M:%S", time.gmtime(end - start))
        print('Time for simulation: ' + time_str + '\n')
        df.to_csv(path_output + file_name_output, encoding='utf-8', index=False)

    # CONFRONTO TRA ALGORITMI
    flag_confronto_algoritmi = False
    if flag_confronto_algoritmi:
        # starting time for simulation
        start = time.time()
        path = "./InputCVRP/Vrp-Set-A/"
        path_output = './OutputFile/'
        file_name_output = 'output.csv'
        df = pd.read_csv(path_output + file_name_output)
        n_iters = 5000
        tabu_size = 120
        seed = 100

        for f in glob(path + "*.vrp"):
            instance_name = f.split("\\")[1]
            customers_imported, base_info = load(path, instance_name)

            start_instance_ortools = datetime.now()
            objective_value_ortools, _ = ortools_solver(customers_imported[1:], base_info, printSolution=False)
            end_instance_ortools = datetime.now()

            start_instance_ts = datetime.now()
            ts = TabuSearch(customers_imported, base_info)
            objective_value_ts = ts.search(n_iters=n_iters, tabu_size=tabu_size, print_solution=False, seed=seed)
            end_instance_ts = datetime.now()

            perc_worst_ortools = (objective_value_ortools - base_info['best_known_solution']) / base_info[
                'best_known_solution'] * 100
            perc_worst_ts = (objective_value_ts - base_info['best_known_solution']) / base_info[
                'best_known_solution'] * 100

            print("solution_ortools    = " + str(objective_value_ortools))
            print("solution_ts         = " + str(objective_value_ts))
            print("best_known_solution = " + str(base_info['best_known_solution']))
            print("perc_worst_ortools  = " + str(perc_worst_ortools)[:5] + str(" %"))
            print("perc_worst_ts       = " + str(perc_worst_ts)[:5] + str(" %"))
            time_str_ortools = str(end_instance_ortools - start_instance_ortools)[:-3]
            time_str_ts = str(end_instance_ts - start_instance_ts)[:-3]
            print('Time for ortools    = ' + time_str_ortools)
            print('Time for TabuSearch = ' + time_str_ts + '\n')
            df = df.append({'path': path,
                            'instance': instance_name,
                            'datetime': datetime.now(),
                            'objective_real': base_info['best_known_solution'],
                            'objective_value': objective_value_ortools,
                            'perc': perc_worst_ortools,
                            'duration': end_instance_ortools - start_instance_ortools,
                            'tool': 'ortools',
                            'n_iters': None,
                            'tabu_size': None,
                            'seed': seed}, ignore_index=True)
            df = df.append({'path': path,
                            'instance': instance_name,
                            'datetime': datetime.now(),
                            'objective_real': base_info['best_known_solution'],
                            'objective_value': objective_value_ts,
                            'perc': perc_worst_ts,
                            'duration': end_instance_ts - start_instance_ts,
                            'tool': 'TabuSearch',
                            'n_iters': n_iters,
                            'tabu_size': tabu_size,
                            'seed': seed}, ignore_index=True)

        # ending time for simulation
        end = time.time()
        time_str = time.strftime("%H:%M:%S", time.gmtime(end - start))
        print('Time for simulation: ' + time_str + '\n')
        df.to_csv(path_output + file_name_output, encoding='utf-8', index=False)

    # FINETUNING NEIGHBOUR POLICY
    flag_finetuning_neighbour_policy = False
    if flag_finetuning_neighbour_policy:
        # starting time for simulation
        start = time.time()
        path_output = './OutputFile/'
        file_name_output = 'finetuning_neighbour_policy.csv'
        df_output = pd.read_csv(path_output + file_name_output)
        num_days = 7
        seed = 202
        print_heat_map = False
        # creazione distribuzione
        df_distribution, base_info, distribution = create_distribution(print_scatter=True, print_heat_map=True,
                                                                       seed=seed)
        list_M = [4, 4.5, 5]
        list_gamma = [0.3, 0.5, 0.7, 0.9]
        list_rho = [0.1, 0.15, 0.2, 0.25, 0.3]
        list_of_lists = [list_M, list_gamma, list_rho]
        for comb in itertools.product(*list_of_lists):
            M = comb[0]
            gamma = comb[1]
            rho = comb[2]
            start_comb = datetime.now()
            # calcolo degli indici di compatibilità tra le celle
            _, list_compatible_cells = compatible_cells(df_distribution, base_info, rho)

            solver = 'ortools'
            policy = 'neighbour'

            np.random.seed(seed)
            customers = pd.DataFrame()
            num_previous_customers = 0
            objective_value_sum = 0
            for day in range(num_days):
                # simulazione clienti del giorno "day"
                if day < num_days - 2:
                    num_new_customers = 200
                else:  # gli ultimi due giorni non vengono generati nuovi clienti
                    num_new_customers = 0

                customers_simulated = simulate_customers(df=df_distribution, num_customer=num_new_customers,
                                                         length=base_info['length'], height=base_info['height'],
                                                         current_day=day, num_days=num_days,
                                                         num_previous_customers=num_previous_customers)
                # aggiunta dei clienti nuovi a quelli vecchi
                customers = pd.concat([customers, customers_simulated], ignore_index=True)
                num_previous_customers += num_new_customers

                # stampa heatmap degli ordini aperti in questo istante
                if print_heat_map:
                    print_heat_map_and_orders(base_info, distribution, customers)

                # scelta dei clienti da servire (ultimo giorno tutti clienti serviti)
                df_customer_selected, df_postponed = selection_customers(customers, day, num_days=num_days,
                                                                         policy=policy,
                                                                         list_compatible_cells=list_compatible_cells,
                                                                         cell_probabilities=df_distribution[
                                                                             'probability'],
                                                                         seed=seed, M=M, gamma=gamma)
                # itera fino a quando trovo un risultato, se non lo trova postpone un cliente
                objective_value = None
                while objective_value is None:
                    # ottimizzazione del percorso sui clienti scelti
                    objective_value, _ = ortools_solver(df_customer_selected, base_info, printSolution=False,
                                                        time=1)
                    if objective_value is None:
                        df_postponed = pd.concat([df_customer_selected[-1:], df_postponed])
                        df_customer_selected = df_customer_selected[:-1]
                objective_value_sum += objective_value

                # segna yet_postponed i vecchi clienti così dopo hanno priorità nella consegna
                df_postponed['yet_postponed'] = True
                # salva come clienti per il giorno successivo quelli posticipati in questo day
                customers = df_postponed
            end_comb = datetime.now()
            df_output = df_output.append({'seed': seed, 'M': M, 'gamma': gamma, 'rho': rho,
                                          'objective_value': objective_value_sum,
                                          'datetime': datetime.now(),
                                          'duration': end_comb - start_comb}, ignore_index=True)
            print('seed: ', seed, 'M:', M, 'gamma: ', gamma, 'rho: ', rho,
                  'objective_value: ' + str(objective_value_sum))
            # ending time for simulation
        end = time.time()
        time_str = time.strftime("%H:%M:%S", time.gmtime(end - start))
        print('Time for simulation: ' + time_str + '\n')
        df_output.to_csv(path_output + file_name_output, encoding='utf-8', index=False)

    # CONFRONTO TRA POLICY DI SELEZIONE DEI CLIENTI (SIMULAZIONE SU PIU' GIORNI CONSECUTIVI)
    flag_simulazione_piu_giorni = False
    if flag_simulazione_piu_giorni:
        # starting time for simulation
        start = time.time()
        path_output = './OutputFile/'
        file_name_output = 'confronto_policy.csv'
        df_output = pd.read_csv(path_output + file_name_output)
        num_days = 7
        seed = 15
        # creazione distribuzione
        df_distribution, base_info, distribution = create_distribution(print_scatter=True, print_heat_map=True,
                                                                       seed=seed)
        rho = 0.3
        M = 4
        gamma = 0.3
        # calcolo degli indici di compatibilità tra le celle
        _, list_compatible_cells = compatible_cells(df_distribution, base_info, rho)

        solver = 'ortools'
        n_iters = 5000
        tabu_size = 120
        objective_value_random = None
        print(solver)
        for policy in ['random']:
            start_comb = datetime.now()
            np.random.seed(seed)
            customers = pd.DataFrame()
            num_previous_customers = 0
            objective_value_sum = 0
            num_overdue = 0
            for day in range(num_days):
                # simulazione clienti del giorno "day"
                if day < num_days - 2:
                    num_new_customers = 200
                else:  # gli ultimi due giorni non vengono generati nuovi clienti
                    num_new_customers = 0

                customers_simulated = simulate_customers(df=df_distribution, num_customer=num_new_customers,
                                                         length=base_info['length'], height=base_info['height'],
                                                         current_day=day, num_days=num_days,
                                                         num_previous_customers=num_previous_customers)
                # aggiunta dei clienti nuovi a quelli vecchi
                customers = pd.concat([customers, customers_simulated], ignore_index=True)
                num_previous_customers += num_new_customers

                # stampa heatmap degli ordini aperti in questo istante
                print_heat_map_and_orders(base_info, distribution, customers, day + 1)

                # scelta dei clienti da servire (ultimo giorno tutti clienti serviti)
                df_customer_selected, df_postponed = selection_customers(customers, day, num_days=num_days,
                                                                         policy=policy,
                                                                         list_compatible_cells=list_compatible_cells,
                                                                         cell_probabilities=df_distribution[
                                                                             'probability'],
                                                                         seed=seed, M=M, gamma=gamma)
                # calcola il numero di clienti la cui consegna è avvenuta dopo la data massima di consegna
                num_overdue += sum(day > df_customer_selected['last_day'])
                # stampa il numero di clienti dell'ultimo giorno
                if day == num_days - 1:
                    print('num_clienti_last_day: ', str(len(df_customer_selected)))

                # itera fino a quando trovo un risultato, se non lo trova postpone un cliente
                objective_value = None
                while objective_value is None:
                    # ottimizzazione del percorso sui clienti scelti
                    if solver == 'ortools':
                        objective_value, _ = ortools_solver(df_customer_selected, base_info, printSolution=False,
                                                            time=3)
                    elif solver == 'TabuSearch':
                        ts = TabuSearch(df_customer_selected, base_info)
                        objective_value = ts.search(n_iters=n_iters, tabu_size=tabu_size, print_solution=False,
                                                    seed=seed)
                    else:
                        raise Exception('The solver has to be ortools or TabuSearch')
                    if objective_value is None:
                        df_postponed = pd.concat([df_customer_selected[-1:], df_postponed])
                        df_customer_selected = df_customer_selected[:-1]
                objective_value_sum += objective_value

                # segna yet_postponed i vecchi clienti così dopo hanno priorità nella consegna
                df_postponed['yet_postponed'] = True
                # salva come clienti per il giorno successivo quelli posticipati in questo day
                customers = df_postponed
            end_comb = datetime.now()
            if policy == 'random':
                objective_value_random = objective_value_sum
            compared_random = (objective_value_random - objective_value_sum) / objective_value_sum
            df_output = df_output.append(
                {'solver': solver, 'policy': policy, 'seed': seed, 'M': M, 'gamma': gamma, 'rho': rho,
                 'objective_value': objective_value_sum, 'num_overdue': num_overdue,
                 'compared_random': compared_random,
                 'datetime': datetime.now(),
                 'duration': end_comb - start_comb}, ignore_index=True)
            print('num_overdue: ', num_overdue)
            print('Policy:', policy, 'objective_value_sum: ' + str(round(objective_value_sum, 1)))

            print('Compared to random: ', compared_random)
        # ending time for simulation
        end = time.time()
        time_str = time.strftime("%H:%M:%S", time.gmtime(end - start))
        print('Time for simulation: ' + time_str + '\n')
        df_output.to_csv(path_output + file_name_output, encoding='utf-8', index=False)

    flag_final = True
    if flag_final:
        # starting time for simulation
        start = time.time()
        num_days = 7
        seed = 15
        # creazione distribuzione
        df_distribution, base_info, distribution = create_distribution(print_scatter=True, print_heat_map=True,
                                                                       seed=seed)
        rho = 0.3
        M = 4
        gamma = 0.3
        # calcolo degli indici di compatibilità tra le celle
        _, list_compatible_cells = compatible_cells(df_distribution, base_info, rho)

        solver = 'ortools'
        policy = 'neighbour'
        np.random.seed(seed)
        customers = pd.DataFrame()
        num_previous_customers = 0
        objective_value_sum = 0
        num_overdue = 0
        for day in range(num_days):
            # simulazione clienti del giorno "day"
            if day < num_days - 2:
                num_new_customers = 200
            else:  # gli ultimi due giorni non vengono generati nuovi clienti
                num_new_customers = 0

            customers_simulated = simulate_customers(df=df_distribution, num_customer=num_new_customers,
                                                     length=base_info['length'], height=base_info['height'],
                                                     current_day=day, num_days=num_days,
                                                     num_previous_customers=num_previous_customers)
            # aggiunta dei clienti nuovi a quelli vecchi
            customers = pd.concat([customers, customers_simulated], ignore_index=True)
            num_previous_customers += num_new_customers

            # stampa heatmap degli ordini aperti in questo istante
            print_heat_map_and_orders(base_info, distribution, customers, day + 1)

            # scelta dei clienti da servire (ultimo giorno tutti clienti serviti)
            df_customer_selected, df_postponed = selection_customers(customers, day, num_days=num_days,
                                                                     policy=policy,
                                                                     list_compatible_cells=list_compatible_cells,
                                                                     cell_probabilities=df_distribution['probability'],
                                                                     seed=seed, M=M, gamma=gamma)
            # calcola il numero di clienti la cui consegna è avvenuta dopo la data massima di consegna
            num_overdue += sum(day > df_customer_selected['last_day'])

            # itera fino a quando trovo un risultato, se non lo trova postpone un cliente
            objective_value = None
            while objective_value is None:
                # ottimizzazione del percorso sui clienti scelti
                objective_value, _ = ortools_solver(df_customer_selected, base_info, printSolution=False,
                                                    time=3)
                if objective_value is None:
                    df_postponed = pd.concat([df_customer_selected[-1:], df_postponed])
                    df_customer_selected = df_customer_selected[:-1]
            objective_value_sum += objective_value

            # segna yet_postponed i vecchi clienti così dopo hanno priorità nella consegna
            df_postponed['yet_postponed'] = True
            # salva come clienti per il giorno successivo quelli posticipati in questo day
            customers = df_postponed
        print('num_overdue: ', num_overdue)
        print('objective_value_sum: ' + str(round(objective_value_sum, 1)))

        # ending time for simulation
        end = time.time()
        time_str = time.strftime("%H:%M:%S", time.gmtime(end - start))
        print('Time for simulation: ' + time_str + '\n')


if __name__ == '__main__':
    main()
