"""
Class for customer

This class contains the class for customers and all the functions needed to simulate and select customers
"""
import numpy as np
from numpy.random import multinomial
import pandas as pd
import constants


class Customer:
    def __init__(self, id: int, demand: int, service_time: int, coord_x: float, coord_y: float):
        """Construction of class Customer.

        Args:
          id: number of the customer.
          demand: customer's demand (kg).
          service_time: customer's service time (min).
          coord_x: customer's position coordinate x.
          coord_y: customer's position coordinate y.
        """
        self.id = id
        self.demand = demand
        self.service_time = service_time
        self.coord_x = coord_x
        self.coord_y = coord_y

    def __str__(self):
        return str(self.__class__) + ": " + str(self.__dict__)


def simulate_customers(df: pd.DataFrame, num_customer: int, length: float, height: float,
                       current_day: int, num_previous_customers: int, num_days: int) -> pd.DataFrame:
    distribution = df['probability']
    demand = multinomial(num_customer, distribution)

    cells_with_demand = df[demand > 0].copy()
    cells_with_demand['demands'] = demand[demand > 0]

    new_orders = {'x': [], 'y': [], 'kg': [], 'service_time': [],
                  'last_day': [], 'yet_postponed': [], 'cell_id': [], 'index': []}

    for _, cell in cells_with_demand.iterrows():
        demand_cell = int(cell['demands'])
        new_orders['cell_id'] += [int(cell['id'])] * demand_cell
        new_orders['x'] += np.random.uniform(cell['x'], cell['x'] + length, demand_cell).tolist()
        new_orders['y'] += np.random.uniform(cell['y'], cell['y'] + height, demand_cell).tolist()
        new_orders['last_day'] += np.random.randint(low=current_day + constants.MIN_DAY,
                                                    high=min(current_day + constants.MAX_DAY, num_days),
                                                    size=demand_cell).tolist()
        new_orders['yet_postponed'] += [False] * demand_cell
        new_orders['service_time'] += np.random.randint(low=constants.MIN_SERVICE_TIME,
                                                        high=constants.MAX_SERVICE_TIME,
                                                        size=demand_cell).tolist()
        new_orders['kg'] += np.random.randint(low=constants.MIN_KG,
                                              high=constants.MAX_KG,
                                              size=demand_cell).tolist()
        new_orders['index'] += list(range(num_previous_customers, num_previous_customers + demand_cell))
        num_previous_customers += demand_cell
    # se il numero di clienti da generare è zero restituisco un dataset vuoto
    if num_customer != 0:
        df_return = pd.DataFrame(new_orders)
    else:
        df_return = pd.DataFrame()
    return df_return


def selection_customers(df: pd.DataFrame, day: int, num_days: int, policy: str = 'random',
                        list_compatible_cells: pd.DataFrame = None, cell_probabilities: pd.Series = None,
                        seed: int = None, M: float = None, gamma: float = None):
    if day in (0, 1):
        num_max_customers = 150
    else:
        num_max_customers = 200

    if policy == 'random':
        df_customer_selected, df_postponed = _random_customer_selection(df, day, num_days, num_max_customers, seed)
    elif policy == 'asap':  # as soon as possible
        df_customer_selected, df_postponed = _asap_customer_selection(df, day, num_days, num_max_customers)
    elif policy == 'asap_2':  # as soon as possible 2
        df_customer_selected, df_postponed = _asap_2_customer_selection(df, day, num_days, num_max_customers)
    elif policy == 'alap':  # as late as possibile
        df_customer_selected, df_postponed = _alap_customer_selection(df, day, num_days, num_max_customers)
    elif policy == 'neighbour':
        df_customer_selected, df_postponed = _neighbour_customer_selection(df, day, num_days, num_max_customers,
                                                                           list_compatible_cells, cell_probabilities, M,
                                                                           gamma)
    else:
        raise Exception("Wrong policy name")

    return df_customer_selected, df_postponed


def _random_customer_selection(df: pd.DataFrame, day: int, num_days: int, num_max_customers: int, seed: int = None):

    # selezione clienti all'ultimo giorno o già scaduti
    df_last_day = df[df['last_day'] <= day]
    df_not_last_day = df[df['last_day'] > day]

    # tra i clienti non all'ultimo giorno ne prendo il numero giusto per completare il numero prefissato
    # ho aggiungo max(num, 0) nel caso il numero di clienti all'ultimo giorno sia maggiore del numero massimo di clienti
    # selezionabili ogni giorno
    df_sample = df_not_last_day.sample(n=max(min(num_max_customers-len(df_last_day), len(df_not_last_day)), 0),
                                       random_state=seed)
    # concateno i clienti all'ultimo giorno e quelli campionati
    df_customer_selected = pd.concat([df_last_day, df_sample])
    if day != num_days-1:
        # prende solo il numero massimo di clienti prendibili (rischiando di far scadere alcuni clienti)
        df_customer_selected = df_customer_selected[0:num_max_customers]
    else:
        # l'ultimo giorno vengono serviti tutti i clienti
        df_customer_selected = df
    # estraggo i clienti non selezionati e quindi postposti
    df_postponed = df[~df['index'].isin(df_customer_selected['index'])].copy()
    return df_customer_selected, df_postponed


def _asap_customer_selection(df: pd.DataFrame, day: int, num_days: int, num_max_customers: int):
    # ordina i clienti in base all'ultimo giorno di consegna disponibile e se sono già stati postposti o no
    df = df.sort_values(by=['last_day', 'yet_postponed'], ascending=[True, False])
    if day != num_days-1:
        # prende solo il numero massimo di clienti selezionabili
        df_customer_selected = df[0:num_max_customers]
    else:
        # l'ultimo giorno vengono serviti tutti i clienti
        df_customer_selected = df
    # estraggo i clienti non selezionati e quindi postposti
    df_postponed = df[~df['index'].isin(df_customer_selected['index'])]

    return df_customer_selected, df_postponed


def _asap_2_customer_selection(df: pd.DataFrame, day: int, num_days: int, num_max_customers: int):
    # ordina i clienti in base a se sono già stati postposti o no e all'ultimo giorno di consegna disponibile
    df = df.sort_values(by=['yet_postponed', 'last_day'], ascending=[False, True])
    if day != num_days-1:
        # prende solo il numero massimo di clienti selezionabili
        df_customer_selected = df[0:num_max_customers]
    else:
        # l'ultimo giorno vengono serviti tutti i clienti
        df_customer_selected = df
    # estraggo i clienti non selezionati e quindi postposti
    df_postponed = df[~df['index'].isin(df_customer_selected['index'])]

    return df_customer_selected, df_postponed


def _alap_customer_selection(df: pd.DataFrame, day: int, num_days: int, num_max_customers: int):
    # ordina i clienti in base a se sono già stati postposti o no
    df = df.sort_values(by=['yet_postponed'], ascending=[False])
    # selezione clienti all'ultimo giorno o già "scaduti"
    df_customer_selected = df[df['last_day'] <= day]

    if day != num_days-1:
        # prende solo il numero massimo di clienti selezionabili (prende prima quelli già postposti)
        df_customer_selected = df_customer_selected[0:num_max_customers]
    else:
        # l'ultimo giorno vengono serviti tutti i clienti
        df_customer_selected = df
    # estraggo i clienti non selezionati e quindi postposti
    df_postponed = df[~df['index'].isin(df_customer_selected['index'])]

    return df_customer_selected, df_postponed


def _neighbour_customer_selection(df: pd.DataFrame, day: int, num_days: int, num_max_customers: int,
                                  list_compatible_cells, cell_probabilities, M, gamma):
    # insieme delle celle con un cliente da servire
    cells_with_orders = set(df['cell_id'])
    # calcolo dell'indice di compatibilità della cella
    df = df.apply(lambda row: _index_compatibility_cell(row, day, list_compatible_cells[int(row['cell_id'])],
                                                        cells_with_orders, cell_probabilities, M, gamma), axis=1)
    # ordine i clienti da servire secondo l'indice di compatibilità
    df = df.sort_values(by=['index_compatibility'], ascending=[False])

    if day != num_days-1:
        # prende solo il numero massimo di clienti selezionabili
        df_customer_selected = df[0:num_max_customers]
    else:
        # l'ultimo giorno vengono serviti tutti i clienti
        df_customer_selected = df

    # estraggo i clienti non selezionati e quindi postposti
    df_postponed = df[~df['index'].isin(df_customer_selected['index'])]

    return df_customer_selected, df_postponed


def _index_compatibility_cell(row, day, compatible_cells, cells_orders, cell_probabilities, M, gamma):
    # numero di giorni che rimangono per la consegna
    time_distance = row['last_day'] - day
    # celle compatibili che al momento non hanno ordini attivi
    compat_no_orders_cells = set(compatible_cells).difference(cells_orders)
    # se oggi è l'ultimo giorno disponibile per la consegna o è già passato
    if time_distance <= 0:
        # suddivisione della casistica di ordini già postposti e nuovi
        if row['yet_postponed']:
            index_compatibility = M + gamma
        else:
            index_compatibility = M
    else:  # se oggi non è l'ultimo giorno per la consegna
        num_cells_no_orders = len(compat_no_orders_cells)
        # se tutte le celle compatibili hanno ordini attivi
        if num_cells_no_orders == 0:
            index_compatibility = M/time_distance
        # se alcune celle compatibili non hanno ordini attivi
        else:
            index_compatibility = 1/time_distance*(1+1/num_cells_no_orders*(num_cells_no_orders-cell_probabilities[list(compat_no_orders_cells)].sum()))
    # aggiunge l'indice di compatibilità alla riga
    row['index_compatibility'] = index_compatibility

    return row
