function k = menu(menuHeader,varargin);
% TXTMENU - text-menu of choices for user input in the command window.
%
% This function was inspired by the MENU.M function by The MathWorks, Inc. It 
% differs from MENU.M in that it will always use the command-window regardless of 
% the graphical capabilities of the workstation. In effect, it will mimic the 
% behavior of the MENU command from older Matlab versions, which did not yet
% provide support for graphical user-interfaces.
%
% Usage:
% ======
% CHOICE = txtmenu(HEADER, ITEM1, ITEM2, ... ) displays the HEADER string 
%   followed in sequence by the menu-item strings: ITEM1, ITEM2, ...
%   ... , ITEMn. Returns the number of the selected menu-item into the
%   scalar variable CHOICE. There is no upper limit to the total number of 
%   menu items.
%
% CHOICE = txtmenu(HEADER, ITEMLIST) where ITEMLIST is a string cell
%   array is also a valid syntax.
%
% Example:
% ========
% K = txtmenu('Choose a color','Red','Blue','Green')
% 
% displays on the screen:
%
%     Choose a color:
%
%        1) Red
%        2) Blue
%        3) Green
%
%     Select a menu number:
%
% The number entered by the user in response to the prompt is returned as K 
% (e.g. K = 2 implies that the user selected Blue from this example menu).
%
% See also MENU.

% Variables
% =========
% k            number corresponding with selected menu item
% menuHeader   string variable, containing the header (title) of the text-menu
% menuItems    cell array, containing the individual menu items 
% numItems     total number of menu items


% Check input. If the number of inputs is zero, display help text for this 
% function. If the number of inputs is one, display a warning message.
% ------------------------------------------------------------------------
if nargin==0
  helpwin txtmenu
elseif nargin==1
  disp('TXTMENU: No menu items to chose from.')
  k=0;
  return;
elseif nargin==2 & iscell(varargin{1}),
  menuItems = varargin{1}; % a cell array was passed in
else,
  menuItems = varargin;    % use the varargin cell array
end


% Display an ascii-based menu in the command window, and return the user's 
% selection from that menu as an index into the menuItems cell array, k.
% ------------------------------------------------------------------------

% Calculate the number of items in the menu
% -----------------------------------------
numItems = length(menuItems);

% Continuous loop to redisplay menu until the user makes a valid choice
% ---------------------------------------------------------------------
while 1,

    % Display the menu-header
    % -----------------------
    disp(' ');
    disp([menuHeader,':']);
    disp(' ');

    % Display items in a numbered list
    % --------------------------------
    for n = 1 : numItems
        disp( [ '      ' int2str(n) ') ' menuItems{n} ] );
    end
    disp(' ');
    
    % Prompt for user input
    % ---------------------
    k = input('Select a menu number: ');
    
    % Make sure k has a value
    % -----------------------
    if isempty(k)
        k = -1;
    end
    
    % Make sure the value of k is valid
    % ---------------------------------
    if  (k < 1) | (k > numItems) ...
        | ~strcmp(class(k),'double') ...
        | ~isreal(k) | (isnan(k)) | isinf(k)
        
        % Failed a key test; ask question again
        % -------------------------------------
        disp(' ');
        disp('Selection out of range. Try again.');
        
    else
        % Passed all tests, exit loop and return k
        % ----------------------------------------
        return
    end
    
end

%-----------------------------------------------------------------------
% The Flight Dynamics and Control Toolbox version 1.4.0. 
% (c) Copyright Marc Rauw, 1994-2005. Licensed under the Open Software 
% License version 2.1; see COPYING.TXT and LICENSE.TXT for more details.
% Last revision of this file: May 22, 2005. 
%-----------------------------------------------------------------------