function dlg_handle = newMsgBox(msg_text,dlg_title,dlg_font)
% newMsgBox - message-box
%
% This function is an alternative for MSGBOX, used to display short messages
% in a dialog window. Although it is somewhat less sophisticated than MSGBOX,
% its readability is better.
%
% Usage:
% ======
% newMsgBox(Message) creates a message box that automatically wraps
%   Message to fit an appropriately sized dialog window. Message is a
%   string vector, string matrix or cell array.
%
% newMsgBox(Message,Title) specifies the title of the message box.
%
% newMsgBox(Message,Title,Font) specifies the title and the font-settings
%   for the message box. Font is a structure that contains the font-definitions.
%   Valid fields are (default values for newMsgBox are displayed in curly braces):
%
%   Font.FontAngle  [{normal} |italic | oblique]
%   Font.FontUnits  [inches | centimeters | normalized | {points} | pixels | data]
%   Font.FontWeight [light | normal | demi | {bold}]
%   Font.FontName   requested font-name | {FactoryUIControlFontName}
%   Font.FontSize   requested font-size | {FactoryUIControlFontSize}
%
% Contrary to MSGBOX, it is not possible to specify an icon. It is also not
% possible to specify the window style and interpreter; use MSBOX if those
% options are required.
%
% See also MSGBOX.

% Variables
% =========
% dlg_font      font for the text in the dialog-window and the OK button
% dlg_handle    handle for the dialog window
% dlg_height    height of the dialog window
% dlg_offset    general offset-value for the dialog box, used as reference for the
%                 other offset-values
% dlg_title     title for the dialog window
% dlg_width     width of the dialog window
% msg_handle    handle for the text-message
% msg_height    height of the formatted message-text
% msg_text      string vector, string matrix, or cell array, containing the
%                 message-text
% msg_width     width of the formatted message-text
% msg_x_offset  amount of free space between left dialog boundary and left side of
%                 the message-text
% msg_y_offset  amount of free space between bottom dialog boundary and bottom side of
%                 the message-text
% new_msg_pos   vector containing information about the size of the dialog window and
%                 message-text, necessary to make the message fit (see TEXTWRAP for
%                 more information)
% ok_handle     handle for the OK-button
% ok_height     height of the OK-button
% ok_width      width of the OK-button
% ok_x_offset   amount of free space between left dialog boundary and left side of
%                 the OK button (= space between right dialog boundary and right side
%                 of the OK button)
% ok_y_offset   amount of free space between bottom dialog boundary and bottom side of
%                 the OK button
% screen_size   screen-size
% wrap_string   version of msg_txt that has been fitted to the dialog-window
% x_dlg         x-coordinate of lower left corner of dialog window
% y_dlg         y-coordinate of lower left corner of dialog window
%

% If function is called without input arguments, display help-text.
% With one input only, use default title for the dialog window.
% -----------------------------------------------------------------
if nargin == 0
    helpwin newMsgBox;
    return;
elseif nargin == 1
    dlg_title = 'Message';
end

% If function is called with less than three input-arguments, use default font
% properties for dialog message and OK button
% ----------------------------------------------------------------------------
if nargin < 3
    dlg_font.FontUnits  = 'points';
    dlg_font.FontName   = get(0,'FactoryUIControlFontName');
    dlg_font.FontSize   = get(0,'FactoryUIControlFontSize');
    dlg_font.FontWeight = 'bold';
    dlg_font.FontAngle  = 'normal';
end

% Convert input argument to cell-array if it isn't one already
% ------------------------------------------------------------
if ~iscell(msg_text)
    msg_text = cellstr(msg_text);
end

% Determine screensize for figure positioning in pixels
% -----------------------------------------------------
screen_size = screensize('characters');

% Size of the dialog-window, expressed in character-units (height will be adjusted later)
% ---------------------------------------------------------------------------------------
dlg_width  = 70;
dlg_height = 8;

% Default position data of message and OK button, expressed in character-units
% ----------------------------------------------------------------------------
ok_width     = 14;
ok_height    = 2.2;
dlg_offset   = 1.2;

ok_x_offset  = floor(0.5*(dlg_width-ok_width));
ok_y_offset  = dlg_offset;
msg_x_offset = 2*dlg_offset;
msg_y_offset = dlg_offset + ok_y_offset + ok_height;

msg_width    = dlg_width - 2*dlg_offset;
msg_height   = dlg_height - dlg_offset - msg_y_offset;

% Initial position of dialog window: centered on the screen. The variables x_dlg and y_dlg
% are used to store the (x,y) coordinate of the lower left corner of the dialog window.
% ----------------------------------------------------------------------------------------
x_dlg = floor(0.5*(screen_size(3)-dlg_width));
y_dlg = floor(0.5*(screen_size(4)-dlg_height));

% Draw dialog window
% ------------------
dlg_handle = dialog('Units','characters', ...
    'InvertHardcopy','off', ...
    'KeyPressFcn',['AsciiVal= abs(get(gcbf,''CurrentCharacter''));  ', ...
                   'if ~isempty(AsciiVal),                          ', ...
                   '  if AsciiVal==32 | AsciiVal==13,               ', ...
                   '    delete(get(0,''CurrentFigure''));           ', ...
                   '  end;                                          ', ...
                   'end                                             '], ...
    'MenuBar','none', ...
    'Name',dlg_title, ...
    'Pointer','arrow', ...
    'Position',[x_dlg y_dlg dlg_width dlg_height], ...
    'Tag','dialog_window');

% Draw text-field for message (used to fit text with textwrap)
% ------------------------------------------------------------
msg_handle = uicontrol(dlg_handle, dlg_font, ...
    'Style','text', ...
    'Units','characters', ...
    'Position',[msg_x_offset  msg_y_offset  msg_width  msg_height], ...
    'String',' ', ...
    'Tag','msg_box', ...
    'HorizontalAlignment','left');

% Determine required width and height of the message
% --------------------------------------------------
[wrap_string,new_msg_pos] = textwrap(msg_handle,msg_text,dlg_width-4);

% Resize dialog-window and text-field if necessary; update dialog-window position
% -------------------------------------------------------------------------------
msg_width   = max(msg_width, ceil(new_msg_pos(3)));
msg_height  = max(msg_height, ceil(new_msg_pos(4)));
dlg_width   = max(dlg_width, ceil(new_msg_pos(3)) + msg_x_offset);
dlg_height  = max(dlg_height, msg_height + dlg_offset + msg_y_offset);
x_dlg       = floor(0.5*(screen_size(3)-dlg_width));
y_dlg       = floor(0.5*(screen_size(4)-dlg_height));

set(dlg_handle,'Position',[x_dlg y_dlg dlg_width dlg_height]);
set(msg_handle,'Position',[msg_x_offset msg_y_offset msg_width msg_height]);

% Update position data for OK button
% ----------------------------------
ok_x_offset = floor(0.5*(dlg_width-ok_width));

% Draw message
% ------------
set(msg_handle,'String',wrap_string);

% Draw OK-button
% --------------
ok_handle = uicontrol(dlg_handle, dlg_font, ...
    'Style','pushbutton', ...
    'Units','characters', ...
    'Position',[ok_x_offset  ok_y_offset  ok_width  ok_height], ...
    'CallBack','delete(get(0,''CurrentFigure''))', ...
    'String','OK', ...
    'HorizontalAlignment','center', ...
    'Tag','ok_button');

%-----------------------------------------------------------------------
% The Flight Dynamics and Control Toolbox version 1.4.0. 
% (c) Copyright Marc Rauw, 1994-2005. Licensed under the Open Software 
% License version 2.1; see COPYING.TXT and LICENSE.TXT for more details.
% Last revision of this file: December 31, 2004. 
%-----------------------------------------------------------------------