function fdcload(extension,datatype)
% FDCLOAD - loads user-specified FDC datafile(s) via UIGETFILE GUI
%
% This function provides a standardized way of retrieving FDC-data from
% file by means of the GUI command UIGETFILE. FDCLOAD uses the default
% data-directory (subdirectory DATA in the FDC toolbox) as a starting
% point to navigate through the datafiles. If that directory can't be
% found, the current working directory will be used instead.
%
% Usage:
% ======
% fdcload('ext');
%   Opens a file browser, listing the files with extension 'ext' in the 
%   data directory (note: wildcards '?' and '*' may be used, and the 
%   extension may have more than 3 characters), and loads the selected 
%   file into the base workspace.
%
% fdcload('ext','datatype');
%   Opens a file browser, listing the files with extension 'ext' in the 
%   data directory (note: wildcards '?' and '*' may be used, and the 
%    extension may have more than 3 characters), and loads the selected 
%   file into the base workspace. 'datatype' is a short description of 
%   the type of data connected with the file-extension 'ext', which is
%   shown in the title bar of the file browser window.
%
% It is possible to specify a default filename by replacing 'ext' with
%   [filesep 'filename.ext']
% where 'filename.ext' is the default file. Wildcards '?' and '*' may 
% be used.
%
% Note: the Matlab file-format will be used, regardless of the file-
% extension (the datafiles are loaded with the '-mat' option). FDCLOAD 
% is not (yet) able to handle ASCII datafiles!
%
% See also DATLOAD, LINLOAD, TRILOAD, MATLOAD, DATADIR.

% Variables
% ---------
% currentdir   used to store current directory
% datatype     file-type description to be displayed in uigetfile window
% defdir       default directory determined by DATADIR.M
% dirname      name of the directory which is selected by the user
% extension    function-argument, which defines the file-extension
% filename     name of the datafile which is selected by the user
% filetype     filterSpec parameter for UIGETFILE (equals *.extension)
% loadcmmnd    stores the resulting command to load the datafile

% Without input arguments, show on-line help for fdcload.m only
% -------------------------------------------------------------
if nargin == 0
    helpwin fdcload
else
    % Define default data-directory
    % -----------------------------
    defdir = feval('datadir');

    % Go to default directory if it exists; else, start load-routine from current directory
    % -------------------------------------------------------------------------------------
    currentdir = pwd;
    try
        cd(defdir);
    catch
        cd(currentdir);
    end

    % Determine list of files to be shown
    % -----------------------------------
    filetype = ['*.' extension];

    % If no data-type defined, display 'data' i.s.o. datatype in uigetfile window
    % ---------------------------------------------------------------------------
    if nargin == 1
       datatype = 'data';
    end

    % Obtain filename and path
    % ------------------------
    [filename,dirname] = uigetfile(filetype,['Retrieve ' datatype ' from file']);

    % If user has not pressed Cancel in the dialog-window, build string for the
    % load-command. Else, return dummy load-command string (a remark-sign '%',
    % which yields no response when it's evaluated by means of the eval command).
    % ---------------------------------------------------------------------------
    if (isstr(filename)~=0 & isstr(dirname)~=0)
       loadcmmnd=['load(''',dirname,filename, ''',''-mat'');'];
    else
       loadcmmnd='%';
    end

    % Evaluate load command in the base workspace
    % -------------------------------------------
    evalin('base',loadcmmnd);

    % Back to previous directory
    % --------------------------
    cd(currentdir);

end

%-----------------------------------------------------------------------
% The Flight Dynamics and Control Toolbox version 1.4.0. 
% (c) Copyright Marc Rauw, 1994-2005. Licensed under the Open Software 
% License version 2.1; see COPYING.TXT and LICENSE.TXT for more details.
% Last revision of this file: May 19, 2005. 
%-----------------------------------------------------------------------