function pathname=fdcdir(varargin)
% FDCDIR - builds pathnames for subdirectories of the FDC toolbox
%
% Due to the fact that the FDC root-directory will be different on different
% systems, referencing FDC subdirectories can be difficult. To facilitate
% this, the function FDCDIR has been constructed. Calling this function for a
% user-specified FDC subdirectory will result in a complete path to this
% directory, that is: the subdirectory will be enhanced with the root-
% directory of the FDC toolbox.
%
% If the resulting pathname does not exist, a warning message will be displayed.
% There are two possible causes for this condition to occur: 1. the specified
% FDC subdirectory, or the currently used root-directory, is invalid, or 2. the
% specified root or subdirectory can't be found because it has been deleted or
% moved to another location. The location of the FDC root-directory will be
% obtained from the file fdc.ini. If that file can't be found, FDCDIR will
% try the default FDC root-location MATLABROOT\TOOLBOX\FDC14.
%
% Usage:
% ======
% D = fdcdir
%   returns the root-directory of the FDC toolbox into the string-variable D.
%
% D = fdcdir(NAME)
%   returns the complete path of the FDC subdirectory specified in the
%   string-variable NAME into the string variable D.
%
% D = fdcdir(NAME1,NAME2)
%   returns the complete path of the FDC sub-subdirectory specified in the
%   string-variable NAME2, which itself is contained in the FDC subdirectory
%   specified in the string-variable NAME1 (see the example below).
%
% More input arguments are allowed to specify deeper subdirectory levels, 
% which may be useful for future toolbox enhancements; currently, this option
% is not yet very practical. All input arguments must be string variables.
%
% Examples:
% =========
% D = fdcdir('MODELS') and C:\MATLAB\TOOLBOX\FDC14 is specified as root-directory
% in fdc.ini. This yields: D = 'C:\MATLAB\TOOLBOX\FDC14\MODELS'.
%
% With the same root-directory, D = fdcdir('MODELS','AIRCRAFT') will yield:
% D = 'C:\MATLAB\TOOLBOX\FDC20\MODELS\AIRCRAFT'.
%
% NOTE: although Windows notation has been used for the pathnames in the examples
% and information above, the program itself makes use of generic path and file-
% separators!
%
% See also DATADIR, HELPDIR.

% Variables:
% ==========
% pathname      complete pathname for the user-specified FDC folders
% rootdir       root-folder of the FDC toolbox
% subdir        sub-folder of the FDC toolbox, sub-sub-folder of the FDC toolbox, etc.
% warning_msg   string-vector to store warning messages for FDCDIR

% Extract FDC root-folder from FDC initialization file fdc.ini, or use default
% root-folder MATLABROOT\TOOLBOX\FDC14 if fdc.ini can't be found
% --------------------------------------------------------------------------------
if exist('fdc.ini','file')
   load fdc.ini -mat                             % extract root-folder from fdc.ini
else
   rootdir = fullfile(matlabroot,'toolbox','fdc14'); % default FDC root-folder
end

% Build pathname for the user-specified FDC folder
% ------------------------------------------------
pathname = rootdir;
for i = 1:length(varargin)
    subdir = char(varargin(i));
    pathname = fullfile(pathname,subdir);
end

% If the resulting pathname does not exist, display warning message.
% ------------------------------------------------------------------
if not(exist(pathname,'dir'))

    if not(exist(rootdir,'dir'))
        warning_msg = ('FDC root-directory not found!');
        h = newMsgBox(warning_msg,'WARNING');
        uiwait(h);
    else
        warning_msg = ['The specified directory  ', ...
                       sprintf('\n') upper(pathname) sprintf('\n'), ...
                       'can''t be found. The result is not a valid directory! '];
        h = newMsgBox(warning_msg,'WARNING');
        uiwait(h);
    end

end

%-----------------------------------------------------------------------
% The Flight Dynamics and Control Toolbox version 1.4.0. 
% (c) Copyright Marc Rauw, 1994-2005. Licensed under the Open Software 
% License version 2.1; see COPYING.TXT and LICENSE.TXT for more details.
% Last revision of this file: March 26, 2005. 
%-----------------------------------------------------------------------