function browse(name,varargin)
% BROWSE - displays HTML files in a webbrowser or the Matlab help browser
%
% This function provides a convenient way to implement an on-line help system
% for Matlab toolboxes and Simulink blocksets, relying on HTML formatting and 
% linking.
%
% Usage:
% ======
% browse name, or browse('name') opens the specified HTML file in the 
%   default webbrowser (see example below for more details). 
%
% browse name -mlhelp, or browse('name','-mlhelp') opens the HTML file
%   in the Matlab help browser (this switch is functional only for Matlab 
%   release 12 and later, as the Matlab help browser didn't exist in earlier
%   Matlab releases).
%
% Example:
% ========
% Consider for example the helpfile 'c:\toolbox\helpfiles\filename.html'.
%
% In order to display this file in the webbrowser, the following entries for
% 'name' are valid:
%
%  - full path plus filename plus extension: c:\toolbox\helpfiles\filename.html
%  - part of the path plus filename plus extension: helpfiles\filename.html
%  - filename plus extension: filename.html
%  - full path plus filename without extension: c:\toolbox\helpfiles\filename
%  - part of the path plus filename without extension: helpfiles\filename
%  - filename only: filename
%
% In order for this function to work, the directory in which the helpfile is 
% located *must* be present in the Matlab search path. If the specified file 
% cannot be found, an error message will be displayed.
%
% Suitable file extensions are: .htm, .html, .HTM, and .HTML. All other 
% extensions will be disregarded and will result in a 'file not found' error, 
% even if the file _does_ exist! For instance, if a file README.TXT exists 
% somewhere on the Matlab path, then browse('README.TXT') will still yield a 
% 'file not found' error, because BROWSE will search for 'README.TXT.htm' 
% and 'README.TXT.html' (both upper and lower-case) only.
%
% Other info:
% ===========
% By default, BROWSE uses the default webbrowser instead of the Matlab help 
% browser; changing this behaviour requires editing of browse.m. Early versions 
% of the Matlab help browser may have problems with modern web technologies such 
% as Cascading Style Sheets, which is one reason why the Matlab help browser is
% not used by default. However, the browser choice in general is mainly a matter 
% of personal preference. 
%

% BROWSE builds on the WEB command of Matlab, which may not work correctly 
% with browsers other than Netscape 4 or Internet Explorer > version 4. If your 
% browser is not supported by the WEB command, it may be more convenient to 
% call BROWSE with the optional argument '-mlhelp'.

% Variables
% ---------
% extension   temporary variable, stores the file extension (if a file-extension can
%              be identified in the inputvariable 'name')
% filename    temporary variable, stores the filename (if a filename can be identified
%              in the inputvariable 'name')
% H           temporary variable to store the handle of a warning or error message window
% location    stores the complete path to the requested HTML helpfile, including filename
%              and file-extension
% name        inputvariable to this function, should at least contain the filename, and
%              may also include (parts of) the pathname and the file-extension (only .htm,
%              .html, .HTM, and .HTML will be recognized as valid file-extensions)
% N,M         temporary variables to count the number of entries in 'location' (only N is
%              needed; M is a dummy variable)


% If no input arguments are specified, display the help-text
% for BROWSE itself in the Matlab command-window.
% ----------------------------------------------------------
if nargin == 0
   helpwin browse

else

   % Check whether the variable 'name' contains the full filename by searching for
   % the extensions '.html' or '.htm' in upper and lower characters. If it does,
   % find the exact location of the file by means of the WHICH command. If neither
   % extension is found, attempt to locate name.html (using both upper and lower
   % case for the extension), and if that doesn't work, try to find name.htm.
   % -----------------------------------------------------------------------------
   [filename,extension] = strtok(lower(name),'.');
   if strcmp(extension,'.htm') | strcmp(extension,'.html') % name contains htm(l) extension
      location = char(which(name,'-all'));
   else                                                    % name does not contain htm(l) extension
      location = char(which([name '.html'],'-all'));
      if isempty(location)
         location = char(which([name '.HTML'],'-all'));
         if isempty(location)
            location = char(which([name '.htm'],'-all'));
            if isempty(location)
               location = char(which([name '.HTM'],'-all'));
            end
         end
      end
   end

   % If 'location' is still an empty variable, the helpfile does not exist anywhere
   % on the Matlab search path. In that case, display an error message and skip the
   % remainder of this function.
   % ------------------------------------------------------------------------------
   if isempty(location)
      newMsgBox(['The specified helpfile can not be found anywhere ', ...
                     'on the Matlab search-path!'],'ERROR');

   else

      % Verify the number of entries in 'location' to find out whether or not the desired
      % helpfile is found at more than one location. If the number of entries is greater
      % than one, apparantly the name of the helpfile is not unique. In that case, display
      % a warning message, and use the first entry from the list.
      % ----------------------------------------------------------------------------------
      [N,M] = size(location);
      if N > 1
         H = newMsgBox(['Found more than one help-file with the name ' name '. ', ...
                    'The first help text from the acquired list will now be displayed; ', ...
                    'it is possible that this is not the help text that was actually ', ...
                    'requested! To prevent this problem in the future, please use a ', ...
                    'more unique name for the helpfile itself and change the calling ', ...
                    'function accordingly.'],'Warning');

         uiwait(H);

         location = location(1,:);

      end

      % Display the helptext in the webbrowser. The try...catch loop ensures compatibility
      % in older Matlab versions of which the WEB command doesn't support the -browser option.
      % --------------------------------------------------------------------------------------
      browserflag = varargin; 
      
      if isempty(browserflag)
         try
            web(['file:' location],'-browser');
         catch
            web(['file:' location]);
         end
      elseif strcmp(browserflag,'-mlhelp')
         web(['file:' location]);
      else
         H = newMsgBox(['The optional second function parameter for BROWSE function-call must ', ...
                    'be equal to ''-mlhelp'' in order to open the HTML file in ', ...
                    'the Matlab help browser. Now using the standard webbrowser.'],'Warning');

         uiwait(H);
         web(['file:' location],'-browser');
      end
      
   end

end

%-----------------------------------------------------------------------
% The Flight Dynamics and Control Toolbox version 1.4.0. 
% (c) Copyright Marc Rauw, 1994-2005. Licensed under the Open Software 
% License version 2.1; see COPYING.TXT and LICENSE.TXT for more details.
% Last revision of this file: April 19, 2005.
%-----------------------------------------------------------------------