% PRAHINIT - initializes the PAH, RAH, and PAHRAH autopilot models
%
% This routine initializes the model of the 'Beaver' aircraft, asks
% the user to specify an initial (trimmed) flight condition, and loads
% the actuator and cable models, using the following initialization 
% scheme:
%
% - The parameters of the 'Beaver' model are loaded from the file 
%   aircraft.dat in the FDC data-directory. 
%
% - The initial flight condition is loaded from a *.tri file from 
%   the FDC data-directory. A wide range of initial trimmed flight
%   conditions is available in the default FDC distribution; if this
%   is not sufficient, use ACTRIM to compute additional trimmed flight
%   conditions and save these results to additional *.tri files.
%
% - The used actuator and cable parameters depends on the initial 
%   value of the airspeed:
%
%      V < 40 m/s        : actmod35.dat
%      40 <= V <= 50 m/s : actmod45.dat
%      V > 50 m/s        : actmod55.dat
%
%   where the number in the actmod*.dat files corresponds with the
%   velocity for which the actuator and cable data were determined.
%
% - The gains are computed for the initial value of the airspeed. 
%   Please notice that this will yield errors if the airspeed varies
%   a lot from the initial value. For simulations involving large air-
%   speed changes, it is advised to use APILOT1 (or APILOT2 or APILOT3)
%   instead, as that model contains continuous gain-scheduling functions.
%
% These initialization rules are quite restrictive and specific: the 
% initialization routine does not try to cover all possible situations! 
% If you want to deviate from these initialization rules (e.g. use 
% cable & actuator parameters, which are not 'compatible' with the 
% initial airspeed, or load the aircraft data from a different file 
% than aircraft.dat) it will be necessary to modify this file, or 
% create a custom initialization routine yourself. 


% Load aircraft model parameters, if not already present in the workspace
% -----------------------------------------------------------------------
if not(exist('AM') & exist('EM') & exist('GM1') & exist('GM2'))
   load('aircraft.dat','-mat');
end

% Load initial flight condition, and display messagebox with 
% information about this step. 
% -----------------------------------------------------------
h_msg=newMsgBox({'Specify initial flight condition. Click OK to continue...'}, ...
                 'Initialize autopilot model');
uiwait(h_msg);
triload;

% If the user has cancelled the previous step by closing the load dialog, 
% or clicking 'Cancel', use a default flight condition (cruise configuration, 
% airspeed = 45 m/s, altitude = 4000 ft), and display warning message.
% Notice that this test does not protect against incorrect definition of 
% xinco!
% ---------------------------------------------------------------------------
if not(exist('xinco'))
   h_msg=newMsgBox({'Initial condition was not specified.', ' ', ...
                    'Using default flight condition (cruise configuration,', ...
                    'airspeed = 45 m/s, altitude = 4000 ft) instead.'},'WARNING');
   uiwait(h_msg);
   load('cr4540.tri','-tri');
end

% Load actuator & cable model that resembles most closely initial velocity
% ------------------------------------------------------------------------
if xinco(1) < 40
   load('actmod35.dat','-mat');
elseif xinco(1) > 50
   load('actmod55.dat','-mat');
else
   load('actmod45.dat','-mat');
end

% Determine gain values for initial velocity
% ------------------------------------------
[Ktheta,Kq,Ki_1,Kphi,Kr,dar,drr,Ki_2] = kpahrah(xinco(1));


% Display ready-message
% ---------------------
h_msg=newMsgBox({'The Beaver model parameters have been set.', ...
                 ['The gains, and the actuator and cable ', ...
                 'models have been initialized for an initial flight velocity of ', ...
                 num2str(xinco(1)) ' m/s.'], ...
                 ['Initial flight altitude is ' num2str(xinco(12)) ' m = ', ...
                  num2str(xinco(12)/.3048) ' ft.']}, ...
                 'PAH/RAH initialization completed.');


%-----------------------------------------------------------------------
% The Flight Dynamics and Control Toolbox version 1.4.0. 
% (c) Copyright Marc Rauw, 1994-2005. Licensed under the Open Software 
% License version 2.1; see COPYING.TXT and LICENSE.TXT for more details.
% Last revision of this file: December 31, 2004.
%-----------------------------------------------------------------------