clc
clear all
close all

%% Geometric and material parameters
r_s= 0.005;         %Sphere radius [m]
d_s= 2*r_s;         %Sphere diameter [m]
E_s= 390*10^9;      %[Pa] Sphere elastic modulus
ni_s= 0.26;         %Sphere Poisson ration

E_d= 218*10^9;      %[Pa] Disc Elastic modulus
ni_d= 0.30;         %Disc Poisson ration

F= 10;              %[N] Applied vertical load
Er= 0.5*( (1-ni_s^2)/E_s + (1-ni_d^2)/E_d ); %Reduced elastic modulus of material pair

l= 0.016;           %[m] stroke length -> wear increment
L= 1000;            %[m] test duration - total sliding distance
N= round(L/l);      %overall number of duration spans -> number of cycles doubled


M=180;                %Number of sector of the wear track




%% Experimental wear law
% Archard coefficients from excel analysis of wear curves
k1= 8*10^(-15); % [m3/(m*N)] dimensional wear rate k = V/(s*F)- wear-in
L1= 23; %[m] end of wear-in
k2= 3*10^(-14); % [m3/(m*N)] dimensional wear rate k = V/(s*F)- wear phase I
L2= 175; %[m] end of wear phase I
k3= 5*10^(-14); % [m3/(m*N)] dimensional wear rate k = V/(s*F)- wear phase II

k= zeros(N,1); %initializing the dimensional wear coefficient vector
k( 1:round(L1/l) )= k1;
k( (round(L1/l)+1) : round(L2/l) )= k2;
k( (round(L2/l)+1) : end )= k3;

% Experimental data import
opts = delimitedTextImportOptions("NumVariables", 2);
opts.DataLines = [1, Inf]; % Specify range and delimiter
opts.Delimiter = "\t";
opts.VariableNames = ["VarName1", "VarName2"]; % Specify column names and types
opts.VariableTypes = ["double", "double"];
opts.ExtraColumnsRule = "ignore"; % Specify file level properties
opts.EmptyLineRule = "read";
opts = setvaropts(opts, ["VarName1", "VarName2"], "DecimalSeparator", ",");% Specify variable properties

% Import the data of wear track cross-sectional area
WearTrackArea = readtable('Wear track area.txt', opts);     %File must be in the same folder as the script
% Convert to numeric matrix
WearTrackArea = table2array(WearTrackArea);

% Import the data of wear track depth
WearTrackDepth = readtable('Wear track depth.txt', opts);   %File must be in the same folder as the script
% Convert to numeric matrix
WearTrackDepth = table2array(WearTrackDepth);

% Clear temporary variables
clear opts

% Table's data import, first have to change load options
opts = delimitedTextImportOptions("NumVariables", 5);
opts.DataLines = [1, Inf]; % Specify range and delimiter
opts.Delimiter = "\t";
opts.VariableNames = ["VarName1", "VarName2", "VarName3", "VarName4", "VarName5"]; % Specify column names and types
opts.VariableTypes = ["double", "double", "double", "double", "double"];
opts.ExtraColumnsRule = "ignore"; % Specify file level properties
opts.EmptyLineRule = "read";
opts = setvaropts(opts, ["VarName1", "VarName2", "VarName3", "VarName4", "VarName5"], "DecimalSeparator", ",");% Specify variable properties

% Import the table
CosTauTable = readtable('Hertz table.txt', opts);   %File must be in the same folder as the script
% Convert to numeric matrix
CosTauTable = table2array(CosTauTable);

% Clear temporary variables
clear opts







%% I make a principal code in order to have some info general about the disc


s= zeros(N,1);          %initializing the incremental sliding distance vector
a= zeros(N,1);          %initializing the major contact axis vector
b= zeros(N,1);          %initializing the minor contact axis vector
delta= zeros(N,1);      %initializing the contact body convergence vector
d= zeros(N+1,1);        %initializing the wear track width vector
Aw= zeros(N,1);         %initializing the cross-sectional area of wear track vector
r_t= zeros(N,1);        %initializing the wear track profile radius vector
theta= zeros(N,1);      %initializing the wear track angular amplitude vector
arc= zeros(N,1);        %initializing the wear track arc profile length vector
hw= zeros(N,1);         %initializing the underfomed wear depth vector
hw_tot= zeros(N,1);     %initializing the wear depth vector + elastic deformation
corrections= zeros(N,2);    %initializing the correction flags matrix


for j=1:N       %j -> stroke counter
    s(j)= j*l;  %cumulative sliding distance
    deltaAw= k(j)*F;
    
    % It is assumed that the wear volume is inscribed into a circular profile if elastic deformation is recovered
    % Circular segment area formula: A = 0.5 * r_t^2 * (2*theta - sen(2*theta))
    % Wear track radius: r_t= a_0 * 1/sen(theta)
    % Wear track angular amplitude -> theta
    % Wear track widht --> d
    % r_t, theta and d are unknown -> system of 2 equations for 3 unknowns: one must add an hypotesys on wear track width d
    
    if j==1 %First iteration (first stroke)
        a_0= ( 3*F/(4/d_s)*Er )^(1/3);      %Hertzian contact RADIUS sphere-plane at first contact (Shaeffler handbook formula)
        Aw(j)= deltaAw;                     %Cross sectional area of the wear track
        a(j)=a_0;
        d(j) =2*a_0;                        %I know the track amplitude from the first iteration -> the width of the sphere-plane hertzian contact
        f= @(x)( (6*Aw(j)/d(j)^2)*( 1 + (x^4)/36 - (x^2)/3 ) - x );     %equation to solve theta with: d(1)=2*a_0
        X0 = fzero(f, atan(a_0/r_s));                                   %solutions vector with 1 positive value only (I hope!!)
        theta(j) = X0(X0>0 && X0<pi);                                   %calculated wear track amplitude,  within its phisical limits
        r_t(j)= a_0 / sin(theta(j));
        
    else %strokes else than the first
        Aw(j)= Aw(j-1) + deltaAw;                                     %Cumulative cross sectional area of the wear track
        f= @(x)( (6*Aw(j)/d(j)^2)*( 1 + (x^4)/36 - (x^2)/3 ) - x );    %equation to solve theta
        X0 = fzero(f, atan(a(j-1)/r_t(j-1)));                          %solutions vector with 1 positive value only (I hope!!)
        %r_t(j-1) is use as guess value for fzero function
        theta(j) = X0(X0>0 && X0<pi);                                  %calculated wear track angle, within its phisical limits
        r_t(j)= d(j) / (2*sin(theta(j)));
    end
    
    %Wear track profile and wear depth calculation
    if  r_t(j)>1.001*r_s                              %Hertz thoery is assumed to hold
        hw(j)= r_t(j) * (1-cos(theta(j)));          % wear depth
        arc(j)= 2* r_t(j) * theta(j); % arc profile of the wear track
        [a(j),b(j),delta(j)]= Hertz_contact(F,E_s,E_d,ni_s,ni_d,r_s,r_t(j),r_s,inf); %Hertz contact calculation at the end of the stroke
        %VERIFY IF CONSISTENT WITH VALUES FORM SCHAEFFLER HANDBOOK FORMULA --> AXIAL BALL BEARINGS
        hw_tot(j)= hw(j) + delta(j+1);
        
        %Hertz contact calculation at the end of the stroke from the Schaeffler handbook formulas
        
        if 2*a(j)<arc(j)
            d(j+1)= 2*a(j);     %HP: wear track amplitude of the following iteration is identified approximately as the major Hertz contact axis
            %but the theoretical major contact axis may expand too much if r_t(j)/r_s -> 1.001, according to Hertz theory
            
        else
            d(j+1)= arc(j);     %HP: wear track amplitude d is artificially limited to the linearized arc profile of the wear track
            corrections(j,2)=1; %flag to identify the correction for track amplitude
            
        end
        
    else %HP: wear track radius is never smaller that the sphere radius for phisical limits
        r_t(j)= 1.001*r_s;                              % Hertz theory does not hold anymore!
        corrections(j,1)=1;                             % flag to identified the correction for radius
        theta(j)=((3/2)*Aw(j)/r_t(j)^2)^(1/3);          % the track radius is known a priori here --> function f is useless
        hw(j)= r_t(j) * (1-cos(theta(j)));              % wear depth
        arc(j)= 2* r_t(j) * theta(j);                   % arc profile of the wear track
        d(j+1)= 2*r_t(j)*sin(theta(j));                 % wear track width is calculated here
        hw_tot(j)= hw(j) + delta(j);
    end
    
end

r_t_disc=r_t;               %I save the variables with the _disc in order to use them after for the real Hp 3 code
d_disc=d;
theta_disc=theta;
hw_disc=hw;


%% I make a principal code in order to have some info general about the pin

%I set the parameters for the pin

a_pin= zeros(N+1,1);          %initializing the major contact axis vector
p_pin= zeros(N+1,1);          %initializing the minor contact axis vector
d_pin= zeros(N+2,1);        %initializing the wear track width vector
hw_pin= zeros(N+2,1);         %initializing the underfomed wear depth vector
hw_tot_pin= zeros(N+1,1);     %initializing the wear depth vector + elastic deformation
% he_pin= zeros(N+1,1);         %initializing the underfomed wear depth vector
% h_tot_pin= zeros(N+1,1);     %initializing the wear depth vector + elastic deformation
Aar= zeros(N+1,1);         %initializing the underfomed wear depth vector
Vsph= zeros(N+1,1);         %initializing the underfomed wear depth vector

k_pin=[k/100;k3/100];

% Wear increment loop
for j=1:N+1                                       %j -> stroke counter
    s(j)= j*l;                                  %cumulative sliding distance
    %     deltaAw= k(j)*F*0.1;                  %At the moment I don't know the exact value of k for the pin so I use 0.1 as first tentative
    
    % It is assumed that the wear volume is inscribed into a circular profile if elastic deformation is recovered
    % Circular segment area formula: A = 0.5 * r_t^2 * (2*theta - sen(2*theta))
    % Wear track radius: r_t= a_0 * 1/sen(theta)
    % Wear track angular amplitude -> theta
    % Wear track widht --> d
    % r_t, theta and d are unknown -> system of 2 equations for 3 unknowns: one must add an hypotesys on wear track width d
    
    if j==1 %First iteration (first stroke)
        
        a_0= ( 3*F/(4/d_s)*Er )^(1/3);                  %Hertzian contact RADIUS sphere-plane at first contact (Shaeffler handbook formula)
        a_pin(j)=a_0;
        d_pin(j) =2*a_0;                                    %I know the track amplitude from the first iteration -> the width of the sphere-plane hertzian contact
        p_pin(j)=F/(3.14*a(j)^2);                           %Pressure of contact of spherical pin
        hw_pin(j+1)=k_pin(j)*p_pin(j)*l;                            %The height of the cutted part of the pin, instead of s there is a delta s --> l
        
        he_0_pin=F*Er/(2*a_0);                              %the initial value of displacement purely elastic
        
        
        a_pin(j+1)=(d_s*hw_pin(j+1)-hw_pin(j+1)^2)^(1/2);     %The radius of the cutted part of the pin
        
        Aar(j)= k(j)*F;                            %Volume of wear for Archard
        %         Asph(j)=3.14*hw(j)/6*(3*a(j)^2+hw(j)^2);        %Volume of wear computed by the geometry of the Sphere
        
        
    else %strokes else than the first
        
        
        p_pin(j)=F/(3.14*a_pin(j)^2);                           %Pressure of contact of spherical pin
        hw_pin(j+1)=k_pin(j)*p_pin(j)*l + hw_pin(j);                    %The height of the cutted part of the pin, instead of s there is a delta s --> l
        d_pin(j)=2*a_pin(j);                                     %The dimension of contact zone is now equal to 2*a
        
        
        a_pin(j+1)=(d_s*hw_pin(j+1)-hw_pin(j+1)^2)^(1/2);     %The radius of the cutted part of the pin
        
        
        Aar(j)= Aar(j-1)+k_pin(j)*F;                            %Volume of wear for Archard
        %         Vsph(j)=3.14*hw(j)/6*(3*a(j)^2+hw(j)^2);        %Volume of wear computed by the geometry of the Sphere
        
        
        
    end
    
end
%% Initialization matrix
%Initializing the vectors for interations containing the END-OF-STROKE pysical and geometrical quantities

s= zeros(N,M);          %initializing the incremental sliding distance vector
A= zeros(N,M);          %initializing the major contact axis vector
B= zeros(N,M);          %initializing the minor contact axis vector
DELTA= zeros(N,M);      %initializing the contact body convergence vector
D= zeros(N+1,M);        %initializing the wear track width vector
AW_MATRIX= zeros(N,M);         %initializing the cross-sectional area of wear track vector
R_T= zeros(N,M);        %initializing the wear track profile radius vector
THETA= zeros(N,M);      %initializing the wear track angular amplitude vector
ARC= zeros(N,M);        %initializing the wear track arc profile length vector
HW= zeros(N,M);         %initializing the underfomed wear depth vector
HW_TOT_MAT= zeros(N,M);     %initializing the wear depth vector + elastic deformation
corrections= zeros(N,M);    %initializing the correction flags matrix
%column 1 for r_t limited to r_s
%column 2 for d limited to linearized arc profile

%initializing the major contact axis vector, as it's done for the Hamrock
%and dawson simplified formulas, it's done for the handbook's formulas
% _t is the subscript to indicate the values came from the Table of
% Schaeffler handbook

a_t= zeros(N,1);
b_t= zeros(N,1);
delta_t= zeros(N,1);
hw_t= zeros(N,1);
hw_tot_t= zeros(N,1);
cos_tau=zeros(N,1);

%A_0= pi*(a_0)^2; Hertzian contact area sphere-plane
%pm_0= F/A_0; Hertzian mean contact pressure

%% Start of the computing of the sector of the disc
% Wear increment loop

sector_DISC=d_disc*1000/0.1;              %sector have the dimension of 0.1mm
sector_flag_DISC=zeros(N,1);             % I set a flag that can useful for find the variation of the form of wear track area
Indices_DISC=zeros(M,1);                 % same as before, Isave the indices in order to use it later

tt=1;

for ii=1:N
    if ii==1
        m=1;
    else
        if sector_DISC<1
            m=1;
        else
            m=round(sector_DISC);
            if m (ii) > m(ii-1)
                sector_flag_DISC(ii)=sector_flag_DISC(ii-1)+1;
                Indices_DISC(tt)=ii;
                tt=tt+1;
            else
                sector_flag_DISC(ii)=0;
            end
        end
    end
end

MAX_SECTOR_DISC=max(m);

%  for jj=1:MAX_SECTOR
%
%     V_Sec=1:MAX_SECTOR;
%     POSITION=find(m,V_Sec(jj));
%
%  end

for jj=1:N
    
    V_Sec=1:MAX_SECTOR_DISC;
    UP=m==V_Sec(1:end);
    
end


%Up to here the code is correct
%Maybe to do a for in order to have manually the point in which sector is
%+1
%% find the maximum sector of the disc

%I multiply *2 in order to have always the center as max wear penetration

Indices_DISC_OK = [0;Indices_DISC];
MAX_SECTOR_DISC


%% Start of the computing of the sector of the PIN
% Wear increment loop

sector_PIN=d_pin*1000/0.1;              %sector have the dimension of 0.1mm
sector_flag_PIN=zeros(N,1);             % I set a flag that can useful for find the variation of the form of wear track area
Indices_PIN=zeros(M,1);                 % same as before, Isave the indices in order to use it later

tt=1;

for ii=1:N
    if ii==1
        m=1;
    else
        if sector_PIN<1
            m=1;
        else
            m=round(sector_PIN);
            if m (ii) > m(ii-1)
                sector_flag_PIN(ii)=sector_flag_PIN(ii-1)+1;
                Indices_PIN(tt)=ii;
                tt=tt+1;
            else
                sector_flag_PIN(ii)=0;
            end
        end
    end
end

MAX_SECTOR_PIN=max(m);

%  for jj=1:MAX_SECTOR
%
%     V_Sec=1:MAX_SECTOR;
%     POSITION=find(m,V_Sec(jj));
%
%  end

for jj=1:N
    
    V_Sec=1:MAX_SECTOR_PIN;
    UP=m==V_Sec(1:end);
    
end


%Up to here the code is correct
%Maybe to do a for in order to have manually the point in which sector is
%+1
%% find the maximum sector of the PIN

%I multiply *2 in order to have always the center as max wear penetration

Indices_PIN_OK = [0;Indices_PIN];
MAX_SECTOR_PIN

%% Compute the wear geometry taking into account the deformation of the disc and the pin

Aw_pin=Aar;


for t=1:MAX_SECTOR_DISC
    
    s= zeros(N,1);          %initializing the incremental sliding distance vector
    a= zeros(N,1);          %initializing the major contact axis vector
    b= zeros(N,1);          %initializing the minor contact axis vector
    delta= zeros(N,1);      %initializing the contact body convergence vector
    d= zeros(N+1,1);        %initializing the wear track width vector
    Aw= zeros(N,1);         %initializing the cross-sectional area of wear track vector
    r_t= zeros(N,1);        %initializing the wear track profile radius vector
    theta= zeros(N,1);      %initializing the wear track angular amplitude vector
    arc= zeros(N,1);        %initializing the wear track arc profile length vector
    hw= zeros(N,1);         %initializing the underfomed wear depth vector
    hw_tot= zeros(N,1);     %initializing the wear depth vector + elastic deformation
    corrections= zeros(N,2);    %initializing the correction flags matrix
    %column 1 for r_t limited to r_s
    %column 2 for d limited to linearized arc profile
    
    %initializing the major contact axis vector, as it's done for the Hamrock
    %and dawson simplified formulas, it's done for the handbook's formulas
    % _t is the subscript to indicate the values came from the Table of
    % Schaeffler handbook
    
    a_t= zeros(N,1);
    b_t= zeros(N,1);
    delta_t= zeros(N,1);
    hw_t= zeros(N,1);
    hw_tot_t= zeros(N,1);
    cos_tau=zeros(N,1);
    
    if t<=MAX_SECTOR_PIN
        
        
        %Inizialize the vector each cycle maybe can speed up the code
        for j=1:N-Indices_DISC_OK(t)      %j -> stroke counter
            hp3_coefficient=0.95+(1.15-0.95)*rand(1,1);    %Applying the random coeff. I canceled the M because is a scalar I think. Every cycle change
            deltaAw= (k(j)*F*hp3_coefficient)-(Aw_pin(j+1)-Aw_pin(j));          % I have to adjust the vector Aw in order to have a proper difference
            s(j)=l*j;
            
            % It is assumed that the wear volume is inscribed into a circular profile if elastic deformation is recovered
            % Circular segment area formula: A = 0.5 * r_t^2 * (2*theta - sen(2*theta))
            % Wear track radius: r_t= a_0 * 1/sen(theta)
            % Wear track angular amplitude -> theta
            % Wear track widht --> d
            % r_t, theta and d are unknown -> system of 2 equations for 3 unknowns: one must add an hypotesys on wear track width d
            
            if j==1 %First iteration (first stroke)
                a_0= ( 3*F/(4/d_s)*Er )^(1/3);      %Hertzian contact RADIUS sphere-plane at first contact (Shaeffler handbook formula)
                Aw(j)= deltaAw;                     %Cross sectional area of the wear track
                a(j)=a_0;
                d(j) =2*a_0;                        %I know the track amplitude from the first iteration -> the width of the sphere-plane hertzian contact
                f= @(x)( (6*Aw(j)/d(j)^2)*( 1 + (x^4)/36 - (x^2)/3 ) - x );     %equation to solve theta with: d(1)=2*a_0
                X0 = fzero(f, atan(a_0/r_s));                                   %solutions vector with 1 positive value only (I hope!!)
                theta(j) = X0(X0>0 && X0<pi);                                   %calculated wear track amplitude,  within its phisical limits
                r_t(j)= a_0 / sin(theta(j));
                
            else %strokes else than the first
                Aw(j)= Aw(j-1) + deltaAw;                                     %Cumulative cross sectional area of the wear track
                f= @(x)( (6*Aw(j)/d(j)^2)*( 1 + (x^4)/36 - (x^2)/3 ) - x );    %equation to solve theta
                X0 = fzero(f, atan(a(j-1)/r_t(j-1)));                          %solutions vector with 1 positive value only (I hope!!)
                %r_t(j-1) is use as guess value for fzero function
                theta(j) = X0(X0>0 && X0<pi);                                  %calculated wear track angle, within its phisical limits
                r_t(j)= d(j) / (2*sin(theta(j)));
            end
            
            %Wear track profile and wear depth calculation
            if  r_t(j)>1.001*r_s                              %Hertz thoery is assumed to hold
                hw(j)= r_t(j) * (1-cos(theta(j)));          % wear depth
                hw_t(j)=r_t(j) * (1-cos(theta(j)));
                arc(j)= 2* r_t(j) * theta(j); % arc profile of the wear track
                [a(j),b(j),delta(j)]= Hertz_contact(F,E_s,E_d,ni_s,ni_d,r_s,r_t(j),r_s,inf); %Hertz contact calculation at the end of the stroke
                %VERIFY IF CONSISTENT WITH VALUES FORM SCHAEFFLER HANDBOOK FORMULA --> AXIAL BALL BEARINGS
                hw_tot(j)= hw(j) + delta(j+1);
                
                %Hertz contact calculation at the end of the stroke from the Schaeffler handbook formulas
                [a_t(j),b_t(j),delta_t(j),cos_tau(j)]= Hertz_contact_table(F,E_s,E_d,ni_s,ni_d,r_s,r_t(j),r_s,inf,CosTauTable);
                hw_tot_t(j)= hw_t(j) + delta_t(j+1);
                
                if 2*a(j)<arc(j)
                    d(j+1)= 2*a(j);     %HP: wear track amplitude of the following iteration is identified approximately as the major Hertz contact axis
                    %but the theoretical major contact axis may expand too much if r_t(j)/r_s -> 1.001, according to Hertz theory
                    
                else
                    d(j+1)= arc(j);     %HP: wear track amplitude d is artificially limited to the linearized arc profile of the wear track
                    corrections(j,2)=1; %flag to identify the correction for track amplitude
                    
                end
                
            else %HP: wear track radius is never smaller that the sphere radius for phisical limits
                r_t(j)= 1.001*r_s;                              % Hertz theory does not hold anymore!
                corrections(j,1)=1;                             % flag to identified the correction for radius
                theta(j)=((3/2)*Aw(j)/r_t(j)^2)^(1/3);          % the track radius is known a priori here --> function f is useless
                hw(j)= r_t(j) * (1-cos(theta(j)));              % wear depth
                arc(j)= 2* r_t(j) * theta(j);                   % arc profile of the wear track
                d(j+1)= 2*r_t(j)*sin(theta(j));                 % wear track width is calculated here
                hw_tot(j)= hw(j) + delta(j);
            end
        end
        
    elseif t<=2*MAX_SECTOR_PIN
        
        
        for j=1:N-Indices_DISC_OK(t)      %j -> stroke counter
            hp3_coefficient=0.95+(1.15-0.95)*rand(1,1);    %Applying the random coeff. I canceled the M because is a scalar I think. Every cycle change
            deltaAw= (k(j)*F*hp3_coefficient)+(Aw_pin(j+1)-Aw_pin(j));          % I have to adjust the vector Aw in order to have a proper difference
            s(j)=l*j;
            
            % It is assumed that the wear volume is inscribed into a circular profile if elastic deformation is recovered
            % Circular segment area formula: A = 0.5 * r_t^2 * (2*theta - sen(2*theta))
            % Wear track radius: r_t= a_0 * 1/sen(theta)
            % Wear track angular amplitude -> theta
            % Wear track widht --> d
            % r_t, theta and d are unknown -> system of 2 equations for 3 unknowns: one must add an hypotesys on wear track width d
            
            if j==1 %First iteration (first stroke)
                a_0= ( 3*F/(4/d_s)*Er )^(1/3);      %Hertzian contact RADIUS sphere-plane at first contact (Shaeffler handbook formula)
                Aw(j)= deltaAw;                     %Cross sectional area of the wear track
                a(j)=a_0;
                d(j) =2*a_0;                        %I know the track amplitude from the first iteration -> the width of the sphere-plane hertzian contact
                f= @(x)( (6*Aw(j)/d(j)^2)*( 1 + (x^4)/36 - (x^2)/3 ) - x );     %equation to solve theta with: d(1)=2*a_0
                X0 = fzero(f, atan(a_0/r_s));                                   %solutions vector with 1 positive value only (I hope!!)
                theta(j) = X0(X0>0 && X0<pi);                                   %calculated wear track amplitude,  within its phisical limits
                r_t(j)= a_0 / sin(theta(j));
                
            else %strokes else than the first
                Aw(j)= Aw(j-1) + deltaAw;                                     %Cumulative cross sectional area of the wear track
                f= @(x)( (6*Aw(j)/d(j)^2)*( 1 + (x^4)/36 - (x^2)/3 ) - x );    %equation to solve theta
                X0 = fzero(f, atan(a(j-1)/r_t(j-1)));                          %solutions vector with 1 positive value only (I hope!!)
                %r_t(j-1) is use as guess value for fzero function
                theta(j) = X0(X0>0 && X0<pi);                                  %calculated wear track angle, within its phisical limits
                r_t(j)= d(j) / (2*sin(theta(j)));
            end
            
            %Wear track profile and wear depth calculation
            if  r_t(j)>1.001*r_s                              %Hertz thoery is assumed to hold
                hw(j)= r_t(j) * (1-cos(theta(j)));          % wear depth
                hw_t(j)=r_t(j) * (1-cos(theta(j)));
                arc(j)= 2* r_t(j) * theta(j); % arc profile of the wear track
                [a(j),b(j),delta(j)]= Hertz_contact(F,E_s,E_d,ni_s,ni_d,r_s,r_t(j),r_s,inf); %Hertz contact calculation at the end of the stroke
                %VERIFY IF CONSISTENT WITH VALUES FORM SCHAEFFLER HANDBOOK FORMULA --> AXIAL BALL BEARINGS
                hw_tot(j)= hw(j) + delta(j+1);
                
                %Hertz contact calculation at the end of the stroke from the Schaeffler handbook formulas
                [a_t(j),b_t(j),delta_t(j),cos_tau(j)]= Hertz_contact_table(F,E_s,E_d,ni_s,ni_d,r_s,r_t(j),r_s,inf,CosTauTable);
                hw_tot_t(j)= hw_t(j) + delta_t(j+1);
                
                if 2*a(j)<arc(j)
                    d(j+1)= 2*a(j);     %HP: wear track amplitude of the following iteration is identified approximately as the major Hertz contact axis
                    %but the theoretical major contact axis may expand too much if r_t(j)/r_s -> 1.001, according to Hertz theory
                    
                else
                    d(j+1)= arc(j);     %HP: wear track amplitude d is artificially limited to the linearized arc profile of the wear track
                    corrections(j,2)=1; %flag to identify the correction for track amplitude
                    
                end
                
            else %HP: wear track radius is never smaller that the sphere radius for phisical limits
                r_t(j)= 1.001*r_s;                              % Hertz theory does not hold anymore!
                corrections(j,1)=1;                             % flag to identified the correction for radius
                theta(j)=((3/2)*Aw(j)/r_t(j)^2)^(1/3);          % the track radius is known a priori here --> function f is useless
                hw(j)= r_t(j) * (1-cos(theta(j)));              % wear depth
                arc(j)= 2* r_t(j) * theta(j);                   % arc profile of the wear track
                d(j+1)= 2*r_t(j)*sin(theta(j));                 % wear track width is calculated here
                hw_tot(j)= hw(j) + delta(j);
            end
        end
        
        
    else
        
        for j=1:N-Indices_DISC_OK(t)      %j -> stroke counter
            hp3_coefficient=0.95+(1.15-0.95)*rand(1,1);    %Applying the random coeff. I canceled the M because is a scalar I think. Every cycle change
            deltaAw= (k(j)*F*hp3_coefficient);
            s(j)=l*j;
            
            % It is assumed that the wear volume is inscribed into a circular profile if elastic deformation is recovered
            % Circular segment area formula: A = 0.5 * r_t^2 * (2*theta - sen(2*theta))
            % Wear track radius: r_t= a_0 * 1/sen(theta)
            % Wear track angular amplitude -> theta
            % Wear track widht --> d
            % r_t, theta and d are unknown -> system of 2 equations for 3 unknowns: one must add an hypotesys on wear track width d
            
            if j==1 %First iteration (first stroke)
                a_0= ( 3*F/(4/d_s)*Er )^(1/3);      %Hertzian contact RADIUS sphere-plane at first contact (Shaeffler handbook formula)
                Aw(j)= deltaAw;                     %Cross sectional area of the wear track
                a(j)=a_0;
                d(j) =2*a_0;                        %I know the track amplitude from the first iteration -> the width of the sphere-plane hertzian contact
                f= @(x)( (6*Aw(j)/d(j)^2)*( 1 + (x^4)/36 - (x^2)/3 ) - x );     %equation to solve theta with: d(1)=2*a_0
                X0 = fzero(f, atan(a_0/r_s));                                   %solutions vector with 1 positive value only (I hope!!)
                theta(j) = X0(X0>0 && X0<pi);                                   %calculated wear track amplitude,  within its phisical limits
                r_t(j)= a_0 / sin(theta(j));
                
            else %strokes else than the first
                Aw(j)= Aw(j-1) + deltaAw;                                     %Cumulative cross sectional area of the wear track
                f= @(x)( (6*Aw(j)/d(j)^2)*( 1 + (x^4)/36 - (x^2)/3 ) - x );    %equation to solve theta
                X0 = fzero(f, atan(a(j-1)/r_t(j-1)));                          %solutions vector with 1 positive value only (I hope!!)
                %r_t(j-1) is use as guess value for fzero function
                theta(j) = X0(X0>0 && X0<pi);                                  %calculated wear track angle, within its phisical limits
                r_t(j)= d(j) / (2*sin(theta(j)));
            end
            
            %Wear track profile and wear depth calculation
            if  r_t(j)>1.001*r_s                              %Hertz thoery is assumed to hold
                hw(j)= r_t(j) * (1-cos(theta(j)));          % wear depth
                hw_t(j)=r_t(j) * (1-cos(theta(j)));
                arc(j)= 2* r_t(j) * theta(j); % arc profile of the wear track
                [a(j),b(j),delta(j)]= Hertz_contact(F,E_s,E_d,ni_s,ni_d,r_s,r_t(j),r_s,inf); %Hertz contact calculation at the end of the stroke
                %VERIFY IF CONSISTENT WITH VALUES FORM SCHAEFFLER HANDBOOK FORMULA --> AXIAL BALL BEARINGS
                hw_tot(j)= hw(j) + delta(j+1);
                
                %Hertz contact calculation at the end of the stroke from the Schaeffler handbook formulas
                [a_t(j),b_t(j),delta_t(j),cos_tau(j)]= Hertz_contact_table(F,E_s,E_d,ni_s,ni_d,r_s,r_t(j),r_s,inf,CosTauTable);
                hw_tot_t(j)= hw_t(j) + delta_t(j+1);
                
                if 2*a(j)<arc(j)
                    d(j+1)= 2*a(j);     %HP: wear track amplitude of the following iteration is identified approximately as the major Hertz contact axis
                    %but the theoretical major contact axis may expand too much if r_t(j)/r_s -> 1.001, according to Hertz theory
                    
                else
                    d(j+1)= arc(j);     %HP: wear track amplitude d is artificially limited to the linearized arc profile of the wear track
                    corrections(j,2)=1; %flag to identify the correction for track amplitude
                    
                end
                
            else %HP: wear track radius is never smaller that the sphere radius for phisical limits
                r_t(j)= 1.001*r_s;                              % Hertz theory does not hold anymore!
                corrections(j,1)=1;                             % flag to identified the correction for radius
                theta(j)=((3/2)*Aw(j)/r_t(j)^2)^(1/3);          % the track radius is known a priori here --> function f is useless
                hw(j)= r_t(j) * (1-cos(theta(j)));              % wear depth
                arc(j)= 2* r_t(j) * theta(j);                   % arc profile of the wear track
                d(j+1)= 2*r_t(j)*sin(theta(j));                 % wear track width is calculated here
                hw_tot(j)= hw(j) + delta(j);
            end
        end
        
    end
    
    
    t
    
    A(:,t)=a(:);
    B(:,t)=b(:);
    DELTA(:,t)=delta(:);
    D(:,t)= d(:);
    AW_MATRIX(:,t)= Aw(:);
    R_T(:,t)= r_t(:);
    THETA(:,t)= theta(:);
    ARC(:,t)=arc(:);
    HW(:,t)=hw(:);
    HW_TOT_MAT(:,t)=hw_tot(:);
    
    
end

% i set a series of for cycle in order to extract the values of the
% coefficient of hertz contact and evaluate if they are correct or not

% size_1=0;
% size_2=0;
%
% for i=1:N
%
%     if corrections(i,1) == 0
%         size_1=size_1+1;
%     else
%         size_2=size_2+1;
%     end
% end
%
% results=rand(size_1,6);  % initializing the resul matrix in the form [a, b, delta, a_t, b_t, delta_t]
%
% for i=1:size_1
%     results(i,1)=a(i);
%     results(i,2)=b(i);
%     results(i,3)=delta(i);
%     results(i,4)=a_t(i);
%     results(i,5)=b_t(i);
%     results(i,6)=delta_t(i);
% end

%% Calculating The frontal PoV of the wear track

% R=r_t(N)*10^3;
% X=3*d(N)*1000;
%
% x=-X:0.01:X;
%
% % y=sqrt(R-x^2);
% z=-sqrt(R-x.^2)+(hw(N)*10^3);


%% Find maximum/minimum value in order to plot the range of possibility of the DISC

vector_Aw_DISC=zeros(M,1);
vector_rt_DISC=zeros(M,1);
vector_d_DISC=zeros(M,1);
vector_hw_DISC=zeros(M,1);
vector_circle_DISC=zeros(M,1);

for tt=1:MAX_SECTOR_DISC
    
    vector_Aw_DISC(tt)=AW_MATRIX(N-Indices_DISC_OK(tt),tt);
    vector_rt_DISC(tt)=R_T(N-Indices_DISC_OK(tt),tt);
    vector_d_DISC(tt)=D(N-Indices_DISC_OK(tt),tt);
    vector_hw_DISC(tt)=HW(N-Indices_DISC_OK(tt),tt);
    %     Vector_circle(tt)=(N-Indices_ok(t),tt);
    
end



[Max_Aw_DISC, I_Max_Aw_DISC]=max(vector_Aw_DISC);   % I found the index of the maximum of Aw in order to plot the right one
[Max_rt_DISC, I_Max_rt_DISC]=max(vector_rt_DISC);
[Max_d_DISC, I_Max_d_DISC]=max(vector_d_DISC);
[Max_hw_DISC, I_Max_hw_DISC]=max(vector_hw_DISC);
% Max_circle=max(vector_circle);

[min_Aw_DISC, I_min_Aw_DISC]=min(vector_Aw_DISC);
[min_rt_DISC, I_min_rt_DISC]=min(vector_rt_DISC);
[min_d_DISC, I_min_d_DISC]=min(vector_d_DISC);
[min_hw_DISC, I_min_hw_DISC]=min(vector_hw_DISC);
% min_rt=min(vector_rt);



%%
% d_med=(vector_d(I_min_d)+vector_d(I_Max_d))/2*1000;           %I find the medium value of d in order to plot correcly the profile of the wear area [mm]

% x=linspace(-d_disc(N)/2,d_disc(N)/2,MAX_SECTOR)';             %Symmetric

x=linspace(0,d_disc(N)/2,MAX_SECTOR_DISC)';

%%


% %% Find maximum/minimum value in order to plot the range of possibility of the PIN
%
% vector_Aw_PIN=zeros(M,1);
% vector_rt_PIN=zeros(M,1);
% vector_d_PIN=zeros(M,1);
% vector_hw_PIN=zeros(M,1);
% vector_circle_PIN=zeros(M,1);
%
% for tt=1:MAX_SECTOR_PIN
%
%     vector_Aw_PIN(tt)=AW_MATRIX(N-Indices_ok(tt),tt);
%     vector_rt_PIN(tt)=R_T(N-Indices_ok(tt),tt);
%     vector_d_PIN(tt)=D(N-Indices_ok(tt),tt);
%     vector_hw_PIN(tt)=HW(N-Indices_ok(tt),tt);
%     %     Vector_circle(tt)=(N-Indices_ok(t),tt);
%
% end
%
%
%
% [Max_Aw_DISC, I_Max_Aw_DISC]=max(vector_Aw_DISC);   % I found the index of the maximum of Aw in order to plot the right one
% [Max_rt_DISC, I_Max_rt_DISC]=max(vector_rt_DISC);
% [Max_d_DISC, I_Max_d_DISC]=max(vector_d_DISC);
% [Max_hw_DISC, I_Max_hw_DISC]=max(vector_hw_DISC);
% % Max_circle=max(vector_circle);
%
% [min_Aw_DISC, I_min_Aw_DISC]=min(vector_Aw_DISC);
% [min_rt_DISC, I_min_rt_DISC]=min(vector_rt_DISC);
% [min_d_DISC, I_min_d_DISC]=min(vector_d_DISC);
% [min_hw_DISC, I_min_hw_DISC]=min(vector_hw_DISC);
% % min_rt=min(vector_rt);
%
%
%
% %%
% % d_med=(vector_d(I_min_d)+vector_d(I_Max_d))/2*1000;           %I find the medium value of d in order to plot correcly the profile of the wear area [mm]
%
% % x=linspace(-d_disc(N)/2,d_disc(N)/2,MAX_SECTOR)';             %Symmetric
%
% x=linspace(0,d_disc(N),MAX_SECTOR_DISC)';


%% Results


for kk=1:N
    s(kk)=kk*l;
end


close all
% figure()
% plot(s,(hw_tot*10^6),'DisplayName','Total wear penetration depth');
% title('Total wear penetration depth [um]');
% xlabel('Test duration [m]');

% figure()
% plot(s,(AW_MATRIX(:,I_Max_Aw_DISC)*10^12),'DisplayName','Simulated Max cross-sectional area')
% hold on
% plot(s,(AW_MATRIX(:,I_min_Aw_DISC,:)*10^12),'DisplayName','Simulated min cross-sectional area')
% hold on
% plot(WearTrackArea(:,1),WearTrackArea(:,2),'xr','DisplayName','Exprimental values')
% title('Corss-sectional wear track area [um^2]')
% xlabel('Test duration [m]')
% legend('location','southeast')
% hold off
%
% figure()
% loglog(s,(R_T(:,I_Max_rt_DISC)*10^3),'DisplayName','Max Wear track profile radius')
% hold on
% loglog(s,(R_T(:,I_min_rt_DISC)*10^3),'DisplayName','Min Wear track profile radius')
% title('Wear track profile radius [mm]')
% xlabel('Test duration [m]')
% hold off
%
% figure()
% plot(s,(D(1:end-1,I_Max_d_DISC)*10^3),'DisplayName','Max Wear track amplitude')
% hold on
% plot(s,(D(1:end-1,I_min_d_DISC)*10^3),'DisplayName','Min Wear track amplitude')
% title('Wear track width [mm]')
% xlabel('Test duration [m]')
% hold off
%
% figure()
% plot(s,(HW(:,I_Max_hw_DISC)*10^6),'DisplayName','Max Simulated wear track depth')
% hold on
% plot(s,(HW(:,I_min_hw_DISC)*10^6),'DisplayName','Min Simulated wear track depth')
% hold on
% plot(WearTrackDepth(:,1),WearTrackDepth(:,2),'xr','DisplayName','Exprimental values')
% title('Wear track maximum depth [um]')
% xlabel('Test duration [m]')
% legend('location','southeast')
% hold off



figure()
plot(s,(AW_MATRIX(:,1)*10^12),'DisplayName','Simulated Max cross-sectional area')
% hold on
% plot(s,(AW_MATRIX(:,1,:)*10^12),'DisplayName','Simulated min cross-sectional area')
hold on
plot(WearTrackArea(:,1),WearTrackArea(:,2),'xr','DisplayName','Exprimental values')
title('Corss-sectional wear track area [um^2]')
xlabel('Test duration [m]')
legend('location','southeast')
hold off

figure()
loglog(s,(R_T(:,1)*10^3),'DisplayName','Max Wear track profile radius')
% hold on
% loglog(s,(R_T(:,1)*10^3),'DisplayName','Min Wear track profile radius')
title('Wear track profile radius [mm]')
xlabel('Test duration [m]')
hold off

figure()
plot(s,(D(1:end-1,1)*10^3),'DisplayName','Max Wear track amplitude')
% hold on
% plot(s,(D(1:end-1,1)*10^3),'DisplayName','Min Wear track amplitude')
title('Wear track width [mm]')
xlabel('Test duration [m]')
hold off

figure()
plot(s,(HW(:,1)*10^6),'DisplayName','Max Simulated wear track depth')
hold on
% plot(s,(HW(:,1)*10^6),'DisplayName','Min Simulated wear track depth')
% hold on
plot(WearTrackDepth(:,1),WearTrackDepth(:,2),'xr','DisplayName','Exprimental values')
title('Wear track maximum depth [um]')
xlabel('Test duration [m]')
legend('location','southeast')
hold off

figure()
plot(x,vector_hw_DISC(1:MAX_SECTOR_DISC)*10^6,'DisplayName','Wear track geometry')
title('Wear track maximum depth [um]')
xlabel('Wear track width [mm]')
legend('location','southeast')
hold off


% figure()
% plot(x,z,'DisplayName','Wear track geometry')
% title('Wear track maximum depth [um]')
% xlabel('Test duration [m]')
% legend('location','southeast')
% hold off

% figure()
% plot(1:size_1, results(:,1)*10^6, 'DisplayName', 'Hamrock & Dawson')
% hold on
% plot(1:size_1, results(:,4)*10^6,'.','MarkerIndices',1:10:size_1, 'DisplayName', 'Hertz theory')
% title('Major contact axis [um]')
% xlabel('Stroke count')
% legend('location','northwest')
% hold off
%
% figure()
% plot(1:size_1, results(:,2)*10^6, 'DisplayName', 'Hamrock & Dawson')
% hold on
% plot(1:size_1, results(:,5)*10^6, '.','MarkerIndices',1:10:size_1, 'DisplayName', 'Hertz theory')
% title('Minor contact axis [um]')
% xlabel('Stroke count')
% legend('location','southwest')
% hold off

% figure()
% plot(1:size_1, results(:,3), 'DisplayName', 'Hamrock & Dawson')
% hold on
% plot(1:size_1, results(:,6), 'DisplayName', 'Hertz theory')
% title('contact body convergence')
% xlabel('Stroke count')
% legend('location','northeast')
% hold off

% figure()
% plot(1:size_1, hw_tot(1:size_1)*10^6, 'DisplayName', 'Hamrock & Dawson')
% hold on
% plot(1:size_1, hw_tot_t(1:size_1)*10^6,'r--', 'DisplayName', 'Hertz theory')
% title('Wear track maximum depth [um]')
% xlabel('Stroke count')
% legend('location','southwest')
% hold off
