import sqlite3
from typing import List, Tuple
from customTypes import *
from log import logger

from patterns import *



class dbAPI(metaclass=SingletonMeta):
    '''API per interfacciarsi col db.
    Syntax: actions_object'''
    def __init__(self):
        # self.DB = "D:/Francesco/Scuola/Politecnico/Magistrale/Tesi/tesiAllocatoreCPLEX/data/dbV1.db"
        # self.DB = "//wsl$/Ubuntu/home/fra/Tesi/Interface-server/dbV1.db"
        self.DB = "D:/Francesco/Scuola/Politecnico/Magistrale/Tesi/sources/2023/dbV2.db"
        self.db = sqlite3.connect(self.DB)    
        self.log:logger = logger()    

    def get_Orientamenti(self):
        '''Return: 
            [orientamento, nomeCdl, tipoCdl]'''
        cur = self.db.cursor()
        cur.execute("SELECT * FROM Orientamento")
        rows = cur.fetchall()
        return rows
    
    def get_Orientamenti_withInsegnamenti(self):
        '''Seleziona solo gli Orientamenti a cui è associato almeno un Insegnamento.\n
        Return:
            [O.orientamento, O.nomdeCdl, O.tipoCdl]'''
        cur = self.db.cursor()
        sql = '''SELECT O.orientamento, O.nomeCdl, O.tipoCdl
                FROM Orientamento O
                WHERE (O.orientamento, O.nomeCdl, O.tipoCdl) IN (
                    SELECT IiO.orientamento, IiO.nomeCdl, IiO.tipoCdl
                    FROM Insegnamento_in_Orientamento IiO
                    GROUP BY IiO.orientamento, IiO.nomeCdl, IiO.tipoCdl
                    HAVING COUNT(*) > 0
                )
                '''
        cur.execute(sql)
        rows = cur.fetchall()
        return rows
    
    def get_Orientamenti_ofInsegnamento(self, ID_INC:int):
        '''Ritorna la lista di Orientamenti per un dato Insegnamento.\n
        Return:
            [tipoCdl, nomeCdl, orientamento]'''
        cur = self.db.cursor()
        sql = '''SELECT tipoCdl, nomeCdl, orientamento
                FROM Insegnamento_in_Orientamento
                WHERE ID_INC = ?'''
        cur.execute(sql, [ID_INC])
        return cur.fetchall()
    
    def get_Alfabetiche_ofOrientamento(self, orientamento, nomeCdl, tipoCdl, periodoDidattico):
        '''Ritorna tutte le alfabetiche (col rispettivo numero) di un dato Orientamento in un dato periodo didattico. Le alfabetiche
        si identificano come gli Insegnamenti con lo stesso codIns e differenti codIns\n
        Return:
            [COUNT(*) AS numAlf, I.titolo, I.CFU]'''
        # per il dato Orientamento, ritorna il numero di Insegnamenti con lo stesso nome che ne fanno parte, per ogni Insegnamento
        # eg: 2,3,1,1 -> nell'Orientamento c'è un Insegnamento con 2 istanze, uno con 3 e due con 1 istanza
        # sarebbe la query giusta ma nel db i ho più codIns assegnati allo stesso Insegnamento nello stesso Orientamento (quello che capita
        # a pescare i dati da più fonti diverse, si spera di aver poi l'accesso definitivo a tutti i dati di interesse dal gof)
        sql0 = '''SELECT COUNT(*) AS numAlf, IlC.codIns, I.titolo, I.CFU
                FROM Insegnamento_in_Orientamento IiO, Insegnamento I, Insegnamento_listCodIns IlC
                WHERE IiO.ID_INC = I.ID_INC AND I.ID_INC = IlC.ID_INC
                    AND IiO.orientamento = ? AND IiO.nomeCdl = ? AND IiO.tipoCdl = ? AND IiO.periodoDidattico = ?
                GROUP BY IlC.codIns, I.titolo, I.CFU
                HAVING COUNT(*) > 1
        ''' 
        sql1 = '''SELECT COUNT(*) AS numAlf, I.titolo, I.CFU
                FROM Insegnamento_in_Orientamento IiO, Insegnamento I
                WHERE IiO.ID_INC = I.ID_INC
                    AND IiO.orientamento = ? AND IiO.nomeCdl = ? AND IiO.tipoCdl = ? AND IiO.periodoDidattico = ?
                GROUP BY I.titolo, I.CFU
                HAVING COUNT(*) > 1
        '''
        cur = self.db.cursor()
        cur.execute(sql1, (orientamento, nomeCdl, tipoCdl, periodoDidattico))
        rows = cur.fetchall()
        return rows

    def get_Alfabetiche_WARNING(self):
        '''DEPRECATED
        \nRitorna tutte le alfabetiche (col rispettivo numero). 
        Le alfabetiche si identificano come gli Insegnamenti con lo stesso codIns e differenti codIns\n
        Return:
            [COUNT(*) AS numAlf, I.titolo, I.CFU]'''
        sql = '''SELECT COUNT(*) AS numAlf, I.titolo, I.CFU
                FROM Insegnamento I
                GROUP BY I.titolo, I.CFU
                HAVING COUNT(*) > 1'''
        cur = self.db.cursor()
        cur.execute(sql)
        return cur.fetchall()
    
    def get_Insegnamenti_inOrientamentoPeriodoDidattico_with_NAlfabetiche(self, orientamento, nomeCdl, tipoCdl, periodoDidattico, N):
        '''DEPRECATED
        \nRitorna tutti gli Insegnamenti con un determinato numero di alfabetiche N\n
        Return:
            [I.titolo, I.CFU]'''
        # dato tra gli Insegnamenti dell'Orientamento il numero di Insegnamenti con lo stesso nome, ritorna gli Insegnamenti
        # che fanno parte dell'Orientamento e che hanno il numero di alfabetiche voluto
        sql1 = '''SELECT DISTINCT I1.titolo, I1.CFU
                FROM Insegnamento I1, Insegnamento_in_Orientamento IiO1
                WHERE (I1.titolo, I1.CFU) IN (SELECT I.titolo, I.CFU
                                        FROM Insegnamento_in_Orientamento IiO, Insegnamento I
                                        WHERE IiO.ID_INC = I.ID_INC AND IiO.orientamento = ? AND IiO.nomeCdl = ? AND IiO.tipoCdl = ?
                                            AND IiO.periodoDidattico = ?
                                        GROUP BY I.titolo, I.CFU
                                        HAVING COUNT(*) = ?)
                    AND I1.ID_INC = IiO1.ID_INC AND IiO1.orientamento = ? AND IiO1.nomeCdl = ? AND IiO1.tipoCdl = ? AND IiO1.periodoDidattico = ?
                '''
        cur = self.db.cursor()
        cur.execute(sql1, (orientamento, nomeCdl, tipoCdl, periodoDidattico, N, orientamento, nomeCdl, tipoCdl, periodoDidattico))
        return cur.fetchall()
    
    def get_ID_INC_Insegnamento_ofOrientamento_WARNING(self, orientamento, nomeCdl, tipoCdl, periodoDidattico, titolo, CFU):
        '''Dato un Insegnamento (non posso usare l'ID_INC) ritrova la lista di codIns. In una versione definitiva questa funzione non
        sarà necessaria poichè potrò estrarre i codIns delle alfabetiche direttamente dal GOF (spero)\n
        Return:
            [IlC.ID_INC]'''
        sql = '''SELECT DISTINCT IlC.ID_INC
                FROM Insegnamento_listCodIns IlC, Insegnamento_in_Orientamento IiO, Insegnamento I
                WHERE IlC.ID_INC = IiO.ID_INC AND I.ID_INC = IiO.ID_INC AND
                    IiO.orientamento = ? AND IiO.nomeCdl = ? AND IiO.tipoCdl = ? AND IiO.periodoDidattico = ? AND I.titolo = ? AND I.CFU = ?
                ORDER BY IlC.ID_INC ASC'''
        cur = self.db.cursor()
        cur.execute(sql, (orientamento, nomeCdl, tipoCdl, periodoDidattico, titolo, CFU))
        return cur.fetchall()
    
    def get_ID_INC_Insegnamento_WARNING(self, titolo, CFU):
        ''''DEPRECATED
        \nDato un Insegnamento (non posso usare l'ID_INC) ritrova la lista di codIns, QUI NON CONSIDERO I VARI ORIENTAMENTI. 
        In una versione definitiva questa funzione non sarà necessaria poichè potrò estrarre i codIns delle alfabetiche direttamente dal GOF.\n
        Return:
            [ID_INC]'''
        sql = '''SELECT ID_INC
                FROM Insegnamento
                WHERE titolo = ? AND CFU = ?'''
        cur = self.db.cursor()
        cur.execute(sql, (titolo, CFU))
        return cur.fetchall()

    def update_Alfabetica_ofInsegnamentoInOrientamento(self, ID_INC, orientamento, nomeCdl, tipoCdl, periodoDidattico, alfabetica:str) -> None:
        '''Aggiorna SOLO per gli Insegnamenti nell'Orientamento l'alfabetica corrispondente'''
        sql = '''UPDATE Insegnamento_in_Orientamento
                SET alfabetica = ?
                WHERE ID_INC = ? AND orientamento = ? AND nomeCdl = ? AND tipoCdl = ? AND periodoDidattico = ?'''
        cur = self.db.cursor()
        cur.execute(sql, (alfabetica, ID_INC, orientamento, nomeCdl, tipoCdl, periodoDidattico))
        self.db.commit()
        
    def reset_Alfabetica(self) -> None:
        '''Reset di tutte le alfabetiche per tutti gli Insegnamenti in tutti gli Orientamenti'''
        sql = '''UPDATE Insegnamento_in_Orientamento SET alfabetica = 0'''
        cur = self.db.cursor()
        cur.execute(sql)
        self.db.commit()
    
    def get_Insegnamenti_withListCodIns(self):
        '''Return: 
            [I.ID_INC, IL.codIns, I.nStudenti, I.nStudentiFreq, I.collegio, I.titolo, I.CFU, I.oreLez, I.titolare'''
        cur = self.db.cursor()
        sql = '''SELECT I.ID_INC, IL.codIns, nStudenti, nStudentiFreq, collegio, titolo, CFU, oreLez, titolare 
                FROM Insegnamento I, Insegnamento_listCodIns IL
                WHERE I.ID_INC = IL.ID_INC'''    
        cur.execute(sql)
        return cur.fetchall()

    def update_Insegnamenti_setNStudenti(self, ID_INC, nStudenti, nStudentiFreq):
        cur = self.db.cursor()
        sql = '''UPDATE Insegnamento
                SET nStudenti = ?, nStudentiFreq = ?
                WHERE ID_INC = ?'''
        cur.execute(sql, (nStudenti, nStudentiFreq, ID_INC))
        self.db.commit()
        
    def update_Insegnamento_setTitolare(self, ID_INC, titolare:str):
        cur = self.db.cursor()
        sql = '''UPDATE Insegnamento
                SET titolare = ?
                WHERE ID_INC = ?'''
        cur.execute(sql, (titolare, ID_INC))
        self.db.commit()
        
    def get_Insegnamenti_withOrientamento(self, orientamento, nomeCdl, tipoCdl):
        '''Ritorna gli Insegnamenti di un Orientamento
        nb: tipoCdl dev'essere compliant con i valori nel db (ie 1 o Z)
        
        Return: 
            [I.ID_INC, I.nStudenti, I.nStudentiFreq, I.collegio, I.titolo, I.CFU, I.oreLez, I.titolare, IiO.nStudenti, IiO.tipoInsegnamento
            IiO.periodoDidattico, IiO.alfabetica]'''
        cur = self.db.cursor()
        sql = '''SELECT I.ID_INC, I.nStudenti, I.nStudentiFreq, I.collegio, I.titolo, I.CFU, I.oreLez, I.titolare, IiO.nStudenti, IiO.tipoInsegnamento, IiO.periodoDidattico, IiO.alfabetica
                FROM Insegnamento_in_Orientamento IiO, Insegnamento I
                WHERE IiO.ID_INC = I.ID_INC AND IiO.orientamento = ? AND IiO.nomeCdl = ? AND Iio.tipoCdl = ?'''
        cur.execute(sql, (orientamento, nomeCdl, tipoCdl))
        return cur.fetchall()
    
    def get_Insegnamenti(self):
        '''Return:
            [ID_INC, nStudenti, nStudentiFreq, collegio, titolo, CFU, oreLez, titolare]'''
        cur = self.db.cursor()
        sql = '''SELECT * FROM Insegnamento'''
        cur.execute(sql)
        return cur.fetchall()
    
    def get_Insegnamento(self, ID_INC:int):
        '''Return:
            [ID_INC, nStudenti, nStudentiFreq, collegio, titolo, CFU, oreLez, titolare]'''
        cur = self.db.cursor()
        sql = '''SELECT * From Insegnamento WHERE ID_INC = ?'''
        cur.execute(sql, [ID_INC])
        return cur.fetchall()

    def get_Insegnamenti_withCodIns(self):
        '''Oltre ai dati della tabella Insegnamento ho anche il codIns (quello che si vede sul sito del Poli).
        Return:
            [I.ID_INC, I.nStudenti, I.nStudentiFreq, I.collegio, I.titolo, I.CFU, I.oreLez, I.titolare, IlC.codIns]
        '''
        cur = self.db.cursor()
        sql = '''SELECT I.ID_INC, I.nStudenti, I.nStudentiFreq, I.collegio, I.titolo, I.CFU, I.oreLez, I.titolare, IlC.codIns
                FROM Insegnamento I, Insegnamento_listCodIns IlC
                WHERE I.ID_INC = IlC.ID_INC'''
        cur.execute(sql)
        return cur.fetchall()
    
    def get_CodInsInsegnamento(self, ID_INC:str):
        '''Return: 
            [codIns]
        '''
        cur = self.db.cursor()
        sql = '''SELECT codIns
                FROM Insegnamento_listCodIns
                WHERE ID_INC = ?'''
        cur.execute(sql, [ID_INC])
        return cur.fetchall()
    
    def delete_InsegnamentoInOrientamento(self, ID_INC, orientamento, nomeCdl, tipoInsegnamento, tipoCdl):
        '''Cancella una entry dalla tabella Insegnamento_in_Orientamento'''
        cur = self.db.cursor()
        sql = '''DELETE FROM Insegnamento_in_Orientamento
                WHERE ID_INC = ? AND orientamento = ? AND nomeCdl = ? AND tipoInsegnamento = ? AND tipoCdl = ?'''
        cur.execute(sql, (ID_INC, orientamento, nomeCdl, tipoInsegnamento, tipoCdl))
        self.db.commit()
    
    def delete_allInsegnamentiInOrientamento(self, orientamento, nomeCdl, tipoCdl):
        '''Cancella tutte le entry relative ad un Orientamento dalla tabella Insegnamento_in_Orientamento'''
        cur = self.db.cursor()
        sql = '''DELETE FROM Insegnamento_in_Orientamento
                WHERE orientamento = ? AND nomeCdl = ? AND tipoCdl = ?'''
        cur.execute(sql, (orientamento, nomeCdl, tipoCdl))
        if cur.rowcount == 0:
            self.log.info_logUtils("WARN dbAPI.delete_allInsegnamentiInOrientamento(): non è stata cancellata nessuna riga")
        self.db.commit()
        
       
       
    # API aggiunta dati puntuale a db (usate solo per ritoccare i dati)
        
    def add_Insegnamento(self, ID_INC:int, nStudenti:int, nStudentiFreq:int, collegio:str, titolo:str, CFU:str, oreLez:int, titolare:str):
        '''Aggiunge un Insegnamento al db. Da usare con estrema catuela'''
        cur = self.db.cursor()
        sql = '''INSERT INTO Insegnamento(ID_INC,nStudenti,nStudentiFreq,collegio,titolo,CFU,oreLez,titolare)
                VALUES (?,?,?,?,?,?,?,?)'''
        cur.execute(sql, (ID_INC,nStudenti,nStudentiFreq,collegio,titolo,CFU,oreLez,titolare))
        self.db.commit()
    
    def add_InsegnamentiInOrientamento(self, orientamento, nomeCdl, tipoCdl, periodoDidattico, listInsegnamenti:List[Tuple[str,TipoInsegnamento]], 
                                       nStudenti=0):
        '''Aggiunge a Insegnamento_in_Orientamento tutti gli Insegnamenti passati nella lista come Insegnamenti per l'Orientamento e il cdl 
        correnti; con il rispettivo grado di importanza (TipoInsegnamento) e il periodo didattico.\n
        listInsegnamenti = [(ID_INC,TipoInsegnamento)]. Alfabetica settata a 0 per default'''
        cur = self.db.cursor()
        sql = '''INSERT INTO Insegnamento_in_Orientamento(ID_INC,orientamento,nomeCdl,tipoInsegnamento,tipoCdl,periodoDidattico,nStudenti,alfabetica)
                VALUES (?,?,?,?,?,?,?,?)'''
        for ins in listInsegnamenti:
            try:
                cur.execute(sql, (ins[0], orientamento, nomeCdl, getStrFromTipoInsegnamento(ins[1]), tipoCdl, periodoDidattico, nStudenti, '0'))
            except Exception as e:
                self.log.info_logUtils("ERR dbAPI.add_InsegnamentiInOrientamento(): " + str(e))
                self.log.info_logUtils("ERR for data: " + str(ins[0]) + ", " + str(orientamento) + ", " + str(nomeCdl) + ", " + str(tipoCdl) + ", " + 
                          str(periodoDidattico) + ", " + getStrFromTipoInsegnamento(ins[1]))
        self.db.commit()
        
    def update_InsegnamentiInOrientamento(self, orientamento:str, nomeCdl:str, listInsegnamenti:List[Tuple[str,TipoInsegnamento]], 
                                          tipoCdl:str, periodoDidattico:str):
        '''Aggiorna in Insegnamento_in_Orientamento tutti gli Insegnamenti passati nella lista come Insegnamenti per l'Orientamento e il cdl 
        correnti; con il relativo grado di importanza (TipoInsegnamento) e il periodo didattico.'''
        cur = self.db.cursor()
        sql = '''UPDATE Insegnamento_in_Orientamento
                SET tipoInsegnamento = ?, periodoDidattico = ?
                WHERE ID_INC = ? AND orientamento = ? AND nomeCdl = ? AND tipoCdl = ?'''
        for ins in listInsegnamenti:
            try:
                '''Modify only if of a less important level'''
                sql1 = '''SELECT *
                        FROM Insegnamento_in_Orientamento
                        WHERE ID_INC = ? AND orientamento = ? AND nomeCdl = ? AND tipoInsegnamento <> "Sconosciuto"'''
                cur.execute(sql1, (ins[0], orientamento, nomeCdl))
                res = cur.fetchall()
                if len(res) > 0:
                    print(res)
                
                cur.execute(sql, (getStrFromTipoInsegnamento(ins[1]), periodoDidattico, ins[0], orientamento, nomeCdl, tipoCdl))
            except Exception as e:
                self.log.info_logUtils("ERR dbAPI.update_InsegnamentiInOrientamento(): " + str(e))
                self.log.info_logUtils("ERR for data: " + str(ins[0]) + ", " + str(orientamento) + ", " + str(nomeCdl) + ", " + str(tipoCdl) 
                        + ", " + getStrFromTipoInsegnamento(ins[1]))
        self.db.commit()              
    
        
    def add_InsegnamentoInOrientamento(self, ID_INC:int, orientamento:str, nomeCdl:str, tipoInsegnamento:TipoInsegnamento, tipoCdl:str, periodoDidattico:str,
                                       nStudenti:int, alfabetica:str, commit:bool = False):
        '''Aggiunge a Orientamento un nuovo Orientamento.
            - tipoCdl in ('1','Z')
            - alfabetica in ('0', '1', '2', '3') '''
        cur = self.db.cursor()
        sql = '''INSERT INTO Insegnamento_in_Orientamento(ID_INC,orientamento,nomeCdl,tipoInsegnamento,tipoCdl,periodoDidattico,nStudenti,alfabetica)
                VALUES (?,?,?,?,?,?,?,?)'''
        if not commit:
            cur.execute(sql, (ID_INC, orientamento, nomeCdl, getStrFromTipoInsegnamento(tipoInsegnamento), tipoCdl, periodoDidattico, nStudenti, alfabetica))
            return            
        try:
            cur.execute(sql, (ID_INC, orientamento, nomeCdl, getStrFromTipoInsegnamento(tipoInsegnamento), tipoCdl, periodoDidattico, nStudenti, alfabetica))
        except Exception as e:
            print(e)
            self.log.error_log(str(e))
            self.db.rollback()
            self.log.info_logUtils("ERR dbAPI.add_InsegnamentiInOrientamento(): " + str(e))
            self.log.info_logUtils("ERR for data: " + str(ID_INC) + ", " + str(orientamento) + ", " + str(nomeCdl) + ", " + str(tipoCdl) + ", " + 
                        str(periodoDidattico) + ", " + getStrFromTipoInsegnamento(tipoInsegnamento))
        else:
            self.db.commit()
        
    def add_Insegnamento_codIns(self, ID_INC:int, codIns:str):
        '''Aggiunge a Insegnamento_listCodIns la nuova tupla (ID_INC, codIns). Serve per il mapping tra codIns (identificativi degli 
        Insegnamenti nell'Orientamento, ie sul sito) e ID_INC'''
        cur = self.db.cursor()
        sql = '''INSERT INTO Insegnamento_listCodIns(ID_INC,codIns) VALUES (?,?)'''
        cur.execute(sql,(ID_INC, codIns))
        self.db.commit()
        
    def add_Docente(self, docente:str):
        '''Aggiunge a Docete un nuovo Docente presente nel piano allocazione'''
        cur = self.db.cursor()
        sql = '''INSERT INTO Docente(Cognome) VALUES (?)'''
        cur.execute(sql, [docente])
        self.db.commit()
        
    def add_DocenteInInsegnamento(self, docente:str, ID_INC:int, nOre:int = 0, tipoLez:str = "L"):
        '''Aggiunge a Docente_in_Insegnamento la nuova entry'''
        cur = self.db.cursor()
        sql = '''INSERT INTO Docente_in_Insegnamento(Cognome,ID_INC,nOre,tipoLez) VALUES (?,?,?,?)'''
        cur.execute(sql, (docente,ID_INC,nOre,tipoLez))
        self.db.commit()
        
    def add_Corso_di_laurea(self, nomeCdl:str, tipoCdl:str):
        '''Aggiunge un Corso di Laurea nel db. tipoCdl in ('1','Z')'''
        cur = self.db.cursor()
        sql = '''INSERT INTO Corso_di_laurea(tipoCdl,nomeCdl) VALUES (?,?)'''
        cur.execute(sql, (tipoCdl,nomeCdl))
        self.db.commit()
        
    def add_Orientamento(self, orientamento:str, nomeCdl:str, tipoCdl:str):
        '''Aggiunge un Orientamento nel db. tipoCdl in ('1','Z')'''
        cur = self.db.cursor()
        sql = '''INSERT INTO Orientamento(orientamento,nomeCdl,tipoCdl) VALUES (?,?,?)'''
        cur.execute(sql, (orientamento,nomeCdl,tipoCdl))
        self.db.commit()
        
    def update_ID_INCInsegnamento(self, oldID_INC:str, newID_INC:str):
        '''Modifica in tutte le tabelle l'ID_INC associato ad un Insegnamento'''
        cur = self.db.cursor()
        try:
            sql = '''UPDATE Insegnamento
                    SET ID_INC = ?
                    WHERE ID_INC = ?'''
            cur.execute(sql, [newID_INC, oldID_INC])
            sql = '''UPDATE Insegnamento_in_Orientamento
                    SET ID_INC = ?
                    WHERE ID_INC = ?'''
            cur.execute(sql, [newID_INC, oldID_INC])
            sql = '''UPDATE Docente_in_Insegnamento
                    SET ID_INC = ?
                    WHERE ID_INC = ?'''
            cur.execute(sql, [newID_INC, oldID_INC])     
            sql = '''UPDATE Insegnamento_listCodIns
                    SET ID_INC = ?
                    WHERE ID_INC = ?'''
            cur.execute(sql, [newID_INC, oldID_INC])           
            sql = '''UPDATE Slot
                    SET ID_INC = ?
                    WHERE ID_INC = ?'''
            cur.execute(sql, [newID_INC, oldID_INC])        
            self.db.commit()
        except Exception as e:
            self.db.rollback()
            print(e)
            self.log.error_log(str(e))
            
    def delete_ALLForImportFromExcel(self):
        cur = self.db.cursor()
        try:
            sql = '''DELETE FROM Docente'''
            cur.execute(sql)
            sql = '''DELETE FROM Insegnamento'''
            cur.execute(sql)
            sql = '''DELETE FROM Docente_in_Insegnamento'''
            cur.execute(sql)
            sql = '''DELETE FROM Corso_di_laurea'''
            cur.execute(sql)
            sql = '''DELETE FROM Orientamento'''
            cur.execute(sql)
            sql = '''DELETE FROM Insegnamento_in_Orientamento'''
            cur.execute(sql)
            sql = '''DELETE FROM Insegnamento_listCodIns'''
            cur.execute(sql)
            self.db.commit()
        except Exception as e:
            self.db.rollback()
            self.log.error_log(str(e))
            print(e)
        
        
        
    def get_Insegnamento_fromTitolo(self, titolo):
        '''Ritorna l'Insegnamento/i in base al titolo.\n
        Return:
            [ID_INC, nStudenti, nStudentiFreq, collegio, titolo, CFU, oreLez, titolare]'''
        cur = self.db.cursor()
        sql = '''SELECT * 
                FROM Insegnamento 
                WHERE titolo = ?'''
        cur.execute(sql, [titolo])
        return cur.fetchall()
    
    def get_ID_INC_InsegnamentiInCdl(self, cdlMagistrale:bool = True):
        '''Ritorna una lista di ID_INC di tutti gli insegnamenti in corsi magistrali'''
        cur = self.db.cursor()
        if cdlMagistrale:
            sql = '''SELECT ID_INC FROM Insegnamento_in_Orientamento WHERE tipoCdl = \'Z\''''
        else:
            sql = '''SELECT ID_INC FROM Insegnamento_in_Orientamento WHERE tipoCdl = \'1\''''            
        cur.execute(sql)
        return cur.fetchall()
    
    def get_ID_INC_Insegnamenti_inOrientamento(self, cdl, orientamento, tipoCdl):
        '''Ritorna una lista di ID_INC di tutti gli insegnamenti in corsi magistrali'''
        cur = self.db.cursor()
        sql = '''SELECT ID_INC 
                FROM Insegnamento_in_Orientamento 
                WHERE orientamento = ? AND nomeCdl = ? AND tipoCdl = ?'''
        cur.execute(sql, (orientamento, cdl, tipoCdl))
        return cur.fetchall()
    
    
    
    # API per interrogazione db ed analisi risultati
    
    def get_groupPianiAllocazione(self, pianoAllocazionePrefix:str):
        '''Ritorna la lista dei PianiAllocazione che cominciano con lo stesso prefisso (ie che verosimilmente sono stati aggiunti al db
        nella stessa esecuzione del solver in modalità Optimizer).\n
        Return:
            [pianoAllocazione]'''
        cur = self.db.cursor()
        sql = "SELECT pianoAllocazione FROM PianoAllocazione WHERE pianoAllocazione LIKE ?"
        cur.execute(sql, [pianoAllocazionePrefix + '%'])
        return cur.fetchall()        
    
    def get_pianiAllocazione(self):
        '''Ritorna la lista dei PianiAllocazione presenti nel db.\n
        Return:
            [pianoAllocazione, descrizione]'''
        cur = self.db.cursor()
        sql = "SELECT pianoAllocazione, descrizione FROM PianoAllocazione"
        cur.execute(sql)
        return cur.fetchall()
    
    def get_pianoAllocazioneDocente(self, pianoAllocazione:str, docente:str):
        '''Ritorna tutti gli Slot allocati per un Docente in un certo PianoAllocazione.\n
        Return:
            [S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, S.capienzaAula, S.squadra, S.preseElettriche]'''
        cur = self.db.cursor()
        sql = '''SELECT S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, S.capienzaAula, S.squadra, S.preseElettriche
                FROM Slot S, Docente_in_Slot DiS
                WHERE DiS.idSlot = S.idSlot AND DiS.pianoAllocazione = S.pianoAllocazione AND DiS.Cognome = ? AND DiS.pianoAllocazione = ?'''
        cur.execute(sql, (docente, pianoAllocazione))
        return cur.fetchall()
    
    def get_pianoAllocazioneID_INC_withDocenti(self, pianoAllocazione:str, ID_INC:int):
        '''Ritorna tutti gli Slot allocati per un Insegnamento in un certo PianoAllocazione con i relativi Docenti.\n
        Return:
            [S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, DiS.Cognome]
            '''
        cur = self.db.cursor()
        sql = '''SELECT S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, DiS.Cognome
                FROM Slot S, Docente_in_Slot DiS
                WHERE S.pianoAllocazione = ? AND S.ID_INC = ? AND S.idSlot = DiS.idSlot AND S.pianoAllocazione = DiS.pianoAllocazione
                ORDER BY S.idSlot'''
        cur.execute(sql, (pianoAllocazione, ID_INC))
        return cur.fetchall()
    
    def get_slotInDayFasciaOrariaFromPianoAllocazione(self, pianoAllocazione:str, day:str, fasciaOraria:str):
        '''Ritorna tutti gli Slot allocati in un certo (Day,FasciaOraria) per il determinato PianoAllocazione\n
        Return:
            [S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, I.titolo, I.titolare, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, S.capienzaAula, S.squadra, S.preseElettriche]
        '''
        cur = self.db.cursor()
        sql = '''SELECT S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, I.titolo, I.titolare, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, S.capienzaAula, S.squadra, S.preseElettriche
                FROM Slot S, Insegnamento I
                WHERE I.ID_INC = S.ID_INC AND S.pianoAllocazione = ? AND S.giorno = ? AND S.fasciaOraria = ?
                '''
        cur.execute(sql, (pianoAllocazione, day, fasciaOraria))
        return cur.fetchall()
    
    def get_slotInDayFromPianoAllocazione(self, pianoAllocazione:str, day:str):
        '''Ritorna tutti gli Slot allocati in un certo Day per il determinato PianoAllocazione\n
        Return:
            [S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, I.titolo, I.titolare, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, S.capienzaAula, S.squadra, S.preseElettriche]
        '''
        cur = self.db.cursor()
        sql = '''SELECT S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, I.titolo, I.titolare, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, S.capienzaAula, S.squadra, S.preseElettriche
                FROM Slot S, Insegnamento I
                WHERE I.ID_INC = S.ID_INC AND S.pianoAllocazione = ? AND S.giorno = ?
                ORDER BY S.idSlot
                '''
        cur.execute(sql, (pianoAllocazione, day))
        return cur.fetchall()
    
    def get_slotInFasciaOrariaFromPianoAllocazione(self, pianoAllocazione:str, fasciaOraria:str):
        '''Ritorna tutti gli Slot allocati in una certa FasciaOraria (Day anche diverso) per il determinato PianoAllocazione\n
        Return:
            [S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, I.titolo, I.titolare, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, S.capienzaAula, S.squadra, S.preseElettriche]
        '''
        cur = self.db.cursor()
        sql = '''SELECT S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, I.titolo, I.titolare, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, S.capienzaAula, S.squadra, S.preseElettriche
                FROM Slot S, Insegnamento I
                WHERE I.ID_INC = S.ID_INC AND S.pianoAllocazione = ? AND S.fasciaOraria = ?
                ORDER BY S.idSlot
                '''
        cur.execute(sql, (pianoAllocazione, fasciaOraria))
        return cur.fetchall()
    
    def get_slotFromPianoAllocazione(self, pianoAllocazione:str):
        '''Ritorna tutti gli Slot allocati in un determinato PianoAllocazione\n
        Return:
            [S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, I.titolo, I.titolare, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, S.capienzaAula, S.squadra, S.preseElettriche]
        '''
        cur = self.db.cursor()
        sql = '''SELECT S.pianoAllocazione, S.idSlot, S.nStudentiAssegnati, S.tipoLez, S.numSlotConsecutivi, S.ID_INC, I.titolo, I.titolare, S.giorno, S.fasciaOraria, S.tipoLocale, S.tipoErogazione, S.capienzaAula, S.squadra, S.preseElettriche
                FROM Slot S, Insegnamento I
                WHERE I.ID_INC = S.ID_INC AND S.pianoAllocazione = ?
                ORDER BY S.Giorno
                '''
        cur.execute(sql, [pianoAllocazione])
        return cur.fetchall()            
        
    def get_ID_INCFromPianoAllocazione(self, pianoAllocazione:str):
        '''Ritorna la lista di ID_INC presenti all'interno di un PianoAllocazione'''
        cur = self.db.cursor()
        sql = '''SELECT DISTINCT ID_INC
                FROM Slot
                WHERE pianoAllocazione = ?'''
        cur.execute(sql, [pianoAllocazione])
        return cur.fetchall()
    
    def get_DocentiFromSlotPianoAllocazione(self, pianoAllocazione:str, idSlot:str):
        '''Ritorna la lista di Docenti in uno Slot di un PianoAllocazione'''
        cur = self.db.cursor()
        sql = '''SELECT Cognome 
                FROM Docente_in_Slot
                WHERE idSlot = ? AND pianoAllocazione = ?'''
        cur.execute(sql, (idSlot, pianoAllocazione))
        return cur.fetchall()
    
    
    # API per scrittura sol a db
    
    def add_pianoAllocazione(self, pianoAllocazione:str, descrizione:str) -> bool:
        '''Crea un nuovo pianoAllocazione se non ancora presente in PianoAllocazione'''
        cur = self.db.cursor()
        sql = '''SELECT * FROM PianoAllocazione WHERE pianoAllocazione = ?'''
        cur.execute(sql, [pianoAllocazione])
        if len(cur.fetchall()) > 0:
            self.log.info_logUtils("WARN dbAPI.add_pianoAllocazione(): pianoAllocazione " + pianoAllocazione + " già presente")
            return False
        sql1 = '''INSERT INTO PianoAllocazione(pianoAllocazione, descrizione) VALUES (?, ?)'''
        cur.execute(sql1, (pianoAllocazione, descrizione))
        self.db.commit()
        return True
    
    def add_docenteInSlot(self, docente:str, idSlot:str, pianoAllocazione:str, commit:bool = True):
        '''Aggiunge una entry in Docente_in_Slot'''
        cur = self.db.cursor()
        sql = '''INSERT INTO Docente_in_Slot(Cognome, idSlot, pianoAllocazione) VALUES (?,?,?)'''
        cur.execute(sql, (docente, idSlot, pianoAllocazione))
        if commit:
            self.db.commit()
    
    def add_slotPianoAllocazione(self, pianoAllocazione:str, idSlot:str, nStudentiAssegnati:int, tipoLez:str, numSlotConsecutivi:int,
                                 ID_INC:int, giorno:str, fasciaOraria:str, tipoLocale:str, tipoErogazione:str,
                                 capienzaAula:str, squadra:str, preseElettriche:str, commit:bool = True):
        '''Aggiunge uno Slot allocato per un determinato PianoAllocazione alla tabella Slot'''
        cur = self.db.cursor()
        sql = '''INSERT INTO Slot(pianoAllocazione, idSlot, nStudentiAssegnati, tipoLez, numSlotConsecutivi, ID_INC, giorno, fasciaOraria, tipoLocale, tipoErogazione, capienzaAula, squadra, preseElettriche)
                VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?)'''
        cur.execute(sql, (pianoAllocazione, idSlot, nStudentiAssegnati, tipoLez, numSlotConsecutivi, ID_INC, giorno, fasciaOraria, tipoLocale, tipoErogazione, capienzaAula, squadra, preseElettriche))
        if commit:
            self.db.commit()
    
    def delete_pianoAllocazione(self, pianoAllocazione:str, commit:bool = True):
        '''Cancellazione di un pianoAllocazione da PianoAllocazione e di tutti i suoi slots da Slot'''
        cur = self.db.cursor()
        sql = '''DELETE FROM Slot
                WHERE pianoAllocazione = ?'''
        cur.execute(sql, [pianoAllocazione])
        if cur.rowcount == 0:
            self.log.info_logUtils("WARN dbAPI.delete_pianoAllocazione(): non è stata cancellata nessuna riga di Slot")
        sql1 = '''DELETE FROM Docente_in_Slot WHERE pianoAllocazione = ?'''
        cur.execute(sql1, [pianoAllocazione])
        if cur.rowcount == 0:
            self.log.info_logUtils("dbAPI.delete_pianoAllocazione(): non è stata cancellata nessuna riga di Docente_in_Slot")        
        sql2 = '''DELETE FROM PianoAllocazione WHERE pianoAllocazione = ?'''
        cur.execute(sql2, [pianoAllocazione])
        if commit:
            self.db.commit()
        
    def commit(self):
        self.db.commit()
        
    def rollback(self):
        self.db.rollback()
