from typing import List, Dict, Set
from log import logger

from Slot import *
from Vincoli import *
from templates import MetaInsegnamento, Orientamento, Insegnamento
from customTypes import *
from patterns import *

from dbAPI import dbAPI
from data.tables import list_ID_INC_parallelizzabili
from Parameter import Parameter



class AuxiliaryStruct(metaclass=SingletonMeta):
    def __init__(self):
        '''Contiene tutte le strutture ausiliarie'''
        self.NUM_DAY = 5
        self.NUM_SLOT_PER_DAY = 7
        self.testing:bool = True
        self.log:logger = logger()
        
        # parametri di comando
        self.PARAM:Parameter = Parameter()        
        # per interfacciarsi col db
        self.dbAPI:dbAPI = dbAPI()
        
        self.endLoadSlotReq:bool = False
        self.allowedStuffCreation:bool = False
        # per permettere creazione map_limitHourDocentePerDay e map_indisponibilitaDocenti
        self.loadFileHardConst:bool = False
        # per permettere la creazione della lista di insegnamenti di ogni orientamento
        self.loadInsegnamenti:bool = False
        self.loadOrientamenti:bool = False
        # per permettere la creazione del mapping Orientamento -> slotId (per ogni slotId presente nell'orientamento)
        self.loadInsegnamenti_in_Orientamento:bool = False
        
        self.load_map_strDocenti_to_idDocenti:bool = False
        self.map_strDocenti_to_idDocenti:Dict[str, int] = dict()
        '''mapping strDocente -> idDocente'''
        self.load_map_idDocenti_to_strDocenti:bool = False
        self.map_idDocenti_to_strDocenti:List[str] = list()
        '''mappimg idDocente -> strDocente'''
        
        self.load_map_strSlotId_to_idSlot:bool = False
        self.map_strSlotId_to_idSlot:Dict[str, int] = dict()
        '''mapping strSlotId -> idSlot'''
        self.load_pianoAllocazione:bool = False
        self.pianoAllocazione:List[SlotReq] = list()
        '''Lista di tutti gli slot da allocare. Tramite l'id nella lista ho anche il mapping idSlot -> strSlotId'''
        
        self.load_list_vincoliInsegnamenti:bool = False
        self.list_vincoliInsegnamenti:List[Vincolo] = list()
        '''Lista di tutti i Vincoli degli Insegnamenti caricati dai vari file xml'''
        self.load_list_operatoriInsegnamenti:bool = False
        self.list_operatoriInsegnamenti:List[Operatore] = list()
        '''Lista di tutti gli Operatori degli Insegnamenti caricati dai vari file xml'''
        
        self.load_list_slotInDocente:bool = False
        self.list_slotInDocente:List[List[int]] = list()
        '''Per ogni docente ho la lista degli slot in cui deve essere presente'''
        self.load_list_docentiInSlot:bool = False
        self.list_docentiInSlot:List[List[int]] = list()
        '''Per ogni Slot ho la lista dei docenti che devono essere presenti'''
        
        self.load_list_slotInInsegnamento:bool = False
        self.list_slotInInsegnamento:List[List[int]] = list()
        '''Per ogni Insegnamento ho la lista di TUTTI gli Slot che lo costituiscono'''
        
        self.load_list_metaInsegnamenti:bool = False
        self.list_metaInsegnamenti:List[MetaInsegnamento] = list()
        '''Lista di tutti gli Insegnamenti, ogni SlotReq ha un ptr a un item di questa lista'''
        self.load_map_MetaInsegnamento_to_IdInsegnamento:bool = False
        self.map_MetaInsegnamento_to_IdInsegnamento:Dict[str,int] = dict()
        '''mapping ID_INC -> idInsegnamento'''
        self.load_list_Insegnamenti:bool = False
        self.list_Insegnamenti:List[Insegnamento] = list()
        '''Lista di tutti gli Insegnamenti, ogni SlotReq ha un ptr a un item di questa lista. E' parallela a self.list_metaInsegnamenti
        ma contiene le info pescate dal db'''
        
        self.load_list_FileDone:bool = False
        self.list_FileDone:List[str] = list()
        '''Lista file di TemplateOrario caricati'''
        
        self.load_list_Orientamenti:bool = False
        self.list_Orientamenti:List[Orientamento] = list()
        '''Lista di tutti gli orientamenti nel db'''
        self.load_map_Orientamento_to_IdOrientamento:bool = False
        self.map_Orientamento_to_IdOrientamento:Dict[str,int] = dict()
        '''mapping Orientamento (orientamento.nomeCdl.tipoCdl) -> idOrientamento. Corrispondente all'index in self.list_Orientamenti'''
        
        self.load_list_InsegnamentiInOrientamento:bool = False
        self.list_InsegnamentiInOrientamento:List[List[int]] = list()
        '''Per ogni Orientamento contiene la lista di Insegnamenti che ne fanno parte'''
        self.load_list_InsegnamentiInOrientamento_numIscritti:bool = False
        self.list_InsegnamentiInOrientamento_numIscritti:List[List[int]] = list()
        '''Per ogni Orientamento ho nella posizione corrispondente alla struttura list_InsegnamentiInOrientamento il
        numero di iscritti a quell'orientamento (TODO: dove posso recuperare i dati?)\n
        list_InsegnamentiInOrientamento[0][1] -> idInsegnamento=0; list_InsegnamentiInOrientamento_numIscritti[0][1] -> 20
        significa che per quell'Insegnamento con id=0 ci sono 20 studenti iscritti a quel determinato Orientamento'''
        self.load_list_InsegnamentiInOrientamento_tipo:bool = False
        self.list_InsegnamentiInOrientamento_tipo:List[List[TipoInsegnamento]] = list()
        '''Stesso discorso di list_InsegnamentiInOrientamento_numIscritti ma tengo traccia del TipoInsegnamento'''
        self.load_list_InsegnamentiInOrientamento_periodoDidattico:bool = False
        self.list_InsegnamentiInOrientamento_periodoDidattico:List[List[PeriodoDidattico]] = list()
        '''Stesso discorso di list_InsegnamentiInOrientamento_numIscritti ma tengo traccia del PeriodoDidattico'''
        self.load_list_InsegnamentiInOrientamento_alfabetica:bool = False
        self.list_InsegnamentiInOrientamento_alfabetica:List[List[Alfabetica]] = list()
        '''Stesso discorso di list_InsegnamentiInOrientamento_numIscritti ma tengo traccia dell'Alfabetica (se presente)'''
        
        # le strutture sopra mi servono come step intermedi per arrivare a queste che rappresentano la stessa cosa
        # ma a livello di slotId
        self.load_list_slotIdInOrientamento:bool = False
        self.list_slotIdInOrientamento:List[List[int]] = list()
        '''Per ogni Orientamento contiene la lista di slotId che ne fanno parte'''
        self.load_list_slotIdInOrientamento_tipo:bool = False
        self.list_slotIdInOrientamento_tipo:List[List[TipoInsegnamento]] = list()
        '''Per ogni Orientamento contiene per ogni slotId il corrispondente TipoDiInsegnamento rispetto all'Orientamento'''
        self.load_list_slotIdInOrientamento_periodoDidattico:bool = False
        self.list_slotIdInOrientamento_periodoDidattico:List[List[PeriodoDidattico]] = list()
        '''Per ogni Orientamento contiene per ogni slotId il corrispondente PeriodoDidattico rispetto all'Orientamento'''
        self.load_list_slotIdInOrientamento_alfabetica:bool = False
        self.list_slotIdInOrientamento_alfabetica:List[List[Alfabetica]] = list()
        '''Per ogni Orientamento contiene per ogni slotId la corrispondente Alfabetica (se presente) rispetto all'Orientamento'''

        self.load_list_InsegnamentiParallelizzabili:bool = False
        self.list_InsegnamentiParallelizzabili:List[List[int]] = list()
        '''Lista di gruppi di slotId allocabili tra loro in parallelo '''    
 
    
    
    def get_NUM_DAY(self):
        '''Ritorna il numero di giorni in cui è possibile allocare'''
        if self.PARAM.sabatoEnabled:
            return self.NUM_DAY + 1
        return self.NUM_DAY
    
    def add_slotToPianoAllocazione(self, slot:SlotReq):
        '''Aggiungo uno Slot alla lista degli slot del piano di allocazione'''
        if self.load_pianoAllocazione:
            self.log.error_log("AuxiliaryStruct.add_slotToPianoAllocazioe(): non posso più aggiungere uno SlotReq")
            return  
        self.pianoAllocazione.append(slot)
        
    def end_pianoAllocazione(self):
        '''Setto come terminato il caricamento di self.pianoAllocazione'''
        if self.load_pianoAllocazione:
            self.log.error_log("AuxiliaryStruct.end_pianoAllocazione(): non posso più aggiungere uno SlotReq")
            return
        self.load_pianoAllocazione = True
        
    
    def add_vincoloToListVincoliInsegnamenti(self, vincolo:Vincolo):
        '''Aggiungo un Vincolo alla lista dei Vincoli caricati da xml'''
        if self.load_list_vincoliInsegnamenti:
            self.log.error_log("AuxiliaryStruct.add_vincoloToList(): non posso più aggiungere un Vincolo")
            return
        self.list_vincoliInsegnamenti.append(vincolo)
        
    def end_listVincoliInsegnamenti(self):
        '''Completo il caricamento dei Vincoli e li finalizzo'''
        if self.load_list_vincoliInsegnamenti:
            self.log.error_log("AuxiliaryStruct.end_listVincoliInsegnamenti(): lista già finalizzata")
            return
        if not self.load_map_strSlotId_to_idSlot or not self.load_pianoAllocazione or not self.load_list_Insegnamenti:
            self.log.error_log("AuxiliaryStruct.end_listVincoliInsegnamenti(): strutture non pronte")
            return
        for vincolo in self.list_vincoliInsegnamenti:
            if vincolo.slotId_iStr not in self.map_strSlotId_to_idSlot.keys() or vincolo.slotId_jStr not in self.map_strSlotId_to_idSlot.keys():
                self.log.error_log("AuxiliaryStruct.end_listVincoliInsegnamenti(): dati non validi -> Vincolo invalido per " + vincolo.slotId_iStr + " e " + vincolo.slotId_jStr)
                # return
            else:
                slotId_i = self.map_strSlotId_to_idSlot[vincolo.slotId_iStr]
                slotId_j = self.map_strSlotId_to_idSlot[vincolo.slotId_jStr]
                vincolo.finalizeVincolo(slotId_i, slotId_j, self.pianoAllocazione[slotId_i].numSlot, self.pianoAllocazione[slotId_j].numSlot)
                
                if self.PARAM.usePianoAllocazioneAsBase:
                        # se entrambi gli Slot appartengono ad Insegnamenti da NON riallocare => il vincolo è inutile
                        # di solito non dovrebbe succedere poichè l'xml con il vincolo non viene proprio letto se l'Insegnameto viene
                        # precaricato dal db
                        if int(self.list_Insegnamenti[self.pianoAllocazione[slotId_i].idInsegnamento].ID_INC) not in self.PARAM.listID_INCToModify and int(self.list_Insegnamenti[self.pianoAllocazione[slotId_j].idInsegnamento].ID_INC) not in self.PARAM.listID_INCToModify:
                            self.log.info_log("auxiliary.end_listVincoliInsegnamenti(): skip Vincolo di Slot appartenenti a Insegnamenti: " + self.list_Insegnamenti[self.pianoAllocazione[slotId_i].idInsegnamento].ID_INC + " e " + self.list_Insegnamenti[self.pianoAllocazione[slotId_j].idInsegnamento].ID_INC)
                            vincolo.valid = False     
                        
        self.load_list_vincoliInsegnamenti = True
        
    def get_listDocentiOfVincolo(self, vincoloId) -> List[int]:
        '''Ritorna la lista di docenti a cui si riferisce un Vincolo'''
        if not self.load_list_vincoliInsegnamenti or not self.load_list_docentiInSlot:
            self.log.error_log("AuxiliaryStruct.get_listDocentiOfVincolo(): strutture non pronte")
            return list()
        
        setDocId:Set[int] = set()
        vincolo:Vincolo = self.list_vincoliInsegnamenti[vincoloId]
        
        if not vincolo:
            self.log.error_log("AuxiliaryStruct.get_listDocentiOfVincolo(): Vincolo non valido")
            return list()
        
        for docId in self.list_docentiInSlot[vincolo.slotId_i]:
            setDocId.add(docId)        
        for docId in self.list_docentiInSlot[vincolo.slotId_j]:
            setDocId.add(docId)        
        return list(setDocId)
    
    
    def add_operatoreToListOperatoriInsegnamenti(self, operatore:Operatore):
        '''Aggiungo un Operatore alla lista degli Operatori caricati da xml'''
        if self.load_list_operatoriInsegnamenti:
            self.log.error_log("AuxiliaryStruct.add_operatore(): non posso più aggiungere un Operatore")
            return
        self.list_operatoriInsegnamenti.append(operatore)
        
    def end_listOperatoriInsegnamenti(self):
        '''Completo il caricamento degli Operatori e li finalizzo'''
        if not self.load_map_strSlotId_to_idSlot or not self.load_pianoAllocazione:
            self.log.error_log("AuxiliaryStruct.end_listOperatoriInsegnamenti(): strutture non pronte")
            return
        if self.load_list_operatoriInsegnamenti:
            self.log.error_log("AuxiliaryStruct.end_listOperatoriInsegnamenti(): lista già finalizzata")
            return
        for operatore in self.list_operatoriInsegnamenti:
            self.rEnd_listOperatoriInsegnamenti(operatore)
        self.load_list_operatoriInsegnamenti = True      
        
    def rEnd_listOperatoriInsegnamenti(self, operatore:Operatore):
        '''Funzione di utility di end_listOperatoriInsegnamenti(). Chiamabile SOLO da essa'''
        if operatore is None:
            return
        # provo a finalizzare tutti i possibili figli
        self.rEnd_listOperatoriInsegnamenti(operatore.op1)
        self.rEnd_listOperatoriInsegnamenti(operatore.op2)
        if operatore.vinc1 is not None:
            vincolo = operatore.vinc1
            if vincolo.slotId_iStr not in self.map_strSlotId_to_idSlot.keys() or vincolo.slotId_jStr not in self.map_strSlotId_to_idSlot.keys():
                self.log.error_log("AuxiliaryStruct.rEnd_listOperatoriInsegnamenti(): dati non validi -> Vincolo invalido")
            else:
                slotId_i = self.map_strSlotId_to_idSlot[vincolo.slotId_iStr]
                slotId_j = self.map_strSlotId_to_idSlot[vincolo.slotId_jStr]
                vincolo.finalizeVincolo(slotId_i, slotId_j, self.pianoAllocazione[slotId_i].numSlot, self.pianoAllocazione[slotId_j].numSlot)
        if operatore.vinc2 is not None:
            vincolo = operatore.vinc2
            if vincolo.slotId_iStr not in self.map_strSlotId_to_idSlot.keys() or vincolo.slotId_jStr not in self.map_strSlotId_to_idSlot.keys():
                self.log.error_log("AuxiliaryStruct.rEnd_listOperatoriInsegnamenti(): dati non validi -> Vincolo invalido")
            else:
                slotId_i = self.map_strSlotId_to_idSlot[vincolo.slotId_iStr]
                slotId_j = self.map_strSlotId_to_idSlot[vincolo.slotId_jStr]
                vincolo.finalizeVincolo(slotId_i, slotId_j, self.pianoAllocazione[slotId_i].numSlot, self.pianoAllocazione[slotId_j].numSlot)        
                
                if self.PARAM.usePianoAllocazioneAsBase:
                    # se entrambi gli Slot appartengono ad Insegnamenti da NON riallocare => il vincolo è inutile
                    if int(self.list_Insegnamenti[self.pianoAllocazione[slotId_i].idInsegnamento].ID_INC) not in self.PARAM.listID_INCToModify and int(self.list_Insegnamenti[self.pianoAllocazione[slotId_j].idInsegnamento].ID_INC) not in self.PARAM.listID_INCToModify:
                        self.log.info_log("auxiliary.rEnd_listOperatoriInsegnamenti(): skip Vincolo di Slot appartenenti a Insegnamenti: " + self.list_Insegnamenti[self.pianoAllocazione[slotId_i].idInsegnamento].ID_INC + " e " + self.list_Insegnamenti[self.pianoAllocazione[slotId_j].idInsegnamento].ID_INC)
                        vincolo.valid = False                
    
    def get_listDocentiOfOperatore(self, operatoreId:int) -> List[int]:
        '''Ritorna la lista di Docenti a cui si riferisce un Operatore'''
        setDoc:Set[int] = set()
        
        if not self.load_list_operatoriInsegnamenti or not self.load_list_docentiInSlot:
            self.log.error_log("AuxiliaryStruct.get_listDocentiOfOperatore(): strutture non pronte")
            return list()
        
        self.rGet_listDocentiOfOperatore(self.list_operatoriInsegnamenti[operatoreId], setDoc)
        return list(setDoc)
    
    def rGet_listDocentiOfOperatore(self, operatore:Operatore, setDocRes:Set[int]) -> None:
        '''Funzione di utility di get_listDocentiOfOperatore(). Chiamabile SOLO da essa'''
        if operatore is None:
            return
        # chiamo sugli Operatori child
        self.rGet_listDocentiOfOperatore(operatore.op1, setDocRes)
        self.rGet_listDocentiOfOperatore(operatore.op2, setDocRes)
        if operatore.vinc1 is not None:
            if not operatore.vinc1:
               self.log.error_log("AuxiliaryStruct.rGet_listDocentiOfOperatore(): Vincolo non valido")
            else:
                for docId in self.list_docentiInSlot[operatore.vinc1.slotId_i]:
                    setDocRes.add(docId)        
                for docId in self.list_docentiInSlot[operatore.vinc1.slotId_j]:
                    setDocRes.add(docId) 
        if operatore.vinc2 is not None:
            if not operatore.vinc2:
               self.log.error_log("AuxiliaryStruct.rGet_listDocentiOfOperatore(): Vincolo non valido")
            else:
                for docId in self.list_docentiInSlot[operatore.vinc2.slotId_i]:
                    setDocRes.add(docId)        
                for docId in self.list_docentiInSlot[operatore.vinc2.slotId_j]:
                    setDocRes.add(docId)                                    
            
        
    def add_OrientamentoToList(self, orient:Orientamento):
        '''Aggiungo un Orientamento solo se non è già presente'''
        if self.load_list_Orientamenti:
            self.log.error_log("auxiliary.add_OrientamentoToList(): caricamento di list_Orientamenti già concluso")
            return
        idOrient = str(orient)
        if idOrient in self.map_Orientamento_to_IdOrientamento.keys():
            self.log.error_log("auxiliary.add_OrientamentoToList(): Orientamento già caricato")
            return
        self.list_Orientamenti.append(orient)
        self.map_Orientamento_to_IdOrientamento[idOrient] = len(self.list_Orientamenti)-1
        
    def endlist_Orientamenti(self):
        '''Setto il completamento del caricamento della lista'''
        self.load_list_Orientamenti = True # per controllare che list_InsegnamentiInOrientamento viene creata solo quando ho tutti i dati
        self.load_map_Orientamento_to_IdOrientamento = True

        
    def add_fileToListFileDone(self, file:str):
        '''Aggiungo un file contenente almeno un TemplateOrario alla lista dei file caricati'''
        if self.load_list_FileDone:
            self.log.error_log("auxiliary.add_fileToListFileDone(): non posso più caricare file")            
            return
        if file in self.list_FileDone:
            self.log.error_log("auxiliary.add_fileToListFileDone(): si sta caricando lo stesso file più volte")
            return
        self.list_FileDone.append(file)
    
    
    def add_metaInsegnamentoToList(self, metaIns:MetaInsegnamento) -> int:
        '''Aggiungo un MetaInsegnamento alla lista degli insegnamenti. Aggiorno anche mapMetaInsegnamento_to_IdInsegnamento.\n
        Return:
            l'id corrispondente all'Insegnamento appena aggiunto'''
        if self.load_list_metaInsegnamenti or self.load_map_MetaInsegnamento_to_IdInsegnamento:
            self.log.error_log("auxiliary.add_metaInsegnamentoToList(): caricamento già concluso")
            return
        
        self.list_metaInsegnamenti.append(metaIns)
        if metaIns.ID_INC in self.map_MetaInsegnamento_to_IdInsegnamento:
            self.log.error_log("auxiliary.add_metaInsegnamentoToList(): esiste già un MetaInsegnamento con lo stesso ID_INC: " + str(metaIns.ID_INC))
        self.map_MetaInsegnamento_to_IdInsegnamento[metaIns.ID_INC] = len(self.list_metaInsegnamenti)-1
        
        return len(self.list_metaInsegnamenti)-1
    
    def end_metaInsegnamentoToList(self):
        '''Setta come concluso il caricamento di self.list_metaInsegnamenti'''
        if self.load_list_metaInsegnamenti or self.load_map_MetaInsegnamento_to_IdInsegnamento:
            self.log.error_log("auxiliary.end_metaInsegnamentoToList(): caricamento già concluso")
            return
        self.load_list_metaInsegnamenti = True
        self.load_map_MetaInsegnamento_to_IdInsegnamento = True
    
    
    def createList_InsegnamentoInOrientamento(self):
        '''Creazione struttura list_InsegnamentiInOrientamento. Per ogni Orientamento ho la lista di Insegnamenti 
        (con il corrispettivo numero di iscritti per quell'insegnamento)'''
        if not self.load_map_Orientamento_to_IdOrientamento or not self.load_list_Orientamenti or not self.load_map_MetaInsegnamento_to_IdInsegnamento:
            self.log.error_log("auxiliary.createList_InsegnamentoInOrientamento(): non posso ancora generare list_InsegnamentiInOrientamento")
            return
        if self.load_list_InsegnamentiInOrientamento:
            self.log.error_log("auxiliary.createList_InsegnamentoInOrientamento(): list_InsegnamentiInOrientamento già creata")
            return
        
        # corretto dimensionamento di list_InsegnamentiInOrientamento, list_InsegnamentiInOrientamento_numIscritti
        # e list_InsegnamentiInOrientamento_tipo
        for idOrient in self.map_Orientamento_to_IdOrientamento.values():
            self.list_InsegnamentiInOrientamento.append(list())
            self.list_InsegnamentiInOrientamento_numIscritti.append(list())
            self.list_InsegnamentiInOrientamento_tipo.append(list())
            self.list_InsegnamentiInOrientamento_periodoDidattico.append(list())
            self.list_InsegnamentiInOrientamento_alfabetica.append(list())
        
        for idOrient in self.map_Orientamento_to_IdOrientamento.values():            
            if self.list_Orientamenti[idOrient].tipoCdl == TipoCdl.Triennale:
                rows = self.dbAPI.get_Insegnamenti_withOrientamento(self.list_Orientamenti[idOrient].orientamento,
                                                             self.list_Orientamenti[idOrient].nomeCdl,'1')
            else:
                rows = self.dbAPI.get_Insegnamenti_withOrientamento(self.list_Orientamenti[idOrient].orientamento,
                                                             self.list_Orientamenti[idOrient].nomeCdl,'Z')
            
            # per ogni Insegnamento appartenente all'Orientamento
            for rowIns in rows:
                idInc = str(rowIns[0])
                numIscritti = int(rowIns[8])
                if idInc not in self.map_MetaInsegnamento_to_IdInsegnamento.keys():
                    # scazza perchè non carico tutto il db
                    if not self.testing:
                        self.log.error_log("auxiliary.createList_InsegnamentoInOrientamento(): ID_INC sconosciuto")
                    continue
                self.list_InsegnamentiInOrientamento[idOrient].append(self.map_MetaInsegnamento_to_IdInsegnamento[idInc])
                self.list_InsegnamentiInOrientamento_numIscritti[idOrient].append(numIscritti)
                self.list_InsegnamentiInOrientamento_tipo[idOrient].append(getTipoInsegnamentoFromStr(rowIns[9]))
                self.list_InsegnamentiInOrientamento_periodoDidattico[idOrient].append(getPeriodoDidatticoFromStr(rowIns[10]))
                self.list_InsegnamentiInOrientamento_alfabetica[idOrient].append(getAlfabeticaFromStr(rowIns[11]))
                
        # ora posso passare dal mapping per Insegnamento a quello per slotId
        self.load_list_InsegnamentiInOrientamento = True
        self.load_list_InsegnamentiInOrientamento_numIscritti = True
        self.load_list_InsegnamentiInOrientamento_tipo = True
        self.load_list_InsegnamentiInOrientamento_periodoDidattico = True     
        self.load_list_InsegnamentiInOrientamento_alfabetica = True   
                
    def createList_slotIdInOrientamento(self):
        '''Creazione delle strutture list_slotIdInOrientamento e list_slotIdInOrientamento_tipo'''
        if not self.load_map_Orientamento_to_IdOrientamento or not self.load_list_InsegnamentiInOrientamento or not self.load_list_slotInInsegnamento or not self.load_list_InsegnamentiInOrientamento_periodoDidattico or not self.load_list_InsegnamentiInOrientamento_alfabetica:
            self.log.error_log("auxiliary.createList_slotIdInOrientamento(): non posso ancora creare list_slotIdInOrientamento")
            return
        if self.load_list_slotIdInOrientamento or self.load_list_slotIdInOrientamento_tipo or self.load_list_slotIdInOrientamento_periodoDidattico or self.load_list_slotIdInOrientamento_alfabetica:
            self.log.error_log("auxiliary.createList_slotIdInOrientamento(): strutture già create")
            return
        # preallocazione strutture
        for orientId in self.map_Orientamento_to_IdOrientamento.values():
            self.list_slotIdInOrientamento.append(list())
            self.list_slotIdInOrientamento_tipo.append(list())
            self.list_slotIdInOrientamento_periodoDidattico.append(list())
            self.list_slotIdInOrientamento_alfabetica.append(list())
                    
        # per ogni Orientamento salvo tutti gliSlotId
        for orientId in self.map_Orientamento_to_IdOrientamento.values():
            # per ogni Insegnamento dell'Orientamento
            for index_ins in range(len(self.list_InsegnamentiInOrientamento[orientId])):
                insId = self.list_InsegnamentiInOrientamento[orientId][index_ins]
                
                # per ogni slotId dell'Insegnameto
                for slotId in self.list_slotInInsegnamento[insId]:
                    self.list_slotIdInOrientamento[orientId].append(slotId)
                    self.list_slotIdInOrientamento_tipo[orientId].append(self.list_InsegnamentiInOrientamento_tipo[orientId][index_ins])
                    self.list_slotIdInOrientamento_periodoDidattico[orientId].append(self.list_InsegnamentiInOrientamento_periodoDidattico[orientId][index_ins])
                    self.list_slotIdInOrientamento_alfabetica[orientId].append(self.list_InsegnamentiInOrientamento_alfabetica[orientId][index_ins])
        self.load_list_slotIdInOrientamento = True
        self.load_list_slotIdInOrientamento_tipo = True
        self.load_list_slotIdInOrientamento_periodoDidattico = True
        self.load_list_slotIdInOrientamento_alfabetica = True

            
    def createList_Insegnamenti(self):
        '''List_Insegnamenti è parallela a list_metaInsegnamenti'''
        if not self.load_map_MetaInsegnamento_to_IdInsegnamento:
            self.log.error_log("auxiliary.createList_Insegnamenti(): non posso ancora creare la struttura")
            return            
        if self.load_list_Insegnamenti:
            self.log.error_log("auxiliary.createList_Insegnamenti(): list_Insegnamenti già creata")
            return
        for idIns in self.map_MetaInsegnamento_to_IdInsegnamento.values():
            self.list_Insegnamenti.append(Insegnamento("",0,0,"","","",0,"")) # tmp
    
    def add_InsegnamentoToList(self, ins:Insegnamento):
        '''Aggiungo un insegnamento alla lista degli insegnamenti'''
        if not self.load_map_MetaInsegnamento_to_IdInsegnamento:
            self.log.error_log("auxiliary.add_InsegnamentoToList(): non posso ancora creare la struttura")
            return            
        if self.load_list_Insegnamenti:
            self.log.error_log("auxiliary.add_InsegnamentoToList(): list_Insegnamenti già creata")
            return
        if ins.ID_INC not in self.map_MetaInsegnamento_to_IdInsegnamento.keys():
            # se non lavoro con tutti gli xml scazza
            if not self.testing:
                self.log.error_log("auxiliary.add_InsegnamentoToList(): non esiste questo Insegnamento tra i MetaInsegnamenti")
            return
        self.list_Insegnamenti[self.map_MetaInsegnamento_to_IdInsegnamento[ins.ID_INC]] = ins
        
    def endList_Insegnamenti(self):
        '''Dopo il caricamento della lista devo settare gli id del docente titolare dentro list_Insegnamenti'''
        if not self.load_map_MetaInsegnamento_to_IdInsegnamento or not self.load_map_strDocenti_to_idDocenti:
            self.log.error_log("auxiliary.endList_Insegnamenti(): non posso ancora creare la struttura")
            return
        if self.load_list_Insegnamenti:
            self.log.error_log("auxiliary.endList_Insegnamenti(): list_Insegnamenti già creata")
            return
        for idIns in self.map_MetaInsegnamento_to_IdInsegnamento.values():
            if self.list_Insegnamenti[idIns].docenteTitolare in self.map_strDocenti_to_idDocenti.keys():
                self.list_Insegnamenti[idIns].idDocenteTitolare = self.map_strDocenti_to_idDocenti[self.list_Insegnamenti[idIns].docenteTitolare]
            else:
                self.log.error_log("auxiliary.endList_Insegnamenti(): Docente titolare sconosciuto per: " + str(self.list_Insegnamenti[idIns]))
        self.load_list_Insegnamenti = True
    
    
    def createMap_strDocenti_to_IdDocenti(self, setDocenti:Set[str]):
        '''Partendo dal set con tutti i docenti crea il dizionario di mapping: docStr -> docId'''
        if self.load_map_idDocenti_to_strDocenti or self.load_map_strDocenti_to_idDocenti:
            self.log.error_log("auxiliary.createMap_strDocenti_to_IdDocenti(): strutture già create")
            return
        i = 0
        for strDoc in setDocenti:
            self.map_strDocenti_to_idDocenti[strDoc] = i
            self.map_idDocenti_to_strDocenti.append(strDoc)
            i+=1
        self.load_map_strDocenti_to_idDocenti = True
        self.load_map_idDocenti_to_strDocenti = True
    
            
    def createMap_strSlotId_to_IdSlot(self):
        '''Partendo dalla lista di tutti gli Slot creo il dizionario di mapping: slotIdStr -> slotId'''
        if not self.load_pianoAllocazione:
            self.log.error_log("auxiliary.createMap_strSlotId_to_IdSlot(): strutture non pronte")
            return
        if self.load_map_strSlotId_to_idSlot:
            self.log.error_log("auxiliary.createMap_strSlotId_to_IdSlot(): strutture già create")
            return            
        for i in range(len(self.pianoAllocazione)):
            # lo slot 1 nel piano di allocazione sarà anche lo slot 1 nel dict
            if self.pianoAllocazione[i].slotId in self.map_strSlotId_to_idSlot.keys():
                self.log.error_log("auxiliary.createMap_strSlotId_to_IdSlot(): esiste già uno Slot con lo stesso slotId: " + str(self.pianoAllocazione[i].slotId))
            self.map_strSlotId_to_idSlot[self.pianoAllocazione[i].slotId] = i
        self.load_map_strSlotId_to_idSlot = True
            
    def createList_docentiInSlot(self):
        '''Per ogni slot ho la lista dei docenti che sono necessari in quello slot'''
        if not self.load_map_strSlotId_to_idSlot or not self.load_pianoAllocazione:
            self.log.error_log("auxiliary.createList_docentiInSlot(): non posso ancora crearlo")
            return
        if self.load_list_docentiInSlot:
            self.log.error_log("auxiliary.createList_docentiInSlot(): strutture già create")
            return        
        
        for i in self.map_strSlotId_to_idSlot:
            self.list_docentiInSlot.append(list())
                
        for slot in self.pianoAllocazione:
            # list di idDocenti per il singolo slot
            l = list(slot.setDocenti)
            l.sort()
            self.list_docentiInSlot[self.map_strSlotId_to_idSlot[slot.slotId]] = l
        self.load_list_docentiInSlot = True
    
    def createList_slotInDocente(self):
        '''Per ogni docente ho la lista degli slot in cui deve essere presente'''
        if not self.load_map_strDocenti_to_idDocenti or not self.load_map_strSlotId_to_idSlot or not self.load_list_docentiInSlot:
            self.log.error_log("auxiliary.createList_slotInDocente(): non posso ancora crearlo")
            return
        if self.load_list_slotInDocente:
            self.log.error_log("auxiliary.createList_slotInDocente(): strutture già create")
            return
        
        for i in range(len(self.map_strDocenti_to_idDocenti)):
            self.list_slotInDocente.append(list())
        
        for docId in self.map_strDocenti_to_idDocenti.values():
            for slotId in self.map_strSlotId_to_idSlot.values():
                if docId in self.list_docentiInSlot[slotId]:
                    self.list_slotInDocente[docId].append(slotId)
            # sort the list
            self.list_slotInDocente[docId].sort()
        self.load_list_slotInDocente = True
           
            
    def createList_slotInInsegnamento(self):
        '''Per ogni insegnamento ho la lista degli slotId'''
        if not self.load_list_metaInsegnamenti or not self.load_map_strSlotId_to_idSlot or not self.load_pianoAllocazione:
            self.log.error_log("auxiliary.createList_slotInInsegnamento(): non posso ancora crearlo")
            return
        if self.load_list_slotInInsegnamento:
            self.log.error_log("auxiliary.createList_slotInInsegnamento(): strutture già create")
            return
                
        for i in self.list_metaInsegnamenti:
            self.list_slotInInsegnamento.append(list())
        
        # ogni slotId lo aggiungo all'Insegnamento corrispondente -> siccome gli slotId sono crescenti ->
        # anche l'ordine in cui in ogni Insegnamento ho la lista degli slot sarà crescente
        for slotId in self.map_strSlotId_to_idSlot.values():
            self.list_slotInInsegnamento[self.pianoAllocazione[slotId].idInsegnamento].append(slotId)
        self.load_list_slotInInsegnamento = True
                    
    
    def createList_InsegnamentiParallelizzabili(self):
        '''Crea la struttura self.list_InsegnamentiParallelizzabili per gestire gli slot parallelizzabili'''
        if self.load_list_InsegnamentiParallelizzabili:
            self.log.error_log("auxiliary.createList_InsegnamentiParallelizzabili(): struttura già creata")
            return
        if not self.load_map_MetaInsegnamento_to_IdInsegnamento or not self.load_list_slotInInsegnamento:
            self.log.error_log("auxiliary.createList_InsegnamentiParallelizzabili(): strutture non pronte")
            return
                
        for comb in list_ID_INC_parallelizzabili:
            index_insegnamentiParal = len(self.list_InsegnamentiParallelizzabili)
            self.list_InsegnamentiParallelizzabili.append(list())
            
            for ID_INC in comb:
                if ID_INC not in self.map_MetaInsegnamento_to_IdInsegnamento.keys():
                    if not self.testing: # scazza perchè non carico tutti gli Insegnamenti
                        self.log.error_log("auxiliary.createList_InsegnamentiParallelizzabili(): ID_INC: " + ID_INC + " sconosciuto")
                    continue # skip it
                idIns = self.map_MetaInsegnamento_to_IdInsegnamento[ID_INC]
                # per ogni slotId di quell'Insegnamento
                for slotId in self.list_slotInInsegnamento[idIns]:
                    self.list_InsegnamentiParallelizzabili[index_insegnamentiParal].append(slotId)
        self.load_list_InsegnamentiParallelizzabili = True
                
                  
    def check_slotParallelizzabili(self, slotId_i:int, slotId_j:int) -> bool:
        '''Verifica se due slot possono essere allocati in parallelo. Gli slot sono passati come slotId all'interno di AUX (da chiamare solo
        su id ricavati da AUX)'''
        if not self.load_list_InsegnamentiParallelizzabili:
            self.log.error_log("auxiliary.check_slotParallelizzabili(): strutture non pronte")            
            return False
        for comb in self.list_InsegnamentiParallelizzabili:
            if slotId_i in comb and slotId_j in comb:
                return True
        return False
    
    def check_slotStessoAnno(self, index_i:int, index_j:int, orientId:int) -> bool:
        '''Verifica se per l'Orientamento corrente, i due Slot appartengono ad Insegnamenti dello stesso anno oppure no (e quindi possono
        essere allocati in parallelo dal punto di vista di queso Orientamento)'''
        if not self.load_list_slotIdInOrientamento_periodoDidattico:
            self.log.error_log("auxiliary.check_slotStessoAnno(): strutture non pronte")
            return True # scelta più conservativa
           
        pd_i:PeriodoDidattico = self.list_slotIdInOrientamento_periodoDidattico[orientId][index_i]
        pd_j:PeriodoDidattico = self.list_slotIdInOrientamento_periodoDidattico[orientId][index_j]

        if pd_i == PeriodoDidattico.Altro or pd_j == PeriodoDidattico.Altro:
            self.log.warning_logning("auxiliary.check_slotStessoAnno(): Slot con PeriodoDidattico sconosciuto")
            return True # per sicurezza, non dovrebbe mai capitare
        if pd_i in [PeriodoDidattico.PrimoAnno_PrimoSemestre, PeriodoDidattico.PrimoAnno_SecondoSemestre] and pd_j in [PeriodoDidattico.PrimoAnno_PrimoSemestre, PeriodoDidattico.PrimoAnno_SecondoSemestre]:
            return True
        if pd_i in [PeriodoDidattico.SecondoAnno_PrimoSemestre, PeriodoDidattico.SecondoAnno_SecondoSemestre] and pd_j in [PeriodoDidattico.SecondoAnno_PrimoSemestre, PeriodoDidattico.SecondoAnno_SecondoSemestre]:
            return True
        if pd_i in [PeriodoDidattico.TerzoAnno_PrimoSemestre, PeriodoDidattico.TerzoAnno_SecondoSemestre] and pd_j in [PeriodoDidattico.TerzoAnno_PrimoSemestre, PeriodoDidattico.TerzoAnno_SecondoSemestre]:
            return True
        return False
    
    def check_slotAlfabeticheDiverse(self, index_i:int, index_j:int, orientId:int):
        '''Verifica se per l'Orientamento corrente, i due Slot appartengono ad Alfabetiche diverse. Se anche solo uno dei due Slot
        non appartenesse a nessuna Alfabetica è come se gli Slot appartenessero alla stessa Alfabetica'''
        if not self.load_list_slotIdInOrientamento_alfabetica:
            self.log.error_log("auxiliary.check_slotAlfabeticheDiverse(): strutture non pronte")
            return False
        
        alf_i:Alfabetica = self.list_slotIdInOrientamento_alfabetica[orientId][index_i]
        alf_j:Alfabetica = self.list_slotIdInOrientamento_alfabetica[orientId][index_j]
        
        if alf_i == Alfabetica.NoAlfabetica or alf_j == Alfabetica.NoAlfabetica:
            return False
        if alf_i != alf_j:
            return True # Alfabetiche diverse ed esistenti
        return False
    
    
    def debug(self):
        self.log.info_log("auxiliary.debug()")
        self.log.info_log("\nfile TemplateOrario caricati:")
        # self.log.info_log(self.list_FileDone)
        self.log.info_log("\nmap_strDocenti_to_idDocenti:")
        # self.log.info_log(self.map_strDocenti_to_idDocenti)
        self.log.info_log("\nlist_docentiInSlot:")
        for s in self.map_strSlotId_to_idSlot.values():
            # self.log.info_log(self.pianoAllocazione[s].slotId + " " + str(s)+ ": " + str(self.list_docentiInSlot[s]))
            pass
        self.log.info_log("\nlist_slotInDocente:")
        for s in self.map_strDocenti_to_idDocenti.values():
            # self.log.info_log(self.map_idDocenti_to_strDocenti[s] + ": " + str(self.list_slotInDocente[s]))
            pass
        self.log.info_log("\nlist_slotInInsegnamento with list_metaInsegnamenti:")
        for idIns in self.map_MetaInsegnamento_to_IdInsegnamento.values():
            self.log.info_log(self.list_metaInsegnamenti[idIns])
            self.log.info_log(self.list_slotInInsegnamento[idIns])
        self.log.info_log("\npiano allocazione:")
        for s in self.pianoAllocazione:
            # self.log.info_log(s)
            pass
        self.log.info_log("\nlist_InsegnamentiInOrientamento:")
        for idOrient in self.map_Orientamento_to_IdOrientamento.values():            
            if len(self.list_InsegnamentiInOrientamento[idOrient]) > 0:
                self.log.info_log(self.list_Orientamenti[idOrient])
                for index in range(len(self.list_InsegnamentiInOrientamento[idOrient])):
                    idIns = self.list_InsegnamentiInOrientamento[idOrient][index]
                    self.log.info_log(str(self.list_metaInsegnamenti[idIns]) + " level: " + str(self.list_InsegnamentiInOrientamento_tipo[idOrient][index]) + 
                             " anno: " + str(self.list_InsegnamentiInOrientamento_periodoDidattico[idOrient][index]))
                    # for index_slot in range(len(self.list_slotIdInOrientamento[idOrient])):
                    #     slotId = self.list_slotIdInOrientamento[idOrient][index_slot]
                    #     if self.pianoAllocazione[slotId].idInsegnamento == idIns:
                    #         self.log.info_log(self.list_slotIdInOrientamento_periodoDidattico[idOrient][index_slot])
                        
        self.log.info_log("\nlist_Orientamenti:")
        for orientId in self.map_Orientamento_to_IdOrientamento.values():
            self.log.info_log(str(self.list_Orientamenti[orientId]) + " - id: " + str(orientId) + " numInsegnamenti: " + str(len(self.list_slotIdInOrientamento[orientId])))
            # if len(self.list_slotIdInOrientamento[orientId]) > 0:
            #     for index in range(len(self.list_slotIdInOrientamento[orientId])):
            #         slotId = self.list_slotIdInOrientamento[orientId][index]
            #         self.log.info_log(str(self.pianoAllocazione[slotId]) + " level: " + str(self.list_slotIdInOrientamento_tipo[orientId][index]))
        self.log.info_log("\nlist_Vincoli:")
        for vincolo in self.list_vincoliInsegnamenti:
            self.log.info_log(vincolo)
        self.log.info_log("\nlist_Operatori:")
        for operatore in self.list_operatoriInsegnamenti:
            self.log.info_log(operatore)
        
    
    
    def get_nSlotId(self) -> int:
        return len(self.map_strSlotId_to_idSlot)
    
    def get_nDocenti(self) -> int:
        return len(self.map_strDocenti_to_idDocenti)
    
    def get_nOrientamenti(self) -> int:
        return len(self.list_Orientamenti)
    
    def get_nSlotSettimana(self) -> int:
        return self.get_NUM_DAY()*self.NUM_SLOT_PER_DAY
    
    def get_nPeriodiDidattici_Orientamenti(self) -> int:
        '''Per ogni Orientamento conteggia il numero di anni che dura (2/3)'''
        nRes:int = 0
        for orient in self.list_Orientamenti:
            if orient.tipoCdl == TipoCdl.Magistrale:
                nRes += 2
            elif orient.tipoCdl == TipoCdl.Triennale:
                nRes += 3 # in realtà non dovrei allocare nulla per il primo anno di Triennale, per usi futuri
            else:
                self.log.error_log("auxiliary.get_nPeriodiDidattici_Orientamenti(): TipoCdl sconosciuto")
        return nRes