from docplex.cp.model import *

from enum import Enum

from log import logger
from RulePenalty import *


log:logger = logger()


class TipoVincolo(Enum):
    Nessuno = -1
    AlmenoUnGiornoDopo = 0
    SubitoDopo = 1
    StessoGiorno = 2
    GiornoDiverso = 3
    Parallelo = 4 # a livello di Slot, non di Insegnamento

def getTipoVincoloFromStr(tipoVincolo:str) -> TipoVincolo:
    '''Compliant con schemaUseCase.xsd'''
    if tipoVincolo == "almenoUnGiornoDopo":
        return TipoVincolo.AlmenoUnGiornoDopo
    if tipoVincolo == "stessoGiorno":
        return TipoVincolo.StessoGiorno
    if tipoVincolo == "giornoDiverso":
        return TipoVincolo.GiornoDiverso
    if tipoVincolo == "parallelo":
        return TipoVincolo.Parallelo
    if tipoVincolo == "subitoDopo":
        return TipoVincolo.SubitoDopo
    log.error_log("vincoli.getTipoVincoloFromStr(): tipoVincolo sconosciuto")
    return TipoVincolo.Nessuno


class Vincolo:
    def __init__(self, vincoloId:str, tipoVincolo:TipoVincolo, slotId_iStr:str, slotId_jStr:str, penalita:LV_Penalties = LV_Penalties.LV_0, standAlone:bool = False):
        '''Gestisce un Vincolo foglia -> non contiene Operatori al suo interno'''
        self.vincoloId:str = vincoloId
        self.tipoVincolo:TipoVincolo = tipoVincolo
        self.slotId_iStr:str = slotId_iStr
        self.slotId_jStr:str = slotId_jStr
        # verranno recuperati successivamente tramite AUX
        self.slotId_i:int = -1
        self.slotId_j:int = -1
        self.valid:bool = False # diventa valido dopo l'assegnazione degli slotId
        self.penalita:LV_Penalties = penalita
        # per tipoVincolo == SubitoDopo
        self.numSlot_i:int = 0
        self.numSlot_j:int = 0
        # per fare debugging e vedere se il vincolo è stato soddisfatto o meno (se soft constraint)
        self.index_X_penaltiesDoc:int = -1
        # è un Vincolo o fa parte di un Operatore (quindi non mi interessa la penalità)
        self.standAlone:bool = standAlone
        self.log:logger = logger()
        
    def __str__(self) -> str:
        if self.valid:
            res = "Vincolo finalized: "        
        else:
            res = "Vincolo: "
        res += self.vincoloId + " " + str(self.tipoVincolo) + " " + str(self.penalita) + ". " + self.slotId_iStr + "-" + self.slotId_jStr
        return res
    
    def __bool__(self):
        if self.slotId_i == -1 or self.slotId_j == -1:
            self.valid = False
        if self.slotId_jStr == "" and self.tipoVincolo in [TipoVincolo.AlmenoUnGiornoDopo, TipoVincolo.GiornoDiverso, TipoVincolo.Parallelo, TipoVincolo.StessoGiorno, TipoVincolo.SubitoDopo]:
            self.valid = False
        if self.slotId_jStr != "" and self.slotId_j == -1:
            self.valid = False
        # se è dentro un nodo foglia non mi interessa la penalità (salvata nell'Operatore root)
        if self.penalita == LV_Penalties.LV_0 and self.standAlone:
            self.valid = False
        if self.tipoVincolo == TipoVincolo.SubitoDopo and (self.numSlot_j == 0 or self.numSlot_i == 0):
            self.valid = False
        if self.tipoVincolo == TipoVincolo.Nessuno:
            self.valid = False
        return self.valid
    
    def finalizeVincolo(self, slotId_i:int, slotId_j:int, numSlot_i:int = 0, numSlot_j:int = 0) -> bool:
        if slotId_i == slotId_j:
            self.log.error_log("Vincolo.finalizeVincolo(): il Vincolo deve riferirsi a due Slot differenti")
            self.valid = False
        elif self.penalita == LV_Penalties.LV_0:
            # TODO: da scommentare
            # self.log.warning_log("Vincolo.finalizeVincolo(): la penalità è nulla")
            self.valid = False
        else:
            self.slotId_i = slotId_i
            self.slotId_j = slotId_j
            self.numSlot_i = numSlot_i
            self.numSlot_j = numSlot_j
            self.valid = True
        return self.valid

    def getEquation(self, X_d, X_h):
        '''Genera l'equazione corrispondente al Vincolo (se valido) e la ritorna.\n
        NOTE: non genera la parte relativa alla penalità perchè va fatta a livello globale del Vincolo (in caso di espressioni complesse)'''
        if not self.valid:
            self.log.error_log("Vincolo.getEquation(): Vincolo non valido: " + str(self))
            return None
        
        if self.tipoVincolo == TipoVincolo.AlmenoUnGiornoDopo:
            return X_d[self.slotId_i] > X_d[self.slotId_j]
        if self.tipoVincolo == TipoVincolo.GiornoDiverso:
            return X_d[self.slotId_i] != X_d[self.slotId_j]
        if self.tipoVincolo == TipoVincolo.StessoGiorno:
            return X_d[self.slotId_i] == X_d[self.slotId_j]
        if self.tipoVincolo == TipoVincolo.Parallelo:
            return logical_and(X_d[self.slotId_i] == X_d[self.slotId_j], X_h[self.slotId_i] == X_h[self.slotId_j])
        if self.tipoVincolo == TipoVincolo.SubitoDopo:
            return logical_and(X_d[self.slotId_i] == X_d[self.slotId_j], X_h[self.slotId_i] == X_h[self.slotId_j] + self.numSlot_j)
        self.log.error_log("Vincolo.getEquation(): TipoVincolo non valido: " + str(self.tipoVincolo))
        return None


class TipoOperatore(Enum):
    Nessuno = -1
    OR = 0
    AND = 1
    NOT = 2
    
class Operatore:
    class TipoChild(Enum):
        Sconosciuto = -1
        Vinc_Vinc = 0
        Vinc_Op = 1
        Op_Op = 2
        Op = 3 # Solo per NOT
    
    def __init__(self, tipoOperatore:TipoOperatore, penalita:LV_Penalties = LV_Penalties.LV_0, root:bool = False):
        '''Gestisce un Operatore che al suo interno contiene in mutua esclusione:
            - 2 Vincoli (foglie)
            - 1 Vincolo (foglia) + 1 Operatore
            - 2 Operatori
            - 1 Operatore (NOT)'''
        self.vinc1:Vincolo = None
        self.vinc2:Vincolo = None
        self.op1:Operatore = None
        self.op2:Operatore = None
        self.tipoOperatore:TipoOperatore = tipoOperatore
        self.penalita:LV_Penalties = penalita
        self.root:bool = root
        self.tipoChild:Operatore.TipoChild = Operatore.TipoChild.Sconosciuto
        '''Indica se ho già definito i figli dell'Operatore corrente ed il loro tipo. Si può fare una sola volta'''        
        # per sapere se l'Operatore è stato soddisfatto o meno (se soft constraint)
        self.index_X_penaltiesDoc:int = -1
        
      
    def getEquation(self, X_d, X_h):
        '''Ritorna l'eq (senza l'eventuale then della penalità)'''
        if not bool(self):
            self.log.error_log("Operatore.getEquation(): Operatore non valido")
            return None
        if self.tipoOperatore == TipoOperatore.NOT:
            return logical_not(self.op1.getEquation(X_d, X_h))
        
        if self.tipoOperatore == TipoOperatore.AND:
            if self.tipoChild == Operatore.TipoChild.Vinc_Vinc:
                return logical_and(self.vinc1.getEquation(X_d, X_h), self.vinc2.getEquation(X_d, X_h))
            if self.tipoChild == Operatore.TipoChild.Vinc_Op:
                return logical_and(self.vinc1.getEquation(X_d, X_h), self.op1.getEquation(X_d, X_h))
            if self.tipoChild == Operatore.TipoChild.Op_Op:
                return logical_and(self.op1.getEquation(X_d, X_h), self.op2.getEquation(X_d, X_h))
            
        if self.tipoOperatore == TipoOperatore.OR:
            if self.tipoChild == Operatore.TipoChild.Vinc_Vinc:
                return logical_or(self.vinc1.getEquation(X_d, X_h), self.vinc2.getEquation(X_d, X_h))
            if self.tipoChild == Operatore.TipoChild.Vinc_Op:
                return logical_or(self.vinc1.getEquation(X_d, X_h), self.op1.getEquation(X_d, X_h))
            if self.tipoChild == Operatore.TipoChild.Op_Op:
                return logical_or(self.op1.getEquation(X_d, X_h), self.op2.getEquation(X_d, X_h))
            
        return None
    
    def isRoot(self) -> bool:
        return self.root
        
    def __str__(self) -> str:
        res = "(\n"
        if self.tipoChild in [Operatore.TipoChild.Vinc_Op, Operatore.TipoChild.Vinc_Vinc]:
            res += str(self.vinc1)
        elif self.tipoChild in [Operatore.TipoChild.Op_Op, Operatore.TipoChild.Op]:
            res += str(self.op1)
        elif self.tipoChild == Operatore.TipoChild.Op:
            res += "\n)"
            return res
        else:
            res += "None"
        res += "\n" + str(self.tipoOperatore) + "\n"
        if self.tipoChild == Operatore.TipoChild.Vinc_Vinc:
            res += str(self.vinc2)
        elif self.tipoChild == Operatore.TipoChild.Vinc_Op:
            res += str(self.op1)
        elif self.tipoChild == Operatore.TipoChild.Op_Op:
            res += str(self.op2)
        res += "\n) -> " + str(self.penalita)
        return res
        
    def __bool__(self) -> bool:
        '''Calculated at runtime'''
        # la root deve avere settata la penalità
        if self.root and self.penalita == LV_Penalties.LV_0:
            return False
        # l'Operatore dev'essere stato compilato
        if self.tipoChild == Operatore.TipoChild.Sconosciuto:
            return False
        
        # analisi dipendente dal tipo di figli dell'Operatore
        if self.tipoChild == Operatore.TipoChild.Vinc_Vinc:
            if self.vinc1 is None or self.vinc2 is None:
                return False
            return bool(self.vinc1) and bool(self.vinc2)
        if self.tipoChild == Operatore.TipoChild.Vinc_Op:
            if self.vinc1 is None or self.op1 is None:
                return False
            return bool(self.vinc1) and bool(self.op1)
        if self.tipoChild == Operatore.TipoChild.Op_Op:
            if self.op1 is None or self.op2 is None:
                return False
            return bool(self.op1) and bool(self.op2)
        if self.tipoChild == Operatore.TipoChild.Op:
            if self.op1 is None or self.tipoOperatore != TipoOperatore.NOT:
                return False
            return bool(self.op1)
        
        return False
            
        
    def set2Vincoli(self, vinc1:Vincolo, vinc2:Vincolo) -> bool:
        '''I figli sono 2 Vincoli'''
        if self.tipoChild != Operatore.TipoChild.Sconosciuto:
            self.log.error_log("Operatore.set2Vincoli(): già definiti i figli")
            return False
        if self.tipoOperatore == TipoOperatore.NOT:
            self.log.error_log("Operatore.set2Vincoli(): tipoOpeatore invalido")
            return False        
        self.vinc1 = vinc1
        self.vinc2 = vinc2
        self.tipoChild = Operatore.TipoChild.Vinc_Vinc
        return True
    
    def set1Vincolo1Operatore(self, vinc1:Vincolo, op1) -> bool:
        if self.tipoChild != Operatore.TipoChild.Sconosciuto:
            self.log.error_log("Operatore.set1Vincolo1Operatore(): già definiti i figli")
            return False
        if self.tipoOperatore == TipoOperatore.NOT:
            self.log.error_log("Operatore.set1Vincolo1Operatore(): tipoOpeatore invalido")
            return False
        self.vinc1 = vinc1
        self.op1 = op1      
        self.tipoChild = Operatore.TipoChild.Vinc_Op
        return True              
    
    def set2Operatori(self, op1, op2) -> bool:
        if self.tipoChild != Operatore.TipoChild.Sconosciuto:
            self.log.error_log("Operatore.set2Operatori(): già definiti i figli")
            return False
        if self.tipoOperatore == TipoOperatore.NOT:
            self.log.error_log("Operatore.set2Operatori(): tipoOpeatore invalido")
            return False
        self.op1 = op1
        self.op2 = op2
        self.tipoChild = Operatore.TipoChild.Op_Op
        return True
    
    def set1Operatore(self, op1) -> bool:
        if self.tipoChild != Operatore.TipoChild.Sconosciuto:
            self.log.error_log("Operatore.set1Operatore(): già definiti i figli")
            return False
        if self.tipoOperatore != TipoOperatore.NOT:
            self.log.error_log("Operatore.set1Operatore(): tipoOpeatore invalido")
            return False
        self.op1 = op1
        self.tipoChild = Operatore.TipoChild.Op
        return True        
    

