from docplex.cp.model import *

from datetime import datetime

from log import logger
from dbAPI import dbAPI
from auxiliary import AuxiliaryStruct
from ConstraintBuilder import ConstraintBuilder
from LocaliHandler import LocaliHandler
from RuleDocente import *
from RuleStudente import *
from RuleInsegnamento import *
from customTypes import *

from Parameter import Parameter



class SolutionHandler(CpoSolverListener):
    def __init__(self) -> None:
        super().__init__()
        self.PARAM = Parameter()
        self.dbAPI:dbAPI = dbAPI()
        self.AUX:AuxiliaryStruct = AuxiliaryStruct()
        self.CB:ConstraintBuilder = ConstraintBuilder()
        self.LH:LocaliHandler = LocaliHandler()
        self.log:logger = logger()
        
        self.solutionsName:str = self.PARAM.nomeSolutions
        self.X_d = self.CB.X_d
        self.X_h = self.CB.X_h
        self.X_penaltiesDoc = self.CB.X_penaltiesDoc
        self.X_penaltiesStud = self.CB.X_penaltiesStud
        self.X_penalties = self.CB.X_penalties
        
        self.prevPenalty:int = -1
        self.currPenalty:int = -1
        self.maxTimeWithoutRes:int = self.PARAM.timeLimitTraSolutions
        self.prevTime:datetime = None
        self.currTime:datetime = None
        self.MAX_PENALTY:int = -1
        self.SOGLIA_STORE_RES:float = self.PARAM.sogliaInterestingSol
        
        
    def result_found(self, solver:CpoSolver, sres:CpoSolveResult):
        # override
        
        # sol valida + modalità minimizzazione funzione obiettivo
        if sres.is_solution() and sres.get_objective_values() is not None:
            self.currPenalty = int(str(sres.get_objective_values()[0]))
            self.currTime = datetime.now()
            # solo la prima volta
            if self.MAX_PENALTY == -1:
                self.MAX_PENALTY = self.currPenalty
            
            if self.outputSolNeeded():
                self.log.info_logRes("*"*60)
                self.log.info_logRes("Soluzione con costo: " + str(self.currPenalty) + ", prima era: " + str(self.prevPenalty)) 
                self.log.info_baseLogRes("STORED - Trovata sol con costo: " + str(self.currPenalty) + ", prima era: " + str(self.prevPenalty))                 
                # self.printModel(sres) # only for debug 
                self.showPenalties(sres)
                self.storeSolution(sres)
                self.printStats(sres)
                self.log.info_logRes("\n")
                
                self.prevPenalty = self.currPenalty
                self.prevTime = self.currTime
            else:
                self.log.info_baseLogRes("IGNORED - Trovata sol con costo: " + str(self.currPenalty) + ", prima era: " + str(self.prevPenalty)) 
        # ho trovato una sol valida e non devo ottimizzare niente
        elif sres.is_solution() and sres.get_objective_values() is None:
            # sol valida, non so con quale costo della funzione obiettivo
            self.currPenalty = -1
            self.log.info_baseLogRes("SOLUTION - Esiste una sol valida con questi Hard Constraint")
            self.log.info_logRes("*"*60)
            self.log.info_logRes("Soluzione con costo sconosciuto")
            self.storeSolution(sres)            
            # self.printModel(sres) # only for debug
            self.printStats(sres)
            # stop the current search
            solver.end_search()
          
    def end_solve(self, solver:CpoSolver):
        # override        
        lastRes:CpoSolveResult = solver.get_last_solution()
        
        if lastRes.get_objective_values() is not None:
            # sono in modalità minimizzazione funzione obiettivo
            if int(str(lastRes.get_objective_values()[0])) != self.currPenalty:
                # non ho ancora una sol con questa penalità
                self.currPenalty = int(str(lastRes.get_objective_values()[0]))
                
                self.log.info_baseLogRes("LAST SOLUTION - Trovata sol con costo: " + str(self.currPenalty) + ", prima era: " + str(self.prevPenalty))
                self.log.info_logRes("*"*60)
                self.log.info_logRes("Soluzione con costo: " + str(self.currPenalty) + ", prima era: " + str(self.prevPenalty)) 
                self.showPenalties(lastRes)
                self.storeSolution(lastRes)
            else:
                self.log.info_baseLogRes("LAST SOLUTION - Trovata sol con costo: " + str(self.currPenalty) + " già salvata nel db")

                
    def new_log_data(self, solver, data):
        if self.PARAM.CPLEX_logging:
            self.log.info_cplexLog(str(data).strip("\n"))
    
     
                
    def outputSolNeeded(self) -> bool:
        '''Stabilisce se la soluzione corrente vada salvata oppure no'''
        # inizio a salvare le sol solo quando il costo in termini di penalità è inferiore ad una certa soglia della sol peggiore trovata
        if self.currPenalty > self.MAX_PENALTY*self.SOGLIA_STORE_RES:
            return False
        
        # la prima sol la salvo
        if self.prevPenalty == -1: 
            return True
        # se ho almeno dimezzato il costo della funzione obiettivo
        if self.prevPenalty/2 >= self.currPenalty: 
            return True
        # se è passato un tempo maggiore a self.maxTimeWithoutRes dall'ultima sol salvata (chiaramente in quel lasso di tempo ho trovato
        # una sol con costo della funzione obiettivo minore ma non dimezzato)
        delta = self.currTime - self.prevTime
        if delta.total_seconds() > self.maxTimeWithoutRes:
            # self.log.info_logRes("Trascorso troppo tempo da: " + self.prevTime.strftime("%H:%M:%S") + " a: " + self.currTime.strftime("%H:%M:%S") + " -> " + str(delta.total_seconds()) + " secondi")
            return True
        
        return False
    
    # output functions     
    def storeSolution(self, solution:CpoSolveResult):
        '''Salva nel db il PianoAllocazione'''
        pianoAllocazione:str = self.solutionsName + "_" + str(self.currPenalty)
        
        # la delete è garantita dal builder.py
        # self.dbAPI.delete_pianoAllocazione(pianoAllocazione)
        self.dbAPI.add_pianoAllocazione(pianoAllocazione, self.PARAM.get_description())
        
        try:
            for strIdSlot,slotId in self.AUX.map_strSlotId_to_idSlot.items():
                if int(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC) < 0:
                    continue
                
                nStudentiAssegnati:int = self.AUX.pianoAllocazione[slotId].numIscritti
                tipoLez:str = getStringFromTipoLezione(self.AUX.pianoAllocazione[slotId].tipoLezione)
                tipoLocale:str = getStringFromTipoLocale(self.AUX.pianoAllocazione[slotId].tipoLocale)
                tipoErogazione:str = getStringFromTipoErogazione(self.AUX.pianoAllocazione[slotId].tipoErogazione)
                capienzaAula:str = getStringFromCapienzaLocale(getCapienzaFromNumStudenti(self.AUX.pianoAllocazione[slotId].numIscritti))
                preseElelettriche:str = getStringFromPreseElettriche(self.AUX.pianoAllocazione[slotId].preseElettriche)
                squadra:str = getStringFromSquadra(self.AUX.pianoAllocazione[slotId].squadra)
                # fix bug stats
                if self.AUX.pianoAllocazione[slotId].tipoLocale not in [TipoLocale.Aula, TipoLocale.Aula_CA2, TipoLocale.Aula_CA, TipoLocale.Aula_TB, TipoLocale.Aula_WB]:
                    capienzaAula = "NonDisponibile"
                numSlotConsecutivi:int = self.AUX.pianoAllocazione[slotId].numSlot
                ID_INC:int = int(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC)
                giorno, fasciaOraria = getStringsFromDayHour(solution[self.X_d[slotId]], solution[self.X_h[slotId]])
                
                self.dbAPI.add_slotPianoAllocazione(pianoAllocazione, strIdSlot, nStudentiAssegnati, tipoLez, numSlotConsecutivi, ID_INC,
                                            giorno, fasciaOraria, tipoLocale, tipoErogazione, capienzaAula, squadra, preseElelettriche, False)
                
                for docId in self.AUX.list_docentiInSlot[slotId]:
                    docente = self.AUX.map_idDocenti_to_strDocenti[docId]
                    self.dbAPI.add_docenteInSlot(docente, strIdSlot, pianoAllocazione, False)
        except Exception as e:
            self.log.info_logRes("ERR builder.storeSolution(): error on store data: " + str(e))
            self.dbAPI.rollback()
        else:
            self.dbAPI.commit()
            
    def printModel(self, solution):
        self.log.info_logRes('*'*60)
        self.log.info_logRes("model results:\n")

        # solo per fare testing
        # print per Insegnamento (Slot)
        # for i in range(self.AUX.get_nSlotId()):
        #     # solo 1 Orientamento
        #     if i in self.AUX.list_slotIdInOrientamento[self.AUX.map_Orientamento_to_IdOrientamento["Cybersecurity.INGEGNERIA INFORMATICA (COMPUTER ENGINEERING).Magistrale"]]:
            
        #         if i == 0 or self.AUX.pianoAllocazione[i-1].idInsegnamento != self.AUX.pianoAllocazione[i].idInsegnamento:
        #             self.log.info_logRes(self.AUX.list_metaInsegnamenti[self.AUX.pianoAllocazione[i].idInsegnamento])
        #         out:str = self.AUX.pianoAllocazione[i].slotId + ": " + getSlotFromDayHour(solution[self.X_d[i]], 
        #                                                                                   solution[self.X_h[i]], 
        #                                                                                   self.AUX.pianoAllocazione[i].numSlot)
        #         # print list docenti
        #         docenti:str = "("
        #         for doc in self.AUX.list_docentiInSlot[i]:
        #             docenti += self.AUX.map_idDocenti_to_strDocenti[doc] + ", "
        #         docenti = docenti[:-2]
        #         docenti+=")"
        #         self.log.info_logRes(out + " " + docenti)
                
        # self.log.info_logRes('*'*60)
                
        # print per Giorno - FasciaOraria
        for day in range(self.AUX.get_NUM_DAY()):            
            for hour in range(self.AUX.NUM_SLOT_PER_DAY):
                dayStr, hourStr = getStringsFromDayHour(day, hour)
                self.log.info_logRes("Day: " + dayStr + " FasciaOraria: " + hourStr)
                
                for slotId in self.AUX.map_strSlotId_to_idSlot.values():
                    if int(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC) < 0:
                        continue                    
                    
                    if solution[self.X_d[slotId]] == day and solution[self.X_h[slotId]] == hour:
                        out:str = self.AUX.pianoAllocazione[slotId].slotId + ": allocato per " + str(self.AUX.pianoAllocazione[slotId].numSlot)
                        out+= " slots. Aula di Capienza: " + getStringFromCapienzaLocale(getCapienzaFromNumStudenti(self.AUX.pianoAllocazione[slotId].numIscritti))
                        self.log.info_logRes(out)            
                self.log.info_logRes("")
            # iPrec = 0
            # for i in range(self.AUX.get_nSlotId()):
            #     # solo 1 Orientamento
            #     # if i in self.AUX.list_slotIdInOrientamento[self.AUX.map_Orientamento_to_IdOrientamento["Cybersecurity.INGEGNERIA INFORMATICA (COMPUTER ENGINEERING).Magistrale"]] or i in self.AUX.list_slotIdInOrientamento[self.AUX.map_Orientamento_to_IdOrientamento["Percorso.INGEGNERIA INFORMATICA.Triennale"]]:
                   
            #         if solution[self.X_d[i]] == day:
            #             if i == 0 or self.AUX.pianoAllocazione[iPrec].idInsegnamento != self.AUX.pianoAllocazione[i].idInsegnamento:
            #                 self.log.info_logRes(self.AUX.list_metaInsegnamenti[self.AUX.pianoAllocazione[i].idInsegnamento])
            #             iPrec = i    
                        
            #             out:str = self.AUX.pianoAllocazione[i].slotId + ": " + getSlotFromDayHour(solution[self.X_d[i]], 
            #                                                                                       solution[self.X_h[i]], 
            #                                                                                       self.AUX.pianoAllocazione[i].numSlot)
            #             # print list docenti
            #             docenti:str = "("
            #             for doc in self.AUX.list_docentiInSlot[i]:
            #                 docenti += self.AUX.map_idDocenti_to_strDocenti[doc] + ", "
            #             docenti = docenti[:-2]
            #             docenti+=")"
            #             self.log.info_logRes(out + " " + docenti)
                
        self.log.info_logRes('*'*60)
     
    def printStats(self, solution:CpoSolveResult):
        '''Stampa alcune statistiche utili. C'è un bug che non riesco a trovare!!'''
        return
        self.log.info_logRes("\nSTAT: numero di Slot allocati per ogni Day_FasciaOraria")
        for day in range(self.AUX.get_NUM_DAY()):
            for hour in range(self.AUX.NUM_SLOT_PER_DAY):
                nSlotAllocati:int = sum((solution[self.X_d[slotId]] == day)*
                                    (solution[self.X_h[slotId]] <= hour)*
                                    (solution[self.X_h[slotId]]+self.AUX.pianoAllocazione[slotId].numSlot > hour) 
                                    for slotId in self.AUX.map_strSlotId_to_idSlot.values())
                dayStr, hourStr = getStringsFromDayHour(day, hour)
                self.log.info_logRes(dayStr + "-" + hourStr + ": allocati " + str(nSlotAllocati) + " Slots")
        
        self.log.info_logRes("\nSTAT: numero di Slot allocati per ogni Day_FasciaOraria per ogni Capienza delle Aule")        
        for day in range(self.AUX.get_NUM_DAY()):
            for hour in range(self.AUX.NUM_SLOT_PER_DAY):
                for capienza in CapienzaLocale:
                    nSlotAllocati = 0
                    for slotId in self.AUX.map_strSlotId_to_idSlot.values():
                        if (solution[self.X_d[slotId]] == day) and (solution[self.X_h[slotId]] <= hour) and (solution[self.X_h[slotId]]+self.AUX.pianoAllocazione[slotId].numSlot > hour) and (getCapienzaFromNumStudenti(self.AUX.pianoAllocazione[slotId].numIscritti) == capienza):
                            nSlotAllocati+=1
                    # nSlotAllocati:int = sum((solution[self.X_d[slotId]] == day)*
                    #                 (solution[self.X_h[slotId]] <= hour)*
                    #                 (solution[self.X_h[slotId]]+self.AUX.pianoAllocazione[slotId].numSlot > hour)* 
                    #                 (getCapienzaFromNumStudenti(self.AUX.pianoAllocazione[slotId].numIscritti) == capienza) 
                    #     for slotId in self.AUX.map_strSlotId_to_idSlot.values())   
                    nSlotDisponibili:int = self.LH.getLimit(day, hour, TipoLocale.Aula, capienza)
                    sogliaOccupazione:float = -1
                    if nSlotDisponibili != 0:
                        sogliaOccupazione = nSlotAllocati / nSlotDisponibili
                    dayStr, hourStr = getStringsFromDayHour(day, hour)
                    capienzaStr = getStringFromCapienzaLocale(capienza)
                    self.log.info_logRes(dayStr + "-" + hourStr + ": allocati " + str(nSlotAllocati) + " Slots di capienza: " + capienzaStr 
                             + ". La soglia di utilizzo è: {:.2f}".format(sogliaOccupazione))   
                    if sogliaOccupazione > 1:
                        self.log.info_logRes("ERR ho allocato contemporaneamente più Aule di quelle che potevo -> uso extension levels") 
                
     
        
    def showPenalties(self, solution:CpoSolveResult):
        '''Stampa le soft penalità attive lato Docente'''
        find = lambda funSearch, listRule: next((rule for rule in listRule if funSearch(rule)), None)
       
        # Penalità Orientamenti            
        self.log.info_logRes("Penalties Orientamenti:")
        for index_pen in range(len(self.X_penalties)):

            # se la penalità è attiva
            if solution[self.X_penalties[index_pen]] != 0:
                # ricerca in listRulesRes
                ruleRes:RuleInsegnamento = find(lambda rule: rule.index_X_penalties == index_pen, self.CB.RIH.listRulesRes)
                
                if ruleRes is None:
                    ruleRes:RulePreferenza = find(lambda rule: rule.index_X_penaltiesDoc == index_pen, self.CB.RIH.listRulePrefernzaInsegnamenti)
                    
                    if solution[self.X_penalties[index_pen]] > 0:
                        self.log.info_logRes("penalità: " + str(solution[self.X_penalties[index_pen]]) + " " + str(ruleRes))
                    elif solution[self.X_penalties[index_pen]] < 0:
                        self.log.info_logRes("bonus: " + str(solution[self.X_penalties[index_pen]]) + " " + str(ruleRes))                    
                else:
                    self.log.info_logRes(ruleRes.logRule())
        
        self.log.info_logRes("\nPenalities Docenti:")   
        # Penalità Docenti
        for docId in self.CB.RDH.map_penaltiesInDocente.keys():
            self.log.info_logRes("Penalità per il Docente: " + self.AUX.map_idDocenti_to_strDocenti[docId])
            for index_docPen in self.CB.RDH.map_penaltiesInDocente[docId]:
                # start searching
                rule = find(lambda rule: rule.index_X_penaltiesDoc == index_docPen, self.CB.RDH.listResPreferenze)
                if rule is None:
                    rule = find(lambda rule: rule.index_X_penaltiesDoc == index_docPen, self.CB.RDH.listResLezSameDay)
                if rule is None:
                    rule = find(lambda rule: rule.index_X_penaltiesDoc == index_docPen, self.CB.RDH.listResDistanze)
                if rule is None:
                    rule = find(lambda rule: rule.index_X_penaltiesDoc == index_docPen, self.AUX.list_vincoliInsegnamenti)
                if rule is None:
                    rule = find(lambda rule: rule.index_X_penaltiesDoc == index_docPen, self.AUX.list_operatoriInsegnamenti)

                if solution[self.X_penaltiesDoc[index_docPen]] > 0:
                    self.log.info_logRes("penalità: " + str(solution[self.X_penaltiesDoc[index_docPen]]) + " " + str(rule))
                elif solution[self.X_penaltiesDoc[index_docPen]] < 0:
                    self.log.info_logRes("bonus: " + str(solution[self.X_penaltiesDoc[index_docPen]]) + " " + str(rule))
        
        self.log.info_logRes("\nPenalties Studenti:")        
        # Penalità Studenti
        oldDay = -1
        for rule in self.CB.RSH.listRuleStudente:
            if solution[self.X_penaltiesStud[rule.index_X_penaltiesStud]] > 0:
                if oldDay != -1 and oldDay == rule.day:
                    continue
                oldDay = rule.day
                day:Day = getDayFromInt(rule.day)
                pd:PeriodoDidattico = getPeriodoDidatticoFromStr(rule.periodoDidattico)
                pianoAlloc = str([solution[self.CB.RSH.X_orient[i]] for i in range(rule.index_X_orient, rule.index_X_orient+self.AUX.NUM_SLOT_PER_DAY)])
                orientamento:str = str(self.AUX.list_Orientamenti[rule.orientId])
                if rule.tipoRuleStud == TipoRuleStudente.SlotConsecutivi:
                    self.log.info_logRes(orientamento + "." + rule.periodoDidattico + ": SlotConsecutivi -> penalita: "  + str(solution[self.X_penaltiesStud[rule.index_X_penaltiesStud]]) + ". Piano allocazione di " + str(day) + ":" + pianoAlloc)
                elif rule.tipoRuleStud == TipoRuleStudente._1BucoOrario:
                    self.log.info_logRes(orientamento + "." + rule.periodoDidattico + ": 1 BucoOrario -> penalita: "  + str(solution[self.X_penaltiesStud[rule.index_X_penaltiesStud]]) + ". Piano allocazione di " + str(day) + ":" + pianoAlloc)
                else:
                    self.log.info_logRes(orientamento + "." + rule.periodoDidattico + ": 2 BuchiOrario -> penalita: "  + str(solution[self.X_penaltiesStud[rule.index_X_penaltiesStud]]) + ". Piano allocazione di " + str(day) + ":" + pianoAlloc)
                    
