from docplex.cp.model import if_then, logical_and, logical_not, logical_or

from enum import Enum
from typing import List, Tuple
import json

from log import logger
from auxiliary import AuxiliaryStruct
from RulePenalty import *
from Parameter import Parameter
from patterns import *
from RuleDocente import RulePreferenza
from customTypes import *

    
    
    
class RuleInsegnamento:
    '''Per la gestione delle Rule al fine di limitare gli hard constraint da inserire e gestire corretamente multipli
    soft constraint sugli stessi slot'''
    def __init__(self, slotId_i:int, slotId_j:int, penalita:LV_Penalties = LV_Penalties.LV_0):
        '''args:
            aux (AuxiliaryStruct): solo per avere un debug fatto bene
            slotId_i (int): id del primo slot del constraint
            slotId_j (int): id del secondo slot del constraint (interscambiabili)
            penalita (int): usata solo in caso di soft constraint\n
        NB: due vincoli sugli stessi slot e dello stesso TipoConstraint sono diversi se uno rappresenta un soft constraint
        mentre l'altro è un hard constraint'''
        self.AUX:AuxiliaryStruct = AuxiliaryStruct() # per fare una print fatta bene
        self.log:logger = logger()
        self.slotId_i:int = slotId_i
        self.slotId_j:int = slotId_j
        self.penalita:LV_Penalties = penalita
        self.valid:bool = True # flag per disabilitare la Rule -> NON ha effetto su __eq__()
        self.valPenalita:int = 0 # settato da RuleHandler.finalizeRules()
        
        # log penalità violate
        self.index_X_penalties = -1 # viene settato da ConstraintBuilder per poter loggare i soft constraint violati
        
        if self.slotId_i == self.slotId_j:
            self.log.error_log("RuleInsegnamento.RuleInsegnamento(): il constraint non è valido")

    def __eq__(self, other) -> bool:
        if (self.slotId_i == other.slotId_i and self.slotId_j == other.slotId_j) or (self.slotId_i == other.slotId_j and self.slotId_j == other.slotId_i):
            # stessi slotId
            if (self.penalita == LV_Penalties.LV_H and other.penalita == LV_Penalties.LV_H) or (self.penalita != LV_Penalties.LV_H and other.penalita != LV_Penalties.LV_H):
                # stesso tipo di constraint e di equazione -> NON DEVE ESSERE UGUALE LA PENALITA
                return True
        return False
    
    def __ne__(self, other) -> bool:
        return not self.__eq__(other)
    
    def __str__(self) -> str:
        res:str = self.AUX.pianoAllocazione[self.slotId_i].slotId + ": " + str(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[self.slotId_i].idInsegnamento])
        res += " - " + self.AUX.pianoAllocazione[self.slotId_j].slotId + ": " + str(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[self.slotId_j].idInsegnamento])
        res += "\n" + str(self.slotId_i) + " " + str(self.slotId_j) + " -> " + str(self.penalita)
        return res
    
    def logRule(self) -> str:
        '''Per stampare la RuleInsegnamento in logRes'''
        res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[self.slotId_i].idInsegnamento].titolo + "_"
        res += self.AUX.pianoAllocazione[self.slotId_i].slotId + " "
        res += self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[self.slotId_j].idInsegnamento].titolo + "_"
        res += self.AUX.pianoAllocazione[self.slotId_j].slotId + " -> penalita: " + str(self.valPenalita)
        return res
    
    def __bool__(self):
        '''Un vincolo sullo stesso slotId non ha senso, così come anche un soft constraint con penalità nulla'''
        if not self.valid:
            return False # alternativa per disabilitare la Rule
        if self.slotId_i == self.slotId_j:
            return False
        return True
        

class RuleInsegnamentoHandler(metaclass=SingletonMeta):
    '''Gestisce tutti i constraint, hard e soft, che riguardano due differenti slotId al fine di ottimizzare i constraint aggiunti a CPLEX
    e di dare un peso ponderato globale a tutte le penalità.\nWorkflow:
    - creazione di tutte le Rules tramite addRule()
    - finalizzazione delle Rules tramite finalizeRules() -> non sarà più possibile aggiungere Rulese
    - aggiunta delle Rules a CPLEX (esterna a questo modulo)'''
    def __init__(self):
        self.AUX:AuxiliaryStruct = AuxiliaryStruct()
        self.log:logger = logger()
        self.listRules:List[RuleInsegnamento] = list()
        self.listRulesRes:List[RuleInsegnamento] = list()
        self.debugging:bool = False
        self.load_listRules:bool = False
        self.DELTA:float = 0.1
        self.PARAM = Parameter()
        
        self.jsonData = None
        self.listRulePrefernzaInsegnamenti:List[RulePreferenza] = list()
        self.nRulePreferenzaSoft:int = 0
        
    def debug(self) -> None:
        '''Stampa le Rules valide'''
        self.log.info_log("RuleInsegnamentoHandler.debug():")
        for rule in self.listRulesRes:
            if rule.valid:
                self.log.info_log(rule)
                
    
    def loadJson(self):
        '''Carica il file json con i vincoli sugli orari di lezione dei vari Insegnamenti'''
        if self.jsonData is not None:
            self.log.warning_log("RuleInsegnamentoHandler.loadJson(): file già caricato")
            return
        try:
            f = open(self.PARAM.jsonFileInsegnamenti)
            self.jsonData = json.load(f)        
            f.close()
        except Exception as e:
            self.log.error_log("RuleInsegnamentoHandler.loadJson(): error loading JSON file " + str(e))
            exit(-1)   

    def loadPreferenza(self, tipo:str, value:str) -> Tuple[RulePreferenza.TimeLogic, Day, FasciaOraria, LV_Penalties]:                   
        '''Legge una entry in json per una RulePreferenza.\n
        Formato: "tipoTemporale": "Day_FasciaOraria_LVPenalità"'''
        tipoTemporale:RulePreferenza.TimeLogic = RulePreferenza.TimeLogic.Nessuno
        if tipo == "primaDi":
            tipoTemporale = RulePreferenza.TimeLogic.PrimaDi
        elif tipo == "ugualeA":
            tipoTemporale = RulePreferenza.TimeLogic.UgualeA
        elif tipo == "dopoDi":
            tipoTemporale = RulePreferenza.TimeLogic.DopoDi
            
        day, fasciaOraria = getDayFasciaOrariaFromString(value.split("_")[0], value.split("_")[1])
        penalita = getLV_PenaltiesFromString(value.split("_")[2])
        return tipoTemporale, day, fasciaOraria, penalita
            
    def prepareRules(self):
        '''Prepara le Rule riguardanti i vincolo sulle fasce orarie in cui allocare gli Insegnamenti'''
        # RulePreferenza
        for ID_INC in dict(self.jsonData["preferenze"]).keys():
            if ID_INC not in self.AUX.map_MetaInsegnamento_to_IdInsegnamento.keys():
                self.log.error_log("RuleInsegnamentoHandler.prepareRules(): Insegnamento sconosciuto: " + ID_INC)
                continue
            
            insId:int = self.AUX.map_MetaInsegnamento_to_IdInsegnamento[ID_INC]
            for tipoPref in dict(self.jsonData["preferenze"][ID_INC]).keys():
                if tipoPref == "favorevole":
                    for v, tipo in dict(self.jsonData["preferenze"][ID_INC]["favorevole"]).items():
                        tipoTemporale, day, fasciaOraria, penalita = self.loadPreferenza(tipo, v)
                        
                        # creo una RulePreferenza per ogni Slot di quell'Insegnamento
                        for slotId in self.AUX.list_slotInInsegnamento[insId]:
                            # create the RulePreferenza
                            rule:RulePreferenza = RulePreferenza(0, penalita, True) # fake docente
                            if not rule.make_ruleSpecifica(RulePreferenza.Preferenza.Favorevole, tipoTemporale, slotId, fasciaOraria, day):
                                continue
                            if rule:
                                self.listRulePrefernzaInsegnamenti.append(rule)
                            
                elif str(tipoPref) == "contrario":
                    for v, tipo in dict(self.jsonData["preferenze"][ID_INC]["contrario"]).items():
                        tipoTemporale, day, fasciaOraria, penalita = self.loadPreferenza(tipo, v)
                        
                        # creo una RulePreferenza per ogni Slot di quell'Insegnamento
                        for slotId in self.AUX.list_slotInInsegnamento[insId]:
                            # create the RulePreferenza
                            rule:RulePreferenza = RulePreferenza(0, penalita, True) # fake docente
                            if not rule.make_ruleSpecifica(RulePreferenza.Preferenza.Contrario, tipoTemporale, slotId, fasciaOraria, day):
                                continue
                            if rule:
                                self.listRulePrefernzaInsegnamenti.append(rule)          
        
    def finalizeRules(self) -> int:
        '''Finalizza la creazione delle Rules. Da chiamare una sola volta. Mette insieme tutte le Rules che si riferiscono allo stesso
        constraint e ignore le Rules invalidate:\n
        p = p' * k. Dove k=(studenti_Insegnamento_Orientamento/studenti_Orientamento), p'=LV_->int 
            -> TODO:sarebbe la versione perfetta se potessi sapere quanti studenti sono iscritti all'Insegnamento per ogni Orientamento.
        Per ora mi limito ad usare: p = p'
        PTot = MAX(p) + SUM(p*DELTA).\n
        Return: il numero di Rules modellate come soft constraint'''
        if self.load_listRules:
            self.log.warning_log("RuleInsegnamentoHandler.finalizeRules(): Rules già finalizzate")
            return
        self.load_listRules = True
        
        # 1 - Rule sovrapposizione Insegnamenti
        self.log.info_log("RuleInsegnamentoHandler.finalizeRules(): number of Rules before finalize: " + str(len(self.listRules)))
        
        for indexRule_i in range(len(self.listRules)):
            if self.listRules[indexRule_i].penalita == LV_Penalties.LV_H:
                # gli HardConstraint sono presenti in istanza singola in listRules
                self.listRulesRes.append(self.listRules[indexRule_i])
            elif self.listRules[indexRule_i].penalita != LV_Penalties.LV_H and self.listRules[indexRule_i] in self.listRulesRes:
                # SoftConstraint già presente in listRulesRes -> skip
                continue
            elif not self.listRules[indexRule_i].valid:
                continue 
            else:
                # SoftConstraint non ancora presente in listRulesRes -> mixing con tutti gli altri SoftConstraint dello stesso tipo (se attivi)
                pList:List[float] = list()
                pList.append(float(getIntFromLV_Penalties(self.listRules[indexRule_i].penalita, TipoPenalty.Orientamento)))
                                
                for indexRule_j in range(indexRule_i+1, len(self.listRules)):
                    if self.listRules[indexRule_j] == self.listRules[indexRule_i]:
                        pList.append(float(getIntFromLV_Penalties(self.listRules[indexRule_j].penalita, TipoPenalty.Orientamento)))
                        self.listRules[indexRule_j].valid = False
                
                maxP = max(pList)
                pList.remove(maxP)
                sumP = sum([comp*self.DELTA for comp in pList])
                ruleRes = RuleInsegnamento(self.listRules[indexRule_i].slotId_i, self.listRules[indexRule_i].slotId_j)
                ruleRes.valPenalita = min(round(maxP + sumP), getIntFromLV_Penalties(LV_Penalties.LV_MAX, TipoPenalty.Orientamento))
                if ruleRes.valPenalita > 0:
                    self.listRulesRes.append(ruleRes)

        self.log.info_log("RuleInsegnamentoHandler.finalizeRules(): number of Rules after finalize: " + str(len(self.listRulesRes)))
        
        # 2 - Rule preferenza orari di lezione
        # sono create direttamente con RulePreferenza.make_ruleSpecifica() -> non devo fare niente, solo contare quanti soft constraint ho
        self.nRulePreferenzaSoft = sum(rule.penalita != LV_Penalties.LV_H and rule.penalita != LV_Penalties.LV_0 for rule in self.listRulePrefernzaInsegnamenti)
        self.log.info_log("RuleInsegnamentoHandler.finalizeRules(): numero di RulePreferenza Insegnamento: " + str(self.nRulePreferenzaSoft))
        
        if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Orientamento == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Orientamento == 0:
            if self.PARAM.debuggingPenalty == False:
                return 0
        
        return self.getNumberSoftRules() + self.nRulePreferenzaSoft
    
    
    def model_AddRules(self, index:int, X_penalties, X_d, X_h):
        '''Aggiunge le Rules al modello CPLEX.\n
        Return: la lista di equazione da aggiungere al modello'''
        eqs = list()
        
        # Rule sovrapposizione Insegnamenti
        for index_rule in range(len(self.listRulesRes)):
            rule = self.listRulesRes[index_rule]
            # self.log.info_log(rule)
            if rule.penalita == LV_Penalties.LV_H:
                eq = if_then(X_d[rule.slotId_i] == X_d[rule.slotId_j],
                    logical_or(X_h[rule.slotId_i] >= X_h[rule.slotId_j] + self.AUX.pianoAllocazione[rule.slotId_j].numSlot,
                                X_h[rule.slotId_j] >= X_h[rule.slotId_i] + self.AUX.pianoAllocazione[rule.slotId_i].numSlot))
                eqs.append(eq)
            elif rule.penalita != LV_Penalties.LV_H:
                # se non mi interessano le penalità dei soft constraint (e quindi non ho imposto nessun vincolo su di essi)
                # non aggiungo questa equazioni al modello
                if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Orientamento == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Orientamento == 0:
                    if self.PARAM.debuggingPenalty == False:
                        # self.log.info_log("skip penalty Orientamento")
                        continue
                
                eq = if_then(
                        logical_and(
                            logical_and(X_d[rule.slotId_i] == X_d[rule.slotId_j],
                                logical_not(X_h[rule.slotId_i] >= X_h[rule.slotId_j] + self.AUX.pianoAllocazione[rule.slotId_j].numSlot)
                            ), logical_not(X_h[rule.slotId_j] >= X_h[rule.slotId_i] + self.AUX.pianoAllocazione[rule.slotId_i].numSlot)
                        ),
                        X_penalties[index] == rule.valPenalita)
                rule.index_X_penalties = index
                eqs.append(eq)
                index += 1
                
        # Rule preferenza Insegnamenti      
        for index_rule in range(len(self.listRulePrefernzaInsegnamenti)):
            rule = self.listRulePrefernzaInsegnamenti[index_rule]
            if self.debugging:
                self.log.info_log(rule)
            
            if rule.tipoPreferenza == RulePreferenza.Preferenza.Contrario:
                # hard constraint
                if rule.penalita == LV_Penalties.LV_H:
                    if rule.tipoTemporale == RulePreferenza.TimeLogic.PrimaDi:
                        
                        if rule.day == Day.Indifferente:
                            eq = X_h[rule.slotId] >= getIntFromFasciaOraria(rule.fasciaOraria)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:  
                            eq = X_d[rule.slotId] >= getIntFromDay(rule.day)                      
                        else: # eg: non voglio fare lezione il Gio prima delle ore ...
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                         X_h[rule.slotId] >= getIntFromFasciaOraria(rule.fasciaOraria))
                        
                        if self.debugging:
                            self.log.info_log(eq)
                        eqs.append(eq)    
                    
                    # mi basta che lo Slot finisca dopo lo Slot limite (lo Slot limite mi va ancora bene)
                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.DopoDi:
                        
                        if rule.day == Day.Indifferente:
                            eq = X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 <= getIntFromFasciaOraria(rule.fasciaOraria)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:       
                            eq = X_d[rule.slotId] <= getIntFromDay(rule.day)
                        else: # eg: il Gio non voglio fare lezione dopo le ore ...
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                         X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 <= getIntFromFasciaOraria(rule.fasciaOraria))

                        if self.debugging:
                            self.log.info_log(eq)
                        eqs.append(eq)    

                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.UgualeA:
                        
                        if rule.day == Day.Indifferente:
                            eq = X_h[rule.slotId] != getIntFromFasciaOraria(rule.fasciaOraria)
                            self.log.warning_log("RuleInsegnamentoHandler.model_AddRules(): strange RulePreferenza added")
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:   
                            # eg: un Insegnamento non può far lezione il giovedì
                            eq = X_d[rule.slotId] != getIntFromDay(rule.day)
                        else: # per indicare indisponibilità Insegnamento
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                         logical_or( # inizia dopo o finisce prima
                                             X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria),
                                             X_h[rule.slotId] + self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                                         ))
                        if self.debugging:
                            self.log.info_log(eq)
                        eqs.append(eq)                    
                    
                # soft constraint
                else:
                    # se non mi interessano le penalità dei soft constraint (e quindi non ho imposto nessun vincolo su di essi)
                    # non aggiungo questa equazioni al modello
                    if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Orientamento == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Orientamento == 0:
                        if self.PARAM.debuggingPenalty == False:
                            # self.log.info_log("skip penalty Orientamento (PreferenzaContraria)")
                            continue
                                        
                    rule.index_X_penaltiesDoc = index
                    
                    if rule.tipoTemporale == RulePreferenza.TimeLogic.PrimaDi:
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId] < getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(X_h[rule.slotId] >= getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = if_then(X_d[rule.slotId] < getIntFromDay(rule.day),
                                        X_penalties[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(X_d[rule.slotId] >= getIntFromDay(rule.day),
                                        X_penalties[index] == 0)
                        else: # eg: non voglio fare lezione il Gio prima delle ore ...
                            eq = if_then(logical_and(
                                    X_h[rule.slotId] < getIntFromFasciaOraria(rule.fasciaOraria), X_d[rule.slotId] == getIntFromDay(rule.day)),
                                    X_penalties[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(logical_not(
                                    logical_and(
                                        X_h[rule.slotId] < getIntFromFasciaOraria(rule.fasciaOraria), X_d[rule.slotId] == getIntFromDay(rule.day))
                                    ), X_penalties[index] == 0)
                        index += 1
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq, eq1])  
                        
                    # mi basta che lo non Slot finisca dopo lo Slot limite (lo Slot limite mi va ancora bene)
                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.DopoDi:
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 > getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 <= getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = if_then(X_d[rule.slotId] > getIntFromDay(rule.day),
                                        X_penalties[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(X_d[rule.slotId] <= getIntFromDay(rule.day),
                                        X_penalties[index] == 0)
                        else:
                            eq = if_then(logical_and(
                                        X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 > getIntFromFasciaOraria(rule.fasciaOraria), 
                                        X_d[rule.slotId] == getIntFromDay(rule.day)),
                                    X_penalties[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(logical_not(
                                    logical_and(
                                        X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 > getIntFromFasciaOraria(rule.fasciaOraria), 
                                        X_d[rule.slotId] == getIntFromDay(rule.day))
                                    ), X_penalties[index] == 0)                            
                        index += 1
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq, eq1])    
                        
                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.UgualeA:
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId] == getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(X_h[rule.slotId] != getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                        X_penalties[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(X_d[rule.slotId] != getIntFromDay(rule.day),
                                        X_penalties[rule.slotId] == 0)
                        else: # se quello Slot è allocato (anche se iniziasse prima)
                            # se lo Slot è allocato quel giorno e NON inizia dopo O NON finisce prima
                            eq = if_then(logical_and( 
                                        X_d[rule.slotId] == getIntFromDay(rule.day),
                                    logical_not(logical_or(                                            
                                        X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_h[rule.slotId] + self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                                    ))),
                                    X_penalties[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(logical_not(logical_and( 
                                        X_d[rule.slotId] == getIntFromDay(rule.day),
                                    logical_not(logical_or(                                            
                                        X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_h[rule.slotId] + self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                                    )))),
                                    X_penalties[index] == 0)                        
                        index += 1
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq, eq1])  
                    
            elif rule.tipoPreferenza == RulePreferenza.Preferenza.Favorevole:
                # lo Slot limite non mi va bene
                
                # hard constraint
                if rule.penalita == LV_Penalties.LV_H:

                    if rule.tipoTemporale == RulePreferenza.TimeLogic.PrimaDi:
                        if rule.day == Day.Indifferente:
                            eq = X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:
                            eq = X_d[rule.slotId] < getIntFromDay(rule.day)
                        else: # eg: il gio prima delle ore ...
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                    X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria))
                        
                        if self.debugging:
                            self.log.info_log(eq)
                        eqs.append(eq)      

                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.DopoDi:
                        if rule.day == Day.Indifferente:
                            eq = X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = X_d[rule.slotId] > getIntFromDay(rule.day)
                        else: # eg voglio fare lezione il Gio dopo le ore ...
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                        X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria))
                        if self.debugging:
                            self.log.info_log(eq)
                        eqs.append(eq) 
                        
                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.UgualeA:
                        if rule.day == Day.Indifferente or rule.fasciaOraria == FasciaOraria._Indifferente:
                            self.log.warning_log("RuleInsegnamentoHandler.model_AddRules(): questa RulePreferenza non ha senso: " + str(rule))
                            continue
                        
                        eq1 = X_d[rule.slotId] == getIntFromDay(rule.day)
                        eq2 = X_h[rule.slotId] == getIntFromFasciaOraria(rule.fasciaOraria)
                        if self.debugging:
                            self.log.info_log(eq1)
                            self.log.info_log(eq2)
                        eqs.extend([eq1, eq2])      
                    
                # soft constraint                
                else:
                    # se non mi interessano le penalità dei soft constraint (e quindi non ho imposto nessun vincolo su di essi)
                    # non aggiungo questa equazioni al modello
                    if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Orientamento == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Orientamento == 0:
                        if self.PARAM.debuggingPenalty == False:
                            # self.log.info_log("skip penalty Orientamento (PreferenzaFavorevole)")
                            continue
                                            
                    rule.index_X_penaltiesDoc = index                    
                    
                    if rule.tipoTemporale == RulePreferenza.TimeLogic.PrimaDi:
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot > getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:
                            eq = if_then(X_d[rule.slotId] < getIntFromDay(rule.day),
                                        X_penalties[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))                        
                            eq1 = if_then(X_d[rule.slotId] > getIntFromDay(rule.day),
                                        X_penalties[index] == 0)
                        else: # eg: il gio prima delle ore ...
                            eq = if_then(logical_and(
                                X_d[rule.slotId] == getIntFromDay(rule.day),
                                X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                            ), X_penalties[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(logical_not(
                                logical_and(
                                    X_d[rule.slotId] == getIntFromDay(rule.day),
                                    X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                                )
                            ), X_penalties[index] == 0)
                        
                        index += 1
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq,eq1])                     

                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.DopoDi:
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(X_h[rule.slotId] <= getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = if_then(X_d[rule.slotId] > getIntFromDay(rule.day),
                                        X_penalties[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(X_d[rule.slotId] <= getIntFromDay(rule.day),
                                        X_penalties[index] == 0)
                        else: # eg voglio fare lezione il Gio dopo le ore ...
                            eq = if_then(logical_and(
                                    X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria), X_d[rule.slotId] == getIntFromDay(rule.day)),
                                    X_penalties[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(logical_not(
                                    logical_and(
                                        X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria), 
                                        X_d[rule.slotId] == getIntFromDay(rule.day))
                                    ), X_penalties[index] == 0)                        
                        index += 1
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq, eq1]) 
                        
                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.UgualeA:
                            
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId] == getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(X_h[rule.slotId] != getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penalties[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                        X_penalties[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(X_d[rule.slotId] != getIntFromDay(rule.day),
                                        X_penalties[index] == 0)
                        else:
                            # eg voglio fare lezione il tale Day ed in tale FasciaOraria
                            eq = if_then(logical_and(
                                    X_h[rule.slotId] == getIntFromFasciaOraria(rule.fasciaOraria), X_d[rule.slotId] == getIntFromDay(rule.day)),
                                    X_penalties[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Orientamento))
                            eq1 = if_then(logical_not(logical_and(
                                    X_h[rule.slotId] == getIntFromFasciaOraria(rule.fasciaOraria), X_d[rule.slotId] == getIntFromDay(rule.day))),
                                    X_penalties[index] == 0)                            
                        index += 1                    
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq, eq1])             
        
        return eqs
    

    def getNumberSoftRules(self) -> int:
        '''Ritorna il numero di soft constraint, mi serve per allocare le variabili binarie per CPLEX'''
        if not self.load_listRules:
            self.log.error_log("RuleInsegnamentoHandler.getNumberSoftRules(): caricamento Rules non ancora terminato")
            return 0
        nSoftConstr:int = 0
        for rule in self.listRulesRes:
            if rule.penalita != LV_Penalties.LV_H and rule.valid:
                nSoftConstr += 1
        return nSoftConstr
        
    def addRule(self, rule:RuleInsegnamento) -> int:
        '''Gestisce l'aggiunta di un constraint, verifica se non è già presente e eventualemente aggiorna la penalità di un soft constraint
        già presente.
        Return:
            -1: errors
            0: il constraint è già presente nella lista e non ha fatto niente
            1: ha aggiunto un constraint alla lista
            2: ha modificato un constraint della lista
            3: ha trasformato un constraint da soft a hard\n
        '''
        if self.load_listRules:
            self.log.error_log("RuleInsegnamentoHandler.addRule(): aggiunta Rules già terminata")
            return -1
        if rule in self.listRules:
            # esiste già un constraint dello stesso TipoConstraint nella lista, ANCHE CON PENALITA' DIVERSA
            if rule.penalita == LV_Penalties.LV_H:
                # ho la ripetizione di un hard constraint
                if self.debugging:
                    self.log.info_log("RuleInsegnamentoHandler.addRule(): già presente: " + str(rule))
                return 0
            else:
                # devo aggiornare la penalità del soft constraint
                # manetengo entrambe le Rule nella lista (anche se logicamente uguali) -> 
                # durante la finalizzazione le fondo in un'unica Rule
                self.listRules.append(rule)
                if self.debugging:
                    self.log.info_log("RuleInsegnamentoHandler.addRule(): update (added): " + str(self.listRules[self.listRules.index(rule)].penalita))
                return 2
        elif rule.penalita == LV_Penalties.LV_H:
            # se la nuova Rule è un hard constraint devo verificare che non esista lo stesso constraint implementato come soft -> 
            # se così fosse lo devo sovrascrivere
            rTmp = RuleInsegnamento(rule.slotId_i, rule.slotId_j)
            indexes = [index_r for index_r, r in enumerate(self.listRules) if r == rTmp]
            
            if len(indexes) > 0:
                # Rule già presente in almeno un'instanza di tipo soft constraint ->
                # overwrite della prima istanza soft constraint con l'hard constraint + invalidazione di tutte le altre
                self.listRules[indexes[0]] = rule
                for i in range(1,len(indexes)):
                    self.listRules[indexes[i]].valid = False
                
                if self.debugging:
                    self.log.info_log("RuleInsegnamentoHandler.addRule(): overwrite soft -> hard: " + str(rule))
                return 3
            else:
                # il nuovo constraint non c'è nella lista e va semplicemente aggiunto
                self.listRules.append(rule)
                if self.debugging:
                    self.log.info_log("RuleInsegnamentoHandler.addRule(): nuovo constraint: " + str(rule))
                return 1
        elif rule.penalita != LV_Penalties.LV_H:
            # se c'è già la stessa Rule come hard constraint -> non mi serve a nulla questo soft constraint
            rTmp = RuleInsegnamento(rule.slotId_i, rule.slotId_j)
            if rTmp in self.listRules:
                if self.debugging:
                    self.log.info_log("RuleInsegnamentoHandler.addRule(): c'è già un hard constraint: " + str(rule))
                return 0
            else:
                # ho un soft constraint non ancora presente nella lista semplicemente da aggiungere
                self.listRules.append(rule)
                if self.debugging:
                    self.log.info_log("RuleInsegnamentoHandler.addRule(): nuovo constraint: " + str(rule))
                return 1
        return -1