from docplex.cp.model import *

from enum import Enum
from typing import List, Tuple, Dict

from log import logger
from auxiliary import AuxiliaryStruct
from customTypes import *
from dbAPI import dbAPI
from RulePenalty import *
from Vincoli import *
from Parameter import Parameter
from patterns import *


class TipoRule(Enum):
    Sconosciuto = 0
    PreferenzaLezione = 1
    LezioneInsDifferentiSameDay = 2
    DistInsDiffrerenti = 3
    

class RuleDocente:
    def __init__(self, docId:int):
        '''Creazione generica RuleDocente. Non è sufficiente a creare una RuleDocente valida'''
        self.docId:int = -1
        self.AUX:AuxiliaryStruct = AuxiliaryStruct()
        self.log:logger = logger()
        self.valid:bool = False
        self.tipoRule:TipoRule = TipoRule.Sconosciuto
        self.penalita:LV_Penalties = LV_Penalties.LV_0
        self.index_X_penaltiesDoc = -1
        
        if docId not in self.AUX.map_strDocenti_to_idDocenti.values():
            self.log.error_log("RuleDocente.RuleDocente(): docId non valido")
            self.docId = -1
        else:
            self.docId:int = docId
            
    def __eq__(self, o: object) -> bool:
        '''La penalità NON è criterio di uguaglianza'''
        if not self.valid or not o.valid or -1 in [self.docId, o.docId]:
            self.log.warning_log("RuleDocente.eq(): almeno una RuleDocente è invalida")
            return False
        
        return self.tipoRule == o.tipoRule and self.docId == o.docId

    def __ne__(self, o: object) -> bool:
        return not self.__eq__(o)
    
    def __bool__(self):
        if self.docId == -1:
            self.valid = False
        if self.tipoRule == TipoRule.Sconosciuto:
            self.valid = False
        if self.penalita == LV_Penalties.LV_0:
            self.valid = False
        if self.index_X_penaltiesDoc != -1 and self.penalita == LV_Penalties.LV_H:
            self.valid = False
        return self.valid


class RuleDist(RuleDocente):
    class DistLogic(Enum):
        Nessuno = 0
        DistMinima = 1
        DistMassima = 2
    
    def __init__(self, docId: int, penalita:LV_Penalties, dist:int, distLogic:DistLogic):
        '''Indica la distanza minima/massima in termini di Slot tra due Slot allocati in un determinato giorno, appartenenti a due Insegnamenti
        differenti e tenuti dallo stesso Docente.
        Può essere considerato come un hard constraint, come un soft constraint o non essere proprio implementato'''
        # super class
        super().__init__(docId)
        self.tipoRule = TipoRule.DistInsDiffrerenti
        self.penalita = penalita
        
        # child class
        self.dist:int = dist
        self.distLogic:RuleDist.DistLogic = distLogic
        self.slotId_i = -1
        self.slotId_j = -1
        
    def __str__(self) -> str:
        if not self:
            return "RuleDist non valida"
        return "RuleDist: " + self.AUX.map_idDocenti_to_strDocenti[self.docId] + " " + str(self.penalita) + " " + str(self.distLogic) + ": " + str(self.dist) + " " + self.AUX.pianoAllocazione[self.slotId_i].slotId + "-" + self.AUX.pianoAllocazione[self.slotId_j].slotId
                
    def __eq__(self, o: object) -> bool:
        if not self.valid or not o.valid:
            return self.docId == o.docId and self.distLogic == o.distLogic # la rule diventa valid solo in finalizzazione
        
        if not super().__eq__(o):
            return False
        if (self.slotId_i == o.slotId_i and self.slotId_j == o.slotId_j) or (self.slotId_i == o.slotId_j and self.slotId_j == o.slotId_i):
            return self.distLogic == o.distLogic # non mi interessa la dist, non posso avere 2 rule che si riferiscono agli stessi Slot
        return False
            
    def __bool__(self):
        if not super().__bool__():
            return False
        if self.slotId_i == -1 or self.slotId_j == -1 or self.slotId_i == self.slotId_j or self.distLogic == RuleDist.DistLogic.Nessuno:
            if self.dist <= 0 and self.distLogic == RuleDist.DistLogic.DistMinima:
                self.valid = False
            elif self.dist < 0 and self.distLogic == RuleDist.DistLogic.DistMassima:
                self.valid = False
        return self.valid
        
    def make_ruleDist(self, slotId_i:int, slotId_j:int) -> bool:
        '''Compila la RuleDist, sia di tipo hard che di tipo soft, specifica per due Slot'''
        if self.AUX.pianoAllocazione[slotId_i].idInsegnamento == self.AUX.pianoAllocazione[slotId_j].idInsegnamento:
            self.log.error_log("RuleDist.make_ruleDist(): gli Slot si riferiscono allo stesso Insegnamento")
            self.valid = False
            return False
        if slotId_i not in self.AUX.list_slotInDocente[self.docId] or slotId_j not in self.AUX.list_slotInDocente[self.docId]:
            self.log.error_log("RuleDist.make_ruleDist(): gli Slot non sono tenuti dallo stesso Docente")
            self.valid = False
            return False
        self.slotId_i = slotId_i
        self.slotId_j = slotId_j
        self.valid = True
        return self.valid


class RuleLezSameDay(RuleDocente):
    def __init__(self, docId: int, penalita:LV_Penalties):
        '''Indica la presenza di Slot appartenenti a due Insegnamenti differenti, tenuti dallo stesso Docente e allocati lo stesso Day.
        Può essere considerato come un hard constraint, come un soft constraint o non essere proprio implementato'''
        # super class
        super().__init__(docId)
        self.tipoRule = TipoRule.LezioneInsDifferentiSameDay
        self.penalita = penalita
        
        # child class
        self.slotId_i = -1
        self.slotId_j = -1
        
    def __str__(self) -> str:
        if not self:
            return "RuleLezSameDay non valida"
        return "RuleLezSameDay: " + self.AUX.map_idDocenti_to_strDocenti[self.docId] + " " + str(self.penalita) + " " + self.AUX.pianoAllocazione[self.slotId_i].slotId + "-" + self.AUX.pianoAllocazione[self.slotId_j].slotId
                
    def __eq__(self, o: object) -> bool:
        if not self.valid or not o.valid:
            return self.docId == o.docId # la rule diventa valid solo in finalizzazione
        
        if not super().__eq__(o):
            return False
        if (self.slotId_i == o.slotId_i and self.slotId_j == o.slotId_j) or (self.slotId_i == o.slotId_j and self.slotId_j == o.slotId_i):
            return True # non mi interessa la dist minima, non posso avere 2 rule che si riferiscono agli stessi Slot
        return False
            
    def __bool__(self):
        if not super().__bool__():
            return False
        if self.slotId_i == -1 or self.slotId_j == -1 or self.slotId_i == self.slotId_j:
            self.valid = False
        return self.valid
        
    def make_ruleLezSameDay(self, slotId_i:int, slotId_j:int) -> bool:
        '''Compila la RuleLezSameDay, sia di tipo hard che di tipo soft, specifica per due Slot. La Rule viene validata solo ora'''
        if self.AUX.pianoAllocazione[slotId_i].idInsegnamento == self.AUX.pianoAllocazione[slotId_j].idInsegnamento:
            self.log.error_log("RuleLezSameDay.make_ruleLezSameDay(): gli Slot si riferiscono allo stesso Insegnamento")
            self.valid = False
            return False
        if slotId_i not in self.AUX.list_slotInDocente[self.docId] or slotId_j not in self.AUX.list_slotInDocente[self.docId]:
            self.log.error_log("RuleLezSameDay.make_ruleLezSameDay(): gli Slot non sono tenuti dallo stesso Docente")
            self.valid = False
            return False
        self.slotId_i = slotId_i
        self.slotId_j = slotId_j
        self.valid = True
        return self.valid


class RulePreferenza(RuleDocente):
    class Target(Enum):
        Generica = 0
        Specifica = 1
        
    class Preferenza(Enum):
        Nessuno = -1
        Favorevole = 1
        Contrario = 2
        
    class TimeLogic(Enum):
        Nessuno = -1
        PrimaDi = 0 # lo Slot limite va ancora bene
        DopoDi = 1 # lo Slot limite va ancora bene
        UgualeA = 2
    
    def __init__(self, docId:int, penalita:LV_Penalties, ignoreDocente:bool = False):
        '''Indica la preferenza di un Docente ad effettuare lezione in determinate FasceOrarie.\n
        Modalità d'uso:
            - la RulePreferenza è un'indicazione generica (non riferita ad un particolare slotId) -> l'Handler la espande a tutti gli slotId di
            lezione tenuti dal docente (è una preferenza generica del docente a fare lezioni generiche in determinate fasce orarie).\n
            Target = Generica
            - la RulePreferenza è un'indicazione precisa (riferita ad un particolare slotId) -> per gestire la possibilità che i Docenti preferiscano
            mantenere l'Orario dell'anno precedente.\n
            Target = Specifica.\n
        NB: non è sufficiente a creare una RulePreferenza valida'''
        # super class
        super().__init__(docId)
        self.tipoRule = TipoRule.PreferenzaLezione
        self.penalita = penalita
        self.ignoreDocente:bool = ignoreDocente
        
        # child class
        self.slotId:int = -1 # per default RulePreferenza è generica
        self.target:RulePreferenza.Target = RulePreferenza.Target.Generica
        self.tipoPreferenza:RulePreferenza.Preferenza = RulePreferenza.Preferenza.Nessuno
        self.tipoTemporale:RulePreferenza.TimeLogic = RulePreferenza.TimeLogic.Nessuno
        self.day:Day = Day.Indifferente 
        self.fasciaOraria:FasciaOraria = FasciaOraria._Indifferente

            
    def __eq__(self, other) -> bool:
        '''La penalità NON è criterio di uguaglianza'''        
        if not super().__eq__(other):
            return False
            
        if self.target == other.target:
            if self.target == RulePreferenza.Target.Generica:
                return self.tipoPreferenza == other.tipoPreferenza and self.tipoTemporale == other.tipoTemporale and self.fasciaOraria == other.fasciaOraria and self.day == other.day
            if self.target == RulePreferenza.Target.Specifica:
                if self.slotId == other.slotId:            
                    return self.tipoPreferenza == other.tipoPreferenza and self.tipoTemporale == other.tipoTemporale and self.fasciaOraria == other.fasciaOraria and self.day == other.day
        return False
            
    def __ne__(self, other) -> bool:
        return not self.__eq__(other)
    
    def __str__(self) -> str:
        res:str = "RulePreferenza: "
        if not self.ignoreDocente:        
            res += self.AUX.map_idDocenti_to_strDocenti[self.docId] + " " 
        elif self.slotId != -1:
            res += self.AUX.pianoAllocazione[self.slotId].slotId + " "
        res += str(self.penalita) + " " + str(self.target) + " " + str(self.tipoPreferenza) + " " + str(self.tipoTemporale) + " " + str(self.day) + "-" + str(self.fasciaOraria)
        if self.target == RulePreferenza.Target.Specifica:
            res += " slot: " + self.AUX.pianoAllocazione[self.slotId].slotId
        return res
    
    def __bool__(self):
        '''Se una RulePreferenza non è valida non ha senso aggiungerla alla lista'''        
        if not super().__bool__():
            return False
        if self.slotId == -1 and self.target != RulePreferenza.Target.Generica:
            self.valid = False
        if self.day == Day.Indifferente and self.fasciaOraria == FasciaOraria._Indifferente:
            self.valid = False
        if self.tipoTemporale == RulePreferenza.TimeLogic.Nessuno:
            self.valid = False
        return self.valid
        
        
    def make_ruleGenerica(self, tipoPreferenza:Preferenza, tipoTemporale:TimeLogic,
                          fasciaOraria:FasciaOraria, day:Day) -> bool:
        '''Compila una RulePreferenza con target Generico.\n
        Una RulePreferenza Generica non si riferisce ad un particolare Slot ma indica la preferenza del docente a fare/non fare lezione in una 
        FasciaOraria e/o in un determinato Day.\n
        Return:
            True: la RulePreferenza è stata compilata correttamente
            False: la RulePreferenza non è stata compilata correttamente'''
        self.target = RulePreferenza.Target.Generica
        self.tipoPreferenza = tipoPreferenza
        self.fasciaOraria = fasciaOraria
        self.day = day

        self.tipoTemporale = tipoTemporale
        self.valid = True
        return self.valid
        
    def make_ruleSpecifica(self, tipoPreferenza:Preferenza, tipoTemporale:TimeLogic,
                           slotId:int, fasciaOraria:FasciaOraria, day:Day) -> bool:
        '''Compila una RulePreferenza di tipo Specifica.\n
        Return:
            True: la RulePreferenza è stata compilata correttamente
            False: la RulePreferenza non è stata compilata correttamente'''
        self.target = RulePreferenza.Target.Specifica
        self.tipoPreferenza = tipoPreferenza
        self.tipoTemporale = tipoTemporale
        self.fasciaOraria = fasciaOraria
        self.day = day
        
        # per usarle anche come preferenza di Insegnamento
        if not self.ignoreDocente and slotId not in self.AUX.list_slotInDocente[self.docId]:
            self.log.error_log("RulePreferenza.make_ruleSpecifica(): slotId non valido")
            self.valid = False
            return False
        self.slotId = slotId
        
        # se compilo una RuleDocente per un un vincolo insensato => la RuleDocente diventa invalida
        # eg. voglio fare una lezione da 2 slot prima delle 10.00
        # eg. voglio fare una lezione da 2 slot dopo le 16.00
        if self.fasciaOraria != FasciaOraria._Indifferente:
            if self.tipoTemporale == RulePreferenza.TimeLogic.PrimaDi and self.tipoPreferenza == RulePreferenza.Preferenza.Favorevole:
                if self.AUX.pianoAllocazione[self.slotId].numSlot > getIntFromFasciaOraria(self.fasciaOraria):
                    self.log.error_log("RulePreferenza.make_ruleSpecifica(): RulePreferenza non applicabile allo Slot: " + self.AUX.pianoAllocazione[self.slotId].slotId)
                    self.valid = False
                    return False
            if self.tipoTemporale == RulePreferenza.TimeLogic.DopoDi and self.tipoPreferenza == RulePreferenza.Preferenza.Favorevole:
                if self.AUX.NUM_SLOT_PER_DAY - getIntFromFasciaOraria(self.fasciaOraria)-1 < self.AUX.pianoAllocazione[self.slotId].numSlot:
                    self.log.error_log("RulePreferenza.make_ruleSpecifica(): RulePreferenza non applicabile allo Slot: " + self.AUX.pianoAllocazione[self.slotId].slotId)
                    self.valid = False
                    return False
        if self.day != Day.Indifferente and self.fasciaOraria == FasciaOraria._Indifferente:
            if self.tipoTemporale == RulePreferenza.TimeLogic.DopoDi and self.AUX.get_NUM_DAY()-1 >= getIntFromDay(self.day):
                self.log.error_log("RulePreferenza.make_ruleSpecifica(): RuleDocente invalida")
                self.valid = False
                return False
            if self.tipoTemporale == RulePreferenza.TimeLogic.PrimaDi and getIntFromDay(self.day) <= 0:
                self.log.error_log("ruleDocente.make_ruleSpecifica(): RuleDocente invalida")
                self.valid = False
                return False
        
        self.valid = True
        return True



class RuleDocenteHandler(metaclass=SingletonMeta):
    def __init__(self):
        '''Per gestire tutte le RuleDocente implementate:
            - Preferenze su quando svolgere/non svolgere lezione (RulePreferenza) Hard/Soft (include anche indisponibilità Docente)
            (in caso di conflitto con xml prevale quest'ultimo)
            - Preferenza utilizzo Orario allocazione precedente (RulePreferenza) Hard/Soft (in caso di conflitto con xml prevale quest'ultimo)
            - Il non volere tenere lezione di Insegnamenti diversi lo stesso Day (RuleLezSameDay) Hard/Soft
            - Impostare una distanza minima/massima tra Slot appartenenti a Insegnamenti diversi tenuti dallo stesso Docente lo stesso Day (Hard/Soft)
            - Implementazione vincoli xml
            - Limite giornaliero di Slot allocabili per il singolo Docente Hard (valore di default o personalizzato)    
        '''
        self.AUX:AuxiliaryStruct = AuxiliaryStruct()
        self.dbAPI:dbAPI = dbAPI()
        self.log:logger = logger()
        
        self.jsonData = None
        self.debugging:bool = True
        
        self.load_lists:bool = False
        '''Flag per dire se tutte le liste sono state caricate'''
        
        self.listPreferenze:List[RulePreferenza] = list()
        self.listResPreferenze:List[RulePreferenza] = list() # contiene solo RulePreferenza di tipo Specifico (ho completato l'espansione)
        self.nRulePreferenza:int = 0
        
        self.listDistanze:List[RuleDist] = list()
        self.listResDistanze:List[RuleDist] = list()
        self.nRuleDistanza:int = 0
        
        self.listLezSameDay:List[RuleLezSameDay] = list()
        self.listResLezSameDay:List[RuleLezSameDay] = list()
        self.nRuleLezSameDay:int = 0
        
        self.nRuleXml:int = 0
        
        self.default_limitHourDocentePerDay:int = 0
        self.map_limitHourDocentePerDay:Dict[int,int] = dict()
        '''mapping idDocente -> limite slot per Day personalizzato (solo per i docenti che non usano il limite di default)'''
        self.map_docentiConInsegnamentiSovrapponibili:Dict[int,List[int]] = dict()
        '''mapping idDocente -> lista di ID_INC di Insegnamenti con Slot sovrapponibili anche se allocati allo stesso Docente.\n
        NB: se nello Slot ci sono anche altri Docenti per cui non vale la sovrapposizione gli Slot non potranno comunque essere sovrapposti.'''
        
        self.map_penaltiesInDocente:Dict[int, List[int]] = dict()
        '''mapping idDocente -> list index in X_penaltiesDoc'''
        
        self.PARAM:Parameter = Parameter()
    
    
    def loadJson(self):
        '''Carica da json tutti i dati dei constraint'''
        if self.jsonData is not None:
            self.log.warning_log("RuleDocenteHandler.loadJson(): file già caricato")
            return
        try:
            f = open(self.PARAM.jsonFileDocenti)
            self.jsonData = json.load(f)        
            f.close()
        except Exception as e:
            self.log.error_log("RuleDocenteHandler.loadJson(): error loading JSON file")
            exit(-1)        
    
    def prepareRules(self):
        '''Costruzione di tutti constraint di preferenza dei Docenti'''  
        if self.jsonData is None:
            self.log.error_log("RuleDocenteHandler.prepareRules(): file json non ancora caricato")
            return
        
        # 0 - Limite Slot lezione per Day
        for k,v in dict(self.jsonData["Docenti"]["maxSlotPerDay"]).items():
            if k == "default":
                self.default_limitHourDocentePerDay = int(v)
            else:
                if k in self.AUX.map_strDocenti_to_idDocenti.keys():
                    self.map_limitHourDocentePerDay[self.AUX.map_strDocenti_to_idDocenti[k]] = int(v)
                else:
                    self.log.warning_log("RuleDocenteHandler.createMap_limitHourDocentePerDay(): docente " + str(k) + " sconosciuto")  
                    
        # 0.1 - I Docenti non possono essere allocati contemporaneamente in più Insegnamenti salvo se diversamente specificato
        for k,v in dict(self.jsonData["Docenti"]["insegnamentiSovrapponibili"]).items():
            if k in self.AUX.map_strDocenti_to_idDocenti.keys():
                self.map_docentiConInsegnamentiSovrapponibili[self.AUX.map_strDocenti_to_idDocenti[k]] = list(v)
            else:
                self.log.warning_log("RuleDocenteHandler.createMap_docentiConInsegnamentiSovrapponibili(): docente " + str(k) + " sconosciuto")                  
        
        # 1 - RulePreferenza
        for docente in dict(self.jsonData["Docenti"]["preferenze"]).keys():
            if str(docente) not in self.AUX.map_strDocenti_to_idDocenti.keys():
                self.log.error_log("RuleDocenteHandler.prepareRules(): Docente sconosciuto")
                continue
            
            docId = self.AUX.map_strDocenti_to_idDocenti[str(docente)]
            for tipoPref in dict(self.jsonData["Docenti"]["preferenze"][str(docente)]).keys():
                if tipoPref == "favorevole":
                    for v, tipo in dict(self.jsonData["Docenti"]["preferenze"][docente]["favorevole"]).items():
                        tipoTemporale, day, fasciaOraria, penalita = self.loadPreferenza(tipo, v)
                        
                        # create the RuleDocente
                        rule:RulePreferenza = RulePreferenza(docId, penalita)
                        if not rule.make_ruleGenerica(RulePreferenza.Preferenza.Favorevole, tipoTemporale, fasciaOraria, day):
                            continue
                        self.addRule(rule, TipoRule.PreferenzaLezione)
                
                elif str(tipoPref) == "contrario":
                    for v, tipo in dict(self.jsonData["Docenti"]["preferenze"][docente]["contrario"]).items():
                        tipoTemporale, day, fasciaOraria, penalita = self.loadPreferenza(tipo, v)
                        
                        # create the RuleDocente
                        rule:RulePreferenza = RulePreferenza(docId, penalita)
                        if not rule.make_ruleGenerica(RulePreferenza.Preferenza.Contrario, tipoTemporale, fasciaOraria, day):
                            continue
                        self.addRule(rule, TipoRule.PreferenzaLezione)    
                        
        # 2 - RuleDist
        # distanza minima     
        for doc,distMin in dict(self.jsonData["Docenti"]["distanzaMinima"]).items():
            if doc not in self.AUX.map_strDocenti_to_idDocenti.keys():
                self.log.warning_log("RuleDocenteHandler.prepareRules(): Docente sconosciuto")
                continue
            docId = self.AUX.map_strDocenti_to_idDocenti[doc]
            rule = RuleDist(docId, getLV_PenaltiesFromString(str(distMin).split('_')[1]), int(str(distMin).split('_')[0]), 
                            RuleDist.DistLogic.DistMinima)
            self.addRule(rule, TipoRule.DistInsDiffrerenti) 
        # distanza massima        
        for doc,distMax in dict(self.jsonData["Docenti"]["distanzaMassima"]).items():
            if doc not in self.AUX.map_strDocenti_to_idDocenti.keys():
                self.log.warning_log("RuleDocenteHandler.prepareRules(): Docente sconosciuto")
                continue
            docId = self.AUX.map_strDocenti_to_idDocenti[doc]
            rule = RuleDist(docId, getLV_PenaltiesFromString(str(distMax).split('_')[1]), int(str(distMax).split('_')[0]), 
                            RuleDist.DistLogic.DistMassima)
            self.addRule(rule, TipoRule.DistInsDiffrerenti)  
            
        # 3 - LezioneInsDifferentiSameDay
        for doc,penalita in dict(self.jsonData["Docenti"]["insegnamentiDiversiSameDay"]).items():
            if doc not in self.AUX.map_strDocenti_to_idDocenti.keys():                
                self.log.warning_log("RuleDocenteHandler.prepareRules(): Docente sconosciuto")
                continue
            docId = self.AUX.map_strDocenti_to_idDocenti[doc]
            rule = RuleLezSameDay(docId, getLV_PenaltiesFromString(penalita))
            self.addRule(rule, TipoRule.LezioneInsDifferentiSameDay)
        
        # 4 - RulePreferenza - Anno precedente  
        for doc,strData in dict(self.jsonData["Docenti"]["preferenzaAnnoPrecedente"]).items():
            if doc not in self.AUX.map_strDocenti_to_idDocenti.keys():                
                self.log.warning_log("RuleDocenteHandler.prepareRules(): Docente sconosciuto")
                continue
            # format strData: pianoAllocazione_levelPenalties
            docId:int = self.AUX.map_strDocenti_to_idDocenti[doc]
            namePianoAllocazione:str = str(strData).split('_')[0]
            pianoAlloc = self.dbAPI.get_pianoAllocazioneDocente(namePianoAllocazione, doc)
            penalita:LV_Penalties = getLV_PenaltiesFromString(str(strData).split('_')[1])

            find = lambda fun, pianoAlloc: next((slot for slot in pianoAlloc if fun(slot)), None)
            
            # per ogni Slot tenuto da quel Docente se non è definito nell'xml come uno SlotScelto
            for slotId in self.AUX.list_slotInDocente[docId]:
                if self.AUX.pianoAllocazione[slotId].tipoSlot != TipoSlot.SlotGenerico:
                    self.log.info_log("RuleDocenteHandler.prepareRules(): lo Slot " + self.AUX.pianoAllocazione[slotId].slotId + " non è di tipo generico nella configurazione corrente")
                    continue
             
                # ho a che fare con uno SlotGenerico e lo modello in base ai dati del db
                slotDb = find(lambda slot: slot[1] == self.AUX.pianoAllocazione[slotId].slotId, pianoAlloc)
                if slotDb is None:
                    continue
                day, fasciaOraria = getDayFasciaOrariaFromString(slotDb[6], slotDb[7])
                rule = RulePreferenza(docId, penalita)
                if not rule.make_ruleSpecifica(RulePreferenza.Preferenza.Favorevole, RulePreferenza.TimeLogic.UgualeA, slotId, fasciaOraria, day):
                    self.log.error_log("RuleDocenteHandler.prepareRules(): impossibile caricare RulePreferenza da anno precedente")
                    continue
                self.addRule(rule, TipoRule.PreferenzaLezione)
                    
                            
    def loadPreferenza(self, tipo:str, value:str) -> Tuple[RulePreferenza.TimeLogic, Day, FasciaOraria, LV_Penalties]:                   
        '''Legge una entry in json per una RulePreferenza.\n
        Formato: "tipoTemporale": "Day_FasciaOraria_LVPenalità"'''
        tipoTemporale:RulePreferenza.TimeLogic = RulePreferenza.TimeLogic.Nessuno
        if tipo == "primaDi":
            tipoTemporale = RulePreferenza.TimeLogic.PrimaDi
        elif tipo == "ugualeA":
            tipoTemporale = RulePreferenza.TimeLogic.UgualeA
        elif tipo == "dopoDi":
            tipoTemporale = RulePreferenza.TimeLogic.DopoDi
            
        day, fasciaOraria = getDayFasciaOrariaFromString(value.split("_")[0], value.split("_")[1])
        penalita = getLV_PenaltiesFromString(value.split("_")[2])
        return tipoTemporale, day, fasciaOraria, penalita
            
            
    def addRule(self, rule:RuleDocente, type:TipoRule):
        '''Una RuleDocente viene aggiunta alla lista TipoRule corrispondente.\n
        TipoRule.PreferenzaLezione:\n
            - se non esiste un'altra RulePreferenza di nè di Target.Specifica, nè di Target.Generica
            - se esiste un'altra RulePreferenza di Target.Specifica e lo è pure questa (ma si riferiscono a due slot diversi)
            - se esiste un'altra RulePreferenza di Target.Generica e questa è di TipoRule.Specifica (o viceversa) la RulePreferenza
            di Target.Generica viene mantenuta (è più importante la preferenza di un Docente a mantenere l'Orario dello scorso anno
            rispetto a considerare sue preferenze generiche su quando fare lezione -> in fase di espansione non ci sarà l'espansione della 
            Rule Generica sugli Slot già coperti da Rule Specifiche).\n
        TipoRule.RuleDist:
            - aggiunta semplice alla lista di RuleDocente di quel tipo.\n
        TipoRule.LezioneInsDifferentiSameDay:
            - aggiunta semplice alla lista di RuleDocente di quel tipo.\n'''
        if self.load_lists:
            self.log.error_log("RuleDocenteHandler.addRule(): listRules già caricata")
            return

        if type == TipoRule.PreferenzaLezione:
            if not rule:
                self.log.error_log("RuleDocenteHandler.addRule(): RuleDocente non valida: " + str(rule))
                return
            if rule in self.listPreferenze:
                self.log.warning_log("RuleDocenteHandler.addRule(): RuleDocente già presente: " + str(rule))
                return  
            self.listPreferenze.append(rule)
        elif type == TipoRule.DistInsDiffrerenti:
            # fino alla finalizzazione la Rule è invalida
            # if not rule:
            #     self.log.error_log("RuleDocenteHandler.addRule(): RuleDistMinima non valida")
            #     return
            if rule in self.listDistanze:
                self.log.warning_log("RuleDocenteHandler.addRule(): RuleDist già presente: " + str(rule))
                return
            self.listDistanze.append(rule)
        elif type == TipoRule.LezioneInsDifferentiSameDay: 
            # fino alla finalizzazione la Rule è invalida
            if rule in self.listLezSameDay:
                self.log.warning_log("RuleDocenteHandler.addRule(): RuleLezSameDay già presente: " + str(rule))
                return
            self.listLezSameDay.append(rule)
            
                
    def finalizeRules(self) -> int:
        '''Finalizzazione delle RuleDocente:
            - RulePreferenza con Target == Generica vanno espanse in modo da rifersi ad un unico Slot
            -  RuleDist possono essere soft constraint (=> mi serve una entry nella lista di penalità del modello) o hard constraint
            - RuleLezSameDay possono essere soft constraint o hard constraint.\n
        Return:
            Il numero di RuleDocente definitivo modellate come soft constraint (corrisponde alla dimensione della lista di penalità da 
            modellare con CPLEX)\n'''
        if self.load_lists:
            self.log.error_log("RuleDocenteHandler.finalizeRules(): listResRules già creata")
            return 0
        self.load_lists = True
        
        # Limit Slot lezione per Day
        if not self.check_numSlotConsecutivi_maggioreMaxSlot_perDay():
            self.log.error_log("RuleDocenteHandler.finalizeRules(): ho degli Slot con un numero maggiore di ore del massimo consentito per il docente")
            return -1
        if not self.check_slotPerDocente():
            self.log.error_log("RuleDocenteHandler.finalizeRules(): troppi slot di lezione allocati per il Docente")
            # a causa dei casi eccezionali di Docenti che fanno lezione in periodi differenti del semestre
            # return -1 
        
        # if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Docenti == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Docenti == 0:
        #     if self.PARAM.debuggingPenalty == False:
        #         return 0
        
        
        # RulePreferenza
        # 1 - Caricamento tutte le Rules Specifiche (sono quelle che si riferiscono ad un altro piano di allocazione)
        for rule in self.listPreferenze:
            if rule.target == RulePreferenza.Target.Specifica:
                if rule:
                    self.listResPreferenze.append(rule)
                    if rule.penalita != LV_Penalties.LV_H:
                        self.nRulePreferenza += 1
        # 2 - Espansione delle Rules generiche a tutti gli slot del Docente + inserimento in listResRules solo se non già presenti come
        # Rule specifica
        for rule in self.listPreferenze:
            if rule.target == RulePreferenza.Target.Generica:
                if rule:
                    for slotId in self.AUX.list_slotInDocente[rule.docId]:
                        # se la modalità di exec prevede un Piano Allocazione come base => se i vincoli non riguardano gli Insegnamenti 
                        # modificabili li devo ignorare
                        if self.PARAM.usePianoAllocazioneAsBase:
                            if int(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC) not in self.PARAM.listID_INCToModify:
                                self.log.info_log("RuleDocenteHadler.finalizeRules(): lo Slot " + self.AUX.pianoAllocazione[slotId].slotId + " non appartiene ad un Insegnamento da modificare (ID_INC: " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + ")")
                                continue                         
                        
                        ruleTmp = RulePreferenza(rule.docId, rule.penalita)
                        ruleTmp.make_ruleSpecifica(rule.tipoPreferenza, rule.tipoTemporale, slotId, rule.fasciaOraria, rule.day)
                        if not ruleTmp:
                            self.log.warning_log("RuleDocenteHandler.finalizeRules(): la RuleDocente non può essere espansa per tale Slot: " + str(ruleTmp))
                        elif ruleTmp not in self.listResPreferenze:
                            # se lo slot a cui mi riferisco non è di tipo SlotGenerico => prevale il vincolo nel file xml => scarto la RulePreferenza
                            if self.AUX.pianoAllocazione[slotId].tipoSlot != TipoSlot.SlotGenerico:
                                self.log.warning_log("RuleDocenteHandler.finalizeRules(): la RuleDocente non si può applicare ad uno Slot di tipo non Generico: " + str(self.AUX.pianoAllocazione[slotId]))
                            else:
                                self.listResPreferenze.append(ruleTmp)
                                if ruleTmp.penalita != LV_Penalties.LV_H:
                                    self.nRulePreferenza += 1
        
        # RuleDist
        # devo contare quanti soft constraint ho (e quindi quante penalità dovrò modellizzare)
        # 1 - Espansione to singoli Slot
        for rule in self.listDistanze:
            for slotId_i in self.AUX.list_slotInDocente[rule.docId]:
                for slotId_j in self.AUX.list_slotInDocente[rule.docId]:
                    
                    if self.PARAM.usePianoAllocazioneAsBase:
                        # se entrambi gli Slot appartengono ad Insegnamenti da NON riallocare => il vincolo è inutile
                        if int(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId_i].idInsegnamento].ID_INC) not in self.PARAM.listID_INCToModify and int(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId_j].idInsegnamento].ID_INC) not in self.PARAM.listID_INCToModify:
                            self.log.info_log("RuleDocente.finalizeRules(): skip Slot appartenenti a Insegnamenti: " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId_i].idInsegnamento].ID_INC + " e " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId_j].idInsegnamento].ID_INC)
                            continue                     
                    
                    if slotId_i > slotId_j and self.AUX.pianoAllocazione[slotId_i].idInsegnamento != self.AUX.pianoAllocazione[slotId_j].idInsegnamento:
                        rTmp = RuleDist(rule.docId, rule.penalita, rule.dist, rule.distLogic)
                        if not rTmp.make_ruleDist(slotId_i, slotId_j):
                            self.log.warning_log("RuleDocenteHandler.finalizeRules(): la RuleDist è invalida: " + str(rTmp))
                        if rTmp not in self.listResDistanze and rTmp:
                            self.listResDistanze.append(rTmp)
                            if rTmp.penalita != LV_Penalties.LV_H:
                                self.nRuleDistanza += 1 # solo i soft constraint vanno conteggiati per la lista di penalità
                                
        # RuleLezSameDay
        # 1 - Espansione to singoli Slot
        for rule in self.listLezSameDay:
            for slotId_i in self.AUX.list_slotInDocente[rule.docId]:
                for slotId_j in self.AUX.list_slotInDocente[rule.docId]:
                    
                    if self.PARAM.usePianoAllocazioneAsBase:
                        # se entrambi gli Slot appartengono ad Insegnamenti da NON riallocare => il vincolo è inutile
                        if int(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId_i].idInsegnamento].ID_INC) not in self.PARAM.listID_INCToModify and int(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId_j].idInsegnamento].ID_INC) not in self.PARAM.listID_INCToModify:
                            self.log.info_log("RuleDocente.finalizeRules(): skip Slot appartenenti a Insegnamenti: " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId_i].idInsegnamento].ID_INC + " e " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId_j].idInsegnamento].ID_INC)
                            continue 
                    
                    if slotId_i > slotId_j and self.AUX.pianoAllocazione[slotId_i].idInsegnamento != self.AUX.pianoAllocazione[slotId_j].idInsegnamento:
                        rTmp = RuleLezSameDay(rule.docId, rule.penalita)
                        if not rTmp.make_ruleLezSameDay(slotId_i, slotId_j):
                            self.log.warning_log("RuleDocenteHandler.finalizeRules(): la RuleLezSameDay è invalida: " + str(rTmp))
                        if rTmp not in self.listResLezSameDay and rTmp:
                            self.listResLezSameDay.append(rTmp)
                            if rTmp.penalita != LV_Penalties.LV_H:
                                self.nRuleLezSameDay += 1 # solo per i soft constraint
                                
        # Rule xml
        for vincolo in self.AUX.list_vincoliInsegnamenti:
            if vincolo and vincolo.penalita != LV_Penalties.LV_H:
                self.nRuleXml += 1
            else:
                # self.log.warning_log("RuleDocenteHandler.finalizeRules(): Vincolo invalido: " + str(vincolo))
                pass
        for operatore in self.AUX.list_operatoriInsegnamenti:
            if operatore and operatore.isRoot() and operatore.penalita not in [LV_Penalties.LV_0, LV_Penalties.LV_H]:
                self.nRuleXml += 1
            else:
                log.warn("RuleDocenteHandler.finalizeRules(): Operatore invalido: " + str(operatore))
    
        return self.nRulePreferenza + self.nRuleDistanza + self.nRuleLezSameDay + self.nRuleXml


    def check_numSlotConsecutivi_maggioreMaxSlot_perDay(self) -> bool:
        '''Controlla che il limite giornaliero di ore di un docente non sia inferiore al numero di SlotConsecutivi di un qualche Slot'''
        err = False
        for docId in self.AUX.map_strDocenti_to_idDocenti.values():
            for slotId in self.AUX.map_strSlotId_to_idSlot.values():
                if docId in self.AUX.list_docentiInSlot[slotId]:
                    # se ha un limite personalizzato
                    if docId in self.map_limitHourDocentePerDay.keys():
                        if self.AUX.pianoAllocazione[slotId].numSlot > self.map_limitHourDocentePerDay[docId]:
                            self.log.error_log("RuleDocenteHandler.check_numSlotConsecutivi_maggioreMaxSlot_perDay(): il docente " + 
                                    self.AUX.map_idDocenti_to_strDocenti[docId] + " ha uno slot da " + str(self.AUX.pianoAllocazione[slotId].numSlot) +
                                    " slotConsecutivi e limite giornaliero di slot di " + str(self.map_limitHourDocentePerDay[docId]))
                            err = True
                    else: # non c'è un limite personalizzato
                        if self.AUX.pianoAllocazione[slotId].numSlot > self.default_limitHourDocentePerDay:
                            self.log.error_log("RuleDocenteHandler.check_numSlotConsecutivi_maggioreMaxSlot_perDay(): il docente " + 
                                    self.AUX.map_idDocenti_to_strDocenti[docId] + " ha uno slot da " + str(self.AUX.pianoAllocazione[slotId].numSlot) +
                                    " slotConsecutivi e limite giornaliero di slot di " + str(self.default_limitHourDocentePerDay))
                            err = True
        return not err                    
    
    def check_slotPerDocente(self) -> bool:
        '''Controlla che possa fisicamente allocare nella settimana tutti gli slot di lezione al Docente'''
        err = False
        # i Docenti che devono essere allocati in tanti slot rispetto al loro massimo
        listCriticita:List[Tuple[int,Tuple[int,float]]] = list()
        sogliaCriticita:float = 0.4
        
        for docId in self.AUX.map_strDocenti_to_idDocenti.values():
            numSlot = 0
            for slotId in self.AUX.map_strSlotId_to_idSlot.values():
                if docId in self.AUX.list_docentiInSlot[slotId]:
                    numSlot += self.AUX.pianoAllocazione[slotId].numSlot
            
            # criticità
            if docId in self.map_limitHourDocentePerDay.keys():
                maxNumSlot = self.map_limitHourDocentePerDay[docId]*self.AUX.get_NUM_DAY()
            else:
                maxNumSlot = self.default_limitHourDocentePerDay*self.AUX.get_NUM_DAY()
            if numSlot/maxNumSlot > sogliaCriticita:
                listCriticita.append((docId, (numSlot, numSlot/maxNumSlot)))
            
            if docId in self.map_limitHourDocentePerDay.keys():
                if numSlot >= self.map_limitHourDocentePerDay[docId]*self.AUX.get_NUM_DAY():
                    self.log.error_log("RuleDocenteHandler.sum_slotPerDocente(): ERROR")
                    self.log.info_log(self.AUX.map_idDocenti_to_strDocenti[docId] + " da allocare in " + str(numSlot) + " slot")
                    err = True
            else:
                if numSlot >= self.default_limitHourDocentePerDay*self.AUX.get_NUM_DAY():
                    self.log.error_log("RuleDocenteHandler.sum_slotPerDocente(): ERROR")
                    self.log.info_log(self.AUX.map_idDocenti_to_strDocenti[docId] + " da allocare in " + str(numSlot) + " slot")
                    err = True                      
                        
        # output criticità
        if len(listCriticita) > 0:
            self.log.warning_log("RuleDocenteHandler: criticita' da numero allocazioni per Docente:")
            for crit in listCriticita:
                self.log.info_log(self.AUX.map_idDocenti_to_strDocenti[crit[0]] + " con criticita': " + str(crit[1][1]) + " a da allocare in: " + str(crit[1][0]) + " slot")
            
        return not err
    

    def model_AddRules(self, index:int, X_penaltiesDoc, X_d, X_h, model:CpoModel):
        '''Aggiunge al modello di CPLEX i vincoli definiti in precedenza.\n
        Args:
            index (int) è l'indice da cui partire ad inserire le penalita nella lista di CPLEX che verrà poi minimizzata
            X_penaltiesDoc (list) è la lista del modello di CPLEX dove inserire le penalità
            X_d, X_h variabili del modello di CPLEX'''
        if not self.load_lists:
            self.log.error_log("ruleDocente.model_AddRule(): listsRes non ancora create")
            return
        eqs = list()
        
        # limite slot lezione per Day
        for day in range(self.AUX.get_NUM_DAY()):
            # per ogni giorno
            for docId in self.AUX.map_strDocenti_to_idDocenti.values():
                # per ogni docente
                if docId in self.map_limitHourDocentePerDay.keys(): 
                    # self.log.info_log(model.sum([(X_d[slotId] == day) * self.AUX.pianoAllocazione[slotId].numSlot 
                                    # for slotId in self.AUX.list_slotInDocente[docId]]) <= self.map_limitHourDocentePerDay[docId])
                    eqs.append(model.sum([(X_d[slotId] == day) * self.AUX.pianoAllocazione[slotId].numSlot 
                                    for slotId in self.AUX.list_slotInDocente[docId]]) <= self.map_limitHourDocentePerDay[docId])
                else:
                    # self.log.info_log(model.sum([(X_d[slotId] == day) * self.AUX.pianoAllocazione[slotId].numSlot
                                    # for slotId in self.AUX.list_slotInDocente[docId]]) <= self.default_limitHourDocentePerDay)
                    eqs.append(model.sum([(X_d[slotId] == day) * self.AUX.pianoAllocazione[slotId].numSlot
                                    for slotId in self.AUX.list_slotInDocente[docId]]) <= self.default_limitHourDocentePerDay)   
                    
        # Docenti non allocati contemporaneamente in più Slot
        for docId in self.AUX.map_strDocenti_to_idDocenti.values():
            # per ogni docente
            for slotId_i in self.AUX.list_slotInDocente[docId]:
                for slotId_j in self.AUX.list_slotInDocente[docId]:
                    # se i due Slot appartengono allo stesso docente -> non possono essere allocati contemporaneamente
                    
                    # se i due Slot si riferiscono entrambi ad Insegnamenti sovrapponibili E APPARTENGONO ad INSEGNAMENTI DIFFERENTI
                    if docId in self.map_docentiConInsegnamentiSovrapponibili.keys():
                        if self.AUX.pianoAllocazione[slotId_i].idInsegnamento != self.AUX.pianoAllocazione[slotId_j].idInsegnamento:
                            
                            ID_INC_i:int = int(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId_i].idInsegnamento].ID_INC)
                            ID_INC_j:int = int(self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId_j].idInsegnamento].ID_INC)
                            
                            if ID_INC_i in self.map_docentiConInsegnamentiSovrapponibili[docId] and ID_INC_j in self.map_docentiConInsegnamentiSovrapponibili[docId]:
                                continue
                    
                    if slotId_j > slotId_i: # as matrice triangolare (gli slotId sono ordinati)
                        eq = if_then(X_d[slotId_i] == X_d[slotId_j], 
                                logical_or(X_h[slotId_i] >= X_h[slotId_j] + self.AUX.pianoAllocazione[slotId_j].numSlot,
                                        X_h[slotId_j] >= X_h[slotId_i] + self.AUX.pianoAllocazione[slotId_i].numSlot))
                        eqs.append(eq)                    
        
        if self.debugging:
            self.log.info_log("\nRuleDocente: RulePreferenza")
        # RulePreferenza
        for index_rule in range(len(self.listResPreferenze)):
            rule = self.listResPreferenze[index_rule]
            if self.debugging:
                self.log.info_log(rule)
            
            # hard constraint
            if rule.tipoPreferenza == RulePreferenza.Preferenza.Contrario:
                if rule.penalita == LV_Penalties.LV_H:
                    if rule.tipoTemporale == RulePreferenza.TimeLogic.PrimaDi:
                        
                        if rule.day == Day.Indifferente:
                            eq = X_h[rule.slotId] >= getIntFromFasciaOraria(rule.fasciaOraria)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:  
                            eq = X_d[rule.slotId] >= getIntFromDay(rule.day)                      
                        else: # eg: non voglio fare lezione il Gio prima delle ore ...
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                         X_h[rule.slotId] >= getIntFromFasciaOraria(rule.fasciaOraria))
                        
                        if self.debugging:
                            self.log.info_log(eq)
                        eqs.append(eq)    
                    
                    # mi basta che lo Slot finisca dopo lo Slot limite (lo Slot limite mi va ancora bene)
                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.DopoDi:
                        
                        if rule.day == Day.Indifferente:
                            eq = X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 <= getIntFromFasciaOraria(rule.fasciaOraria)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:       
                            eq = X_d[rule.slotId] <= getIntFromDay(rule.day)
                        else: # eg: il Gio non voglio fare lezione dopo le ore ...
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                         X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 <= getIntFromFasciaOraria(rule.fasciaOraria))

                        if self.debugging:
                            self.log.info_log(eq)
                        eqs.append(eq)    

                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.UgualeA:
                        
                        if rule.day == Day.Indifferente:
                            eq = X_h[rule.slotId] != getIntFromFasciaOraria(rule.fasciaOraria)
                            self.log.warning_log("RuleDocenteHandler.model_AddRules(): strange RulePreferenza added")
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:   
                            # eg: un Docente non può far lezione il giovedì
                            eq = X_d[rule.slotId] != getIntFromDay(rule.day)
                        else: # per indicare indisponibilità docente
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                         logical_or( # inizia dopo o finisce prima
                                             X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria),
                                             X_h[rule.slotId] + self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                                         ))
                        if self.debugging:
                            self.log.info_log(eq)
                        eqs.append(eq)                    
                    
                # soft constraint
                else:
                    # se non mi interessano le penalità dei soft constraint (e quindi non ho imposto nessun vincolo su di essi)
                    # non aggiungo questa equazioni al modello
                    if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Docenti == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Docenti == 0:
                        if self.PARAM.debuggingPenalty == False:
                            # self.log.info_log("skip penalty Docente (PreferenzaContraria)")
                            continue
                    
                    
                    if rule.docId not in self.map_penaltiesInDocente.keys():
                        self.map_penaltiesInDocente[rule.docId] = list()
                    self.map_penaltiesInDocente[rule.docId].append(index)
                    rule.index_X_penaltiesDoc = index
                    
                    if rule.tipoTemporale == RulePreferenza.TimeLogic.PrimaDi:
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId] < getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(X_h[rule.slotId] >= getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = if_then(X_d[rule.slotId] < getIntFromDay(rule.day),
                                        X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(X_d[rule.slotId] >= getIntFromDay(rule.day),
                                        X_penaltiesDoc[index] == 0)
                        else: # eg: non voglio fare lezione il Gio prima delle ore ...
                            eq = if_then(logical_and(
                                    X_h[rule.slotId] < getIntFromFasciaOraria(rule.fasciaOraria), X_d[rule.slotId] == getIntFromDay(rule.day)),
                                    X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(logical_not(
                                    logical_and(
                                        X_h[rule.slotId] < getIntFromFasciaOraria(rule.fasciaOraria), X_d[rule.slotId] == getIntFromDay(rule.day))
                                    ), X_penaltiesDoc[index] == 0)
                        index += 1
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq, eq1])  
                        
                    # mi basta che lo non Slot finisca dopo lo Slot limite (lo Slot limite mi va ancora bene)
                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.DopoDi:
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 > getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 <= getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = if_then(X_d[rule.slotId] > getIntFromDay(rule.day),
                                        X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(X_d[rule.slotId] <= getIntFromDay(rule.day),
                                        X_penaltiesDoc[index] == 0)
                        else:
                            eq = if_then(logical_and(
                                        X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 > getIntFromFasciaOraria(rule.fasciaOraria), 
                                        X_d[rule.slotId] == getIntFromDay(rule.day)),
                                    X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(logical_not(
                                    logical_and(
                                        X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot-1 > getIntFromFasciaOraria(rule.fasciaOraria), 
                                        X_d[rule.slotId] == getIntFromDay(rule.day))
                                    ), X_penaltiesDoc[index] == 0)                            
                        index += 1
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq, eq1])    
                        
                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.UgualeA:
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId] == getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(X_h[rule.slotId] != getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                        X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(X_d[rule.slotId] != getIntFromDay(rule.day),
                                        X_penaltiesDoc[rule.slotId] == 0)
                        else: # se quello Slot è allocato (anche se iniziasse prima)
                            # se lo Slot è allocato quel giorno e NON inizia dopo O NON finisce prima
                            eq = if_then(logical_and( 
                                        X_d[rule.slotId] == getIntFromDay(rule.day),
                                    logical_not(logical_or(                                            
                                        X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_h[rule.slotId] + self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                                    ))),
                                    X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(logical_not(logical_and( 
                                        X_d[rule.slotId] == getIntFromDay(rule.day),
                                    logical_not(logical_or(                                            
                                        X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_h[rule.slotId] + self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                                    )))),
                                    X_penaltiesDoc[index] == 0)                        
                        index += 1
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq, eq1])  
                    
            elif rule.tipoPreferenza == RulePreferenza.Preferenza.Favorevole:
                # lo Slot limite non mi va bene
                
                # hard constraint
                if rule.penalita == LV_Penalties.LV_H:

                    if rule.tipoTemporale == RulePreferenza.TimeLogic.PrimaDi:
                        if rule.day == Day.Indifferente:
                            eq = X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:
                            eq = X_d[rule.slotId] < getIntFromDay(rule.day)
                        else: # eg: il gio prima delle ore ...
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                    X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria))
                        
                        if self.debugging:
                            self.log.info_log(eq)
                        eqs.append(eq)      

                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.DopoDi:
                        if rule.day == Day.Indifferente:
                            eq = X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = X_d[rule.slotId] > getIntFromDay(rule.day)
                        else: # eg voglio fare lezione il Gio dopo le ore ...
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                        X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria))
                        if self.debugging:
                            self.log.info_log(eq)
                        eqs.append(eq) 
                        
                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.UgualeA:
                        if rule.day == Day.Indifferente or rule.fasciaOraria == FasciaOraria._Indifferente:
                            self.log.warning_log("RuleDocenteHandler.model_AddRules(): questa RulePreferenza non ha senso: " + str(rule))
                            continue
                        
                        eq1 = X_d[rule.slotId] == getIntFromDay(rule.day)
                        eq2 = X_h[rule.slotId] == getIntFromFasciaOraria(rule.fasciaOraria)
                        if self.debugging:
                            self.log.info_log(eq1)
                            self.log.info_log(eq2)
                        eqs.extend([eq1, eq2])      
                    
                # soft constraint                
                else:
                    # se non mi interessano le penalità dei soft constraint (e quindi non ho imposto nessun vincolo su di essi)
                    # non aggiungo questa equazioni al modello
                    if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Docenti == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Docenti == 0:
                        if self.PARAM.debuggingPenalty == False:
                            # self.log.info_log("skip penalty Docente (PreferenzaFavorevole)")
                            continue
                                            
                    if rule.docId not in self.map_penaltiesInDocente.keys():
                        self.map_penaltiesInDocente[rule.docId] = list()
                    self.map_penaltiesInDocente[rule.docId].append(index)
                    rule.index_X_penaltiesDoc = index                    
                    
                    if rule.tipoTemporale == RulePreferenza.TimeLogic.PrimaDi:
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot > getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:
                            eq = if_then(X_d[rule.slotId] < getIntFromDay(rule.day),
                                        X_penaltiesDoc[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))                        
                            eq1 = if_then(X_d[rule.slotId] > getIntFromDay(rule.day),
                                        X_penaltiesDoc[index] == 0)
                        else: # eg: il gio prima delle ore ...
                            eq = if_then(logical_and(
                                X_d[rule.slotId] == getIntFromDay(rule.day),
                                X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                            ), X_penaltiesDoc[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(logical_not(
                                logical_and(
                                    X_d[rule.slotId] == getIntFromDay(rule.day),
                                    X_h[rule.slotId]+self.AUX.pianoAllocazione[rule.slotId].numSlot <= getIntFromFasciaOraria(rule.fasciaOraria)
                                )
                            ), X_penaltiesDoc[index] == 0)
                        
                        index += 1
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq,eq1])                     

                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.DopoDi:
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(X_h[rule.slotId] <= getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = if_then(X_d[rule.slotId] > getIntFromDay(rule.day),
                                        X_penaltiesDoc[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(X_d[rule.slotId] <= getIntFromDay(rule.day),
                                        X_penaltiesDoc[index] == 0)
                        else: # eg voglio fare lezione il Gio dopo le ore ...
                            eq = if_then(logical_and(
                                    X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria), X_d[rule.slotId] == getIntFromDay(rule.day)),
                                    X_penaltiesDoc[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(logical_not(
                                    logical_and(
                                        X_h[rule.slotId] > getIntFromFasciaOraria(rule.fasciaOraria), 
                                        X_d[rule.slotId] == getIntFromDay(rule.day))
                                    ), X_penaltiesDoc[index] == 0)                        
                        index += 1
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq, eq1]) 
                        
                    elif rule.tipoTemporale == RulePreferenza.TimeLogic.UgualeA:
                            
                        if rule.day == Day.Indifferente:
                            eq = if_then(X_h[rule.slotId] == getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(X_h[rule.slotId] != getIntFromFasciaOraria(rule.fasciaOraria),
                                        X_penaltiesDoc[index] == 0)
                        elif rule.fasciaOraria == FasciaOraria._Indifferente:                        
                            eq = if_then(X_d[rule.slotId] == getIntFromDay(rule.day),
                                        X_penaltiesDoc[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(X_d[rule.slotId] != getIntFromDay(rule.day),
                                        X_penaltiesDoc[index] == 0)
                        else:
                            # eg voglio fare lezione il tale Day ed in tale FasciaOraria
                            eq = if_then(logical_and(
                                    X_h[rule.slotId] == getIntFromFasciaOraria(rule.fasciaOraria), X_d[rule.slotId] == getIntFromDay(rule.day)),
                                    X_penaltiesDoc[index] == (-1)*getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                            eq1 = if_then(logical_not(logical_and(
                                    X_h[rule.slotId] == getIntFromFasciaOraria(rule.fasciaOraria), X_d[rule.slotId] == getIntFromDay(rule.day))),
                                    X_penaltiesDoc[index] == 0)                            
                        index += 1                    
                        if self.debugging:
                            self.log.info_log(eq)
                            self.log.info_log(eq1)
                        eqs.extend([eq, eq1])       
        
        if self.debugging:
            self.log.info_log("\nRuleDocente: RuleDist")
        # RuleDist
        for rule in self.listResDistanze:
            if self.debugging:
                self.log.info_log(rule)
            
            if rule.distLogic == RuleDist.DistLogic.DistMinima:
                # hard const
                if rule.penalita == LV_Penalties.LV_H:
                    eq = if_then(X_d[rule.slotId_i] == X_d[rule.slotId_j],
                                logical_or(
                                    X_h[rule.slotId_i] >= X_h[rule.slotId_j] + self.AUX.pianoAllocazione[rule.slotId_j].numSlot + rule.dist,
                                    X_h[rule.slotId_j] >= X_h[rule.slotId_i] + self.AUX.pianoAllocazione[rule.slotId_i].numSlot + rule.dist
                                ))
                    if self.debugging:
                        self.log.info_log(eq)
                    eqs.append(eq) 
                # soft constraint
                else:
                    if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Docenti == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Docenti == 0:
                        if self.PARAM.debuggingPenalty == False:
                            # self.log.info_log("skip penalty Docente (DistMinima)")
                            continue                    
                    
                    if rule.docId not in self.map_penaltiesInDocente.keys():
                        self.map_penaltiesInDocente[rule.docId] = list()
                    self.map_penaltiesInDocente[rule.docId].append(index)
                    rule.index_X_penaltiesDoc = index                                        
                    
                    # soft const -> se stesso giorno e non sono a distanza maggiore di dist_minima
                    eq = if_then(logical_and(
                        X_d[rule.slotId_i] == X_d[rule.slotId_j],
                        logical_not(logical_or(
                            X_h[rule.slotId_i] >= X_h[rule.slotId_j] + self.AUX.pianoAllocazione[rule.slotId_j].numSlot + rule.dist,
                            X_h[rule.slotId_j] >= X_h[rule.slotId_i] + self.AUX.pianoAllocazione[rule.slotId_i].numSlot + rule.dist
                        ))
                    ),
                        X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente)
                    )
                    eq1 = if_then(logical_not(logical_and(
                        X_d[rule.slotId_i] == X_d[rule.slotId_j],
                        logical_not(logical_or(
                            X_h[rule.slotId_i] >= X_h[rule.slotId_j] + self.AUX.pianoAllocazione[rule.slotId_j].numSlot + rule.dist,
                            X_h[rule.slotId_j] >= X_h[rule.slotId_i] + self.AUX.pianoAllocazione[rule.slotId_i].numSlot + rule.dist
                        ))
                    )),
                        X_penaltiesDoc[index] == 0
                    )
                    
                    index += 1
                    if self.debugging:
                        self.log.info_log(eq)
                        self.log.info_log(eq1)
                    eqs.extend([eq, eq1])
            
            elif rule.distLogic == RuleDist.DistLogic.DistMassima:
                # hard constraint
                if rule.penalita == LV_Penalties.LV_H:
                    eq = if_then(logical_and(
                                X_d[rule.slotId_i] == X_d[rule.slotId_j],
                                X_h[rule.slotId_i] > X_h[rule.slotId_j]
                            ),
                                X_h[rule.slotId_i] <= X_h[rule.slotId_j] + self.AUX.pianoAllocazione[rule.slotId_j].numSlot + rule.dist
                            )
                    eq1 = if_then(logical_and(
                                X_d[rule.slotId_i] == X_d[rule.slotId_j],
                                X_h[rule.slotId_j] > X_h[rule.slotId_i]
                            ),
                                X_h[rule.slotId_j] <= X_h[rule.slotId_i] + self.AUX.pianoAllocazione[rule.slotId_i].numSlot + rule.dist
                            )
                    if self.debugging:
                        self.log.info_log(eq)
                        self.log.info_log(eq1)
                    eqs.extend([eq, eq1]) 
                # soft constraint
                else:
                    if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Docenti == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Docenti == 0:
                        if self.PARAM.debuggingPenalty == False:
                            # self.log.info_log("skip penalty Docente (DistMassima)")
                            continue                    
                    
                    if rule.docId not in self.map_penaltiesInDocente.keys():
                        self.map_penaltiesInDocente[rule.docId] = list()
                    self.map_penaltiesInDocente[rule.docId].append(index)
                    rule.index_X_penaltiesDoc = index                    

                    # soft const
                    # se i due Slot sono allocati lo stesso giorno AND
                    # (slot_i viene dopo slot_j AND slot_i è a distanza maggiore di distMassima dalla fine di slot_j) OR
                    # (slot_j viene dopo slot_i AND slot_j è a distanza maggiore di distMassima dalla fine di slot_i), => penalità
                    eq = if_then(logical_and(
                        X_d[rule.slotId_i] == X_d[rule.slotId_j],
                        logical_or(
                            logical_and(
                               X_h[rule.slotId_i] > X_h[rule.slotId_j],
                               X_h[rule.slotId_i] > X_h[rule.slotId_j] + self.AUX.pianoAllocazione[rule.slotId_j].numSlot + rule.dist
                            ), logical_and(
                                X_h[rule.slotId_j] > X_h[rule.slotId_i],
                                X_h[rule.slotId_j] > X_h[rule.slotId_i] + self.AUX.pianoAllocazione[rule.slotId_i].numSlot + rule.dist
                            )
                        )),
                        X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente))
                    eq1 = if_then(logical_not(logical_and(
                        X_d[rule.slotId_i] == X_d[rule.slotId_j],
                        logical_or(
                            logical_and(
                               X_h[rule.slotId_i] > X_h[rule.slotId_j],
                               X_h[rule.slotId_i] > X_h[rule.slotId_j] + self.AUX.pianoAllocazione[rule.slotId_j].numSlot + rule.dist
                            ), logical_and(
                                X_h[rule.slotId_j] > X_h[rule.slotId_i],
                                X_h[rule.slotId_j] > X_h[rule.slotId_i] + self.AUX.pianoAllocazione[rule.slotId_i].numSlot + rule.dist
                            )
                        ))),
                        X_penaltiesDoc[index] == 0)                    
                    
                    if self.debugging:
                        self.log.info_log(eq)
                        self.log.info_log(eq1)  
                    index += 1      
                    eqs.extend([eq, eq1])  
        
        if self.debugging:
            self.log.info_log("\nRuleDocente: RuleLezSameDay")
        # RuleLezSameDay
        for rule in self.listResLezSameDay:
            if self.debugging:
                self.log.info_log(rule)
            # hard constraint
            if rule.penalita == LV_Penalties.LV_H:
                eq = X_d[rule.slotId_i] != X_d[rule.slotId_j]
                if self.debugging:
                    self.log.info_log(eq)
                eqs.append(eq)
            # soft constraint
            else:
                if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Docenti == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Docenti == 0:
                    if self.PARAM.debuggingPenalty == False:
                        # self.log.info_log("skip penalty Docente (LezioneSameDay)")
                        continue                
                
                if rule.docId not in self.map_penaltiesInDocente.keys():
                    self.map_penaltiesInDocente[rule.docId] = list()
                self.map_penaltiesInDocente[rule.docId].append(index)
                rule.index_X_penaltiesDoc = index                    
                
                eq = if_then(
                    X_d[rule.slotId_i] == X_d[rule.slotId_j],
                    X_penaltiesDoc[index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Docente)
                )
                eq1 = if_then(
                    X_d[rule.slotId_i] != X_d[rule.slotId_j],
                    X_penaltiesDoc[index] == 0
                )
                index += 1
                
                if self.debugging:
                    self.log.info_log(eq)
                    self.log.info_log(eq1)
                eqs.extend([eq, eq1])
        
        if self.debugging:
            self.log.info_log("\nRuleDocente: Vincoli xml")
        # Vincoli xml
        for vincoloId in range(len(self.AUX.list_vincoliInsegnamenti)):
            vincolo:Vincolo = self.AUX.list_vincoliInsegnamenti[vincoloId]

            if not vincolo:
                continue # non è una lista già filtrata da RuleDocenteHandler.finalizeRules()
            if self.debugging:
                self.log.info_log(vincolo)

            
            # hard constraint
            if vincolo.penalita == LV_Penalties.LV_H:
                eq = vincolo.getEquation(X_d, X_h)
                if self.debugging:
                    self.log.info_log(eq)
                eqs.append(eq)
            # soft constraint
            else: 
                if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Docenti == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Docenti == 0:
                    if self.PARAM.debuggingPenalty == False:
                        # self.log.info_log("skip penalty Docente (Vincolo)")
                        continue                
                
                for docId in self.AUX.get_listDocentiOfVincolo(vincoloId):
                    if docId not in self.map_penaltiesInDocente.keys():
                        self.map_penaltiesInDocente[docId] = list()
                    self.map_penaltiesInDocente[docId].append(index)
                vincolo.index_X_penaltiesDoc = index                    
                
                eq = if_then(vincolo.getEquation(X_d, X_h),
                             X_penaltiesDoc[index] == (-1)*getIntFromLV_Penalties(vincolo.penalita, TipoPenalty.Docente))
                eq1 = if_then(logical_not(vincolo.getEquation(X_d, X_h)),
                              X_penaltiesDoc[index] == 0)
                
                if self.debugging:
                    self.log.info_log(eq)
                    self.log.info_log(eq1)
                eqs.extend([eq, eq1])                

        # Operatori xml
        for operatoreId in range(len(self.AUX.list_operatoriInsegnamenti)):
            operatore:Operatore = self.AUX.list_operatoriInsegnamenti[operatoreId]
            
            if self.debugging:
                self.log.info_log(operatore)
            if not operatore or operatore.penalita == LV_Penalties.LV_0:
                continue # non è una lista già filtrata da RuleDocenteHandler.finalizeRules()
            
            # hard constraint
            if operatore.penalita == LV_Penalties.LV_H:
                eq = operatore.getEquation(X_d, X_h)
                if self.debugging:
                    self.log.info_log(eq)
                eqs.append(eq)
            # soft constraint
            else: 
                if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Docenti == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Docenti == 0:
                    if self.PARAM.debuggingPenalty == False:
                        # self.log.info_log("skip penalty Docente (Operatore)")
                        continue                
                
                for docId in self.AUX.get_listDocentiOfOperatore(operatoreId):
                    if docId not in self.map_penaltiesInDocente.keys():
                        self.map_penaltiesInDocente[docId] = list()
                    self.map_penaltiesInDocente[docId].append(index)
                operatore.index_X_penaltiesDoc = index                    
                
                eq = if_then(operatore.getEquation(X_d, X_h),
                             X_penaltiesDoc[index] == (-1)*getIntFromLV_Penalties(operatore.penalita, TipoPenalty.Docente))
                eq1 = if_then(logical_not(operatore.getEquation(X_d, X_h)),
                              X_penaltiesDoc[index] == 0)
                
                if self.debugging:
                    self.log.info_log(eq)
                    self.log.info_log(eq1)
                eqs.extend([eq, eq1])

        
        return eqs
        