#include <string.h>
#include <stdint.h>
#include <stdbool.h>

#include "app.h"

/* FreeRtos includes */
#include "FreeRTOS.h"
#include "task.h"

#include "debug.h"
#include "crtp.h"
#include "vl53l5cx_api.h"
#include "deck.h"
#include "param.h"



#define  ARM_CM_DEMCR      (*(uint32_t *)0xE000EDFC)
#define  ARM_CM_DWT_CTRL   (*(uint32_t *)0xE0001000)
#define  ARM_CM_DWT_CYCCNT (*(uint32_t *)0xE0001004)

#define DEBUG_MODULE "HELLOWORLD"

//-------------------Control MAcros-------------------//
#define SESNSOR_FORAWARD_ENABLE
//#define SESNSOR_BACKWARD_ENABLE
// #define SEND_DATA
#define ON_BOARD_PROCESS
#define START_FLIGHT

//-------------------Custom libraries-------------------//
#include "I2C_expander.h"
#include "ToF_process.h"
#include "Fly_control.h"

//-------------------TimerCounters-------------------//
// #define start_timer()    *((volatile uint32_t*)0xE0001000) = 0x40000001  // Enable CYCCNT register
// #define stop_timer()   *((volatile uint32_t*)0xE0001000) = 0x40000000  // Disable CYCCNT register
// #define get_timer()   *((volatile uint32_t*)0xE0001004)               // Get value from CYCCNT register
// uint32_t it1, it2, it3;      // start and stop flag 
// uint64_t delta_it,delta_it_2 =0;
// uint16_t it_numer = 0;   
// #define ARM_CM_DEMCR (*(uint32_t *)0xE000EDFC)
// #define ARM_CM_DWT_CTRL (*(uint32_t *)0xE0001000)
// #define ARM_CM_DWT_CYCCNT (*(uint32_t *)0xE0001004)



//-------------------Global Defines-------------------//
//Sensors Addresses
#define VL53L5CX_FORWARD_I2C_ADDRESS            ((uint16_t)(VL53L5CX_DEFAULT_I2C_ADDRESS*4))
#define VL53L5CX_BACKWARD_I2C_ADDRESS            ((uint16_t)(VL53L5CX_FORWARD_I2C_ADDRESS+2))
//Data Length
// #define ToF_DISTANCES_LEN             (128)
// #define ToF_TARGETS_DETECTED_LEN      (64)
// #define ToF_TARGETS_STATUS_LEN        (64)
#define FRONT_SENSOR_OFFSET        (0)
#define BACK_SENSOR_OFFSET        (ToF_DISTANCES_LEN+ToF_TARGETS_DETECTED_LEN+ToF_TARGETS_STATUS_LEN)
//GPIO Pins
#define EXI_PIN      DECK_GPIO_IO2
#define EXI_PortSource   EXTI_PortSourceGPIOB
#define EXI_PinSource    EXTI_PinSource5
#define EXI_LineN 		   EXTI_Line5
// IRQ
#define IRQ_READY     ((uint8_t)(1<<0))
#define IRQ_PIN_IS_LOW     ((uint8_t)(1<<1))
#define IRQ_FRONT     ((uint8_t)(1<<2))
#define IRQ_REAR     ((uint8_t)(1<<3))
#define IRQ_ERROR    ((uint8_t)(1<<4))


//-------------------Global Variables-------------------//
#ifdef SESNSOR_FORAWARD_ENABLE
static VL53L5CX_Configuration vl53l5dev_f;
static VL53L5CX_ResultsData vl53l5_res_f;
#endif
#ifdef SESNSOR_BACKWARD_ENABLE
static VL53L5CX_Configuration vl53l5dev_b;
static VL53L5CX_ResultsData vl53l5_res_b;
#endif

// IRQ
volatile uint8_t irq_status = 0;
//uint8_t vl53l5_buffer[VL53L5CX_MAX_RESULTS_SIZE];
//uint64_t t1, t2, delta;
//
uint32_t timestamp;

//-------------------Functions defins-------------------//
void send_command(uint8_t command, uint8_t arg);
void send_data_packet(uint8_t *data, uint16_t data_len);
void send_data_packet_28b(uint8_t *data, uint8_t size, uint8_t index);
//
bool initialize_sensors_I2C(VL53L5CX_Configuration *p_dev, uint8_t mode);
bool config_sensors(VL53L5CX_Configuration *p_dev, uint16_t new_i2c_address);
bool get_sensor_data(VL53L5CX_Configuration *p_dev,VL53L5CX_ResultsData *p_results);
//
void initialize_GPIOs();
static inline bool clear_irq();
//
void Enable_cycle_counter(void);

//
#ifdef START_FLIGHT
uint16_t ToFly = 0;
#else
uint16_t ToFly = 1;
#endif

//MAIN
void appMain()
{ 
  vTaskDelay(M2T(3000)); //no rush to start
  // flight_test();
  // vTaskDelay(M2T(1000)); 
  // return;

  //-----------------------------------Initilaize The Deck -----------------------------------------------------//
  bool gpio_exp_status =false;
  bool sensors_status = true;
  gpio_exp_status = I2C_expander_initialize();
  DEBUG_PRINT("ToFDeck I2C_GPIO Expander: %s\n", gpio_exp_status ? "OK." : "ERROR!");  
  #ifdef SESNSOR_FORAWARD_ENABLE
    vTaskDelay(M2T(100)); 
    sensors_status = initialize_sensors_I2C(&vl53l5dev_f,1); //forward
    DEBUG_PRINT("ToFDeck Forward Sensor Initlaize 1: %s\n", sensors_status ? "OK." : "ERROR!");
  #endif
  #ifdef SESNSOR_BACKWARD_ENABLE
    vTaskDelay(M2T(100)); 
    sensors_status = initialize_sensors_I2C(&vl53l5dev_b,2); //backward
    DEBUG_PRINT("ToFDeck_B Sensors Initlaize 2: %s\n", sensors_status ? "OK." : "ERROR!");
  #endif
  #if(defined SESNSOR_BACKWARD_ENABLE && defined SESNSOR_FORAWARD_ENABLE)
    vTaskDelay(M2T(100)); 
    sensors_status = initialize_sensors_I2C(NULL,3);
    DEBUG_PRINT("ToFDeck Sensors Initlaize Both 3: %s\n", sensors_status ? "OK." : "ERROR!"); 
  #endif

  if(gpio_exp_status ==false || sensors_status == false)
  {
    DEBUG_PRINT("ERROR LOOP_1!"); 
    while (1)
    {//stay in ERROR LOOP
      vTaskDelay(M2T(10000)); 
    }
  }

  initialize_GPIOs();
  DEBUG_PRINT("ToFDeck GPIO & Interrupt Initlaized. \n");  

  //-----------------------------------Take Drone Off -----------------------------------------------------//
  //wait for start command to fly
  while(ToFly == 0)
    vTaskDelay(200
    );
  #ifdef START_FLIGHT
    fly_task((FlyCommand_t){take_off,0.0f});
  #endif
  //-----------------------------------Satrt Sensors Ranging -----------------------------------------------------//
  #ifdef SESNSOR_FORAWARD_ENABLE
    vTaskDelay(M2T(100));
    uint8_t ranging_start_res_f = vl53l5cx_start_ranging(&vl53l5dev_f);
    DEBUG_PRINT("ToFDeck Start Sensor Forward Ranging: %s\n", (ranging_start_res_f == VL53L5CX_STATUS_OK) ? "OK." : "ERROR!"); 
  #else
    uint8_t ranging_start_res_f = VL53L5CX_STATUS_OK;
  #endif
  #ifdef SESNSOR_BACKWARD_ENABLE
    vTaskDelay(M2T(100));
    uint8_t ranging_start_res_b = vl53l5cx_start_ranging(&vl53l5dev_b);
    DEBUG_PRINT("Start ToFDeck Sensor Bakcward Ranging: %s\n", (ranging_start_res_b == VL53L5CX_STATUS_OK) ? "OK." : "ERROR!"); 
  #else
    uint8_t ranging_start_res_b = VL53L5CX_STATUS_OK;
  #endif
  if(ranging_start_res_f != VL53L5CX_STATUS_OK || ranging_start_res_b != VL53L5CX_STATUS_OK){
    DEBUG_PRINT("ERROR LOOP_2!"); 
    while (1)
    {//stay in ERROR LOOP
      vTaskDelay(M2T(10000)); 
    }
  }

  //-----------------------------------Collect and Send Data------------------------------------------------------//

  #ifdef SESNSOR_FORAWARD_ENABLE
    uint8_t get_data_success_f = false;
    #ifdef SEND_DATA
      uint8_t to_send_buffer_f[ToF_DISTANCES_LEN+ToF_TARGETS_DETECTED_LEN+ToF_TARGETS_STATUS_LEN]; 
    #endif
  #endif
  #ifdef SESNSOR_BACKWARD_ENABLE
    uint8_t get_data_success_b = false;
    #ifdef SEND_DATA
      uint8_t to_send_buffer_b[ToF_DISTANCES_LEN+ToF_TARGETS_DETECTED_LEN+ToF_TARGETS_STATUS_LEN];
    #endif
  #endif
  // Enable_cycle_counter();
  while(1) {
    if (ToFly == 0){
      fly_task((FlyCommand_t){land,0.0f});
      break;
    }
    vTaskDelay(M2T(35)); // Task Delay
    // Collect Data
    if (irq_status){
      if (clear_irq() == false){
        irq_status = IRQ_ERROR;
      }
      //DEBUG_PRINT("irq_status: 0X%x\n",irq_status);
      if (irq_status && (IRQ_READY | IRQ_PIN_IS_LOW)){
        // timestamp = ((long long)xTaskGetTickCount())/portTICK_RATE_MS;
        #ifdef SESNSOR_FORAWARD_ENABLE
          if (irq_status && IRQ_FRONT){
            get_data_success_f = get_sensor_data(&vl53l5dev_f,&vl53l5_res_f);
            if (get_data_success_f == false){
              // ERROR in Interrupt Result
              DEBUG_PRINT("ToFDeck_Interrupt Wrong Flag Forward!\n"); 
            }
          }
        #endif
        #ifdef SESNSOR_BACKWARD_ENABLE
          if (irq_status && IRQ_REAR){
            get_data_success_b = get_sensor_data(&vl53l5dev_b,&vl53l5_res_b);
            if (get_data_success_b == false){
              // ERROR in Interrupt Result
              DEBUG_PRINT("ToFDeck_Interrupt Wrong Flag Back!\n"); 
            }
          }  
        #endif  
      }else if (irq_status && IRQ_ERROR){
        // ERROR in Handling Interrupt
        DEBUG_PRINT("ToFDeck_Interrupt ERROR!\n"); 
      }
      // Clear flag
      irq_status = 0;
    }
    
    //Set data ready to send
    #ifdef SESNSOR_FORAWARD_ENABLE
      if (get_data_success_f == true)
      {
        
        #ifdef SEND_DATA
          //
          send_command(1, (ToF_DISTANCES_LEN+ToF_TARGETS_DETECTED_LEN+ToF_TARGETS_STATUS_LEN)/28 + 1);  
          //
          memcpy(&to_send_buffer_f[0], (uint8_t *)(&vl53l5_res_f.distance_mm[0]), ToF_DISTANCES_LEN);
          memcpy(&to_send_buffer_f[ToF_DISTANCES_LEN], (uint8_t *)(&vl53l5_res_f.nb_target_detected[0]), ToF_TARGETS_DETECTED_LEN);
          memcpy(&to_send_buffer_f[ToF_DISTANCES_LEN+ToF_TARGETS_DETECTED_LEN], (uint8_t *)(&vl53l5_res_f.target_status[0]), ToF_TARGETS_STATUS_LEN);
          //
          send_data_packet(&to_send_buffer_f[0], ToF_DISTANCES_LEN+ToF_TARGETS_DETECTED_LEN+ToF_TARGETS_STATUS_LEN);
        #endif
        #ifdef ON_BOARD_PROCESS
          // start_timer();
          // it1 = get_timer();
          // uint32_t TStart = get_time_stamp();
          // DEBUG_PRINT("Proceess Image ...\n"); 
          // I2C_expander_toggle_output_pin(LED_FORWARD_PIN_NUM);
          // ARM_CM_DWT_CYCCNT = 0;
          // __disable_irq();
          // it1 = ARM_CM_DWT_CYCCNT;
          FlyCommand_t flight_command = Process_ToF_Image(&vl53l5_res_f);
          // it2 = ARM_CM_DWT_CYCCNT;
          bool flight_status = fly_task(flight_command);
          // it3 =ARM_CM_DWT_CYCCNT;
          // __enable_irq();
          // delta_it_2 += it3-it2;
          // delta_it += it2-it1;
          // it_numer++;
          // DEBUG_PRINT("Delay_%d: %lu\t%lu\n",it_numer,(unsigned long)(delta_it/it_numer),(unsigned long)(delta_it_2/it_numer));
          // I2C_expander_set_output_pin(LED_BACKWARD_PIN_NUM, flight_status);
          // DEBUG_PRINT("ToFDeck P.I.: %d_%f\n",flight_command.command,flight_command.severity); 
          // uint32_t TEnd = get_time_stamp();
          // DEBUG_PRINT("ToFDeck Task Delay: %d\n",TEnd - TStart); 
          // DEBUG_PRINT("ToFDeck P.I.: %d\n",flight_command); 
          // it2 = get_timer() - it1;
        #endif

        // Clear Flag
        get_data_success_f = false;

      }
    #endif
    #ifdef SESNSOR_BACKWARD_ENABLE
    if (get_data_success_b == true)
    { 
      #ifdef SEND_DATA
        //
        send_command(1, (ToF_DISTANCES_LEN+ToF_TARGETS_DETECTED_LEN+ToF_TARGETS_STATUS_LEN)/28 + 1);  
        //
        memcpy(&to_send_buffer_b[0], (uint8_t *)(&vl53l5_res_b.distance_mm[0]), ToF_DISTANCES_LEN);
        memcpy(&to_send_buffer_b[ToF_DISTANCES_LEN], (uint8_t *)(&vl53l5_res_b.nb_target_detected[0]), ToF_TARGETS_DETECTED_LEN);
        memcpy(&to_send_buffer_b[ToF_DISTANCES_LEN +ToF_TARGETS_DETECTED_LEN], (uint8_t *)(&vl53l5_res_b.target_status[0]), ToF_TARGETS_STATUS_LEN);
        //
        send_data_packet(&to_send_buffer_b[0], ToF_DISTANCES_LEN+ToF_TARGETS_DETECTED_LEN+ToF_TARGETS_STATUS_LEN);
      #endif
      // Clear Flag
      get_data_success_b = false;
    }
    #endif
  }
  
}

void send_data_packet(uint8_t *data, uint16_t data_len)
{
  uint8_t packets_nr = 0;
  if (data_len%28 > 0)
    packets_nr = data_len/28 + 1;
  else
    packets_nr = data_len/28;

  for (uint8_t idx=0; idx<packets_nr; idx++)
    if(data_len - 28*idx >= 28)
      send_data_packet_28b(&data[28*idx], 28, idx);
    else
      send_data_packet_28b(&data[28*idx], data_len - 28*idx, idx);
}

void send_data_packet_28b(uint8_t *data, uint8_t size, uint8_t index)
{
  CRTPPacket pk;
  pk.header = CRTP_HEADER(1, 0); // first arg is the port number
  pk.size = size + 2;
  pk.data[0] = 'D';
  pk.data[1] = index;
  memcpy(&(pk.data[2]), data, size);
  crtpSendPacketBlock(&pk);
}

// uint32_t get_time_stamp()
// {
//   timestamp = ((long long)xTaskGetTickCount())/portTICK_RATE_MS;
//   //DEBUG_PRINT("ToFDeck timestamp: %d (size: %d)\n", timestamp,sizeof(timestamp));
//   return timestamp;
// }

void send_command(uint8_t command, uint8_t arg)
{
  //uint32_t timestamp = get_time_stamp();
  CRTPPacket pk;
  pk.header = CRTP_HEADER(1, 0); // first arg is the port number
  pk.size = 7;
  pk.data[0] = 'C';
  pk.data[1] = command;
  pk.data[2] = arg;
  memcpy(&pk.data[3], (uint8_t *)(&timestamp), 4);
  //DEBUG_PRINT("cmd PK 1:%d, 2:%d, 3:%d, 4:%d, ",pk.data[3],pk.data[4],pk.data[5],pk.data[6]); //todo delete
  crtpSendPacketBlock(&pk);
}

bool config_sensors(VL53L5CX_Configuration *p_dev, uint16_t new_i2c_address)
{
  p_dev->platform = VL53L5CX_DEFAULT_I2C_ADDRESS; // use default adress for first use

  // initialize the sensor
  uint8_t tof_res = vl53l5cx_init(p_dev);   if (tof_res != VL53L5CX_STATUS_OK) return false ;
  //DEBUG_PRINT("ToF Config Result: %d \n", tof_init_res);

  // Configurations
  //change i2c address
  tof_res = vl53l5cx_set_i2c_address(p_dev, new_i2c_address);if (tof_res != VL53L5CX_STATUS_OK) return false ;
  tof_res = vl53l5cx_set_resolution(p_dev, VL53L5CX_RESOLUTION_8X8);if (tof_res != VL53L5CX_STATUS_OK) return false ; 
  // 15hz
  tof_res = vl53l5cx_set_ranging_frequency_hz(p_dev, 15);if (tof_res != VL53L5CX_STATUS_OK) return false ; 
  tof_res = vl53l5cx_set_target_order(p_dev, VL53L5CX_TARGET_ORDER_CLOSEST);if (tof_res != VL53L5CX_STATUS_OK) return false ;
  tof_res = vl53l5cx_set_ranging_mode(p_dev, VL53L5CX_RANGING_MODE_CONTINUOUS);if (tof_res != VL53L5CX_STATUS_OK) return false ;
  //tof_res = vl53l5cx_set_ranging_mode(p_dev, VL53L5CX_RANGING_MODE_AUTONOMOUS);if (tof_res != VL53L5CX_STATUS_OK) return false ;// TODO test it

  //Check for sensor to be alive
  uint8_t isAlive;
  tof_res =vl53l5cx_is_alive(p_dev,&isAlive);if (tof_res != VL53L5CX_STATUS_OK) return false;
  if (isAlive != 1) return false;
  
  // All Good!
  return true;
}

bool initialize_sensors_I2C(VL53L5CX_Configuration *p_dev, uint8_t mode)
{
  bool status = false;

  //reset I2C  //configure pins out/in for forward only

  //status = I2C_expander_set_register(OUTPUT_PORT_REG_ADDRESS,I2C_RST_BACKWARD_PIN,I2C_RST_FORWARD_PIN);if (status == false)return status;

  if (mode == 1 && p_dev != NULL){
    //enable forward only and config
    status = I2C_expander_set_register(OUTPUT_PORT_REG_ADDRESS,LPN_FORWARD_PIN | LED_FORWARD_PIN );if (status == false)return status; 
    status = config_sensors(p_dev,VL53L5CX_FORWARD_I2C_ADDRESS);if (status == false)return status; 
  }
  if (mode == 2 && p_dev != NULL){
    //enable backward only and config
    status = I2C_expander_set_register(OUTPUT_PORT_REG_ADDRESS,LPN_BACKWARD_PIN | LED_BACKWARD_PIN); if (status == false)return status; 
    status = config_sensors(p_dev,VL53L5CX_BACKWARD_I2C_ADDRESS);if (status == false)return status; 
  }
  //status = I2C_expander_set_register(OUTPUT_PORT_REG_ADDRESS,0x00); //all off
  if (mode == 3){
    //enable both forward & backward
    status = I2C_expander_set_register(OUTPUT_PORT_REG_ADDRESS,LPN_BACKWARD_PIN | LED_BACKWARD_PIN|LPN_FORWARD_PIN | LED_FORWARD_PIN); if (status == false)return status; 
  }
  return status;
}
void initialize_GPIOs(){
  // {.periph= RCC_AHB1Periph_GPIOB, .port= GPIOB, .pin=GPIO_Pin_5,  .adcCh=-1},            /* IO2 */
  // void EXTI5_Callback(void);
  
  EXTI_InitTypeDef EXTI_InitStructure;
  // Set up interrupt
  SYSCFG_EXTILineConfig(EXI_PortSource, EXI_PinSource);

  EXTI_InitStructure.EXTI_Line = EXI_LineN;
  EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
  EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Falling; // EXTI_Trigger_Rising_Falling
  EXTI_InitStructure.EXTI_LineCmd = ENABLE;
  EXTI_Init(&EXTI_InitStructure);

  pinMode(EXI_PIN, INPUT);     // Set 

}
static inline bool clear_irq(){
  int8_t status=-1,result=-1;
  //read input current value.
  status = I2C_expander_get_register(INPUT_PORT_REG_ADDRESS,&result);if (status != true) return false;
  if (result & INTERRUPT_SENSE_FORWARD_PIN){
    irq_status |= IRQ_FRONT;
  } 
  if (result & INTERRUPT_SENSE_BACKWARD_PIN){
    irq_status |= IRQ_REAR;
  }
  //state= I2C_expander_get_input_pin(INTERRUPT_SENSE_FORWARD_PIN_NUM,&result);//clear intrupt
  //DEBUG_PRINT("int_f_status: %d (%d_%d)\n",int_pin_status,state,result);
  return true;
}
bool get_sensor_data(VL53L5CX_Configuration *p_dev,VL53L5CX_ResultsData *p_results){
 
  // Check  for data ready I2c
  uint8_t ranging_ready = 2;
  //ranging_ready --> 0 if data is not ready, or 1 if a new data is ready.
  uint8_t status = vl53l5cx_check_data_ready(p_dev, &ranging_ready);if (status != VL53L5CX_STATUS_OK) return false;

  // 1 Get data in case it is ready
  if (ranging_ready == 1){
    status = vl53l5cx_get_ranging_data(p_dev, p_results);if (status != VL53L5CX_STATUS_OK) return false;
  }else {
    //0  data in not ready yet
    return false;
  }

  // All good then
  //return false;// TODO deleet
  return true;
}
//EXTI IRQ
void __attribute__((used)) EXTI5_Callback(void){
  //DEBUG_PRINT("IRQ\n"); 
  if(irq_status){
    // Missed Image
    //DEBUG_PRINT("an Imaged has been missed !\n"); --> tested ok
  }
  irq_status = 0 ;
  int8_t int_pin_status = digitalRead(EXI_PIN);  //HIGH:LOW
  if (int_pin_status == LOW ){
    irq_status |= IRQ_READY | IRQ_PIN_IS_LOW ;
  }else{
    irq_status |= IRQ_READY ;
  }
  timestamp = ((long long)xTaskGetTickCount())/portTICK_RATE_MS;
  // TODO Delete
  // bool clear_status = false; 
  // clear_status = clear_irq();
  // if (clear_status == false){
  //   irq_status = IRQ_ERROR;
  // }
  //DEBUG_PRINT("TOF_Interrupt\n"); 
}


void Enable_cycle_counter(void)
{
  if (ARM_CM_DWT_CTRL != 0) 
  {
    // See if DWT is available
    ARM_CM_DEMCR|= 1 << 24; // Set bit 24
    ARM_CM_DWT_CYCCNT = 0;
    ARM_CM_DWT_CTRL |= 1 << 0; // Set bit 0 
  }
  
}

PARAM_GROUP_START(ToF_FLY_PARAMS)
PARAM_ADD(PARAM_UINT16, ToFly, &ToFly)
PARAM_GROUP_STOP(ToF_FLY_PARAMS)