/*
 * equalize_hist.c
 * Tommaso Polonelli
 *
 * Copyright (C) 2016 ETH Zurich, University of Bologna
 * Copyright (C) 2018 Tommaso Polonelli
 *
 * This software may be modified and distributed under the terms
 * of the MIT license.  See the LICENSE file for details.
 *
 * Created on: May 25, 2020
 *
 */

#include "equalize_hist.h"
#include "fixed_p.h"
#include <string.h>

/*********** Constants  ****************/

/* Operation block size (number of pixel per operators)
 * this impact the L1 memory but decrease number of transfers L1 -> L3
 */

#define PERF		(1)
#define PRINT_IMG	(0)

/*********** Structures *************/

enum { 
	/* must be equal to the color depth */
	HIST_SZ = 256 
};

/*********** Declarations *************/

/*------ Global variables  -----------*/

/*------ Function prototypes ---------*/


/*********** Functions   ****************/

void EqualizeHistCalcHist_Run
(unsigned char *src, int* histogram, int total)
{
	uint8_t *ptr = src;

	int x = 0;
	for (; x <= total - 4; x += 4)
	{
		int t0 = ptr[x], t1 = ptr[x + 1];
		histogram[t0]++; histogram[t1]++;
		t0 = ptr[x + 2]; t1 = ptr[x + 3];
		histogram[t0]++; histogram[t1]++;
	}

	for (; x < total; ++x)
		histogram[ptr[x]]++;

}

void EqualizeHistLut_Run(int* lut, unsigned char* src, unsigned char* dst, int total)
{

	unsigned char* sptr = src;
	unsigned char* dptr = dst;

	int x = 0;
	for (; x <= total - 4; x += 4)
	{
		int v0 = sptr[x];
		int v1 = sptr[x + 1];
		int x0 = lut[v0];
		int x1 = lut[v1];
		dptr[x] = (unsigned char)x0;
		dptr[x + 1] = (unsigned char)x1;

		v0 = sptr[x + 2];
		v1 = sptr[x + 3];
		x0 = lut[v0];
		x1 = lut[v1];
		dptr[x + 2] = (unsigned char)x0;
		dptr[x + 3] = (unsigned char)x1;
	}

	for (; x < total; ++x)
		dptr[x] = (unsigned char)lut[sptr[x]];


}


void EqualizeScale_Run(int* lut, int* hist, int hist_sz, int total, int i)
{
	/* floating point function */
	float histf = (float)(hist[i]);
	float scale = (((float)hist_sz) - 1.f) / (((float)total) - histf);
	int sum = 0;
	for (lut[i++] = 0; i < hist_sz; ++i)
	{
		sum += hist[i];
		float temp = (float)sum * scale;
		if (temp > 0xFF) {
			temp = 0xFF;
		}
		if (temp < 0x00) {
			temp = 0x00;
		}
		lut[i] = (int)temp;
	}

}

void equalizeHist(uint8_t* src, uint8_t* dst, int width, int height, int hist_sz)
{

#if PERF
	uint32_t tt;
	pi_perf_conf(1 << PI_PERF_CYCLES | 1 << PI_PERF_ACTIVE_CYCLES);
	pi_perf_start();
	tt = pi_perf_read(PI_PERF_ACTIVE_CYCLES);
#endif

	int *hist; //could be initialized to zero
	int *lut;
	int i = 0;
	int total = width * height;

	/* init color depth */
	if ((hist_sz == 0) || (hist_sz > HIST_SZ)){
		/* default color depth */
		hist_sz = HIST_SZ;
	}

	/* Init hist in L1 */
	hist = (int *) pi_fc_l1_malloc((uint32_t) sizeof(int)*HIST_SZ);
	if (hist == NULL)
	{
		printf("buff alloc failed !\n");
		pmsis_exit(-1);
	}
	/* Init lut in L1 */
	lut = (int *) pi_fc_l1_malloc((uint32_t) sizeof(int)*HIST_SZ);
	if (lut == NULL)
	{
		printf("buff alloc failed !\n");
		pmsis_exit(-1);
	}

	/* run body calc */
	memset(hist,0,sizeof(int)*HIST_SZ);
	EqualizeHistCalcHist_Run(src, hist, total);


	/* check Hist */
	/* error if the image has a single color */
	while (!hist[i]) ++i;
	if (hist[i] == total)
	{
		if (dst != src){
			memcpy(dst, src, total);
		}
		return;
	}

	EqualizeScale_Run(lut, hist, hist_sz, total, i);

	EqualizeHistLut_Run(lut , src, dst, total);

	pi_fc_l1_free(hist, (uint32_t) sizeof(int)*HIST_SZ);
	pi_fc_l1_free(lut, (uint32_t) sizeof(int)*HIST_SZ);

#if PERF
	pi_perf_stop();
	printf("Eq. Hist. Cycles %d \n",(int)(pi_perf_read(PI_PERF_ACTIVE_CYCLES)-tt));
	printf("Eq. Hist. cpp %d \n",((int)(pi_perf_read(PI_PERF_ACTIVE_CYCLES)-tt))/((int)(total)));
#endif

#if PRINT_IMG
	for(i = 0; i < PRINT_IMG; i++){
		printf("%d ,",dst[i]);
	}
	printf("\n");
#endif

}

/* End of file. */
