
function [dd]=f_MeshMaterialPreProcessor(geom,dd)
%
% function [dd]=f_MeshMaterialPreProcessor(geom,dd)
%
% This preprocessor, starting from the vector of structures "geom",
% generates vectors containing the mesh and position-dependent material
% properties. geom has the following fields, for each layer
% (geom(indLayer)):
%
% L            : length of the layer (m)
% MeshStep     : mesh step of the layer (m)
% dopType      : doping type ('p' or 'n')
% dopini       : doping value at the left node of the interval
% dopend       : for layers with graded doping, this is the 
% [OPTIONAL]                  right doping value (dopini is the left one) 
%                             (1/m^3)
% material     : material of the layer; possible strings are:
%                              o AlGaAs : (Al_{x} Ga_{1-x} As)
%                              o GaAsSb : (Ga As_{1-x} Sb_{x})
%                              o GaInAs : (Ga_{1-x} In_{x} As)
%                              o Si
%                              o SiO2
%
% xmolini      : xmol value at left node of the interval
% xmolend      : for layers with graded xmol, this is the right xmol value
% [OPTIONAL]     (xmolini is the left one)
% ymolini      : like xmolini for quaternary compounds
% ymolend      : like xmolend for quaternary compounds
%
% For the first and the last layer only (in the others, the command is not
% recognized), it is possible to introduce the following [OPTIONAL]
% argument:
%
% SchottkyBarrier : if specified, the contact is assumed to be Schottky
% [OPTIONAL]                       with barrier (intended as difference of 
%                                  metal workfunction and semiconductor
%                                  affinity) specified as value of this 
%                                  variable; if unspecified, the contact is
%                                  treated as ohmic.
%
% WorkFunction    : alternative to SchottkyBarrier to specify a Schottky
% [OPTIONAL]                       contact, in this case working with a
%                                  contact-on-insulator (e.g., the gate
%                                  of a MOSFET). If this is  specified, 
%                                  overwrites a possible specification 
%                                  of SchottkyBarrier. If both 
%                                  SchottkyBarrier and WorkFunction are
%                                  unspecified, the contact is treated
%                                  as ohmic.
%
% In addition, also the dd structure is an input, to provide additional
% parameters to the preprocessor:
%
% strName          : name of the structure for possible save files
% Temp             : lattice temperature (K)
% tolConv          : tolerance of the solution for BC Newton solver
% tolConv_ferdr    : tolerance for invferdr Newton method
% [OPTIONAL]         if not specified, set equal to tolConv
% tolConv_neutr    : tolerance of the solution for Poisson Newton solver
% [OPTIONAL]         if not specified, set equal to tolConv_ferdr
% max_diter        : max number of iterations for Newton methods
%
%
%
%
% The structure dd contains also the outputs of this function, in the
% following additional fields of the structure dd:
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Fundamental quantities
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% nn            : integer value indicating the total number of mesh points
% node          : real vector of mesh points (dimension (1,nn)) (m)
% ElementLength : real vector containing the lengths of the elements 
%                 (in 1D, the edges connecting the nodes (m))
% ControlBox    : real vector contaning the weights for trapezoidal
%                 integration, useful for finite box methods
%                 (dimension (1,nn)) (m)
% in            : integer vector containing, for each mesh node, the index
%                 of the layer to which it belongs
% dop_D         : real vector containing the donor doping density at the
%                 corresponding node (1/m^3)
% dop_A         : real vector containing the acceptor doping density at the
%                 at the corresponding node (1/m^3)
% xmol          : real vector containing the "x" molar fraction at the
%                 corresponding node
% ymol          : real vector containing the "y" molar fraction at the
%                 corresponding node (useful for quaternary alloys)
% DeltaEMSB     : real vector containing, only in its first and last
%                 elements, the energy jump introduced by a possibly present
%                 Schottky contact (J)
% WorkFunction  : real vector containing, only in its first and last
%                 elements, the energy jump introduced by a possibly present
%                 Schottky contact (J)
% mattype       : type of material
%                 o 0: semiconductor
%                 o 1: oxide
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Material parameters (electrostatics)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% epsr         : real vector containing the relative dielectric
%                permittivity (dimensionless)
% meff_e       : real vector containing the conduction band effective mass
%                (normalized to m0)
% meff_h       : real vector containing the valence band effective mass
%                (normalized to m0)
% BandGap      : real vector containing the bandgap (J)
% VBO          : real vector containing the valence band offset (J)
% affinity     : real vector containing the electron affinity (J)
% DeltaED      : real vector containing the donor doping ionization energy
%                (J)
% DeltaEA      : real vector containing the  acceptor doping ionization
%                energy (J)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Material parameters (transport and recombination)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% eMobility    : real vector containing the intrinsic electron mobility
%                (m^2/s/V)
% hMobility    : real vector containing the intrinsic hole mobility
%                (m^2/s/V)
% tauSRH_n     : electron lifetime for Shockley-Read-Hall recombination (s)
% tauSRH_p     : hole lifetime for Shockley-Read-Hall recombination (s)
% EtrapSRH     : energy (w.r.t. EF) of traps
% BRad         : radiative recombination rate (m^3/s)
% CAug_nnp     : electron-electron Auger coefficient (m^6/s)
% CAug_ppn     : hole-hole Auger coefficient (m^6/s)
%
%
%
% Alberto Tibaldi, 27/01/2022



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Loading physical and unit conversion constants
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
s_LoadConstants


                     
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Mesh generation
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
NLayers = length(geom);  % number of layers of the geom
dd.node = 0;             % using "0" as the first mesh node
ii      = 1;             % ii is used to generate dd.in, which is a vector
                         % long nn and containing, in each element, the number
                         % of the corresponding layer; this is useful to
                         % associate to each node the correct material
                         % properties; ii is a counter, so it is incremented by
                         % a loop
for indLayer=1:NLayers
    %
    Llayer  = geom(indLayer).L;        % length of this layer (indLayer)
    step    = geom(indLayer).MeshStep; % step size of the mesh in this layer
    nnLayer = round(Llayer/step);       % number of nodes in this layer
    %
    % The mesh step in each layer should be chosen in such a way to have an
    % integer number of mesh nodes, otherwise an error is returned
    if(abs(nnLayer-Llayer/step)>1e-9)
        error(['ERROR in the mesh step in layer ',num2str(indLayer), ...
               ': it does not lead to an integer number of mesh points! Use a multiple of the length!'])
    end
    step = geom(indLayer).MeshStep;
    %
    % the first layer should be treated differently to initialize inLayer,
    % which is later saved in dd.in
    if(indLayer == 1)
        inLayer = [ii];
    else
        inLayer = [];
    end
    %
    % the vector inLayer is the vector containing, for the current layer
    % (indLayer), the indexes of the nodes belonging to it; because we know
    % a priori that the layer has nnLayer nodes, it can be obtained as
    % follows:
    for indnn=1:nnLayer
        dd.node = [dd.node,dd.node(end)+step];
        inLayer = [inLayer,ii+1];
        ii      = ii+1;
    end
    %
    % saving in the final variable dd.in
    dd.in(inLayer) = indLayer;
end % for indLayer=1:NLayers
%
nn      = length(dd.node);
dd.nn   = nn;
dd.node = reshape(dd.node,nn,1);         % column vector (m)



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Pre-processing mesh data: element lengths and control boxes
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Element lengths
le = diff(dd.node);
%
% Control boxes (trapezoidal method weights)
Li         = zeros(nn,1);     % initialization of integration segments (m)
Li(1:nn-1) = le/2;            % left nodes contributions (m)
Li(2:nn)   = Li(2:nn) + le/2; % right nodes contributions (m)



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Starting a new layer loop to define doping, alloy molar fractions and,
% if present, the Schottky barrier at contacts
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
dd.DeltaEMSB    = zeros(1,nn); % initializing vector of Schottky barriers
dd.WorkFunction = zeros(1,nn); % initializing vector of Schottky barriers
%
for indLayer=1:NLayers
    iiLayer = find(dd.in==indLayer);
    NPoints = length(iiLayer);
    %
    % If geom(indLayer).dopend is not specified, the layer has constant
    % doping defined on it
    if(not(isfield(geom(indLayer),'dopend')) || isempty(geom(indLayer).dopend))
        geom(indLayer).dopend = geom(indLayer).dopini;
    end
    %
    % If geom(indLayer).xmolend is not specified, the layer has constant
    % x molar fraction defined on it
    if(not(isfield(geom(indLayer),'xmolend')) || isempty(geom(indLayer).xmolend))
        geom(indLayer).xmolend = geom(indLayer).xmolini;
    end
    %
    % If geom(indLayer).ymolend is not specified, the layer has constant
    % y molar fraction defined on it
    if(not(isfield(geom(indLayer),'ymolend')) || isempty(geom(indLayer).ymolend))
        geom(indLayer).ymolend = geom(indLayer).ymolini;
    end
    %
    % Defining n or p dopings as linearly varying from dopini and dopend
    % (if dopend is not specified, doping is constant in the layer)
    if(geom(indLayer).dopType=='n')
        dd.dop_D(iiLayer) = linspace(geom(indLayer).dopini,geom(indLayer).dopend,NPoints);
        dd.dop_A(iiLayer) = zeros(1,NPoints);
    elseif(geom(indLayer).dopType=='p')
        dd.dop_D(iiLayer) = zeros(1,NPoints);
        dd.dop_A(iiLayer) = linspace(geom(indLayer).dopini,geom(indLayer).dopend,NPoints);
    end
    %
    % Defining x and y molar fractions in the layer, possibly varying
    dd.xmol(iiLayer)=linspace(geom(indLayer).xmolini,geom(indLayer).xmolend,NPoints);
    dd.ymol(iiLayer)=linspace(geom(indLayer).ymolini,geom(indLayer).ymolend,NPoints);
    %
    % Looking for Schottky barrier at first contact
    if(indLayer==1 && isfield(geom(indLayer),'SchottkyBarrier')==1 && ...
                      isfield(geom(indLayer),'WorkFunction')==0    && ...
                      isempty(geom(indLayer).SchottkyBarrier)==0)
        dd.DeltaEMSB(1) = geom(indLayer).SchottkyBarrier;
    end
    %
    % Looking for Schottky barrier at last contact
    if(indLayer==length(geom) && isfield(geom(indLayer),'SchottkyBarrier')==1 && ...
                                 isfield(geom(indLayer),'WorkFunction')==0    && ...
                                 isempty(geom(indLayer).SchottkyBarrier)==0)
        dd.DeltaEMSB(end) = geom(indLayer).SchottkyBarrier;
    end
    %
    % Looking for Schottky barrier at first contact
    if(indLayer==1 && isfield(geom(indLayer),'WorkFunction')==1 && ...
                      isempty(geom(indLayer).WorkFunction)==0)
        dd.WorkFunction(1) = geom(indLayer).WorkFunction;
    end
    %
    % Looking for Schottky barrier at last contact
    if(indLayer==length(geom) && isfield(geom(indLayer),'WorkFunction')==1 && ...
                                 isempty(geom(indLayer).WorkFunction)==0)
        dd.WorkFunction(end) = geom(indLayer).WorkFunction;
    end
end % for indLayer=1:NLayers



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Optional arguments of dd structure
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if(not(isfield(dd,'tolConv_ferdr')))
    dd.tolConv_ferdr = dd.tolConv;
end
%
if(not(isfield(dd,'tolConv_neutr')))
    dd.tolConv_neutr = dd.tolConv_ferdr;
end



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Evaluating material properties
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initializing the vectors
dd.epsr      = [];
dd.meff_e    = [];
dd.meff_h    = [];
dd.BandGap   = [];
dd.VBO       = [];
dd.affinity  = [];
dd.DeltaED   = [];
dd.DeltaEA   = [];
%
dd.eMobility = [];
dd.hMobility = [];
%
dd.tauSRH_n  = [];
dd.tauSRH_p  = [];
dd.EtrapSRH  = [];
dd.BRad      = [];
dd.CAug_nnp  = [];
dd.CAug_ppn  = [];
%
dd.mattype   = [];
%
for indLayer=1:length(geom)
    %
    in = find(dd.in==indLayer); % loading layer node indexes
    xmol = dd.xmol(in); % loading layer molar fraction vector
    ymol = dd.ymol(in); % loading layer molar fraction vector
    %
    % Calling the material parameters function
    [macro] = f_MaterialMacro(geom(indLayer).material,xmol,ymol,dd.Temp);
    %
    % Piling macro information in vectors
    dd.mattype   = [dd.mattype,macro.mattype];
    %
    dd.epsr      = [dd.epsr,macro.epsr];
    dd.meff_e    = [dd.meff_e,macro.meff_e];
    dd.meff_h    = [dd.meff_h,macro.meff_h];
    dd.BandGap   = [dd.BandGap,macro.BandGap];
    dd.VBO       = [dd.VBO,macro.VBO];
    dd.affinity  = [dd.affinity,macro.affinity];
    dd.DeltaED   = [dd.DeltaED,macro.DeltaED];
    dd.DeltaEA   = [dd.DeltaEA,macro.DeltaEA];
    %
    dd.eMobility = [dd.eMobility,macro.eMobility];
    dd.hMobility = [dd.hMobility,macro.hMobility];
    %
    dd.tauSRH_n  = [dd.tauSRH_n,macro.tauSRH_n];
    dd.tauSRH_p  = [dd.tauSRH_p,macro.tauSRH_p];
    dd.EtrapSRH  = [dd.EtrapSRH,macro.EtrapSRH];
    dd.BRad      = [dd.BRad,macro.BRad];
    dd.CAug_nnp  = [dd.CAug_nnp,macro.CAug_nnp];
    dd.CAug_ppn  = [dd.CAug_ppn,macro.CAug_ppn];
end % for indLayer=1:length(geom)



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Final unit conversion and storage in the data structure
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Node-wise quantities
dd.node           = reshape(dd.node,nn,1);         % column vector (m)
dd.ElementLength  = le;                            % element lengths (m)
dd.ControlBox     = Li;                            % control boxes (m)
dd.dop_D          = reshape(dd.dop_D,nn,1);        % column vector (1/m^3)
dd.dop_A          = reshape(dd.dop_A,nn,1);        % column vector (1/m^3)
%
dd.meff_e         = reshape(dd.meff_e,nn,1);       % column vector
dd.meff_h         = reshape(dd.meff_h,nn,1);       % column vector
dd.BandGap        = reshape(dd.BandGap,nn,1)*eV;   % column vector (J)
dd.VBO            = reshape(dd.VBO,nn,1)*eV;       % column vector (J)
dd.DeltaED        = reshape(dd.DeltaED,nn,1)*eV;   % column vector (J)
dd.DeltaEA        = reshape(dd.DeltaEA,nn,1)*eV;   % column vector (J)
dd.DeltaEMSB      = reshape(dd.DeltaEMSB,nn,1);    % column vector (J)
%
dd.tauSRH_n       = reshape(dd.tauSRH_n,nn,1);     % column vector (s)
dd.tauSRH_p       = reshape(dd.tauSRH_p,nn,1);     % column vector (s)
dd.EtrapSRH       = reshape(dd.EtrapSRH,nn,1);     % column vector (J)
dd.BRad           = reshape(dd.BRad,nn,1);         % column vector (cm^3/s)
dd.CAug_nnp       = reshape(dd.CAug_nnp,nn,1);     % column vector (cm^6/s)
dd.CAug_ppn       = reshape(dd.CAug_ppn,nn,1);     % column vector (cm^6/s)
%
% For element-wise quantities, performing averages over elements
dd.epsr           = (dd.epsr(1:end-1)+dd.epsr(2:end))/2;
dd.eMobility      = (dd.eMobility(1:end-1)+dd.eMobility(2:end))/2;
dd.hMobility      = (dd.hMobility(1:end-1)+dd.hMobility(2:end))/2;
%
dd.epsr           = reshape(dd.epsr,nn-1,1);       % column vector
dd.eMobility      = reshape(dd.eMobility,nn-1,1);  % column vector (m^2/V/s)
dd.hMobility      = reshape(dd.hMobility,nn-1,1);  % column vector (m^2/V/s)