function [macro]=f_MaterialMacro(material,xmol,ymol,T)
%
% function [macro]=f_MaterialMacro(material,xmol,ymol,T)
%
% This function contains the parameters of the materials of interest useful
% in Poisson's equation.
% For alloys, parameters for several starting materials are introduced, and
% interpolated according to formulas from the literature.
%
% Input parameters:
%
% material: (case-sensitive) string containing the name of the
%             material; available materials are:
%             o AlGaAs : Aluminum gallium arsenide (Al_{x} Ga_{1-x} As)
%             o GaAsSb : Gallium arsenide antimonide (Ga As_{1-x} Sb_{x})
%             o GaInAs : Gallium indium arsenide (Ga_{1-x} In_{x} As)
%             o Si     : Silicon
%
% xmol    : real vector variable containing the "x" molar fraction to be
%           used to interpolate the molecules properties (for both
%           ternary and quaternary materials). Not useful with Si.
%
% ymol    : real vector variable containing the "y" molar fraction to be
%           used to interpolate the molecules properties (to be mainly
%           used in quaternary materials). Not useful with Si.
%
% T       : real scalar variable containing the temperature at which the
%           material parameters should be estimated.
%
% For the sake of simplicity, all the output quantities are introduced in a
% structure "macro", which contains the quantities useful to solve
% Poisson's equation:
%
% macro.epss     : relative dielectric constant (dimensionless)
% macro.meff_e   : conduction band effective mass normalized to m0
% macro.meff_h   : valence band effective mass normalized to m0
% macro.Eg       : bandgap (eV)
% macro.VBO      : valence band offset (eV)
% macro.DeltaED  : donor doping ionization energy (eV)
% macro.DeltaEA  : acceptor doping ionization energy (eV)
%
% References:
%
% [1] I. Vurgaftman, J. R. Meyer and L. R. Ram-Mohan, "Band parameters for
%     III-V compound semiconductors and their alloys," J. Appl. Phys, vol.
%     89, no. 11, pp. 5815-5875, Jun. 2001
%
% [2] Ioffe Physico-Technical Institute, Physical Properties of
% semiconductors (silicon), St. Petersburg, Russia, available online,
% http://www.ioffe.ru/SVA/NSM/Semicond/Si/
%
% Alberto Tibaldi, 27/01/2022



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Loading physical and unit conversion constants
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
s_LoadConstants



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Setting all quantities to default values
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
mattype   = NaN*ones(size(xmol));
DeltaEA   = 0*ones(size(xmol));
DeltaED   = 0*ones(size(xmol));
epsr      = 0*ones(size(xmol));
Eg        = 0*ones(size(xmol));
VBO       = 0*ones(size(xmol));
meff_e    = 0*ones(size(xmol));
meff_h    = 0*ones(size(xmol));
eMobility = 0*ones(size(xmol));
hMobility = 0*ones(size(xmol));
tauSRH_n   = 0*ones(size(xmol));
tauSRH_p   = 0*ones(size(xmol));
EtrapSRH  = 0*ones(size(xmol));
BRad      = 0*ones(size(xmol));
CAug_nnp  = 0*ones(size(xmol));
CAug_ppn  = 0*ones(size(xmol));



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Material parameters of GaSb
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
epsr_GaSb     = 15.70; % http://www.ioffe.ru/SVA/NSM/Semicond/GaSb/basic.html
Eg0_GaSb      = 0.812;                          % [1, Table VII]
alpha_GaSb    = 0.417e-3;                       % [1, Table VII]
beta_GaSb     = 140;                            % [1, Table VII]
meffn_GaSb    = 0.039;                          % [1, Table VII]
gamma1_GaSb   = 13.4;                           % [1, Table VII]
gamma2_GaSb   = 4.7;                            % [1, Table VII]
mhh_GaSb      = 1./(gamma1_GaSb-2*gamma2_GaSb); % [1, eq. (2.16), top]
mlh_GaSb      = 1./(gamma1_GaSb+2*gamma2_GaSb); % [1, eq. (2.17), top]
VBO_GaSb      = -0.03;                          % [1, Table VII]
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Material parameters of GaAs
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
epsr_GaAs   = 12.90; % http://www.ioffe.ru/SVA/NSM/Semicond/GaAs/basic.html
Eg0_GaAs    = 1.519;                          % [1, Table I]
alpha_GaAs  = 0.5405e-3;                      % [1, Table I]
beta_GaAs   = 204;                            % [1, Table I]
meffn_GaAs  = 0.067;                          % [1, Table I]
gamma1_GaAs = 6.98;                           % [1, Table I]
gamma2_GaAs = 2.06;                           % [1, Table I]
mhh_GaAs    = 1./(gamma1_GaAs-2*gamma2_GaAs); % [1, eq. (2.16), top]
mlh_GaAs    = 1./(gamma1_GaAs+2*gamma2_GaAs); % [1, eq. (2.17), top]
VBO_GaAs    = -0.80;                          % [1, Table I]
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Material parameters of AlAs
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
epsr_AlAs   = 10.06; % http://www.ioffe.ru/SVA/NSM/Semicond/AlGaAs/basic.html
Eg0_AlAs    = 3.099;                          % [1, Table II]
alpha_AlAs  = 0.885e-3;                       % [1, Table II]
meffn_AlAs  = 0.15;                           % [1, Table II]
gamma1_AlAs = 3.76;                           % [1, Table II]
gamma2_AlAs = 0.82;                           % [1, Table II]
mhh_AlAs    = 1./(gamma1_AlAs-2*gamma2_AlAs); % [1, eq. (2.16), top]
mlh_AlAs    = 1./(gamma1_AlAs+2*gamma2_AlAs); % [1, eq. (2.17), top]
VBO_AlAs    = -1.33;                          % [1, Table II]
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Material parameters of InAs
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
epsr_InAs   = 13.10; % http://www.ioffe.ru/SVA/NSM/Semicond/InAs/basic.html
Eg0_InAs    = 0.417;                          % [1, Table III]
alpha_InAs  = 0.276e-3;                       % [1, Table III]
beta_AlAs   = 530;                            % [1, Table III]
beta_InAs   = 93;                             % [1, Table III]
meffn_InAs  = 0.026;                          % [1, Table III]
gamma1_InAs = 20.0;                           % [1, Table III]
gamma2_InAs = 8.50;                           % [1, Table III]
mhh_InAs    = 1./(gamma1_InAs-2*gamma2_InAs); % [1, eq. (2.16), top]
mlh_InAs    = 1./(gamma1_InAs+2*gamma2_InAs); % [1, eq. (2.17), top]
VBO_InAs    = -0.59;                          % [1, Table III]
%
affinity_GaSb = 4.06;
Eg_GaSb = Eg0_GaSb - alpha_GaSb.*T.^2./(beta_GaSb + T);
CBO_GaSb = VBO_GaSb + Eg_GaSb;



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Obtaining (possibly, alloy) parameters from single-material definitions
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
switch material
    
    case 'AlGaAs' % Aluminum gallium arsenide (Al_{x} Ga_{1-x} As)
        
        mattype   = semiconductor*ones(size(xmol));
        
        %-- Incomplete ionization energies, eV
        DeltaED   = 5e-3*ones(size(xmol));  % donor energies
        DeltaEA   = 26e-3*ones(size(xmol)); % acceptor energies
        %
        %-- Relative dielectric constant
        epsr      = epsr_AlAs.*xmol + epsr_GaAs.*(1-xmol);
        %
        %-- Bandgap, eV, bowing for [1, eq. (4.1)] from [1, Table XII]
        Eg_GaAs   = Eg0_GaAs - alpha_GaAs.*T.^2./(T+beta_GaAs); % [1, eq. (2.13)]
        Eg_AlAs   = Eg0_AlAs - alpha_AlAs.*T.^2./(T+beta_AlAs); % [1, eq. (2.13)]
        Eg        = Eg_GaAs + (Eg_AlAs - Eg_GaAs).*xmol - (- 0.127 + 1.310.*xmol).*xmol.*(1-xmol);
        %
        %-- Valence band offset, eV, [1, eq. (4.1)]
        VBO       = VBO_GaAs + (VBO_AlAs - VBO_GaAs)*xmol;
        CBO       = VBO + Eg;
        affinity  = affinity_GaSb - CBO + CBO_GaSb;
        %
        %-- Conduction band effective mass (normalized to m0)
        meff_e    = 1./(xmol./meffn_AlAs + (1-xmol)./meffn_GaAs);
        %
        %-- Valence band effective mass (normalized to m0)
        meff_h_lh = 1./(1./mlh_GaAs + (1/mlh_AlAs - 1/mlh_GaAs)*xmol);
        meff_h_hh = 1./(1./mhh_GaAs + (1/mhh_AlAs - 1/mhh_GaAs)*xmol);
        meff_h    = (meff_h_lh.^(3/2) + meff_h_hh.^(3/2)).^(2/3);
        %
        indReg1   = find(xmol>=-1e-10 & xmol<=0.45);
        indReg2   = find(xmol>0.45 & xmol<=1+1e-10);
        eMobility(indReg1) = (8000-22000.*xmol(indReg1)+10000.*xmol(indReg1).^2)/10000;
        eMobility(indReg2) = (-255+1160*xmol(indReg2)-720*xmol(indReg2).^2)/10000;
        hMobility =(400-700*xmol+450*xmol.^2)/10000;
        %
        tauSRH_n   = 2.5e-7*ones(size(xmol)); % electron SRH lifetime, s
        tauSRH_p   = 5e-9*ones(size(xmol));   % hole SRH lifetime, s
        EtrapSRH  = 0*ones(size(xmol));       % position of SRH trap level, eV
        %
        BRad      = 7.2e-10/100^3*ones(size(xmol)); % radiative recombination coefficient, m^3/s
        CAug_nnp  = 1e-30/100^6*ones(size(xmol));   % Auger e-e-h process coefficient, m^6/s
        CAug_ppn  = 1e-30/100^6*ones(size(xmol));   % Auger h-h-e process coefficient, m^6/s

    case 'GaAsSb' % Gallium arsenide antimonide (Ga As_{1-x} Sb_{x})
        
        mattype   = semiconductor*ones(size(xmol));

        %-- Incomplete ionization energies, eV
        DeltaED   = 5e-3*ones(size(xmol));  % donor energies
        DeltaEA   = 26e-3*ones(size(xmol)); % acceptor energies
        %
        %-- Relative dielectric constant
        epss      = epsr_GaSb.*xmol + epsr_GaAs.*(1-xmol);
        %
        %-- Bandgap, eV, bowing from [1, Table XXI]
        Eg_GaAs   = Eg0_GaAs - alpha_GaAs.*T.^2./(T+beta_GaAs); % [1, eq. (2.13)]
        Eg_GaSb   = Eg0_GaSb - alpha_GaSb.*T.^2./(T+beta_GaSb); % [1, eq. (2.13)]
        Eg        = Eg_GaAs + (Eg_GaSb - Eg_GaAs).*xmol - 1.43*xmol.*(1 - xmol); % [1, eq. (4.1)]
        %
        %-- Valence band offset, eV, bowing for [1, eq. (4.1)] from [1, Table XXI]
        VBO       = VBO_GaAs + (VBO_GaSb - VBO_GaAs)*xmol + 1.06*xmol.*(1 - xmol);
        CBO       = VBO + Eg;
        affinity  = affinity_GaSb - CBO + CBO_GaSb;
        %
        %-- Conduction band effective mass (normalized to m0)
        meff_e    = 1./(1/meffn_GaAs + (1/meffn_GaSb - 1/meffn_GaAs)*xmol);
        %
        %-- Valence band effective mass (normalized to m0)
        meff_h_lh = 1./(1/mlh_GaAs + (1/mlh_GaSb - 1/mlh_GaAs)*xmol);
        meff_h_hh = 1./(1/mhh_GaAs + (1/mhh_GaSb - 1/mhh_GaAs)*xmol);
        meff_h    = (meff_h_lh.^(3/2) + meff_h_hh.^(3/2)).^(2/3);
        %
        tauSRH_n   = 2.5e-7*ones(size(xmol)); % electron SRH lifetime, s
        tauSRH_p   = 5e-9*ones(size(xmol));   % hole SRH lifetime, s
        EtrapSRH  = 0*ones(size(xmol));       % position of SRH trap level, eV
        %
        BRad      = 7.2e-10/100^3*ones(size(xmol)); % radiative recombination coefficient, m^3/s
        CAug_nnp  = 1e-30/100^6*ones(size(xmol));   % Auger e-e-h process coefficient, m^6/s
        CAug_ppn  = 1e-30/100^6*ones(size(xmol));   % Auger h-h-e process coefficient, m^6/s
  
    case 'GaInAs' % Gallium indium arsenide (Ga_{1-x} In_{x} As)
        
        mattype   = semiconductor*ones(size(xmol));

        %-- Incomplete ionization energies, eV
        DeltaED   = 5e-3*ones(size(xmol));  % donor energies
        DeltaEA   = 26e-3*ones(size(xmol)); % acceptor energies
        %
        %-- Relative dielectric constant
        epss      = epsr_InAs.*xmol + epsr_GaAs.*(1-xmol);
        %
        %-- Bandgap, eV, bowing for [1, eq. (4.1)], from [1, Table XIII] 
        Eg_GaAs   = Eg0_GaAs - alpha_GaAs.*T.^2./(T+beta_GaAs); % [1, eq. (2.13)]
        Eg_InAs   = Eg0_InAs - alpha_InAs.*T.^2./(T+beta_InAs); % [1, eq. (2.13)]
        Eg        = Eg_GaAs + (Eg_InAs - Eg_GaAs).*xmol - 0.477*xmol.*(1 - xmol);
        %
        %-- Valence band offset, eV, bowing for [1, eq. (4.1)] from [1, Table XIII]    
        VBO       = VBO_GaAs + (VBO_InAs - VBO_GaAs)*xmol + 0.38*xmol.*(1 - xmol);
        CBO       = VBO + Eg;
        affinity  = affinity_GaSb - CBO + CBO_GaSb;
        %
        %-- Conduction band effective mass (normalized to m0)
        meff_e    = 1./(1/meffn_GaAs + (1/meffn_InAs - 1/meffn_GaAs)*xmol + 1/0.0091*xmol.*(1 - xmol));
        %
        %-- Valence band effective mass (normalized to m0)
        meff_h_hh = 1./(1/mhh_GaAs + (1/mhh_InAs - 1/mhh_GaAs)*xmol + 1/0.145*xmol.*(1 - xmol));
        meff_h_lh = 1./(1/mlh_GaAs + (1/mlh_InAs - 1/mlh_GaAs)*xmol - 1/0.0202*xmol.*(1 - xmol));
        meff_h    = (meff_h_lh.^(3/2) + meff_h_hh.^(3/2)).^(2/3);
        %
        tauSRH_n   = 2.5e-7*ones(size(xmol)); % electron SRH lifetime, s
        tauSRH_p   = 5e-9*ones(size(xmol));   % hole SRH lifetime, s
        EtrapSRH  = 0*ones(size(xmol));       % position of SRH trap level, eV
        %
        BRad      = 7.2e-10/100^3*ones(size(xmol)); % radiative recombination coefficient, m^3/s
        CAug_nnp  = 1e-30/100^6*ones(size(xmol));   % Auger e-e-h process coefficient, m^6/s
        CAug_ppn  = 1e-30/100^6*ones(size(xmol));   % Auger h-h-e process coefficient, m^6/s

    case 'Si' % Silicon

        mattype = semiconductor*ones(size(xmol));

        %-- Incomplete ionization energies, eV
        % Shallow donor energies:
        % As: 0.054 eV
        % P: 0.045 eV
        % Sb: 0.043 eV
        DeltaEA = 43e-3*ones(size(xmol)); % Doping with Sb (eV)
        %
        % Shallow acceptor energies:
        % Al: 0.072 eV
        % B: 0.045 eV
        % Ga: 0.074 eV
        % In: 0.157 eV
        DeltaED = 45e-3*ones(size(xmol)); % Doping with B (eV)
        %
        %-- Relative dielectric constant
        epsr    = 11.7*ones(size(xmol));
        %
        %-- Bandgap, eV, from [2] 
        Eg0	    = 1.16964;
    	alpha   = 4.7300e-04;
        beta    = 6.3600e+02;
        Eg      = (Eg0 - alpha.*T.^2./(beta + T))*ones(size(xmol));
        %
        %-- Valence band offset, eV (arbitrary, not useful)
        % CBO(1) - CBO(2) = DeltaEc
        % where DeltaEc = affinity_Si - affinity_SiO2 = (4.07274038462-0.9)
        % this becomes (VBO(1) + Eg(1)) - (VBO(2) + Eg(2)) = DeltaEc
        % where VBO(1) = 0 for example, Eg(1) = 9 (that of SiO2), and so
        % on. Hence, this VBO is compatible with SiO2, see below.
        
        VBO       = (9 - Eg - (4.07274038462-0.9));
        affinity  = 4.07274038462*ones(size(xmol));
        %
        %-- Conduction band effective mass (normalized to m0)
        a         = 0.1905;
        ml        = 0.9163;
        mm        = 0.0000e+00;
        mt        = a.*Eg0./Eg;
        meff_e    = [(6*mt).^2.*ml].^(1/3) + mm;
        %
        %-- Valence band effective mass (normalized to m0)
        meffpa    = 0.443587;
        meffpb    = 0.003609528;
        meffpc    = 0.0001173515;
        meffpd    = 1.263218e-06;
        meffpe    = 3.025581e-09;
        meffpf    = 0.004683382;
        meffpg    = 0.0002286895;
        meffph    = 7.469271e-07;
        meffpi    = 1.727481e-09;
        meffpmm	  = 0;
        meff_h    = (((meffpa+meffpb.*T+meffpc.*T.^2+meffpd.*T.^3+meffpe.*T.^4)./ ...
                    (1+meffpf.*T+meffpg.*T.^2+meffph.*T.^3+meffpi.*T.^4)).^(2/3) + meffpmm)*ones(size(xmol));
    
        eMobility = 1417/1e4*ones(size(xmol));
        hMobility = 470.5/1e4*ones(size(xmol));
        
        tauSRH_n   = 1e-8*ones(size(xmol)); % electron SRH lifetime, s
        tauSRH_p   = 1e-8*ones(size(xmol)); % hole SRH lifetime, s
        EtrapSRH  = 0*ones(size(xmol));     % position of SRH trap level, eV
        
        BRad      = 7.2e-10/100^3*ones(size(xmol)); % radiative recombination coefficient, m^3/s
        CAug_nnp  = 1e-30/100^6*ones(size(xmol));   % Auger e-e-h process coefficient, m^6/s
        CAug_ppn  = 1e-30/100^6*ones(size(xmol));   % Auger h-h-e process coefficient, m^6/s
        
    case 'SiO2' % Silicon oxide
        
        mattype  = oxide*ones(size(xmol));
        
        %-- Relative dielectric constant
        epsr     = 3.9*ones(size(xmol));

        %-- Bandgap, eV, from Sentaurus
        Eg       = 9*ones(size(xmol));

        %-- Electron affinity, eV, from Sentaurus
        affinity = 0.9*ones(size(xmol));

    otherwise
        error('Unknown material')
        
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Saving the function outputs in the structure "macro"
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
macro.mattype = mattype;     % material type
%
macro.epsr      = epsr;      % relative dielectric constant
macro.meff_e    = meff_e;    % conduction band effective mass (norm. to m0)
macro.meff_h    = meff_h;    % valence band effective mass (norm. to m0)
macro.BandGap   = Eg;        % energy gap (eV)
macro.VBO       = VBO;       % valence band offset (eV)
macro.affinity  = affinity;  % electron affinity (eV)
macro.DeltaED   = DeltaED;   % donor doping ionization energy (eV)
macro.DeltaEA   = DeltaEA;   % acceptor doping ionization energy (eV)
%
macro.eMobility = eMobility; % electron mobility (m^2/(V*s))
macro.hMobility = hMobility; % hole mobility (m^2/(V*s))
%
macro.tauSRH_n  = tauSRH_n;  % electron SRH lifetime (s)
macro.tauSRH_p  = tauSRH_p;  % hole SRH lifetime (s)
macro.EtrapSRH  = EtrapSRH;  % energy position, w.r.t. midgap, of trap (eV)
macro.BRad      = BRad;      % radiative recombination rate (m^3/s)
macro.CAug_nnp  = CAug_nnp;  % eeh Auger coefficient (m^6/s)
macro.CAug_ppn  = CAug_ppn;  % hhe Auger coefficient (m^6/s)
