clear
close all
clc

en_QDD = 1;                 % Enable the quantum drift-diffusion model
obs_conv = 1;               % Observe the convergence of the electron density
obs_volt = 1;               % Observe the effect of varying voltage


s_LoadConstants % Loading physical and unit conversion constants
%% 
% Solver damping for each voltage:

gamma = [0.45,1*ones(1,10000)];
%% 
% Schottky contact parameters

qBarr_l = 0.5*eV;
veln_l = 2.573e6*cm/sec;
velp_l = 1.93e6*cm/sec;
%% GEOMETRY DESCRIPTION
% HEMT/PHEMT geometry

indLayer = 0;     % initializing layer index to 0
MeshStep = 0.1*nm;  % setting the mesh step
% 
%------------------------------------------------------------------------------------------------
indLayer                = indLayer + 1;     % increasing layer index
geom(indLayer).L        = 20*nm;          % thickness (m)
geom(indLayer).MeshStep = MeshStep;         % mesh step (m)
geom(indLayer).dopType  = 'n';              % doping type (n or p)
geom(indLayer).dopini   = 1e18*(1/cm^3);    % doping density (1/m^3)
geom(indLayer).material = 'AlGaAs';         % layer material
geom(indLayer).xmolini  = 0.20;             % x molar fraction
geom(indLayer).ymolini  = 0.0;              % y molar fraction
%------------------------------------------------------------------------------------------------
%------------------------------------------------------------------------------------------------
indLayer                = indLayer + 1;     % increasing layer index
geom(indLayer).L        = 5*nm;             % thickness (m)
geom(indLayer).MeshStep = MeshStep;         % mesh step (m)
geom(indLayer).dopType  = 'n';              % doping type (n or p)
geom(indLayer).dopini   = 1e12*(1/cm^3);    % doping density (1/m^3)
geom(indLayer).material = 'AlGaAs';         % layer material
geom(indLayer).xmolini  = 0.20;             % x molar fraction
geom(indLayer).ymolini  = 0.0;              % y molar fraction
%------------------------------------------------------------------------------------------------
%------------------------------------------------------------------------------------------------
indLayer                = indLayer + 1;     % increasing layer index
geom(indLayer).L        = 30*nm;            % thickness (m)
geom(indLayer).MeshStep = MeshStep;         % mesh step (m)
geom(indLayer).dopType  = 'n';              % doping type (n or p)
geom(indLayer).dopini   = 1e12*(1/cm^3);    % doping density (1/m^3)
geom(indLayer).material = 'AlGaAs';         % layer material
geom(indLayer).xmolini  = 0.0;              % x molar fraction
geom(indLayer).ymolini  = 0.0;              % y molar fraction
%------------------------------------------------------------------------------------------------
% indLayer                = indLayer + 1;     % increasing layer index
% geom(indLayer).L        = 12000*nm;           % thickness (m)
% geom(indLayer).MeshStep = MeshStep*20;      % mesh step (m)
% geom(indLayer).dopType  = 'n';              % doping type (n or p)
% geom(indLayer).dopini   = 1e12*(1/cm^3);    % doping density (1/m^3)
% geom(indLayer).material = 'AlGaAs';         % layer material
% geom(indLayer).xmolini  = 0.0;              % x molar fraction
% geom(indLayer).ymolini  = 0.0;              % y molar fraction
%------------------------------------------------------------------------------------------------
% MAKE THE STRUCTURE A PHEMT - COMMENT FOR A HEMT
%------------------------------------------------------------------------------------------------
% indLayer                = indLayer + 1;     % increasing layer index
% geom(indLayer).L        = 100*nm;           % thickness (m)
% geom(indLayer).MeshStep = 10*MeshStep;      % mesh step (m)
% geom(indLayer).dopType  = 'n';              % doping type (n or p)
% geom(indLayer).dopini   = 1e12*(1/cm^3);    % doping density (1/m^3)
% geom(indLayer).material = 'AlGaAs';         % layer material
% geom(indLayer).xmolini  = 0.2;              % x molar fraction
% geom(indLayer).ymolini  = 0.0;              % y molar fraction
% %------------------------------------------------------------------------------------------------
%% SOLVER SETTINGS
% Inputs, options and parameters for the solver:

dd.strName       = 'GaAs_pn';             % name of the structure under investigation
dd.Temp          = 300;                   % temperature (K)
dd.tolConv       = 1e-7;                  % tolerance of the BC solution
dd.tolphi        = 1e-7;                  % tolerance of the potential solution
dd.toln          = 1e15*(cm^(-3));        % tolerance of electron density solution
dd.tolp          = 1e15*(cm^(-3));        % tolerance of hole density solution
dd.max_diter     = 1000;                  % maximum number of iterations for Newton methods (BC, FEM)
dd.GR            = {'SRH'};               % recombination models (SRH, Rad, Auger, Opt)
dd.Rmat          = 0;                     % impedance for circuit equations (ohm*m^2)
Vvet             = [0];                   % vector of input voltages
% Vvet             = [0];                 % vector of input voltages
%% MATERIAL AND STRUCTURE PARAMETERS

dd = f_MeshMaterialPreProcessor(geom,dd);   % compute the material parameters
s_PhysicalParameters                        % compute the physical parameters
%% NEUTRALITY CONDITION
% The neutrality condition is used as initial guess for the classical Boltzmann 
% equilibrium solution

[phi,n,p] = f_Neutrality(pp);
%% 
% Neutrality values are saved

n_neutr = n;
p_neutr = p;
phi_neutr = phi;
%% CLASSICAL BOLTZMANN EQUILIBRIUM SOLUTION
% Equilibrium solver with classical Boltzmann statistics is used as an initial 
% guess for the density gradient problem solution.

disp(['--- CLASSICAL EQUILIBRIUM SOLVER ---']);
%% 
% Impose the boundary conditions on:
% 
% * Schottky boundary condition at the left contact
% * ohmic boundary condition at the right contact

Ef       = 0;
phi(1) = Ef - qBarr_l/q; % schottky contact condition
phi(end) = phi_neutr(end); % ohmic contact condition
%% 
% Variables for Newton method ending conditions

fail_cnt_eq = 0;
dphi = ones(size(phi))*Inf;
%% 
% Discretization of Poisson's equation

E = sparse(pp.N,pp.N);
E(1,1) = 1;
E(pp.N,pp.N) = 1;
for i=2:(pp.N-1)
    E(i,i-1) = pp.eps(i-1)/pp.le(i-1);
    E(i,i) = -pp.eps(i-1)/pp.le(i-1)-pp.eps(i)/pp.le(i);
    E(i,i+1) = pp.eps(i)/pp.le(i);
end
%% 
% Newton's loop

while ((fail_cnt_eq < dd.max_diter)&&(norm(dphi) > dd.tolConv))
    fail_cnt_eq = fail_cnt_eq + 1;
    
    % known term
    n        = sparse(pp.Nc.*exp((q*phi-pp.cbo+Ef)./(kB*pp.T)));
    p        = sparse(pp.Nv.*exp((-q*phi-pp.Eg(1)+pp.vbo-Ef)./(kB*pp.T)));
    dn_phi   = sparse(q/(kB*pp.T)*pp.Nc.*exp((q*phi-pp.cbo+Ef)./(kB*pp.T)));
    dp_phi   = sparse(-q/(kB*pp.T)*pp.Nv.*exp((-q*phi-pp.Eg(1)+pp.vbo-Ef)./(kB*pp.T)));
    u        = sparse(pp.N,1);
    u(1,1)   = -phi(1);
    u(pp.N,1)   = -phi(end);
    for i=2:(pp.N-1)
        u(i,1) = q*(pp.dop_tot(i)+p(i)-n(i))*pp.lb(i);
    end

    % jacobian of known term
    Ju       = sparse(pp.N,pp.N);
    for i=2:(pp.N-1)
        Ju(i,i) = q*(dp_phi(i)-dn_phi(i))*pp.lb(i);
    end

    % residual
    r = E*phi+u;
    % jacobian of residual
    J = E+Ju;

    [L,U,P,Q,R] = lu(J);
    dphi = -Q*(U\(L\(P*(R\r))));
    phi = phi+dphi;
    disp(['iteration #',num2str(fail_cnt_eq),' - residual:',num2str(norm(dphi))]);
end
%% 
% Compute conduction and valence band.

Ec = -q*phi + pp.cbo;
Ev = Ec - pp.Eg;
%% 
% Check convergence of the classical equilibrium solution.

if(norm(dphi) > dd.tolConv)
    disp(['Convergence failed']);
else
    disp(['Convergence achieved in ',num2str(fail_cnt_eq),' iterations']);
end
%% 
% Save equilibrium results.

phi_eq = phi;
n_eq = n;
p_eq = p;
Ec_eq = Ec;
%% 
% Plotting results of classical Boltzmann analysis at equilibrium

figure(1);plot(pp.z,-phi+pp.cbo/q);
hold on;
figure(2);plot(pp.z,n);
hold on;

%2.95 picco
h = h*sqrt(1);
hbar = hbar*sqrt(1);

%% QUANTUM DRIFT-DIFFUSION
%%
% Define some useful indexes
ic = 2:(pp.N-1);
il = 1:(pp.N-2);
ir = 3:(pp.N  );

% save equilibrium potential
phi_eq = phi;
n_eq = n;
p_eq = p;
%% 
% The model considered for the initial guess is:
% 
% * Poisson equation
% * semiclassical Boltzmann distribution
% 
% hence the initial guesses for the quantum potentials $\Lambda_n$ and $\Lambda_p$ 
% are zero (because the model is _semiclassical_).

% add quantum guesses
lambdan = sparse(pp.N,1);
lambdap = sparse(pp.N,1);
%% 
% The solution to the quantum drift-diffusion method is reiterated for every 
% value of voltage difference across the contacts.
% 
% The result of the previous solution is used as an initial guess for the 
% new one.

for indV = 1:length(Vvet)
    V_app = Vvet(indV);
    
    disp(' ');
    disp(['--- V=',num2str(V_app),'V ---']);
%% 
% The applied voltage is shifting the Fermi levels in the metals:
% 
% $$E_{f,m}^{(left)} = E_{f,eq} - q V^{(left)}\qquad\qquad\qquadE_{f,m}^{(right)} 
% = E_{f,eq} - q V^{(right)}$$
% 
% The contact on the right is chosen as a reference.

    Ef_left = -q*V_app;
    Ef_right = 0;
%% 
% The updates to the guesses $\Delta \phi$, $\Delta n$, $\Delta p$ are set 
% to infinity as a starting value so that the loop doesn't end prematurely (the 
% loop ends when they are below the tolerance).

    fail_cnt_dd = 0;
    dphi  = ones(size(phi))*Inf;
    delec = ones(size(phi))*Inf;
    dhole = ones(size(phi))*Inf;
%% 
% The left contact is a Schottky contact, hence the boundary condition is:
% 
% 

    phi(1) = -Ef_left/q - qBarr_l/q + pp.cbo(1)/q;
%% 
% The right contact is an ohmic contact, hence the boundary condition is:
% 
% 

    phi(end) = phi_eq(end)-Ef_right/q;
%% 
% The left contact is a Schottky contact, hence it imposes a Neumann boundary 
% condition on $n$ and $p$ (no B.C. condition on the initial guess).
% 
% The right contact is an ohmic contact, hence it imposes a semiclassical 
% Boltzmann equilibrium condition on $n$ and $p$ (since the guess is already that 
% it is enough to leave it as it is.
% 
% The left contact is a Schottky contact, hence it imposes a Neumann boundary 
% condition on $\Lambda_n$ and $\Lambda_p$ (no B.C. condition on the initial guess).
% 
% The right contact is an ohmic contact, hence it imposes a semiclassical 
% electron and hole distribution:
% 
% $$\Lambda_n = 0\qquad\qquad\qquad\Lambda_p = 0$$

    lambdan(end) = 0;
    lambdap(end) = 0;
%% 
% Newton's iterative method is adopted for the solution

    while ((fail_cnt_dd < dd.max_diter/gamma(indV))&&((norm(dphi) > dd.tolphi*gamma(indV))||(norm(delec) > dd.toln*gamma(indV))||(norm(dhole) > dd.tolp*gamma(indV))))
%% 
% *POISSON EQUATION*
% 
% _*POISSON EQUATION RESIDUAL*_
% 
% The matrix $E$ is defined for the linear part of the discretized equation.

        mdg_E = zeros(pp.N,1);
        ldg_E = zeros(pp.N,1);
        udg_E = zeros(pp.N,1);
%% 
% On the left we have a Schottky contact, hence:
% 
% 

        mdg_E(1,1) = 1;
%% 
% On the right we have an ohmic contact, hence:
% 
% 

        mdg_E(pp.N,1) = 1;
%% 
% In the middle the equation is:
% 
% 

        mdg_E(2:pp.N-1,1) = -pp.eps(1:pp.N-2,1)./pp.le(1:pp.N-2,1)-pp.eps(2:pp.N-1,1)./pp.le(2:pp.N-1,1);
        ldg_E(1:pp.N-2,1) = pp.eps(1:pp.N-2,1)./pp.le(1:pp.N-2,1);
        udg_E(ir  ,1) = pp.eps(2:pp.N-1,1)./pp.le(2:pp.N-1,1);
        
        E = spdiags([ldg_E,mdg_E,udg_E],[-1,0,1],pp.N,pp.N);
%% 
% The system of equations deriving from Poisson's equation can be written 
% with matrix notation as:
% 
% $$E \: \phi + u(n,p) = 0$$

        mdg_u          = zeros(pp.N,1);
%% 
% For the known term $u$ the boundary condition on the left contact (Schottky 
% contact) is:
% 
% 
% 
% Since it is a Dirichlet condition and it was already solved when determining 
% the guess it is sufficient to set:

        mdg_u(1,1)     = -phi(1);
%% 
% For the known term $u$ the boundary condition on the right contact (ohmic 
% contact) is:
% 
% 

        mdg_u(pp.N,1)  = -phi(end);

%% 
% For the inner lines of the term $u$ the expression is:
% 
% 

        mdg_u(ic,1) = q*(pp.dop_tot(ic)+p(ic)-n(ic)).*pp.lb(ic);
        
        u = sparse(mdg_u);
%% 
% To obtain the residual of Poisson's equation compose the two parts as:
% 
% $$r_P = E \: \phi + u(n,p)$$

        % POISSON EQUATION - Residual
        r1 = E*phi + u;
%% 
% _*POISSON EQUATION JACOBIAN w.r.t *_$\phi$
% 
% When differentiating Poisson's equation with respect to $\phi$ in determining 
% the Jacobian for the problem the result is:
% 
% * Schottky boundary condition derivative for the first row:
% 
% 
% 
% * for the rows in the middle:
% 
% 
% 
% * ohmic boundary condition derivative for the last row:
% 
% 
% 
% Hence, the Jacobian $J_{P,\phi}$ of Poisson's equation with respect to 
% the electrostatic potential is equal to the matrix $E$.

        JP_phi = E;
%% 
% _*POISSON EQUATION JACOBIAN w.r.t *_$n$
% 
% When differentiating Poisson's equation with respect to $n$ in determining 
% the Jacobian for the problem the result is:
% 
% * zero first line
% * for the rows in the middle:
% 
% 
% 
% * zero last line:
% 
% Hence, the Jacobian $J_{P,n}$ of Poisson's equation with respect to the 
% electron density is implemented as:

        mdg_JP_n = zeros(pp.N,1);
        mdg_JP_n(ic,1) = -q*pp.lb(ic);
        
        JP_n = spdiags(mdg_JP_n,0,pp.N,pp.N);
%% 
% _*POISSON EQUATION JACOBIAN w.r.t *_$p$
% 
% When differentiating Poisson's equation with respect to $p$ in determining 
% the Jacobian for the problem the result is:
% 
% * zero first line
% * for the rows in the middle:
% 
% 
% 
% * zero last line:
% 
% Hence, the Jacobian $J_{P,p}$ of Poisson's equation with respect to the 
% hole density is implemented as:

        mdg_JP_p = zeros(pp.N,1);
        mdg_JP_p(ic,1) = q*pp.lb(ic);
        
        JP_p = spdiags(mdg_JP_p,0,pp.N,pp.N);
%% 
% _*POISSON EQUATION JACOBIAN w.r.t *_$\Lambda_n$
% 
% The Jacobian $J_{P,\Lambda_n}$ of Poisson's equation with respect to the 
% electron quantum potential is zero everywhere:

        JP_lambdan = sparse(pp.N,pp.N);
%% 
% _*POISSON EQUATION JACOBIAN w.r.t *_$\Lambda_p$
% 
% The Jacobian $J_{P,\Lambda_p}$ of Poisson's equation with respect to the 
% hole quantum potential is zero everywhere:

        JP_lambdap = sparse(pp.N,pp.N);
%% 
% _*JACOBIAN - POISSON EQUATION ASSEMBLY*_
% 
% The first row of the Jacobian is assembled as:
% 
% 

        if (en_QDD == 1)
            J1 = [JP_phi,JP_n,JP_p,JP_lambdan,JP_lambdap];
        else
            J1 = [JP_phi,JP_n,JP_p];
        end
%% 
% *ELECTRON CONTINUITY EQUATION*
% 
% _*ELECTRON CONTINUITY EQUATION RESIDUAL*_
% 
% Defining the Bernoulli function $\mathcal{B}$ and its derivative $\mathcal{B}^\prime$:

        B = @(x) bern(x);
        dB = @(x) dbern(x);
%% 
% Computing the equivalent potential for electrons:
% 
% 

        phieqn = phi-pp.cbo/q+kB*pp.T/q*log(pp.Nc)+lambdan;
%% 
% A matrix $F$ is defined for the linear part of the discretized expression 
% for the residual:
% 
% $$r_{C_n} = F \: n + t$$
% 
% $t$ contains instead the nonlinear terms.

        mdg_F = zeros(pp.N,1);
        ldg_F = zeros(pp.N,1);
        udg_F = zeros(pp.N,1);
%% 
% The first line of the matrix $F$ derives from the boundary condition for 
% the Schottky contact at the left:
% 
% 

        udg_F(2,1) = + pp.Dn(1)/(pp.le(1))*(B(q/(kB*pp.T)*(phieqn(2)-phieqn(1))));
        mdg_F(1,1) = - pp.Dn(1)/(pp.le(1))*(B(-q/(kB*pp.T)*(phieqn(2)-phieqn(1)))) - veln_l;
%% 
% The last line of the matrix $F$ derives from the boundary condition for 
% the ohmic contact at the right:
% 
% 

        mdg_F(pp.N,1) = 1; % ohmic contact B.C.
%% 
% The inner lines of the matrix $F$ are found as:
% 
% 

        mdg_F(ic,1) = - pp.Dn(ic)./(pp.le(ic)).*(B(-q/(kB*pp.T).*(phieqn(ir)-phieqn(ic)))) ...
                              - pp.Dn(il)./(pp.le(il)).*(B(q/(kB*pp.T).*(phieqn(ic)-phieqn(il))));
                          
        udg_F(ir,1)     =   pp.Dn(ic)./(pp.le(ic)).*(B(q/(kB*pp.T).*(phieqn(ir)-phieqn(ic))));
        
        ldg_F(il)   =   pp.Dn(il)./(pp.le(il)).*(B(-q/(kB*pp.T).*(phieqn(ic)-phieqn(il))));
        
        F = spdiags([ldg_F,mdg_F,udg_F],[-1,0,1],pp.N,pp.N);
%% 
% The nonlinear part of the residual of the electron continuity equation 
% is called $t$:

        mdg_t = zeros(pp.N,1);
%% 
% The first row of the nonlinear term $t$ is given by the Schottky boundary 
% condition at the left contact:
% 
% 

        mdg_t(1)   = - pp.lb(1)*(n(1)*p(1)-pp.ni(1)^2)/(pp.taup_SRH(1)*(n(1)+pp.nt(1))+pp.taun_SRH(1)*(p(1)+pp.pt(1))) + veln_l*n_eq(1);
%% 
% The last row of the nonlinear term $t$ is given by the ohmic boundary 
% condition at the right contact:
% 
% 

        mdg_t(end) = -n(end);
%% 
% The inner rows of the nonlinear term $t$ are given by the expression:
% 
% 

        mdg_t(ic) = - pp.lb(ic).*(n(ic).*p(ic)-pp.ni(ic).^2)./(pp.taup_SRH(ic).*(n(ic)+pp.nt(ic))+pp.taun_SRH(ic).*(p(ic)+pp.pt(ic)));
        
        t = sparse(mdg_t);
%% 
% _ *ELECTRON CONTINUITY EQUATION - RESIDUAL ASSEMBLY*_

        r2 = F*n + t;
%% 
% _*ELECTRON CONTINUITY EQUATION JACOBIAN w.r.t *_$\phi$

        mdg_JCn_phi = zeros(pp.N,1);
        ldg_JCn_phi = zeros(pp.N,1);
        udg_JCn_phi = zeros(pp.N,1);
%% 
% The first row of this jacobian is found from the Shottky boundary condition 
% at the left contact:
% 
% 

        mdg_JCn_phi(1,1) = - n(1)*pp.Dn(1)/(pp.le(1))*(dB(-q/(kB*pp.T)*(phieqn(2)-phieqn(1))))*q/(kB*pp.T) ...
            - n(2)*pp.Dn(1)/(pp.le(1))*(dB(q/(kB*pp.T)*(phieqn(2)-phieqn(1))))*q/(kB*pp.T);
%% 
% 

        udg_JCn_phi(2,1) = + n(2)*pp.Dn(1)/(pp.le(1))*(dB(q/(kB*pp.T)*(phieqn(2)-phieqn(1))))*q/(kB*pp.T) ...
             + n(1)*pp.Dn(1)/(pp.le(1))*(dB(-q/(kB*pp.T)*(phieqn(2)-phieqn(1))))*q/(kB*pp.T);
%% 
% The last row of this jacobian is found from the ohmic boundary condition 
% at the right contact and is zero:

        mdg_JCn_phi(pp.N,1) = 0;
%% 
% The inner rows of this jacobian are found from the expressions:
% 
% 

        mdg_JCn_phi(ic,1) = - n(ic).*pp.Dn(ic)./(pp.le(ic)).*(dB(-q/(kB*pp.T).*(phieqn(ir)-phieqn(ic))))*q/(kB*pp.T)  ...
                                    - n(ic).*pp.Dn(il)./(pp.le(il)).*(dB(q/(kB*pp.T).*(phieqn(ic)-phieqn(il))))*q/(kB*pp.T) ...
                                    - n(ir).*pp.Dn(ic)./(pp.le(ic)).*(dB(q/(kB*pp.T).*(phieqn(ir)-phieqn(ic))))*q/(kB*pp.T)         ...
                                    - n(il).*pp.Dn(il)./(pp.le(il)).*(dB(-q/(kB*pp.T).*(phieqn(ic)-phieqn(il))))*q/(kB*pp.T);
%% 
% 

        ldg_JCn_phi(1:pp.N-2)     = + n(ic).*pp.Dn(il)./(pp.le(il)).*(dB(q/(kB*pp.T).*(phieqn(ic)-phieqn(il))))*q/(kB*pp.T) ...
                                    + n(il).*pp.Dn(il)./(pp.le(il)).*(dB(-q/(kB*pp.T).*(phieqn(ic)-phieqn(il))))*q/(kB*pp.T);
%% 
% 

        udg_JCn_phi(ir)       = + n(ir).*pp.Dn(ic)./(pp.le(ic)).*(dB(q/(kB*pp.T).*(phieqn(ir)-phieqn(ic))))*q/(kB*pp.T) ...
                                    + n(ic).*pp.Dn(ic)./(pp.le(ic)).*(dB(-q/(kB*pp.T).*(phieqn(ir)-phieqn(ic))))*q/(kB*pp.T);
        
        JCn_phi = spdiags([ldg_JCn_phi,mdg_JCn_phi,udg_JCn_phi],[-1,0,1],pp.N,pp.N);
%% 
% _*ELECTRON CONTINUITY EQUATION JACOBIAN w.r.t *_$n$
% 
% Differentiating the linear part of the problem $r_{C_n} = F \: n + t$ gives 
% a contribution which is identical to the matrix $F$, hence:

        JCn_n = F;
%% 
% Then, according to the Schottky boundary conditions, the Jacobian of the 
% nonlinear term must be added:
% 
% 

        JCn_n(1,1) = JCn_n(1,1) - pp.lb(1)*( p(1)*( pp.taup_SRH(1)*(n(1)+pp.nt(1)) + pp.taun_SRH(1)*(p(1)+pp.pt(1)) ) - pp.taup_SRH(1)*( n(1)*p(1) - pp.ni(1)^2 ) )/( ( pp.taup_SRH(1)*(n(1)+pp.nt(1)) + pp.taun_SRH(1)*(p(1)+pp.pt(1)) )^2 ); % schottky contact
%% 
% According to the ohmic boundary conditions instead no term needs to be 
% added to the matrix:
% 
% 
% 
% For the inner lines of this jacobian the expression is instead:
% 
% 
% 
% No terms need to be added to the j-1 and j+1 entries:
% 
%                     

        mdg_JCn_n = zeros(pp.N,1);
        mdg_JCn_n(ic,1) = - pp.lb(ic).*( p(ic).*( pp.taup_SRH(ic).*(n(ic)+pp.nt(ic)) + pp.taun_SRH(ic).*(p(ic)+pp.pt(ic)) ) - pp.taup_SRH(ic).*( n(ic).*p(ic) - pp.ni(ic).^2 ) )./( ( pp.taup_SRH(ic).*(n(ic)+pp.nt(ic)) + pp.taun_SRH(ic).*(p(ic)+pp.pt(ic)) ).^2 );
        
        JCn_n = JCn_n + spdiags(mdg_JCn_n,0,pp.N,pp.N);
%% 
% _*ELECTRON CONTINUITY EQUATION JACOBIAN w.r.t *_$p$

        mdg_JCn_p = zeros(pp.N,pp.N);
%% 
% For a Schottky contact on the left side the expression is:
% 
% 

        mdg_JCn_p(1,1) = - pp.lb(1)*( n(1)*( pp.taup_SRH(1)*(n(1)+pp.nt(1)) + pp.taun_SRH(1)*(p(1)+pp.pt(1)) ) - pp.taun_SRH(1)*( n(1)*p(1) - pp.ni(1)^2 ) )/( ( pp.taup_SRH(1)*(n(1)+pp.nt(1)) + pp.taun_SRH(1)*(p(1)+pp.pt(1)) )^2 ); % schottky contact b.c.
%% 
% For an ohmic contact on the right side no contribution needs to be added:

        mdg_JCn_p(pp.N,1) = 0;
%% 
% For the inner lines the expression is:
% 
% 

        mdg_JCn_p(ic,1) = - pp.lb(ic,1).*( n(ic,1).*( pp.taup_SRH(ic,1).*(n(ic,1)+pp.nt(ic,1)) + pp.taun_SRH(ic,1).*(p(ic,1)+pp.pt(ic,1)) ) - pp.taun_SRH(ic,1).*( n(ic,1).*p(ic,1) - pp.ni(ic,1).^2 ) )./( ( pp.taup_SRH(ic,1).*(n(ic,1)+pp.nt(ic,1)) + pp.taun_SRH(ic,1).*(p(ic,1)+pp.pt(ic,1)) ).^2 );
        
        JCn_p = spdiags(mdg_JCn_p,0,pp.N,pp.N);
%% 
% _*ELECTRON CONTINUITY EQUATION JACOBIAN w.r.t *_$\Lambda_n$ !!!!! PROBLEMATIC 
% !!!!!

        mdg_JCn_lambdan = zeros(pp.N,1);
        ldg_JCn_lambdan = zeros(pp.N,1);
        udg_JCn_lambdan = zeros(pp.N,1);
%% 
% For a Schottky boundary condition at the left side:
% 
% 

        mdg_JCn_lambdan(1,1) = - n(1)*pp.Dn(1)/(pp.le(1))*(dB(-q/(kB*pp.T)*(phieqn(2)-phieqn(1))))*q/(kB*pp.T) ...
                               - n(2)*pp.Dn(1)/(pp.le(1))*(dB(q/(kB*pp.T)*(phieqn(2)-phieqn(1))))*q/(kB*pp.T);
%% 
% 

        udg_JCn_lambdan(2,1) = + n(2)*pp.Dn(1)/(pp.le(1))*(dB(q/(kB*pp.T)*(phieqn(2)-phieqn(1))))*q/(kB*pp.T) ...
                               + n(1)*pp.Dn(1)/(pp.le(1))*(dB(-q/(kB*pp.T)*(phieqn(2)-phieqn(1))))*q/(kB*pp.T);
%% 
% For Ohmic boundary conditions at the right side the last row of this Jacobian 
% is zero.

        mdg_JCn_lambdan(pp.N,1) = 0;
%% 
% For the inner rows of the Jacobian:
% 
% 

        mdg_JCn_lambdan(ic,1) = - n(ic).*pp.Dn(ic)./(pp.le(ic)).*(dB(-q./(kB.*pp.T).*(phieqn(ir)-phieqn(ic)))).*q./(kB*pp.T)     ...
                                        - n(ic).*pp.Dn(il)./(pp.le(il)).*(dB(q./(kB.*pp.T).*(phieqn(ic)-phieqn(il)))).*q./(kB.*pp.T) ...
                                        - n(ir).*pp.Dn(ic)./(pp.le(ic)).*(dB(q./(kB.*pp.T).*(phieqn(ir)-phieqn(ic)))).*q./(kB.*pp.T)         ...
                                        - n(il).*pp.Dn(il)./(pp.le(il)).*(dB(-q./(kB.*pp.T).*(phieqn(ic)-phieqn(il)))).*q./(kB.*pp.T);
%% 
% 

        ldg_JCn_lambdan(il,1) = + n(ic).*pp.Dn(il)./(pp.le(il)).*(dB(q./(kB.*pp.T).*(phieqn(ic)-phieqn(il)))).*q./(kB.*pp.T) ...
                                        + n(il).*pp.Dn(il)./(pp.le(il)).*(dB(-q./(kB.*pp.T).*(phieqn(ic)-phieqn(il)))).*q./(kB.*pp.T);
%% 
% 

        udg_JCn_lambdan(ir,1) = + n(ir).*pp.Dn(ic)./(pp.le(ic)).*(dB(q./(kB.*pp.T).*(phieqn(ir)-phieqn(ic)))).*q./(kB.*pp.T) ...
                                    + n(ic).*pp.Dn(ic)./(pp.le(ic)).*(dB(-q./(kB.*pp.T).*(phieqn(ir)-phieqn(ic)))).*q./(kB.*pp.T);
        
        JCn_lambdan = spdiags([ldg_JCn_lambdan,mdg_JCn_lambdan,udg_JCn_lambdan],[-1,0,1],pp.N,pp.N);
%% 
% _*ELECTRON CONTINUITY EQUATION JACOBIAN w.r.t *_$\Lambda_p$
% 
% This Jacobian is zero.

        JCn_lambdap = sparse(pp.N,pp.N);
%% 
% _*JACOBIAN - ELECTRON CONTINUITY EQUATION ASSEMBLY*_
% 
% Assemble the Jacobian of the electron continuity equation as:
% 
% 

        if (en_QDD == 1)
            J2 = [JCn_phi,JCn_n,JCn_p,JCn_lambdan,JCn_lambdap];
        else
            J2 = [JCn_phi,JCn_n,JCn_p];
        end
%% 
% *HOLE CONTINUITY EQUATION*
% 
% _*HOLE CONTINUITY EQUATION RESIDUAL*_
% 
% Defining the Bernoulli function $\mathcal{B}$ and its derivative $\mathcal{B}^\prime$:

        B = @(x) bern(x);
        dB = @(x) dbern(x);
%% 
% Computing the equivalent electrostatic potential for holes:
% 
% 

        phieqp = phi-pp.vbo/q-kB*pp.T/q*log(pp.Nv)+pp.Eg(1)/q-lambdap;
%% 
% A matrix $G$ is defined for the linear part of the discretized expression 
% for the residual:
% 
% $$r_{C_p} = G \: p + s$$
% 
% $s$ contains instead the nonlinear terms.

        mdg_G = zeros(pp.N,1);
        udg_G = zeros(pp.N,1);
        ldg_G = zeros(pp.N,1);
%% 
% The first line of the matrix $G$ is coming from the Schottky boundary 
% condition at the left side:
% 
% 

        mdg_G(1,1) = + pp.Dp(1)/(pp.le(2))*(B(q/(kB*pp.T)*(phieqp(2)-phieqp(1)))) + velp_l;
        udg_G(2,1) = - pp.Dp(1)/(pp.le(1))*(B(-q/(kB*pp.T)*(phieqp(2)-phieqp(1))));
%% 
% The last line of the matrix $G$ is coming from the ohmic boundary condition 
% at the right side:
% 
% 

        mdg_G(pp.N,1) = 1;
%% 
% The inner lines of the matrix $G$ are coming from the equation:
% 
% 

        mdg_G(ic,1) = + pp.Dp(ic)./(pp.le(ic)).*(B(q./(kB.*pp.T).*(phieqp(ir)-phieqp(ic)))) ...
                              + pp.Dp(il)./(pp.le(il)).*(B(-q./(kB.*pp.T).*(phieqp(ic)-phieqp(il))));
                          
        udg_G(ir,1)     = - pp.Dp(ic)./(pp.le(ic)).*(B(-q./(kB.*pp.T).*(phieqp(ir)-phieqp(ic))));
        
        ldg_G(il)   = - pp.Dp(il)./(pp.le(il)).*(B(q./(kB.*pp.T).*(phieqp(ic)-phieqp(il))));
        
        G = spdiags([ldg_G,mdg_G,udg_G],[-1,0,1],pp.N,pp.N);
%% 
% The first entry of the nonlinear term $s$ is coming from the Schottky 
% boundary condition at the left contact:
% 
% 

        mdg_s = zeros(pp.N,1);
        mdg_s(1,1) = pp.lb(1)*(n(1)*p(1)-pp.ni(1)^2)/(pp.taup_SRH(1)*(n(1)+pp.nt(1))+pp.taun_SRH(1)*(p(1)+pp.pt(1))) - velp_l*p_eq(1); % schottky contact B.C.
%% 
% The last entry of the nonlinear term $s$ is coming from the ohmic condition 
% at the right contact:
% 
% 

        mdg_s(pp.N,1) = -p(pp.N);
%% 
% The inner entries of the nonlinear term $s$ are coming from the equation:
% 
% 

        mdg_s(ic,1) = pp.lb(ic).*(n(ic).*p(ic)-pp.ni(ic).^2)./(pp.taup_SRH(ic).*(n(ic)+pp.nt(ic))+pp.taun_SRH(ic).*(p(ic)+pp.pt(ic)));

        s = sparse(mdg_s);
%% 
% Assembling the residual:

        r3 = G*p + s;
%% 
% _*HOLE CONTINUITY EQUATION JACOBIAN w.r.t *_$\phi$

        mdg_JCp_phi = zeros(pp.N,1);
        ldg_JCp_phi = zeros(pp.N,1);
        udg_JCp_phi = zeros(pp.N,1);
%% 
% The first row of this Jacobian is given by the Schottky boundary condition 
% at the left contact:
% 
% 

        mdg_JCp_phi(1,1) = - p(1)*pp.Dp(1)/(pp.le(1))*(dB(q/(kB*pp.T)*(phieqp(2)-phieqp(1))))*q/(kB*pp.T) ...
                           - p(2)*pp.Dp(1)/(pp.le(1))*(dB(-q/(kB*pp.T)*(phieqp(2)-phieqp(1))))*q/(kB*pp.T);
%% 
% 

        udg_JCp_phi(2,1) = + p(2)*pp.Dp(1)/(pp.le(1))*(dB(-q/(kB*pp.T)*(phieqp(2)-phieqp(1))))*q/(kB*pp.T) ...
                           + p(1)*pp.Dp(1)/(pp.le(1))*(dB(q/(kB*pp.T)*(phieqp(2)-phieqp(1))))*q/(kB*pp.T);
%% 
% An ohmic boundary condition at the right contact does not impose any condition:

        mdg_JCp_phi(pp.N,1) = 0;
%% 
% The inner lines of this Jacobian are found from the expressions:
% 
% 

        mdg_JCp_phi(ic,1) = - p(ic).*pp.Dp(ic)./(pp.le(ic)).*(dB(q./(kB.*pp.T).*(phieqp(ir)-phieqp(ic)))).*q./(kB.*pp.T) ...
                                    - p(ic).*pp.Dp(il)./(pp.le(il)).*(dB(-q./(kB.*pp.T).*(phieqp(ic)-phieqp(il)))).*q./(kB.*pp.T) ...
                                    - p(ir).*pp.Dp(ic)./(pp.le(ic)).*(dB(-q./(kB.*pp.T).*(phieqp(ir)-phieqp(ic)))).*q/(kB.*pp.T) ...
                                    - p(il).*pp.Dp(il)./(pp.le(il)).*(dB(q./(kB.*pp.T).*(phieqp(ic)-phieqp(il)))).*q./(kB.*pp.T);
%% 
% 

        ldg_JCp_phi(il,1) = + p(ic).*pp.Dp(il)./(pp.le(il)).*(dB(-q./(kB.*pp.T).*(phieqp(ic)-phieqp(il)))).*q./(kB.*pp.T) ...
                                    + p(il).*pp.Dp(il)./(pp.le(il)).*(dB(+q./(kB.*pp.T).*(phieqp(ic)-phieqp(il)))).*q./(kB.*pp.T);
%% 
% 

        udg_JCp_phi(ir,1) = + p(ir).*pp.Dp(ic)./(pp.le(ic)).*(dB(-q./(kB.*pp.T).*(phieqp(ir)-phieqp(ic)))).*q./(kB.*pp.T) ...
                                + p(ic).*pp.Dp(ic)./(pp.le(ic)).*(dB(q./(kB.*pp.T).*(phieqp(ir)-phieqp(ic)))).*q./(kB.*pp.T);
                            
        JCp_phi = spdiags([ldg_JCp_phi,mdg_JCp_phi,udg_JCp_phi],[-1,0,1],pp.N,pp.N);
%% 
% _*HOLE CONTINUITY EQUATION JACOBIAN w.r.t *_$p$
% 
% The first contribution to this Jacobian is given by the matrix $G$:

        JCp_p = G;
        mdg_JCp_p = zeros(pp.N,1);
%% 
% Then by the Schottky boundary condition at the left contact:
% 
% 

        mdg_JCp_p(1,1) = + pp.lb(1)*( n(1)*( pp.taup_SRH(1)*(n(1)+pp.nt(1)) + pp.taun_SRH(1)*(p(1)+pp.pt(1)) ) - pp.taun_SRH(1)*( n(1)*p(1) - pp.ni(1)^2 ) )/( ( pp.taup_SRH(1)*(n(1)+pp.nt(1)) + pp.taun_SRH(1)*(p(1)+pp.pt(1)) )^2 );
%% 
% By the ohmic boundary condition at the right contact:
% 
% 
% 
% so no terms need to be added:

        mdg_JCp_p(pp.N,1) = 0;
%% 
% For the inner rows of this Jacobian the equation to be considered is:
% 
% 

        mdg_JCp_p(ic,1) = pp.lb(ic).*( n(ic).*( pp.taup_SRH(ic).*(n(ic)+pp.nt(ic)) + pp.taun_SRH(ic).*(p(ic)+pp.pt(ic)) ) - pp.taun_SRH(ic).*( n(ic).*p(ic) - pp.ni(ic).^2 ) )./( ( pp.taup_SRH(ic).*(n(ic)+pp.nt(ic)) + pp.taun_SRH(ic).*(p(ic)+pp.pt(ic)) ).^2 );
%% 
% while no terms need to be added for the other equations:
% 
%                   

        JCp_p = JCp_p + spdiags(mdg_JCp_p,0,pp.N,pp.N);
%% 
% _*HOLE CONTINUITY EQUATION JACOBIAN w.r.t *_$n$

        mdg_JCp_n = zeros(pp.N,1);
%% 
% The first line is coming from the Schottky boundary condition at the left 
% side:
% 
% 

        mdg_JCp_n(1,1) = + pp.lb(1)*( p(1)*( pp.taup_SRH(1)*(n(1)+pp.nt(1)) + pp.taun_SRH(1)*(p(1)+pp.pt(1)) ) - pp.taup_SRH(1)*( n(1)*p(1) - pp.ni(1)^2 ) )/( ( pp.taup_SRH(1)*(n(1)+pp.nt(1)) + pp.taun_SRH(1)*(p(1)+pp.pt(1)) )^2 );
%% 
% An ohmic boundary condition at the right side does not yield any contribution 
% to this Jacobian:

        mdg_JCp_n(pp.N,1) = 0;
%% 
% The inner lines are coming from the equation:
% 
% 

        mdg_JCp_n(ic,1) = + pp.lb(ic).*( p(ic).*( pp.taup_SRH(ic).*(n(ic)+pp.nt(ic)) + pp.taun_SRH(ic).*(p(ic)+pp.pt(ic)) ) - pp.taup_SRH(ic).*( n(ic).*p(ic) - pp.ni(ic).^2 ) )./( ( pp.taup_SRH(ic).*(n(ic)+pp.nt(ic)) + pp.taun_SRH(ic).*(p(ic)+pp.pt(ic)) ).^2 );
        
        JCp_n = spdiags(mdg_JCp_n,0,pp.N,pp.N);
%% 
% _*HOLE CONTINUITY EQUATION JACOBIAN w.r.t *_$\lambda_n$
% 
% This Jacobian is all zero:

        JCp_lambdan = sparse(pp.N,pp.N);
%% 
% _*HOLE CONTINUITY EQUATION JACOBIAN w.r.t *_$\lambda_p$

        mdg_JCp_lambdap = zeros(pp.N,1);
        udg_JCp_lambdap = zeros(pp.N,1);
        ldg_JCp_lambdap = zeros(pp.N,1);
%% 
% The first line of this Jacobian is coming from the Schottky boundary condition 
% at the left contact:
% 
% 

        mdg_JCp_lambdap(1,1) = + p(1)*pp.Dp(1)/(pp.le(1))*(dB(q/(kB*pp.T)*(phieqp(2)-phieqp(1))))*q/(kB*pp.T) ...
                               + p(2)*pp.Dp(1)/(pp.le(1))*(dB(-q/(kB*pp.T)*(phieqp(2)-phieqp(1))))*q/(kB*pp.T);
%% 
% 

        udg_JCp_lambdap(2,1) = - p(2)*pp.Dp(1)/(pp.le(1))*(dB(-q/(kB*pp.T)*(phieqp(2)-phieqp(1))))*q/(kB*pp.T) ...
                               - p(1)*pp.Dp(1)/(pp.le(1))*(dB(q/(kB*pp.T)*(phieqp(2)-phieqp(1))))*q/(kB*pp.T);
%% 
% The last line of this Jacobian is coming from the ohmic boundary condition 
% at the right contact:

        mdg_JCp_lambdap(pp.N,1) = 0;
%% 
% The inner lines of this Jacobian are given by the expression:
% 
% 

        mdg_JCp_lambdap(ic,1) = + p(ic).*pp.Dp(ic)./(pp.le(ic)).*(dB(q./(kB.*pp.T).*(phieqp(ir)-phieqp(ic)))).*q./(kB.*pp.T) ...
                                        + p(ic).*pp.Dp(il)./(pp.le(il)).*(dB(-q./(kB.*pp.T).*(phieqp(ic)-phieqp(il)))).*q./(kB.*pp.T) ...
                                        + p(ir).*pp.Dp(ic)./(pp.le(ic)).*(dB(-q./(kB.*pp.T).*(phieqp(ir)-phieqp(ic)))).*q./(kB.*pp.T) ...
                                        + p(il).*pp.Dp(il)./(pp.le(il)).*(dB(q./(kB.*pp.T).*(phieqp(ic)-phieqp(il)))).*q./(kB.*pp.T);
%% 
% 

        ldg_JCp_lambdap(il,1) = - p(ic).*pp.Dp(il)./(pp.le(il)).*(dB(-q./(kB.*pp.T).*(phieqp(ic)-phieqp(il)))).*q./(kB.*pp.T) ...
                                        - p(il).*pp.Dp(il)./(pp.le(il)).*(dB(+q./(kB.*pp.T).*(phieqp(ic)-phieqp(il)))).*q./(kB.*pp.T);
%% 
% 

        udg_JCp_lambdap(ir,1) = - p(ir).*pp.Dp(ic)./(pp.le(ic)).*(dB(-q./(kB.*pp.T).*(phieqp(ir)-phieqp(ic)))).*q./(kB.*pp.T) ...
                                      - p(ic).*pp.Dp(ic)./(pp.le(ic)).*(dB(q./(kB*pp.T).*(phieqp(ir)-phieqp(ic)))).*q./(kB.*pp.T);
        
        JCp_lambdap = spdiags([ldg_JCp_lambdap,mdg_JCp_lambdap,udg_JCp_lambdap],[-1,0,1],pp.N,pp.N);
%% 
% _*JACOBIAN - HOLE CONTINUITY EQUATION ASSEMBLY*_
% 
% Assemble the Jacobian of the electron continuity equation as:
% 
% 

        if (en_QDD == 1)
            J3 = [JCp_phi,JCp_n,JCp_p,JCp_lambdan,JCp_lambdap];
        else
            J3 = [JCp_phi,JCp_n,JCp_p];
        end
%% 
% *ELECTRON DENSITY GRADIENT EQUATION*
% 
% _*ELECTRON DENSITY GRADIENT EQUATION RESIDUAL*_

        mdg_r4 = zeros(pp.N,1);
%% 
% The first residual is given by the Schottky contact at the left boundary:
% 
% 

        mdg_r4(1,1) = sqrt(n(1))*lambdan(1)*pp.lb(1)-hbar^2/(6*q)*( 1/pp.meffn(1)*(sqrt(n(2))-sqrt(n(1)))/pp.le(1) );
%% 
% The last residual is given by the ohmic contact at the right boundary:
% 
% 

        mdg_r4(pp.N,1) = 0;
%% 
% The inner residuals are given by the equation:
% 
% 

        mdg_r4(ic,1) = sqrt(n(ic)).*lambdan(ic).*pp.lb(ic)-hbar.^2./(6.*q).*( 1./pp.meffn(ic).*(sqrt(n(ir))-sqrt(n(ic)))./pp.le(ic) - 1./pp.meffn(il).*(sqrt(n(ic))-sqrt(n(il)))./pp.le(il) );

        r4 = sparse(mdg_r4);
%% 
% _*ELECTRON DENSITY GRADIENT EQUATION JACOBIAN w.r.t *_$n$

        mdg_J4_n = zeros(pp.N,1);
        udg_J4_n = zeros(pp.N,1);
        ldg_J4_n = zeros(pp.N,1);
%% 
% The first line of this Jacobian is given by the Schottky boundary condition 
% at the left contact:
% 
% 

        mdg_J4_n(1,1) = 1/(2*sqrt(n(1)))*lambdan(1)*pp.lb(1)+hbar^2/(6*q)*( 1/(pp.meffn(1)*2*sqrt(n(1))*pp.le(1)) );
%% 
% 

        udg_J4_n(2,1) = -hbar^2/(6*q)*( 1/(pp.meffn(1)*2*sqrt(n(2))*pp.le(1)) );
%% 
% For the inner rows the equations are:
% 
% 

        mdg_J4_n(ic,1) = 1./(2.*sqrt(n(ic))).*lambdan(ic).*pp.lb(ic)+hbar.^2./(6.*q).*( 1./(pp.meffn(ic).*2.*sqrt(n(ic)).*pp.le(ic)) + 1./(pp.meffn(il).*2.*sqrt(n(ic)).*pp.le(il)) );
%% 
% 

        ldg_J4_n(il,1) = -hbar.^2./(6.*q).*( 1./(pp.meffn(il).*2.*sqrt(n(il)).*pp.le(il)) );
%% 
% 

        udg_J4_n(ir,1) = -hbar.^2./(6.*q).*( 1./(pp.meffn(ic).*2.*sqrt(n(ir)).*pp.le(ic)) );
        
        J4_n = spdiags([ldg_J4_n,mdg_J4_n,udg_J4_n],[-1,0,1],pp.N,pp.N);
%% 
% _*ELECTRON DENSITY GRADIENT EQUATION JACOBIAN w.r.t *_$p$
% 
% This Jacobian is zero

        J4_p = sparse(pp.N,pp.N);
%% 
% _*ELECTRON DENSITY GRADIENT EQUATION JACOBIAN w.r.t *_$\phi$
% 
% This Jacobian is zero

        J4_phi = sparse(pp.N,pp.N);
%% 
% _*ELECTRON DENSITY GRADIENT EQUATION JACOBIAN w.r.t *_$\Lambda_n$

        mdg_J4_lambdan = zeros(pp.N,1);
%% 
% The first line of this matrix is given by the Schottky boundary condition 
% at the left contact:
% 
% 

        mdg_J4_lambdan(1,1) = sqrt(n(1))*pp.lb(1);
%% 
% The last line of this matrix is given by the ohmic boundary condition 
% at the right contact:
% 
% 

        mdg_J4_lambdan(pp.N,1) = 1;
%% 
% For the inner lines the equation is instead:
% 
% 

        mdg_J4_lambdan(ic,1) = sqrt(n(ic)).*pp.lb(ic);
        J4_lambdan = spdiags(mdg_J4_lambdan,0,pp.N,pp.N);
%% 
% _*ELECTRON DENSITY GRADIENT EQUATION JACOBIAN w.r.t *_$\Lambda_p$
% 
% This Jacobian is zero:

        J4_lambdap = sparse(pp.N,pp.N);        
%% 
% _*JACOBIAN - ELECTRON DENSITY EQUATION ASSEMBLY*_
% 
% Jacobian assembled according to the equation:
% 
% 

        J4 = [J4_phi,J4_n,J4_p,J4_lambdan,J4_lambdap];
%% 
% *HOLE DENSITY GRADIENT EQUATION*
% 
% _*HOLE DENSITY GRADIENT EQUATION RESIDUAL*_

        mdg_r5 = zeros(pp.N,1);
%% 
% The first residual is given by the Schottky boundary condition at the 
% left contact:
% 
% 

        mdg_r5(1) = sqrt(p(1))*lambdap(1)*pp.lb(1)-hbar^2/(6*q)*( 1/pp.meffp(1)*(sqrt(p(2))-sqrt(p(1)))/pp.le(1) );
%% 
% The last residual is zero by the ohmic boundary condition at the right 
% contact:

        mdg_r5(pp.N) = 0;
%% 
% The inner residuals are given by the expression:
% 
% 

        mdg_r5(ic) = sqrt(p(ic)).*lambdap(ic).*pp.lb(ic)-hbar.^2./(6.*q).*( 1./pp.meffp(ic).*(sqrt(p(ir))-sqrt(p(ic)))./pp.le(ic) - 1./pp.meffp(il).*(sqrt(p(ic))-sqrt(p(il)))./pp.le(il) );
        
        r5 = sparse(mdg_r5);
%% 
% _*HOLE DENSITY GRADIENT EQUATION JACOBIAN w.r.t *_$p$

        mdg_J5_p = zeros(pp.N,1);
        udg_J5_p = zeros(pp.N,1);
        ldg_J5_p = zeros(pp.N,1);
%% 
% The first row is given by the Schottky boundary condition at the left 
% contact:
% 
% 

        mdg_J5_p(1,1) = 1/(2*sqrt(p(1)))*lambdap(1)*pp.lb(1)+hbar^2/(6*q)*( 1/(pp.meffp(1)*2*sqrt(p(1))*pp.le(1)) );
%% 
% 

        udg_J5_p(2,1) = -hbar^2/(6*q)*( 1/(pp.meffp(1)*2*sqrt(p(2))*pp.le(1)) );
%% 
% The inner rows are given by the expressions:
% 
% 

        mdg_J5_p(ic,1) = 1./(2.*sqrt(p(ic))).*lambdap(ic).*pp.lb(ic)+hbar.^2./(6.*q).*( 1./(pp.meffp(ic).*2.*sqrt(p(ic)).*pp.le(ic)) + 1./(pp.meffp(il).*2.*sqrt(p(ic)).*pp.le(il)) );
%% 
% 

        ldg_J5_p(il,1) = -hbar.^2./(6.*q).*( 1./(pp.meffp(il).*2.*sqrt(p(il)).*pp.le(il)) );
%% 
% 

        udg_J5_p(ir,1) = -hbar.^2./(6.*q).*( 1./(pp.meffp(ic).*2.*sqrt(p(ir)).*pp.le(ic)) );
        
        J5_p = spdiags([ldg_J5_p,mdg_J5_p,udg_J5_p],[-1,0,1],pp.N,pp.N);
%% 
% _*HOLE DENSITY GRADIENT EQUATION JACOBIAN w.r.t *_$n$
% 
% This Jacobian is zero:

        J5_n = sparse(pp.N,pp.N);
%% 
% _*HOLE DENSITY GRADIENT EQUATION JACOBIAN w.r.t *_$\phi$
% 
% This Jacobian is zero:

        J5_phi = sparse(pp.N,pp.N);
%% 
% _*HOLE DENSITY GRADIENT EQUATION JACOBIAN w.r.t *_$\Lambda_p$

        mdg_J5_lambdap = zeros(pp.N,1);
%% 
% The first row is given by the Schottky boundary condition at the left 
% contact:
% 
% 

        mdg_J5_lambdap(1,1) = sqrt(p(1))*pp.lb(1);
%% 
% The last row is given by the ohmic boundary condition at the right contact:
% 
% 

        mdg_J5_lambdap(pp.N,1) = 1;
%% 
% The inner rows are given by the equation:
% 
% 

        mdg_J5_lambdap(ic,1) = sqrt(p(ic)).*pp.lb(ic);
        
        J5_lambdap = spdiags(mdg_J5_lambdap,0,pp.N,pp.N);
%% 
% _*HOLE DENSITY GRADIENT EQUATION JACOBIAN w.r.t *_$\Lambda_n$
% 
% This Jacobian is zero

        J5_lambdan = sparse(pp.N,pp.N);        
%% 
% _*JACOBIAN - HOLE DENSITY EQUATION ASSEMBLY*_
% 
% Jacobian assembled according to the equation:
% 
% 

        J5 = [J5_phi,J5_n,J5_p,J5_lambdan,J5_lambdap];        
%% 
% *APPLYING NEWTON'S ITERATION*

        %--- ASSEMBLE JACOBIAN AND RESIDUAL ---%
        if (en_QDD == 1)
            r = [r1;r2;r3;r4;r5];
            J = [J1;J2;J3;J4;J5];
        else
            r = [r1;r2;r3];
            J = [J1;J2;J3];
        end

        % Solving the linearized system (is giving problems)
        [U,C] = dgsequ(J);
        res = norm(U*r);
        J = U*J*C;
        Delta_u = -C*(J\(U*r));

        dphi  = gamma(indV)*Delta_u(1:pp.N);
        delec = gamma(indV)*Delta_u(pp.N+1:2*pp.N);
        dhole = gamma(indV)*Delta_u(2*pp.N+1:3*pp.N);
        if( en_QDD == 1)
            dlambdan = gamma(indV)*Delta_u(3*pp.N+1:4*pp.N);
            dlambdap = gamma(indV)*Delta_u(4*pp.N+1:5*pp.N);
        end
        
        if (obs_conv == 1)
            figure(100);
            plot(pp.z,n)
        end

        phi = real(phi + dphi);
        n = abs(n + delec);
        p = abs(p + dhole);
        if( en_QDD == 1)
            lambdan = real(lambdan + dlambdan);
            lambdap = real(lambdap + dlambdap);
        end

        fail_cnt_dd = fail_cnt_dd + 1;
        disp(['iteration #',num2str(fail_cnt_dd),' - variations (phi,n,p): ',num2str(norm(dphi)),' , ',num2str(norm(delec)),' , ',num2str(norm(dhole)),' - norm: ',num2str(norm(res))]);
    end
    
    % Compute all wanted quantities
    Ec = -q*phi+pp.cbo;
    Efn = kB.*pp.T.*log(n./pp.Nc)-q*phi+pp.cbo-q*lambdan;
    
    if((fail_cnt_dd >= dd.max_diter/gamma(indV)))
        disp(['Convergence failed']);
        return;
    else
        disp(['Convergence achieved in ',num2str(fail_cnt_dd),' iterations']);
    end
    
    if (obs_volt == 1)
        figure(1); hold on;
        plot(pp.z,real(-phi+pp.cbo/q));
        figure(2); hold on;
        plot(pp.z,real(n));
    end
    
    V_QDD{indV} = Vvet(indV);
    phi_QDD{indV} = phi;
    Ec_QDD{indV} = Ec;
    n_QDD{indV} = n;
    p_QDD{indV} = p;
    lambdan_QDD{indV} = lambdan;
    lambdap_QDD{indV} = lambdap;
    
    phi_BZM = phi_eq;
    Ec_BZM = Ec_eq;
    n_BZM = n_eq;
    p_BZM = p_eq;
    
    z = pp.z;
end
save('DG_HEMT_tuned');