import model, entity, constants, common, algorithm, jsonpickle, notice
from config import db
from datetime import datetime


def doAnalysis(body, loginUser: entity.LoginUser) -> entity.AnalysisResult:
    # datasets
    fileIdList = []
    selectAll = body.get("selectAll")
    if selectAll is True:
        if loginUser.roleId != 3:
            fileList = db.session.query(model.AnalysisFile).filter(model.AnalysisFile.createUserId == loginUser.userId, model.AnalysisFile.enableFlag == True)
        else:
            fileList = db.session.query(model.AnalysisFile).filter(model.AnalysisFile.enableFlag == True)
        for file in fileList:
            fileIdList.append(file.id)
    else:
        fileIdList = body.get("fileIdList")
        if fileIdList == None or len(fileIdList) == 0:
            raise Exception("please upload at least one dataset file")
    
    fileList = []
    for id in fileIdList:
        item = db.session.query(model.AnalysisFile).filter(model.AnalysisFile.id == id, model.AnalysisFile.enableFlag == True).first()
        fileList.append(entity.FileEntity(item))
    if len(fileList) == 0:
        raise Exception("no datatset")
    # measures
    measures = body.get("measures")
    if len(measures) == 0:
        raise Exception("please at least one measure for analysis")
    resultByDataset = analysisByDataset(fileList, measures)
    resultByMeasure = analysisByMeasure(resultByDataset, measures)
    result = entity.AnalysisResult(None)
    result.resultByDataset = resultByDataset
    result.resultByMeasure = resultByMeasure
    return result


def analysisByDataset(fileList, measures):
    # analysis result
    analysisResults = []
    for file in fileList:
        result = algorithm.file_handler(file.fileUrl)
        measureList = []
        for i in range(0, len(constants.measures) - 1):
            measureName = constants.measures[i]
            if measureName not in measures:
                continue
            measureValue = str(result[i])
            item = entity.MeasureEntity(measureName, measureValue)
            measureList.append(item)
        analysisResult = entity.AnalysisResultByDataset(file.id, file.fileName, measureList)
        analysisResults.append(analysisResult)
    return analysisResults


def analysisByMeasure(resultByDataset, measures):
    analysisResults = []
    for i in range(0, len(measures) - 1):
        measureName = measures[i]
        datasetEntityList = []
        for item in resultByDataset:
            fileId = item.fileId
            fileName = item.fileName
            measureEntityList = item.measureList
            value = str(measureEntityList[i].value)
            datasetEntity = entity.DatasetEntity(fileId, fileName, value)
            datasetEntityList.append(datasetEntity)
        analysisResult = entity.AnalysisResultByMeasure(measureName, datasetEntityList)
        analysisResults.append(analysisResult)
    return analysisResults


def getAnalysisSetupDetail(loginUser: entity.LoginUser, setupId: str) -> entity.AnalysisSetupDetail:
    setup = db.session.query(model.AnalysisSetup).filter(model.AnalysisSetup.setupId == setupId, model.AnalysisSetup.enableFlag == True).first()
    if not setup:
        raise Exception("setup not found")
    measureList = setup.measures.split(";")
    fileIdList = []
    if setup.selectAll:
        fileList = db.session.query(model.AnalysisFile).filter(model.AnalysisFile.createUserId == loginUser.userId, model.AnalysisFile.enableFlag == True)
        for file in fileList:
            fileIdList.append(file.id)
    else:
        if setup.fileIds:
            fileIdList = setup.fileIds.split(";")
    
    fileList = []
    for fileId in fileIdList:
        fileModel = db.session.query(model.AnalysisFile).filter(model.AnalysisFile.id == int(fileId), model.AnalysisFile.enableFlag == True).first()
        if fileModel:
            fileList.append(entity.FileEntity(fileModel))
    vo = entity.AnalysisSetupDetail(setup)
    vo.fileList = fileList
    vo.measureList = measureList
    return vo
    

def saveAnalysisSetup(body, loginUser: entity.LoginUser) -> str:
    # check at least dataset file in setup
    fileIdList = []
    selectAll = body.get("selectAll")
    if selectAll is True:
        fileList = db.session.query(model.AnalysisFile).filter(model.AnalysisFile.createUserId == loginUser.userId, model.AnalysisFile.enableFlag == True)
        for file in fileList:
            fileIdList.append(file.id)
    else:
        fileIdList = body.get("fileIdList")
        if fileIdList == None or len(fileIdList) == 0:
            raise Exception("please upload at least one dataset file")
    # check setup name valid
    setupName = body.get("setupName")
    if not setupName:
        raise Exception("please enter a name for this analysis setup")
    # check at least one measure selected
    measures = body.get("measures")
    if measures == None or len(measures) == 0:
        raise Exception("please at least one measure")
    # if setup id not exist, this is a new setup
    setupId = body.get("setupId")
    if not setupId:
        return createAnalysisSetup(body, loginUser)
    # update setup info
    setup = db.session.query(model.AnalysisSetup).filter(model.AnalysisSetup.setupId == setupId, model.AnalysisSetup.enableFlag == True).first()
    setup.setupName = setupName
    setup.measures = ";".join(measures)
    fileIds = ";".join(map(str, fileIdList))
    setup.fileIds = fileIds
    setup.selectAll = selectAll
    db.session.commit()
    noticeContent = "Ciao, your analysis setup named {} has been updated by {}.".format(setupName, loginUser.userName)
    notice.createNotice(noticeContent, loginUser.userId)
    return setupId
        
    
def createAnalysisSetup(body, loginUser: entity.LoginUser) -> str:
    setupId = common.generateUUID()
    setupName = body.get("setupName")
    measures = ";".join(body.get("measures"))
    createUserId = loginUser.userId
    createUserName = loginUser.userName
    setup = model.AnalysisSetup(setupId, setupName, measures, createUserId, createUserName)
    fileIdList = body.get("fileIdList")
    fileIds = ";".join([str(fileId) for fileId in fileIdList])
    # bind file
    setup.fileIds = fileIds
    setup.selectAll = body.get("selectAll")
    # save database
    try:
        db.session.add(setup)
        db.session.commit()
        noticeContent = "Ciao, you have created a analysis setup successfully."
        notice.createNotice(noticeContent, loginUser.userId)
    except:
        db.session.rollback()
        raise Exception("save failed")
    return setupId


def deleteAnalysisSetup(loginUser: entity.LoginUser, id: str):
    setup = db.session.query(model.AnalysisSetup).filter(model.AnalysisSetup.setupId == id, model.AnalysisSetup.enableFlag == True).first()
    if not setup:
        raise Exception("setup not exist")
    setup.enableFlag = False
    db.session.commit()
    noticeContent = "Ciao, the analysis setup named {} has been deleted by {}.".format(setup.setupName, loginUser.userName)
    notice.createNotice(noticeContent, setup.createUserId)
    
    
def saveAnalysisResult(body, loginUser: entity.LoginUser):
    resultName = body.get("resultName")
    if not resultName:
        raise Exception("please enter a result name to save")
    resultContent = body.get("resultContent")
    if not resultContent:
        raise Exception("please result content is empty")
    result = model.AnalysisResult()
    result.resultId = common.generateUUID();
    result.resultName = resultName
    result.resultContent = jsonpickle.dumps(resultContent)
    result.createUserId = loginUser.userId
    result.createUserName = loginUser.userName
    result.createDateTime = datetime.now()
    result.enableFlag = True
    # save database
    db.session.add(result)
    db.session.commit()
    noticeContent = "Ciao, you have saved a analysis result successfully."
    notice.createNotice(noticeContent, loginUser.userId)


def renameAnalysisResult(loginUser: entity.LoginUser, body):
    resultId = body.get("resultId")
    if not resultId:
        raise Exception("resultId is empty")
    resultName = body.get("resultName")
    if not resultName:
        raise Exception("resultName is empty")
    result = db.session.query(model.AnalysisResult).filter(model.AnalysisResult.resultId == resultId, model.AnalysisResult.enableFlag == True).first()
    result.resultName = resultName
    db.session.commit()
    noticeContent = "Ciao, your analysis result has been renamed by {}. Now its name is {}.".format(loginUser.userName, resultName)
    notice.createNotice(noticeContent, result.createUserId)

    
def getAnalysisResultDetail(resultId: str):
    if not resultId:
        raise Exception("resultId is empty")
    result = db.session.query(model.AnalysisResult).filter(model.AnalysisResult.resultId == resultId, model.AnalysisResult.enableFlag == True).first()
    resultContent = jsonpickle.decode(result.resultContent)
    return resultContent, result.resultName


def deleteAnalysisResult(resultId: str):
    if not resultId:
            raise Exception("resultId is empty")
    result = db.session.query(model.AnalysisResult).filter(model.AnalysisResult.resultId == resultId, model.AnalysisResult.enableFlag == True).first()
    result.enableFlag = False
    db.session.commit()
    noticeContent = "Ciao, your analysis result has deleted."
    notice.createNotice(noticeContent, result.createUserId)
    

def getAnalysisSetupPageList(body, loginUser: entity.LoginUser) -> entity.AnalysisSetupPageList:
    pageIndex = body.get("pageIndex")
    if not pageIndex:
        raise Exception("page index is none")
    pageSize = body.get("pageSize")
    if not pageSize:
        raise Exception("page size is none")
    paginate = None
    if loginUser.roleId != 3:
        paginate = db.session.query(model.AnalysisSetup).filter(model.AnalysisSetup.createUserId == loginUser.userId, model.AnalysisSetup.enableFlag == True).paginate(page=pageIndex, per_page=pageSize)
    else:
        paginate = db.session.query(model.AnalysisSetup).filter(model.AnalysisSetup.enableFlag == True).paginate(page=pageIndex, per_page=pageSize)
    content = []
    for item in paginate.items:
        obj = entity.AnalysisSetupDetail(item)
        content.append(obj)
    totalCount = paginate.total
    pageList = entity.AnalysisSetupPageList(content, pageIndex, pageSize, totalCount)
    return pageList


def getAnalysisResultPageList(body, loginUser: entity.LoginUser) -> entity.AnalysisResultPageList:
    pageIndex = body.get("pageIndex")
    if not pageIndex:
        raise Exception("page index is none")
    pageSize = body.get("pageSize")
    if not pageSize:
        raise Exception("page size is none")
    paginate = None
    if loginUser.roleId != 3:
        paginate = db.session.query(model.AnalysisResult).filter(model.AnalysisResult.createUserId == loginUser.userId, model.AnalysisResult.enableFlag == True).paginate(page=pageIndex, per_page=pageSize)
    else:
        paginate = db.session.query(model.AnalysisResult).filter(model.AnalysisResult.enableFlag == True).paginate(page=pageIndex, per_page=pageSize)
    content = []
    for item in paginate.items:
        obj = entity.AnalysisResult(item)
        content.append(obj)
    totalCount = paginate.total
    pageList = entity.AnalysisResultPageList(content, pageIndex, pageSize, totalCount)
    return pageList


def createSampleResult(member: model.Member):
    resultName = "analysis_result_sample"
    resultContent = constants.sampleResultContent
    # new result model
    result = model.AnalysisResult()
    result.resultId = common.generateUUID();
    result.resultName = resultName
    result.resultContent = resultContent
    result.createUserId = member.userId
    result.createUserName = member.userName
    result.createDateTime = datetime.now()
    result.enableFlag = True
    # save database
    db.session.add(result)
    db.session.commit()
    return