// Copyright 2017 ETH Zurich and University of Bologna.
// Copyright and related rights are licensed under the Solderpad Hardware
// License, Version 0.51 (the “License”); you may not use this file except in
// compliance with the License.  You may obtain a copy of the License at
// http://solderpad.org/licenses/SHL-0.51. Unless required by applicable law
// or agreed to in writing, software, hardware and materials distributed under
// this License is distributed on an “AS IS” BASIS, WITHOUT WARRANTIES OR
// CONDITIONS OF ANY KIND, either express or implied. See the License for the
// specific language governing permissions and limitations under the License.

  `define SPI_STD     2'b00
  `define SPI_QUAD_TX 2'b01
  `define SPI_QUAD_RX 2'b10

  `define SPI_SEMIPERIOD      50ns    //10Mhz SPI CLK

  `define DELAY_BETWEEN_SPI   100ns

  int                   num_stim_r,num_exp_r,num_cycles_r,num_err_r = 0;   // counters for statistics

  logic                 more_stim_r = 1;

  logic [31:0]          spi_data_r;
  logic [31:0]          spi_data_r_recv;
  logic [31:0]          spi_addr_r;
  logic [31:0]          spi_addr_r_old;

  logic [63:0]          stimuli_r  [10000:0];                // array for the stimulus vectors

  task spi_send_cmd_addr_f;
    input          use_qspi;
    input    [7:0] command;
    input   [31:0] addr;
    begin
      if (use_qspi)
      begin
        for (int i = 2; i > 0; i--)
        begin
          spi_sdo3_f = command[4*i-1];
          spi_sdo2_f = command[4*i-2];
          spi_sdo1_f = command[4*i-3];
          spi_sdo0_f = command[4*i-4];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
      else
      begin
        for (int i = 7; i >= 0; i--)
        begin
          spi_sdo0_f = command[i];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end

      if (use_qspi)
      begin
        for (int i = 8; i > 0; i--)
        begin
          spi_sdo3_f = addr[4*i-1];
          spi_sdo2_f = addr[4*i-2];
          spi_sdo1_f = addr[4*i-3];
          spi_sdo0_f = addr[4*i-4];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
      else
      begin
        for (int i = 31; i >= 0; i--)
        begin
          spi_sdo0_f = addr[i];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
    end
  endtask

  task spi_send_data_f;
    input          use_qspi;
    input   [31:0] data;
    begin
      if (use_qspi)
      begin
        for (int i = 8; i > 0; i--)
        begin
          spi_sdo3_f = data[4*i-1];
          spi_sdo2_f = data[4*i-2];
          spi_sdo1_f = data[4*i-3];
          spi_sdo0_f = data[4*i-4];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
      else
      begin
        for (int i = 31; i >= 0; i--)
        begin
          spi_sdo0_f = data[i];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
    end
  endtask

  task spi_recv_data_f;
    input          use_qspi;
    output  [31:0] data;
    begin
      if (use_qspi)
      begin
        for (int i = 8; i > 0; i--)
        begin
          data[4*i-1] = spi_sdi3_f;
          data[4*i-2] = spi_sdi2_f;
          data[4*i-3] = spi_sdi1_f;
          data[4*i-4] = spi_sdi0_f;
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
      else
      begin
        for (int i = 31; i >= 0; i--)
        begin
          data[i] = spi_sdi0_f;
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
    end
  endtask

  task spi_load_f;
    input  use_qspi;
    begin
      $readmemh("./slm_files/spi_stim.txt", stimuli_r);  // read in the stimuli_r vectors  == address_value

      spi_addr_r        = stimuli_r[num_stim_r][63:32]; // assign address
      spi_data_r        = stimuli_r[num_stim_r][31:0];  // assign data

      $display("[SPI] Loading Instruction RAM");
      spi_csn_f  = 1'b0;
      #100  spi_send_cmd_addr_f(use_qspi,8'h2,spi_addr_r);

      spi_addr_r_old = spi_addr_r - 32'h4;

      while (more_stim_r)                        // loop until we have no more stimuli_r)
      begin
        spi_addr_r        = stimuli_r[num_stim_r][63:32]; // assign address
        spi_data_r        = stimuli_r[num_stim_r][31:0];  // assign data

        if (spi_addr_r != (spi_addr_r_old + 32'h4))
        begin
          $display("[SPI] Prev address %h current addr %h",spi_addr_r_old,spi_addr_r);
          $display("[SPI] Loading Data RAM");
          #100 spi_csn_f  = 1'b1;
          #`DELAY_BETWEEN_SPI;
          spi_csn_f  = 1'b0;
          #100  spi_send_cmd_addr_f(use_qspi,8'h2,spi_addr_r);
        end
        spi_send_data_f(use_qspi,spi_data_r[31:0]);

        num_stim_r     = num_stim_r + 1;             // increment stimuli_r
        spi_addr_r_old = spi_addr_r;
        if (num_stim_r > 9999 | stimuli_r[num_stim_r]===64'bx ) // make sure we have more stimuli_r
          more_stim_r = 0;                    // if not set variable to 0, will prevent additional stimuli_r to be applied
      end                                   // end while loop
      #100 spi_csn_f  = 1'b1;
      #`DELAY_BETWEEN_SPI;
    end
  endtask

  task spi_check_f;
    input  use_qspi;
    begin
      num_stim_r  = 0;
      more_stim_r = 1;

      spi_addr_r        = stimuli_r[num_stim_r][63:32]; // assign address
      spi_data_r        = stimuli_r[num_stim_r][31:0];  // assign data

      $display("[SPI] Checking Instruction RAM");
      spi_csn_f  = 1'b0;
      #100  spi_send_cmd_addr_f(use_qspi,8'hB,spi_addr_r);
      spi_addr_r_old = spi_addr_r - 32'h4;

      // dummy cycles
      padmode_spi_master_f = use_qspi ? `SPI_QUAD_RX : `SPI_STD;
      for (int i = 33; i >= 0; i--)
      begin
        #`SPI_SEMIPERIOD spi_sck_f = 1;
        #`SPI_SEMIPERIOD spi_sck_f = 0;
      end

      while (more_stim_r)                        // loop until we have no more stimuli_r)
      begin
        spi_addr_r        = stimuli_r[num_stim_r][63:32]; // assign address
        spi_data_r        = stimuli_r[num_stim_r][31:0];  // assign data

        if (spi_addr_r != (spi_addr_r_old + 32'h4))
        begin
          $display("[SPI] Prev address %h current addr %h",spi_addr_r_old,spi_addr_r);
          $display("[SPI] Checking Data RAM");
          #100 spi_csn_f  = 1'b1;
          #`DELAY_BETWEEN_SPI;
          spi_csn_f  = 1'b0;
          padmode_spi_master_f = use_qspi ? `SPI_QUAD_TX : `SPI_STD;
          #100  spi_send_cmd_addr_f(use_qspi,8'hB,spi_addr_r);

          // dummy cycles
          padmode_spi_master_f = use_qspi ? `SPI_QUAD_RX : `SPI_STD;
          for (int i = 33; i >= 0; i--)
          begin
            #`SPI_SEMIPERIOD spi_sck_f = 1;
            #`SPI_SEMIPERIOD spi_sck_f = 0;
          end
        end
        spi_recv_data(use_qspi,spi_data_r_recv[31:0]);

        if (spi_data_r_recv != spi_data_r)
          $display("%t: [SPI] Readback has failed, expected %X, got %X", $time, spi_data_r, spi_data_r_recv);

        num_stim_r     = num_stim_r + 1;             // increment stimuli_r
        spi_addr_r_old = spi_addr_r;
        if (num_stim_r > 9999 | stimuli_r[num_stim_r]===64'bx ) // make sure we have more stimuli_r
          more_stim_r = 0;                    // if not set variable to 0, will prevent additional stimuli_r to be applied
      end                                   // end while loop
      #100 spi_csn_f  = 1'b1;
      #`DELAY_BETWEEN_SPI;
      padmode_spi_master_f = use_qspi ? `SPI_QUAD_TX : `SPI_STD;
    end
  endtask

  task spi_write_reg_f;
    input          use_qspi;
    input    [7:0] command;
    input    [7:0] reg_val;
    begin
      padmode_spi_master_f = use_qspi ? `SPI_QUAD_TX : `SPI_STD;
      spi_csn_f  = 1'b0;
      #100;
      if (use_qspi)
      begin
        for (int i = 2; i > 0; i--)
        begin
          spi_sdo3_f = command[4*i-1];
          spi_sdo2_f = command[4*i-2];
          spi_sdo1_f = command[4*i-3];
          spi_sdo0_f = command[4*i-4];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
      else
      begin
        for (int i = 7; i >= 0; i--)
        begin
          spi_sdo0_f = command[i];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end

      if (use_qspi)
      begin
        for (int i = 2; i > 0; i--)
        begin
          spi_sdo3_f = reg_val[4*i-1];
          spi_sdo2_f = reg_val[4*i-2];
          spi_sdo1_f = reg_val[4*i-3];
          spi_sdo0_f = reg_val[4*i-4];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
      else
      begin
        for (int i = 7; i >= 0; i--)
        begin
          spi_sdo0_f = reg_val[i];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
      #100 spi_csn_f  = 1'b1;
      #`DELAY_BETWEEN_SPI;
    end
  endtask

  task spi_write_word_f;
    input          use_qspi;
    input   [31:0] addr;
    input   [31:0] data;
    begin
      spi_csn_f  = 1'b0;
      #`DELAY_BETWEEN_SPI;
      spi_send_cmd_addr_f(use_qspi, 8'h2, addr);
      spi_send_data_f(use_qspi, data);
      #100 spi_csn_f  = 1'b1;
      #`DELAY_BETWEEN_SPI;
    end
  endtask

  task spi_read_nword_f;
    input          use_qspi;
    input   [31:0] addr;
    input int      n;
    inout   [31:0] data[];

    logic    [7:0] command;
    int i;
    int j;
    begin
      command = 8'hB;
      padmode_spi_master_f = use_qspi ? `SPI_QUAD_TX : `SPI_STD;
      spi_sck_f = 0;
      #`SPI_SEMIPERIOD spi_sck_f = 0;
      #`SPI_SEMIPERIOD spi_sck_f = 0;
      spi_csn_f = 0;
      #`SPI_SEMIPERIOD spi_sck_f = 0;
      #`SPI_SEMIPERIOD spi_sck_f = 0;
      if (use_qspi)
      begin
        for (i = 2; i > 0; i--)
        begin
          spi_sdo3_f = command[4*i-1];
          spi_sdo2_f = command[4*i-2];
          spi_sdo1_f = command[4*i-3];
          spi_sdo0_f = command[4*i-4];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
      else
      begin
        for (i = 7; i >= 0; i--)
        begin
          spi_sdo0_f = command[i];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
      if (use_qspi)
      begin
        for (i = 8; i > 0; i--)
        begin
          spi_sdo3_f = addr[4*i-1];
          spi_sdo2_f = addr[4*i-2];
          spi_sdo1_f = addr[4*i-3];
          spi_sdo0_f = addr[4*i-4];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
      else
      begin
        for (i = 31; i >= 0; i--)
        begin
          spi_sdo0_f = addr[i];
          #`SPI_SEMIPERIOD spi_sck_f = 1;
          #`SPI_SEMIPERIOD spi_sck_f = 0;
        end
      end
      padmode_spi_master_f = use_qspi ? `SPI_QUAD_RX : `SPI_STD;
      for (i = 32; i >= 0; i--)
      begin
        #`SPI_SEMIPERIOD spi_sck_f = 1;
        #`SPI_SEMIPERIOD spi_sck_f = 0;
      end
      if (use_qspi)
      begin
        for (j = 0; j < n; j++)
        begin
          for (i = 8; i > 0; i--)
          begin
            #`SPI_SEMIPERIOD spi_sck_f = 1;
            data[j][4*i-1] = spi_sdi3_f;
            data[j][4*i-2] = spi_sdi2_f;
            data[j][4*i-3] = spi_sdi1_f;
            data[j][4*i-4] = spi_sdi0_f;
            #`SPI_SEMIPERIOD spi_sck_f = 0;
          end
        end
      end
      else
      begin
        for (j = 0; j < n; j++)
        begin
          for (i = 31; i >= 0; i--)
          begin
            #`SPI_SEMIPERIOD spi_sck_f = 1;
            data[j][i] = spi_sdi0_f;
            #`SPI_SEMIPERIOD spi_sck_f = 0;
          end
        end
      end
      #`SPI_SEMIPERIOD spi_sck_f = 0;
      #`SPI_SEMIPERIOD spi_sck_f = 0;
      spi_csn_f   = 1;
      #`SPI_SEMIPERIOD spi_sck_f = 0;
      #`SPI_SEMIPERIOD spi_sck_f = 0;
      padmode_spi_master_f = use_qspi ? `SPI_QUAD_TX : `SPI_STD;
    end
  endtask

  task spi_read_word_f;
    input          use_qspi;
    input   [31:0] addr;
    output  [31:0] data;

    logic   [31:0] tmp[1];
    begin
      spi_read_nword_f(use_qspi, addr, 1, tmp);
      data = tmp[0];
    end
  endtask

  task spi_write_halfword_f;
    input          use_qspi;
    input   [31:0] addr;
    input   [15:0] data;

    logic   [31:0] temp;
    begin
      spi_read_word_f(use_qspi, {addr[31:2], 2'b00}, temp);

      case (addr[1])
        1'b0: temp[15: 0] = data[15:0];
        1'b1: temp[31:16] = data[15:0];
      endcase

      spi_write_word_f(use_qspi, {addr[31:2], 2'b00}, temp);
    end
  endtask

  task spi_write_byte_f;
    input          use_qspi;
    input   [31:0] addr;
    input   [ 7:0] data;

    logic   [31:0] temp;
    begin
      spi_read_word_f(use_qspi, {addr[31:2], 2'b00}, temp);

      case (addr[1:0])
        2'b00: temp[ 7: 0] = data[7:0];
        2'b01: temp[15: 8] = data[7:0];
        2'b10: temp[23:16] = data[7:0];
        2'b11: temp[31:24] = data[7:0];
      endcase

      spi_write_word_f(use_qspi, {addr[31:2], 2'b00}, temp);
    end
  endtask

  task spi_read_halfword_f;
    input          use_qspi;
    input   [31:0] addr;
    output  [15:0] data;

    logic   [31:0] temp;
    begin
      spi_read_word_f(use_qspi, {addr[31:2], 2'b00}, temp);

      case (addr[1])
        1'b0: data[15:0] = temp[15: 0];
        1'b1: data[15:0] = temp[31:16];
      endcase
    end
  endtask

  task spi_read_byte_f;
    input          use_qspi;
    input   [31:0] addr;
    output  [ 7:0] data;

    logic   [31:0] temp;
    begin
      spi_read_word_f(use_qspi, {addr[31:2], 2'b00}, temp);

      case (addr[1:0])
        2'b00: data[7:0] = temp[ 7: 0];
        2'b01: data[7:0] = temp[15: 8];
        2'b10: data[7:0] = temp[23:16];
        2'b11: data[7:0] = temp[31:24];
      endcase
    end
  endtask

  task spi_enable_qpi_f;
    $display("[SPI] Enabling QPI mode");
    //Sets QPI mode
    spi_write_reg_f(0,8'h1,8'h1);

    padmode_spi_master_f = `SPI_QUAD_TX;
  endtask

  task spi_check_return_codes_f;
    output exit_code;

    spi_read_word_f(use_qspi_f, 32'h1A10_7014, recv_data_f);
    $display("[SPI] Received %X", recv_data_f);
    if (recv_data_f != '0) begin
      exit_code = `EXIT_FAIL;
      $display("[SPI] Test FAILED");
    end else begin
      exit_code = `EXIT_SUCCESS;
      $display("[SPI] Test OK");
    end
  endtask

