// Copyright 2017 ETH Zurich and University of Bologna.
// Copyright and related rights are licensed under the Solderpad Hardware
// License, Version 0.51 (the “License”); you may not use this file except in
// compliance with the License.  You may obtain a copy of the License at
// http://solderpad.org/licenses/SHL-0.51. Unless required by applicable law
// or agreed to in writing, software, hardware and materials distributed under
// this License is distributed on an “AS IS” BASIS, WITHOUT WARRANTIES OR
// CONDITIONS OF ANY KIND, either express or implied. See the License for the
// specific language governing permissions and limitations under the License.

`include "config.sv"
`include "tb_jtag_pkg.sv"

`define REF_CLK_PERIOD   (2*15.25us)  // 32.786 kHz --> FLL reset value --> 50 MHz
`define CLK_PERIOD       40.00ns      // 25 MHz

`define EXIT_SUCCESS  0
`define EXIT_FAIL     1
`define EXIT_ERROR   -1

module tb;
  timeunit      1ns;
  timeprecision 1ps;

  // +MEMLOAD= valid values are "SPI", "STANDALONE" "PRELOAD", "" (no load of L2)
  parameter  SPI            = "QUAD";    // valid values are "SINGLE", "QUAD"
  parameter  BAUDRATE       = 781250;    // 1562500
  parameter  CLK_USE_FLL    = 0;  // 0 or 1
  parameter  TEST           = ""; //valid values are "" (NONE), "DEBUG"
  parameter  TEST_F         = ""; //valid values are "" (NONE), "DEBUG"
  parameter  USE_ZERO_RISCY = 0;
  parameter  RISCY_RV32F    = 0;
  parameter  ZERO_RV32M     = 1;
  parameter  ZERO_RV32E     = 0;

  int           exit_status = `EXIT_ERROR; // modelsim exit code, will be overwritten when successful
  
   //fault_model_signals
  string        memload_f;
  logic         s_clk_f   = 1'b0;
  logic         s_rst_n_f = 1'b0;

  logic         fetch_enable_f = 1'b0;

  logic [1:0]   padmode_spi_master_f;
  logic         spi_sck_f   = 1'b0;
  logic         spi_csn_f   = 1'b1;
  logic [1:0]   spi_mode_f;
  logic         spi_sdo0_f;
  logic         spi_sdo1_f;
  logic         spi_sdo2_f;
  logic         spi_sdo3_f;
  logic         spi_sdi0_f;
  logic         spi_sdi1_f;
  logic         spi_sdi2_f;
  logic         spi_sdi3_f;

  logic         uart_tx_f;
  logic         uart_rx_f;
  logic         s_uart_dtr_f;
  logic         s_uart_rts_f;

  logic         scl_pad_i_f;
  logic         scl_pad_o_f;
  logic         scl_padoen_o_f;

  logic         sda_pad_i_f;
  logic         sda_pad_o_f;
  logic         sda_padoen_o_f;

  tri1          scl_io_f;
  tri1          sda_io_f;

  logic [31:0]  gpio_in_f = '0;
  logic [31:0]  gpio_dir_f;
  logic [31:0]  gpio_out_f;

  logic [31:0]  recv_data_f;

  jtag_i jtag_if_f();

  adv_dbg_if_t adv_dbg_if_f = new(jtag_if_f);
  
  // use 8N1
  uart_bus
  #(
    .BAUD_RATE(BAUDRATE),
    .PARITY_EN(0)
  )
  uart_f
  (
    .rx         ( uart_rx_f ),
    .tx         ( uart_tx_f ),
    .rx_en      ( 1'b1    )
  );

  spi_slave
  spi_master_f();

  i2c_buf i2c_buf_i_f
  (
    .scl_io       ( scl_io_f       ),
    .sda_io       ( sda_io_f       ),
    .scl_pad_i    ( scl_pad_i_f    ),
    .scl_pad_o    ( scl_pad_o_f    ),
    .scl_padoen_o ( scl_padoen_o_f ),
    .sda_pad_i    ( sda_pad_i_f    ),
    .sda_pad_o    ( sda_pad_o_f    ),
    .sda_padoen_o ( sda_padoen_o_f )
  );

  i2c_eeprom_model
  #(
    .ADDRESS ( 7'b1010_000 )
  )
  i2c_eeprom_model_i_f
  (
    .scl_io ( scl_io_f  ),
    .sda_io ( sda_io_f  ),
    .rst_ni ( s_rst_n_f )
  );

   pulpino_top
  #(
    .USE_ZERO_RISCY    ( USE_ZERO_RISCY ),
    .RISCY_RV32F       ( RISCY_RV32F    ),
    .ZERO_RV32M        ( ZERO_RV32M     ),
    .ZERO_RV32E        ( ZERO_RV32E     )
   )
  top_i_f
  (
    .clk               ( s_clk        ),
    .rst_n             ( s_rst_n_f      ),

    .clk_sel_i         ( 1'b0         ),
    .testmode_i        ( 1'b0         ),
    .fetch_enable_i    ( fetch_enable_f ),

    .spi_clk_i         ( spi_sck_f      ),
    .spi_cs_i          ( spi_csn_f      ),
    .spi_mode_o        ( spi_mode_f     ),
    .spi_sdo0_o        ( spi_sdi0_f     ),
    .spi_sdo1_o        ( spi_sdi1_f     ),
    .spi_sdo2_o        ( spi_sdi2_f     ),
    .spi_sdo3_o        ( spi_sdi3_f     ),
    .spi_sdi0_i        ( spi_sdo0_f     ),
    .spi_sdi1_i        ( spi_sdo1_f     ),
    .spi_sdi2_i        ( spi_sdo2_f     ),
    .spi_sdi3_i        ( spi_sdo3_f     ),

    .spi_master_clk_o  ( spi_master_f.clk     ),
    .spi_master_csn0_o ( spi_master_f.csn     ),
    .spi_master_csn1_o (                    ),
    .spi_master_csn2_o (                    ),
    .spi_master_csn3_o (                    ),
    .spi_master_mode_o ( spi_master_f.padmode ),
    .spi_master_sdo0_o ( spi_master_f.sdo[0]  ),
    .spi_master_sdo1_o ( spi_master_f.sdo[1]  ),
    .spi_master_sdo2_o ( spi_master_f.sdo[2]  ),
    .spi_master_sdo3_o ( spi_master_f.sdo[3]  ),
    .spi_master_sdi0_i ( spi_master_f.sdi[0]  ),
    .spi_master_sdi1_i ( spi_master_f.sdi[1]  ),
    .spi_master_sdi2_i ( spi_master_f.sdi[2]  ),
    .spi_master_sdi3_i ( spi_master_f.sdi[3]  ),

    .scl_pad_i         ( scl_pad_i_f    ),
    .scl_pad_o         ( scl_pad_o_f    ),
    .scl_padoen_o      ( scl_padoen_o_f ),
    .sda_pad_i         ( sda_pad_i_f    ),
    .sda_pad_o         ( sda_pad_o_f    ),
    .sda_padoen_o      ( sda_padoen_o_f ),


    .uart_tx           ( uart_rx_f      ),
    .uart_rx           ( uart_tx_f      ),
    .uart_rts          ( s_uart_rts_f   ),
    .uart_dtr          ( s_uart_dtr_f   ),
    .uart_cts          ( 1'b0         ),
    .uart_dsr          ( 1'b0         ),

    .gpio_in           ( gpio_in_f      ),
    .gpio_out          ( gpio_out_f     ),
    .gpio_dir          ( gpio_dir_f     ),
    .gpio_padcfg       (              ),

    .tck_i             ( jtag_if_f.tck     ),
    .trstn_i           ( jtag_if_f.trstn   ),
    .tms_i             ( jtag_if_f.tms     ),
    .tdi_i             ( jtag_if_f.tdi     ),
    .tdo_o             ( jtag_if_f.tdo     )
  );

generate
    if (CLK_USE_FLL) begin
      initial
      begin
        #(`REF_CLK_PERIOD/2);
        s_clk_f = 1'b1;
        forever s_clk_f = #(`REF_CLK_PERIOD/2) ~s_clk_f;
      end
    end else begin
      initial
      begin
        #(`CLK_PERIOD/2);
        s_clk_f = 1'b1;
        forever s_clk_f = #(`CLK_PERIOD/2) ~s_clk_f;
      end
    end
  endgenerate

  logic use_qspi_f;

  initial
  begin
    int i;

    if(!$value$plusargs("MEMLOAD_f=%s", memload_f))
      memload_f = "PRELOAD";

    $display("Using MEMLOAD_f method: %s", memload_f);

    $display("Using %s core", USE_ZERO_RISCY ? "zero-riscy" : "ri5cy");

    use_qspi_f = SPI == "QUAD" ? 1'b1 : 1'b0;

    s_rst_n_f      = 1'b0;
    fetch_enable_f = 1'b0;

    #500ns;

    s_rst_n_f = 1'b1;

    #500ns;
    if (use_qspi_f)
      spi_enable_qpi_f(); //pay attention


    if (memload_f != "STANDALONE")
    begin
      /* Configure JTAG and set boot address */
      adv_dbg_if_f.jtag_reset();
      adv_dbg_if_f.jtag_softreset();
      adv_dbg_if_f.init();
      adv_dbg_if_f.axi4_write32(32'h1A10_7008, 1, 32'h0000_0000);
	  
    end

    if (memload_f == "PRELOAD")
    begin
      // preload memories
      mem_preload_f();
    end
    else if (memload_f == "SPI")
    begin
      spi_load_f(use_qspi_f);
      spi_check_f(use_qspi_f);
    end

    #200ns;
    fetch_enable_f = 1'b1;

    if(TEST_F == "DEBUG") begin
      debug_tests_f();
    end else if (TEST_F == "DEBUG_IRQ") begin
      debug_irq_tests_f();
    end else if (TEST_F == "MEM_DPI") begin
      mem_dpi_f(4567);
    end else if (TEST_F == "ARDUINO_UART") begin
      if (~gpio_out_f[0])
        wait(gpio_out_f[0]);
      uart_f.send_char(8'h65);
    end else if (TEST_F == "ARDUINO_GPIO") begin
      // Here  test for GPIO Starts
      if (~gpio_out_f[0])
        wait(gpio_out_f[0]);

      gpio_in_f[4]=1'b1;

      if (~gpio_out_f[1])
        wait(gpio_out_f[1]);
      if (~gpio_out_f[2])
        wait(gpio_out_f[2]);
      if (~gpio_out_f[3])
        wait(gpio_out_f[3]);

      gpio_in_f[7]=1'b1;

    end else if (TEST_F == "ARDUINO_SHIFT") begin

      if (~gpio_out_f[0])
        wait(gpio_out_f[0]);
      //start TEST

      if (~gpio_out_f[4])
        wait(gpio_out_f[4]);
      gpio_in_f[3]=1'b1;
      if (gpio_out_f[4])
        wait(~gpio_out_f[4]);

      if (~gpio_out_f[4])
        wait(gpio_out_f[4]);
      gpio_in_f[3]=1'b1;
      if (gpio_out_f[4])
        wait(~gpio_out_f[4]);

      if (~gpio_out_f[4])
        wait(gpio_out_f[4]);
      gpio_in_f[3]=1'b0;
      if (gpio_out_f[4])
        wait(~gpio_out_f[4]);

      if (~gpio_out_f[4])
        wait(gpio_out_f[4]);
      gpio_in_f[3]=1'b0;
      if (gpio_out_f[4])
        wait(~gpio_out_f[4]);

      if (~gpio_out_f[4])
        wait(gpio_out_f[4]);
      gpio_in_f[3]=1'b1;
      if (gpio_out_f[4])
        wait(~gpio_out_f[4]);

      if (~gpio_out_f[4])
        wait(gpio_out_f[4]);
      gpio_in_f[3]=1'b0;
      if (gpio_out_f[4])
        wait(~gpio_out_f[4]);

      if (~gpio_out_f[4])
        wait(gpio_out_f[4]);
      gpio_in_f[3]=1'b0;
      if (gpio_out_f[4])
        wait(~gpio_out_f[4]);

      if (~gpio_out_f[4])
        wait(gpio_out_f[4]);
      gpio_in_f[3]=1'b1;
      if (gpio_out_f[4])
        wait(~gpio_out_f[4]);

    end else if (TEST_F == "ARDUINO_PULSEIN") begin
      if (~gpio_out_f[0])
        wait(gpio_out_f[0]);
      #50us;
      gpio_in_f[4]=1'b1;
      #500us;
      gpio_in_f[4]=1'b0;
      #1ms;
      gpio_in_f[4]=1'b1;
      #500us;
      gpio_in_f[4]=1'b0;
    end else if (TEST_F == "ARDUINO_INT") begin
      if (~gpio_out_f[0])
        wait(gpio_out_f[0]);
      #50us;
      gpio_in_f[1]=1'b1;
      #20us;
      gpio_in_f[1]=1'b0;
      #20us;
      gpio_in_f[1]=1'b1;
      #20us;
      gpio_in_f[2]=1'b1;
      #20us;
    end else if (TEST_F == "ARDUINO_SPI") begin
      for(i = 0; i < 2; i++) begin
        spi_master.wait_csn(1'b0);
        spi_master.send(0, {>>{8'h38}});
      end
    end



    // end of computation
    if (~gpio_out_f[8])
      wait(gpio_out_f[8]);

    spi_check_return_codes(exit_status);

    $fflush();
    $stop();
  end
  
  `include "tb_spi_pkg_f.sv"
  `include "tb_mem_pkg_f.sv"
  `include "spi_debug_test_f.svh"
  `include "mem_dpi_f.svh"
  
endmodule
