LIBRARY IEEE;
USE IEEE.STD_LOGIC_1164.ALL;
USE IEEE.NUMERIC_STD.ALL;
use work.constants.all;
use work.func_external.all;


--------------------------------------------------------------------------------
-- 
--------------------------------------------------------------------------------


entity FSM is
	port (
		fsm_index					: in integer;
		fsm_clock                   : in std_logic;
		fsm_reset                   : in std_logic;
		fsm_enable					: in std_logic;
		tracker_address             : in std_logic_vector (SIZE_sram_address-1 downto 0);
		memory_segment              : in std_logic_vector (SM_segment_bit-1 downto 0);
		tracker_ready 				: in std_logic;
		all_fsm_done				: in std_logic;
		--sm_value					: in std_logic_vector (SM_segment_bit-1 downto 0);
		sm_or_not					: out integer;   --used as state index    0 means invalid, 1 means sm, 2 means pixel 
		--sm_out 						: out std_logic_vector(SM_segment_bit-1 downto 0);
		fsm_output_address          : out std_logic_vector (SIZE_sram_address -1 downto 0);
		fsm_sram_control		    : out std_logic;
		fsm_stripe_done				: out std_logic;
		fsm_tracker_address			: out integer;
		pixel_coor_x				: out integer;
		pixel_coor_y 				: out integer;
		pixel_coor_z				: out integer
	);
end FSM;

architecture FSM_arc of FSM is

	type type_state is (reset,value_true,value_false,new_start,update,prepare_sm);
	signal current_state,next_state : type_state;
	signal current_segment_value : std_logic;
	signal index : integer;
	signal segment_full : std_logic;
	----------------------------------------------------------------------------
	-- sm_value is used as a register to store the sparsity map
	-- So the register isn't built independently, it's merged in the file "FSM.vhd"
	----------------------------------------------------------------------------
	
	signal sm_value : std_logic_vector (SM_segment_bit-1 downto 0);
	signal fsm_reg_address : std_logic_vector (SIZE_sram_address-1 downto 0);
	signal fsm_tracker_address_buffer : integer;
	signal coor_x,coor_y,coor_z : integer;
	signal update_time_control : integer;
	signal sm_value_index : integer;
	signal sm_or_not_buffer : integer;
begin

synchronous_pixel : process(fsm_clock)
begin
	if (fsm_reset = '1') then
		pixel_coor_x  <= fsm_index; 
		pixel_coor_y  <= 0; 
		pixel_coor_z  <= 0; 
		sm_or_not  <= 0;
	elsif (rising_edge (fsm_clock)) then
		pixel_coor_x  <= coor_x;
		pixel_coor_y  <= coor_y;
		pixel_coor_z  <= coor_z;
		sm_or_not  <= sm_or_not_buffer;
	end if;
end process;

	fsm_sram_control  <= fsm_enable;
	fsm_tracker_address  <= fsm_tracker_address_buffer;
-----------------------------------------------------------------------------
--  next state
-----------------------------------------------------------------------------	

	next_state_proc: process(fsm_clock,fsm_reset)
	begin
			if (fsm_reset = '1') then
				current_state  <= reset;
				index  <= -2;


			elsif (rising_edge(fsm_clock)) then
				if (fsm_enable = '1') then
					if (tracker_ready = '1') then
						current_state  <= next_state;
						
							if( index  <= 16) then 
								index  <= index + 1;
							else 
								index  <= -1;
							end if;
					end	 if;
				else 
					current_state  <= reset;
				end if;
			end if;
	end process next_state_proc;

--------------------------------------------------------------------------------
-- future state network
--------------------------------------------------------------------------------

	future_state_proc: process(current_state,index,memory_segment)
	begin
		
		case current_state is

			when reset => 
				if (fsm_enable = '1') then
					next_state  <= new_start;
				else
					next_state  <= reset;
				end if;

			when new_start  => 
				if (fsm_enable = '1') then
					next_state  <= prepare_sm;
				else
					next_state  <= reset;
				end if;
			
			when prepare_sm => 
				if (fsm_enable = '1') then
					if (memory_segment(0) = '1') then
						next_state  <= value_true;
					elsif (memory_segment(0) = '0') then
						next_state  <= value_false;
					end if;
				else
					next_state  <= reset;
				end if;

			when value_true => 
				if(fsm_enable = '1') then
					if ( sm_value_index < 15 ) then
						if (sm_value( sm_value_index + 1 ) = '1') then
							next_state  <= value_true;
						elsif (sm_value( sm_value_index + 1 ) = '0') then
							next_state  <= value_false;
						end if;
					else
						next_state  <= update;
					end if;
 				else 
 					next_state  <= reset;
 				end if;

 			when value_false =>

				if(fsm_enable = '1') then
					if ( sm_value_index < 15 ) then
						if (sm_value( sm_value_index + 1 ) = '1') then
							next_state  <= value_true;
						elsif (sm_value( sm_value_index + 1 ) = '0') then
							next_state  <= value_false;
						end if;
					else 
						next_state  <=  update;
					end if;
				else 
					next_state  <= reset;
				end if;

			when update  => 
				if (fsm_enable = '1') then
					--if (tracker_ready = '1') then 
					if ( all_fsm_done = '1') then
						next_state  <= new_start;
					else 
						next_state   <= update;

					end if;
				else
					next_state  <= reset;
				end if;



		end case;
	end process future_state_proc;


--------------------------------------------------------------------------------
-- output network
--------------------------------------------------------------------------------

fsm_output_address  <= fsm_reg_address;

	output_network: process(current_state,index)
	begin
			case current_state is

				when reset => 
					fsm_tracker_address_buffer  <= 0;
					fsm_stripe_done  <= '1';
					sm_or_not_buffer  <= 0;
					fsm_reg_address  <= (others  => '0');
					coor_x  <= fsm_index;
					coor_y  <= -1;
					coor_z  <= -1;
					update_time_control  <= 1;
					sm_value_index  <= 0;

				when new_start => 
					fsm_reg_address  <=  std_logic_vector(unsigned(tracker_address));						
					fsm_stripe_done  <= '0';
					sm_or_not_buffer  <= 0;	-- 0 means it's invalid value
					update_time_control  <= 1;  --this one is used to control the reading address of tracker increment
												--in case we stay in update for multiple cycles, we do the increment for 1 time.
					sm_value_index  <= 0;

					if (coor_y  < SIZE_input_image_horizontal - 1) then
						coor_y  <= coor_y + 1;
					else
						coor_y  <= 0;
						coor_x  <= coor_x + SIZE_kernel_number; 
					end if;

					if (coor_z  >= SIZE_input_image_channel) then 
						coor_z  <= 0;
					end if;

				when prepare_sm  => 
					sm_value  <= memory_segment;
					sm_or_not_buffer  <= 1;  -- 1 means it's sm
					sm_value_index  <= 0;



				when value_true => 

						current_segment_value  <= sm_value(sm_value_index);
						sm_or_not_buffer  <= 2;  -- 2 means it's pixel value
						fsm_reg_address  <=  std_logic_vector(unsigned(fsm_reg_address) + 1);
						coor_z  <= coor_z + 1; 
						if ( sm_value_index  <= 14) then
						sm_value_index  <= sm_value_index + 1;
						end if;


				when value_false => 
						current_segment_value  <= sm_value(sm_value_index);
						sm_or_not_buffer  <= 3;  -- 3 means it's a zero pixel 
						coor_z  <= coor_z + 1;
						if ( sm_value_index  <= 14) then
						sm_value_index  <= sm_value_index + 1;
						end if;


				when update  => 
						if update_time_control = 1 then
							fsm_tracker_address_buffer  <= fsm_tracker_address_buffer + SIZE_kernel_number + 1;
							update_time_control  <= 0;
						end if;
						fsm_stripe_done  <= '1';
						sm_or_not_buffer  <= 0;
						sm_value_index  <= 0;

				end case;

	end process output_network;



end FSM_arc;